/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/AppContext.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/DataBaseRegistry.h>

#include <util_algorithm/GAutoDeleteList.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVConstants.h>
#include <util_ov_annotated_dna/ADVUtils.h>

#include <selection/DNASequenceSelection.h>
#include <gobjects/DNASequenceObject.h>
#include <util_gui/GUIUtils.h>

#include <test_framework/xmltest/XMLTestFormat.h>
#include <test_framework/GTest.h>
#include <test_framework/GTestFrameworkComponents.h>

#include <QtGui/QMenu>
#include <QtCore/QCoreApplication>
#include <QtCore/QDir>
#include <QtCore/QDirIterator>
#include <QtGui/QMessageBox>

#include "RemoteQueryPlugin.h"



namespace GB2 {

extern "C" Q_DECL_EXPORT Plugin * GB2_PLUGIN_INIT_FUNC() {
    RemoteQueryPlugin * plug = new RemoteQueryPlugin();
    return plug;
}

RemoteQueryPlugin::RemoteQueryPlugin():Plugin(tr("Remote query plugin"),tr("Performs remote database queries: BLAST, CDD, etc...")), ctx(NULL) {
    if(AppContext::getMainWindow()) {
        ctx = new RemoteQueryViewContext(this);
        ctx->init();
    }

    DataBaseRegistry *reg = AppContext::getDataBaseRegistry();
    reg->registerDataBase(new BLASTFactory(),"blastn");
    reg->registerDataBase(new BLASTFactory(),"blastp");
    reg->registerDataBase(new CDDFactory(),"cdd");

    LocalWorkflow::RemoteQueryWorkerFactory::init();

    GTestFormatRegistry *tfr = AppContext::getTestFramework()->getTestFormatRegistry();
    XMLTestFormat *xmlTestFormat = qobject_cast<XMLTestFormat*>(tfr->findFormat("XML"));
    assert(xmlTestFormat!=NULL);

    GAutoDeleteList<XMLTestFactory> *l = new GAutoDeleteList<XMLTestFactory>(this);
    l->qlist = RemoteQueryPluginTests::createTestFactories();

    foreach(XMLTestFactory *f,l->qlist) {
        bool res = xmlTestFormat->registerTestFactory(f);
        Q_UNUSED(res);
        assert(res);
    }
}

RemoteQueryViewContext::RemoteQueryViewContext(QObject *p):GObjectViewWindowContext(p,ANNOTATED_DNA_VIEW_FACTORY_ID)
{
}

void RemoteQueryViewContext::initViewContext(GObjectView * view) {
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
    ADVGlobalAction * a = new ADVGlobalAction(av, QIcon(":/remote_query/images/remote_db_request.png"), tr("Query remote database"), 60);
    connect( a, SIGNAL(triggered()), SLOT(sl_showDialog()) );
}

void RemoteQueryViewContext::sl_showDialog() {
    QAction* a = (QAction*)sender();
    GObjectViewAction* viewAction = qobject_cast<GObjectViewAction*>(a);
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(viewAction->getObjectView());
    assert(av);

    ADVSequenceObjectContext* seqCtx = av->getSequenceInFocus();

    bool isAminoSeq = seqCtx->getAlphabet()->isAmino();
    SendSelectionDialog dlg( seqCtx->getSequenceObject(), isAminoSeq, av->getWidget() );
    if( QDialog::Accepted == dlg.exec() ) {

        //prepare query
        const QByteArray& sequence = seqCtx->getSequenceData();
        DNASequenceSelection* s = seqCtx->getSequenceSelection();
        QList<LRegion> regions;
        if(s->isEmpty()) {
            int seqLen = seqCtx->getSequenceLen();
            regions.append(LRegion(0, seqLen));
        }
        else {
            regions =  s->getSelectedRegions();
        }
        foreach(const LRegion& r, regions) {
        QByteArray query( sequence.constData() + r.startPos, r.len );

        DNATranslation * aminoT = (dlg.translateToAmino ? seqCtx->getAminoTT() : 0);
        DNATranslation * complT = (dlg.translateToAmino ? seqCtx->getComplementTT() : 0);

        Task * t = new RemoteQueryToAnnotationsTask( dlg.db, aminoT, complT, query, r.startPos, 
            dlg.getAnnotationObject(), dlg.getUrl(),dlg.getGroupName(), dlg.requestParameters, dlg.retries );
        AppContext::getTaskScheduler()->registerTopLevelTask( t );
        }
    }
}

QList<XMLTestFactory*> RemoteQueryPluginTests::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_RemoteQuery::createFactory());
    return res;
}

}
