/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CoreLib.h"
#include "BaseDocWorker.h"
#include "DocWorkers.h"
#include "DocActors.h"
#include "GenericReadActor.h"
#include "FindWorker.h"
#include "SWWorker.h"
#include "SequenceSplitWorker.h"

#include <core_api/DocumentFormats.h>
#include <core_api/DocumentModel.h>
#include <core_api/AppContext.h>
#include <gobjects/GObjectTypes.h>

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowManager.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow/WorkflowEnv.h>

#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/LocalDomain.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>

#include <core_api/core_api.h>
#include <util_gui/DialogUtils.h>
#include <util_gui/GUIUtils.h>

/* TRANSLATOR GB2::Workflow::CoreLib */

namespace GB2 {
using namespace LocalWorkflow;
namespace Workflow {

void CoreLib::init() {

    Descriptor apnd(CoreLibConstants::APPEND_ATTR_ID, tr("Accumulate objects"), 
        tr("Accumulate all incoming data in one file or create separate files for each input."
        "In the latter case, an incremental numerical suffix is added to the file name."));
    Descriptor writeUrlD(CoreLibConstants::URL_SLOT_ID, tr("Location"), tr("Location for writing data"));
    Descriptor fastaHd(CoreLibConstants::FASTA_HEADER_SLOT_ID, tr("FASTA header"), tr("A header line for the FASTA record."));
    Descriptor dfd(CoreLibConstants::DOCUMENT_FORMAT_ATTR_ID, tr("Document format"), tr("Document format of output file"));
    //Descriptor gbAccd(GENBANK_ACN_SLOT_ID, tr("Genbank Accession Number"), tr("Accession Number within NCBI GenBank nucleotide sequence database"));

    DataTypeRegistry* dr = WorkflowEnv::getDataTypeRegistry();
    assert(dr);

//     DataTypePtr readMAType;
//     {
//         QMap<Descriptor, DataTypePtr> m;
//         m[CoreLib::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
//         m[BioActorLibrary::MA_SLOT()] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
//         readMAType = new MapDataType(Descriptor(CoreLib::MA_TYPESET_ID), m);
//         dr->registerEntry(readMAType);
//     }
    DataTypePtr writeMAType;
    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[BioActorLibrary::MA_SLOT()] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
        writeMAType = new MapDataType(Descriptor(CoreLibConstants::MA_TYPESET_ID), m);
        //dr->registerEntry(writeMAType);
    }

    WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
    assert(r);
    r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), new GenericMAActorProto());
    r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), new GenericSeqActorProto());

    // WRITE FASTA actor proto
    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        m[fastaHd] = CoreDataTypes::STRING_TYPE();
        DataTypePtr fastaTypeSet(new MapDataType(Descriptor(CoreLibConstants::FASTA_TYPESET_ID), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLibConstants::WRITE_FASTA_ACTOR, tr("Write FASTA"), tr("Writes all supplied sequences to file(s) in FASTA format."));
        Descriptor pd(CoreLibConstants::DATA_PORT_ID, tr("Sequence"), tr("A sequence along with FASTA header line."));
        p << new PortDescriptor(pd, fastaTypeSet, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new WriteDocActorProto(BaseDocumentFormats::PLAIN_FASTA, acd, p, a);
        proto->setPrompter(new WriteFastaPrompter());
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }
    // WRITE FASTQ actor proto
    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        DataTypePtr fastqTypeSet(new MapDataType(Descriptor(CoreLibConstants::FASTQ_TYPESET_ID), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLibConstants::WRITE_FASTQ_ACTOR, tr("Write FASTQ"), tr("Writes all supplied sequences to file(s) in FASTQ format."));
        Descriptor pd(CoreLibConstants::DATA_PORT_ID, tr("Sequence"), tr("A sequence in FASTQ format along with PHRED quality scores."));
        p << new PortDescriptor(pd, fastqTypeSet, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new WriteDocActorProto(BaseDocumentFormats::FASTQ, acd, p, a);
        proto->setPrompter(new WriteFastaPrompter());
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }
    // WRITE GENBANK actor proto
    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        m[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_LIST_TYPE();
        //m[gbAccd] = CoreDataTypes::STRING_TYPE();
        DataTypePtr genbankTypeSet(new MapDataType(Descriptor(CoreLibConstants::GENBANK_TYPESET_ID), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLibConstants::WRITE_GENBANK_ACTOR, tr("Write Genbank"), tr("Writes all supplied sequences and related annotations to file(s) in Genbank format."));
        Descriptor pd(CoreLibConstants::DATA_PORT_ID, tr("Input data"), tr("Sequence and set of annotations"));
        p << new PortDescriptor(pd, genbankTypeSet, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new WriteDocActorProto(BaseDocumentFormats::PLAIN_GENBANK, acd, p, a);
        proto->setPrompter(new WriteGenbankPrompter());
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }
    // READ PLAIN TEXT actor proto
    Descriptor txtd(CoreLibConstants::DATA_PORT_ID, tr("Plain Text"), tr("Just a plain text w/o splitting to strings."));
    {
        QMap<Descriptor, DataTypePtr> m;
        m[CoreLibConstants::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
        m[txtd] = CoreDataTypes::STRING_TYPE();
        DataTypePtr dtl(new MapDataType(Descriptor(CoreLibConstants::TEXT_TYPESET_ID), m));
        dr->registerEntry(dtl);

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLibConstants::READ_TEXT_ACTOR, tr("Read plain text"), tr("Plain text file reader"));
        p << new PortDescriptor(txtd, dtl, false, true);
        BusActorPrototype* proto = new ReadDocActorProto(BaseDocumentFormats::PLAIN_TEXT, acd, p, a);
        proto->setPrompter(new ReadDocPrompter(tr("Read the whole text from <u>%1</u> and output it as a single string.")));
        proto->setIcon( GUIUtils::createRoundIcon(QColor(Qt::darkRed), 22) );
        r->registerProto(BioActorLibrary::CATEGORY_OTHER(), proto);
    }
    // WRITE PLAIN TEXT actor proto
    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[txtd] = new ListDataType("string-list", CoreDataTypes::STRING_TYPE());
        DataTypePtr dtl(new MapDataType(Descriptor("in.text"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLibConstants::WRITE_TEXT_ACTOR, tr("Write plain text"), tr("Write strings to a file"));
        p << new PortDescriptor(txtd, dtl, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new WriteDocActorProto(BaseDocumentFormats::PLAIN_TEXT, acd, p, a);
        proto->setPrompter(new WriteDocPrompter(tr("Save text from <u>%1</u> to document <u>%2</u>."), CoreLibConstants::DATA_PORT_ID));
        proto->setIcon( GUIUtils::createRoundIcon(QColor(Qt::darkRed), 22) );
        r->registerProto(BioActorLibrary::CATEGORY_OTHER(), proto);
    }
    // WRITE CLUSTAL actor proto
    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLibConstants::WRITE_CLUSTAL_ACTOR, tr("Write ClustalW"), tr("Writes all supplied alignments to file(s) in CLUSTALW format."));
        Descriptor pd(CoreLibConstants::DATA_PORT_ID, tr("Multiple sequence alignment"), tr("Multiple sequence alignment"));
        p << new PortDescriptor(pd, writeMAType, true);
        BusActorPrototype* proto = new WriteDocActorProto(BaseDocumentFormats::CLUSTAL_ALN, acd, p, a);
        proto->setPrompter(new WriteDocPrompter(tr("Save all MSAs from <u>%1</u> to document <u>%2</u>."), BioActorLibrary::MA_SLOT_ID));
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }
    // WRITE STOCKHOLM actor proto
    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLibConstants::WRITE_STOCKHOLM_ACTOR, tr("Write Stockholm"), tr("Writes all supplied alignments to file(s) in Stockholm format."));
        Descriptor pd(CoreLibConstants::DATA_PORT_ID, tr("Multiple sequence alignment"), tr("Multiple sequence alignment"));
        p << new PortDescriptor(pd, writeMAType, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new WriteDocActorProto(BaseDocumentFormats::STOCKHOLM, acd, p, a);
        proto->setPrompter(new WriteDocPrompter(tr("Save all MSAs from <u>%1</u> to document <u>%2</u>."), BioActorLibrary::MA_SLOT_ID));
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }
    // GENERIC WRITE MSA actor proto
    {
        DocumentFormatConstraints constr;
        constr.supportedObjectTypes.insert( GObjectTypes::MULTIPLE_ALIGNMENT );
        constr.addFlagToSupport(DocumentFormatFlag_SupportWriting);
        QList<DocumentFormatId> supportedFormats = AppContext::getDocumentFormatRegistry()->selectFormats( constr );

        if( !supportedFormats.isEmpty() ) {
            QList<PortDescriptor*> p; QList<Attribute*> a;
            Descriptor acd(CoreLibConstants::WRITE_MSA_ACTOR, tr("Write alignment"), tr("Writes all supplied alignments to file(s) in selected format"));
            Descriptor pd(CoreLibConstants::DATA_PORT_ID, tr("Multiple sequence alignment"), tr("Multiple sequence alignment"));
            p << new PortDescriptor(pd, writeMAType, true);
            a << new Attribute(dfd, CoreDataTypes::STRING_TYPE(), false, 
                supportedFormats.contains( BaseDocumentFormats::CLUSTAL_ALN ) ? BaseDocumentFormats::CLUSTAL_ALN : supportedFormats.first() );
            BusActorPrototype * proto = new WriteDocActorProto( acd, GObjectTypes::MULTIPLE_ALIGNMENT, p, a );
            
            QVariantMap m;
            foreach( const DocumentFormatId & fid, supportedFormats ) {
                m[fid] = fid;
            }
            proto->getEditor()->addDelegate(new ComboBoxDelegate(m), CoreLibConstants::DOCUMENT_FORMAT_ATTR_ID);
            proto->setPrompter(new WriteDocPrompter(tr("Save all MSAs from <u>%1</u> to document <u>%2</u>."), BioActorLibrary::MA_SLOT_ID));
            r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
        }
    }
    
    // GENERIC WRITE SEQ actor proto
    {
        DocumentFormatConstraints constr;
        constr.supportedObjectTypes.insert( GObjectTypes::SEQUENCE );
        constr.addFlagToSupport(DocumentFormatFlag_SupportWriting);
        QList<DocumentFormatId> supportedFormats = AppContext::getDocumentFormatRegistry()->selectFormats( constr );
        
        if( !supportedFormats.isEmpty() ) {
            QMap<Descriptor, DataTypePtr> typeMap;
            typeMap[writeUrlD] = CoreDataTypes::STRING_TYPE();
            typeMap[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
            DataTypePtr typeSet( new MapDataType(Descriptor(CoreLibConstants::SEQ_TYPESET_ID), typeMap));
            
            QList<PortDescriptor*> p; QList<Attribute*> a;
            Descriptor acd(CoreLibConstants::WRITE_SEQ_ACTOR, tr("Write sequence"), tr("Writes all supplied alignments to file(s) in selected format"));
            Descriptor pd(CoreLibConstants::DATA_PORT_ID, tr("Sequence"), tr("Sequence"));
            p << new PortDescriptor(pd, typeSet, true);
            a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
            a << new Attribute(dfd, CoreDataTypes::STRING_TYPE(), false, 
                supportedFormats.contains( BaseDocumentFormats::PLAIN_FASTA ) ? BaseDocumentFormats::PLAIN_FASTA : supportedFormats.first() );
            BusActorPrototype * proto = new WriteDocActorProto( acd, GObjectTypes::SEQUENCE, p, a );
            
            QVariantMap m;
            foreach( const DocumentFormatId & fid, supportedFormats ) {
                m[fid] = fid;
            }
            proto->getEditor()->addDelegate(new ComboBoxDelegate(m), CoreLibConstants::DOCUMENT_FORMAT_ATTR_ID);
            proto->setPrompter(new WriteDocPrompter(tr("Save all MSAs from <u>%1</u> to document <u>%2</u>."), BioActorLibrary::MA_SLOT_ID));
            r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
        }
    }
    DataWorkerFactory::init();
    FindWorkerFactory::init();
    SWWorkerFactory::init();
    SequenceSplitWorkerFactory::init();

}

}//Workflow namespace
}//GB2 namespace
