/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DocActors.h"
#include "CoreLib.h"

#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <core_api/DocumentModel.h>
#include <util_tasks/SaveDocumentTask.h>
#include <util_gui/DialogUtils.h>

namespace GB2 {
namespace Workflow {

/****************************
 * DocActorProto
 *****************************/
DocActorProto::DocActorProto(const DocumentFormatId& _fid, const Descriptor& _desc, const QList<PortDescriptor*>& _ports, 
                             const QList<Attribute*>& _attrs ) : BusActorPrototype(_desc, _ports, _attrs), fid(_fid) {
}

DocActorProto::DocActorProto(const Descriptor& _desc, const GObjectType& t, const QList<PortDescriptor*>& _ports,
                             const QList<Attribute*>& _attrs ) : BusActorPrototype(_desc, _ports, _attrs), type(t) {
}

bool DocActorProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params, const QString & urlAttrId ) const {
    QList<DocumentFormat*> fs;
    QString url = DesignerUtils::getDropUrl(fs, md);
    foreach(DocumentFormat* df, fs) {
        if (fid == df->getFormatId()) {
            if (params) {
                params->insert( urlAttrId, url );
            }
            return true;
        }
    }
    return false;
}

QString DocActorProto::prepareDocumentFilter() {
    if( !fid.isEmpty() ) {
        return DialogUtils::prepareDocumentsFileFilter( fid, true );
    } else {
        assert( !type.isEmpty() );
        return DialogUtils::prepareDocumentsFileFilterByObjType( type, true );
    }
}

/****************************
 * ReadDocActorProto
 *****************************/
ReadDocActorProto::ReadDocActorProto(const DocumentFormatId& _fid, const Descriptor& _desc, const QList<PortDescriptor*>& _ports, 
                                     const QList<Attribute*>& _attrs ) : DocActorProto( _fid, _desc, _ports, _attrs ) {
    attrs << new Attribute( CoreLibConstants::URL_IN_ATTR(), CoreDataTypes::STRING_TYPE(), true );
    QMap< QString, PropertyDelegate* > delegateMap;
    delegateMap[CoreLibConstants::URL_IN_ATTR_ID] = new URLDelegate( prepareDocumentFilter(), QString(), true );
    setEditor( new DelegateEditor( delegateMap ) );
}

bool ReadDocActorProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return DocActorProto::isAcceptableDrop( md, params, CoreLibConstants::URL_IN_ATTR_ID );
}

/****************************
 * WriteDocActorProto
 *****************************/
WriteDocActorProto::WriteDocActorProto(const DocumentFormatId& _fid, const Descriptor& _desc, const QList<PortDescriptor*>& _ports, 
                                       const QList<Attribute*>& _attrs ) : DocActorProto( _fid, _desc, _ports, _attrs ) {
    construct();
}

WriteDocActorProto::WriteDocActorProto(const Descriptor& _desc, const GObjectType & t, const QList<PortDescriptor*>& _ports, 
                                       const QList<Attribute*>& _attrs ) : DocActorProto(_desc, t, _ports, _attrs) {
    construct();
}

bool WriteDocActorProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return DocActorProto::isAcceptableDrop( md, params, CoreLibConstants::URL_OUT_ATTR_ID );
}

void WriteDocActorProto::construct() {
    attrs << new Attribute(CoreLibConstants::URL_OUT_ATTR(), CoreDataTypes::STRING_TYPE(), false );
    attrs << new Attribute(BioActorLibrary::FILE_MODE_ATTR(), CoreDataTypes::NUM_TYPE(), false, SaveDoc_Roll);

    QMap< QString, PropertyDelegate* > delegateMap;
    delegateMap[CoreLibConstants::URL_OUT_ATTR_ID] = new URLDelegate(prepareDocumentFilter(), QString(), false );
    delegateMap[BioActorLibrary::FILE_MODE_ATTR_ID] = new FileModeDelegate(attrs.size() > 2);

    setEditor(new DelegateEditor(delegateMap));
    setValidator(new ScreenedParamValidator(CoreLibConstants::URL_OUT_ATTR_ID, ports.first()->getId(), CoreLibConstants::URL_SLOT_ID));
    setPortValidator(CoreLibConstants::DATA_PORT_ID, new ScreenedSlotValidator(CoreLibConstants::URL_SLOT_ID));
}

/****************************
 * WriteGenbankPrompter
 *****************************/
QString WriteGenbankPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::DATA_PORT_ID));

    Actor* seqProducer = input->getProducer(BioActorLibrary::SEQ_SLOT_ID);
    QString seqName = seqProducer ? tr(" sequence from <u>%1</u>").arg(seqProducer->getLabel()) : "";
    QString annName = getProducers(CoreLibConstants::DATA_PORT_ID, BioActorLibrary::FEATURE_TABLE_SLOT_ID);
    if (!annName.isEmpty()) {
        annName = tr(" set of annotations from <u>%1</u>").arg(annName);
    }

    QString url = getScreenedURL( qobject_cast<BusPort*>(target->getPort( CoreLibConstants::DATA_PORT_ID )), 
        CoreLibConstants::URL_OUT_ATTR_ID, CoreLibConstants::URL_SLOT_ID );

    QString data;
    if (seqName.isEmpty() && annName.isEmpty()) {
        return url;
    } else if (!seqName.isEmpty() && !annName.isEmpty()) {
        data = tr("For each %1 and %2").arg(seqName).arg(annName);
    } else {
        data = tr("For each ") + seqName + annName;
    }

    QString extra;
    //     Actor* acnProducer = input->getProducer(CoreLib::GENBANK_ACN_SLOT.getId());
    //     if (!acnProducer) {
    //         extra = tr("<br>Autogenerate ACN.");
    //     }

    QString doc = tr("%1, add Genbank entry to document <u>%2</u>.%3")
        .arg(data) //sequence from Read Fasta 1
        .arg(url) 
        .arg(extra);

    return doc;
}

/****************************
 * WriteFastaPrompter
 *****************************/
QString WriteFastaPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::DATA_PORT_ID));
    QString url = getScreenedURL( qobject_cast<BusPort*>(target->getPort( CoreLibConstants::DATA_PORT_ID )), 
        CoreLibConstants::URL_OUT_ATTR_ID, CoreLibConstants::URL_SLOT_ID );

    Actor* seqProducer = input->getProducer(BioActorLibrary::SEQ_SLOT_ID);
    if (!seqProducer) {
        return url;
    }

    QString doc = tr("For each sequence from <u>%1</u>, add entry to document <u>%2</u>.")
        .arg(seqProducer->getLabel())
        .arg(url);

    return doc;
}

/****************************
* WriteDocPrompter
*****************************/
QString WriteDocPrompter::composeRichDoc() {
    QString url = getScreenedURL( qobject_cast<BusPort*>(target->getPort( CoreLibConstants::DATA_PORT_ID )), 
        CoreLibConstants::URL_OUT_ATTR_ID, CoreLibConstants::URL_SLOT_ID );
    
    QString producers = getProducers(CoreLibConstants::DATA_PORT_ID, slot);
    if (producers.isEmpty()) {
        return url;
    }
    return spec.arg(producers).arg(url);
}

/****************************
* ReadDocPrompter
*****************************/
QString ReadDocPrompter::composeRichDoc() {
    return spec.arg(getURL(CoreLibConstants::URL_IN_ATTR_ID));
}

}//namespace Workflow
}//namespace GB2
