/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DocWorkers.h"
#include "CoreLib.h"
#include "GenericReadWorker.h"

#include <workflow/WorkflowEnv.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/QVariantUtils.h>
#include <workflow_support/CoreLibConstants.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <core_api/Log.h>
#include <core_api/DocumentModel.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/MAlignmentObject.h>
#include <gobjects/TextObject.h>
#include <gobjects/GObjectRelationRoles.h>
#include <datatype/DNASequence.h>
#include <datatype/AnnotationData.h>

#include <document_format/EMBLGenbankAbstractDocument.h>

namespace GB2 {
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);
static int ct = 0;

/*************************************
 * TextReader
 *************************************/
void TextReader::doc2data(Document* doc) {
    log.info(tr("Reading text from %1").arg(doc->getURLString()));
    foreach(GObject* go, GObjectUtils::select(doc->getObjects(), GObjectTypes::TEXT, UOF_LoadedOnly)) {
        TextObject* txtObject = qobject_cast<TextObject*>(go);
        assert(txtObject);
        QVariantMap m; 
        m[CoreLibConstants::DATA_PORT_ID] = txtObject->getText();
        m[CoreLibConstants::URL_SLOT_ID] = doc->getURLString();
        cache << Message(mtype, m);
    }
}

/*************************************
* TextWriter
*************************************/
void TextWriter::data2doc(Document* doc, const QVariantMap& data) {
    QStringList list = data.value(CoreLibConstants::DATA_PORT_ID).toStringList();
    QString text = list.join("\n");
    TextObject* to = qobject_cast<TextObject*>(GObjectUtils::selectOne(doc->getObjects(), GObjectTypes::TEXT, UOF_LoadedOnly));
    if (!to) {
        to = new TextObject(text, QString("Text %1").arg(++ct));
        doc->addObject(to);
    } else {
        to->setText(to->getText() + "\n" + text);
    }
}

/*************************************
* FastaWriter
*************************************/
void FastaWriter::data2doc(Document* doc, const QVariantMap& data) {
    data2document(doc, data);
}

void FastaWriter::data2document(Document* doc, const QVariantMap& data) {
    DNASequence seq = qVariantValue<DNASequence>(data.value(BioActorLibrary::SEQ_SLOT_ID));
    QString sequenceName = data.value(CoreLibConstants::FASTA_HEADER_SLOT_ID).toString();
    if (sequenceName.isEmpty()) {
        sequenceName = seq.getName();
    } else {
        seq.info.insert(DNAInfo::FASTA_HDR, sequenceName);
    }
    if (sequenceName.isEmpty()) {
        sequenceName = QString("unknown sequence %1").arg(doc->getObjects().size());
    }
    if (seq.alphabet && seq.length() != 0 && !doc->findGObjectByName(sequenceName)) {
        log.trace(QString("Adding seq [%1] to FASTA doc %2").arg(sequenceName).arg(doc->getURLString()));
        doc->addObject(new DNASequenceObject(sequenceName, seq));
    }
}

/*************************************
* FastQWriter
*************************************/
void FastQWriter::data2doc(Document* doc, const QVariantMap& data) {
    data2document(doc, data);
}

void FastQWriter::data2document(Document* doc, const QVariantMap& data) {
    DNASequence seq = qVariantValue<DNASequence>(data.value(BioActorLibrary::SEQ_SLOT_ID));
    QString sequenceName = seq.getName(); 
    if (sequenceName.isEmpty()) {
        sequenceName = QString("unknown sequence %1").arg(doc->getObjects().size());
    }
    if (seq.alphabet && seq.length() != 0 && !doc->findGObjectByName(sequenceName)) {
        log.trace(QString("Adding seq [%1] to FASTQ doc %2").arg(sequenceName).arg(doc->getURLString()));
        doc->addObject(new DNASequenceObject(sequenceName, seq));
    }
}

/*************************************
* GenbankWriter
*************************************/
void GenbankWriter::data2doc(Document* doc, const QVariantMap& data) {
    data2document(doc, data);
}

void GenbankWriter::data2document(Document* doc, const QVariantMap& data) {
    DNASequence seq = qVariantValue<DNASequence>(data.value(BioActorLibrary::SEQ_SLOT_ID));

	/*QStringList order;
	order << DNAInfo::DEFINITION << DNAInfo::ACCESSION << DNAInfo::VERSION;
	order << DNAInfo::PROJECT << DNAInfo::KEYWORDS << DNAInfo::SEGMENT;
	foreach(const QString& key, order) {
		if (seq.info.contains(key)) {
			QVariant v = seq.info.take(key);
			if (!(v.canConvert(QVariant::String)||v.canConvert(QVariant::StringList))) {
				seq.info.remove(key);
			}
		}
	}*/

	QMapIterator<QString, QVariant> it(seq.info);
	while (it.hasNext())
	{
		it.next();
		if ( !(it.value().type() == QVariant::String || it.value().type() == QVariant::StringList) ) {
			seq.info.remove(it.key());
		}
	}


    QString sequenceName = seq.getName(); //data.value(CoreLib::GENBANK_ACN_SLOT_ID).toString();
    QString annotationName;
    if (sequenceName.isEmpty()) {
        int num = doc->findGObjectByType(GObjectTypes::SEQUENCE).size();
        sequenceName = QString("unknown sequence %1").arg(num);
        annotationName = QString("unknown features %1").arg(num);
    } else {
        annotationName = sequenceName + " features";
    }

    QList<SharedAnnotationData> atl = QVariantUtils::var2ftl(data.value(BioActorLibrary::FEATURE_TABLE_SLOT_ID).toList());
    DNASequenceObject* dna = qobject_cast<DNASequenceObject*>(doc->findGObjectByName(sequenceName));
    if (!dna && !seq.isNull()) {
        doc->addObject(dna = new DNASequenceObject(sequenceName, seq));
        log.trace(QString("Adding seq [%1] to GB doc %2").arg(sequenceName).arg(doc->getURLString()));
    }
    if (!atl.isEmpty()) {
        AnnotationTableObject* att = NULL;
        if (dna) {
            QList<GObject*> relAnns = GObjectUtils::findObjectsRelatedToObjectByRole(dna, GObjectTypes::ANNOTATION_TABLE, GObjectRelationRole::SEQUENCE, doc->getObjects(), UOF_LoadedOnly);
            att = relAnns.isEmpty() ? NULL : qobject_cast<AnnotationTableObject*>(relAnns.first());
        }
        if (!att) {
            doc->addObject(att = new AnnotationTableObject(annotationName));
            if (dna) {
                att->addObjectRelation(dna, GObjectRelationRole::SEQUENCE);
            }
            log.trace(QString("Adding features [%1] to GB doc %2").arg(annotationName).arg(doc->getURLString()));
        }
        foreach(SharedAnnotationData sad, atl) {
            att->addAnnotation(new Annotation(sad), QString());
        }
    }
}

/*************************************
* SeqWriter
*************************************/
void SeqWriter::data2doc(Document* doc, const QVariantMap& data){
    if( format == NULL ) {
        return;
    }
    DocumentFormatId fid = format->getFormatId();
    if( fid == BaseDocumentFormats::PLAIN_FASTA ) {
        FastaWriter::data2document( doc, data );
    }
    else if( fid == BaseDocumentFormats::PLAIN_GENBANK ) {
        GenbankWriter::data2document( doc, data );
    } else if ( fid == BaseDocumentFormats::FASTQ) {
        FastQWriter::data2document( doc, data );    
    }else {
        assert(0);
        log.error(QString("Unknown data format for writing: %1").arg(fid));
    }
    
}

/*************************************
* MSAWriter
*************************************/
void MSAWriter::data2doc(Document* doc, const QVariantMap& data) {
    MAlignment ma = data.value(BioActorLibrary::MA_SLOT_ID).value<MAlignment>();
    if (ma.isEmpty()) {
        log.error(tr("Empty alignment passed for writing to %1").arg(doc->getURLString()));
        return; //FIXME
    }
    if (ma.getName().isEmpty()) {
        ma.setName( QString(MA_OBJECT_NAME + "_%1").arg(ct++) );
    }
    doc->addObject(new MAlignmentObject(ma));
}

/*************************************
* DataWorkerFactory
*************************************/
Worker* DataWorkerFactory::createWorker(Actor* a) {
    // TODO: wtf is this??
    //  each actor must have own factory 

    BaseWorker* w = NULL;
    QString protoId = a->getProto()->getId();
    if (CoreLibConstants::READ_TEXT_ACTOR == protoId ) {
        TextReader* t = new TextReader(a);
        w = t;
    } 
    else if (CoreLibConstants::WRITE_TEXT_ACTOR == protoId) {
        w = new TextWriter(a);
    } 
    else if (CoreLibConstants::WRITE_FASTA_ACTOR == protoId) {
        w = new FastaWriter(a);
    }
    else if (CoreLibConstants::WRITE_GENBANK_ACTOR == protoId) {
        w = new GenbankWriter(a);
    }
    else if (CoreLibConstants::WRITE_CLUSTAL_ACTOR == protoId) {
        w = new MSAWriter(a, BaseDocumentFormats::CLUSTAL_ALN);
    }
    else if (CoreLibConstants::WRITE_STOCKHOLM_ACTOR == protoId) {
        w = new MSAWriter(a, BaseDocumentFormats::STOCKHOLM);
    }
    else if (CoreLibConstants::GENERIC_READ_MA_ACTOR == protoId) {
        w = new GenericMSAReader(a);
    }
    else if (CoreLibConstants::GENERIC_READ_SEQ_ACTOR == protoId) {
        w = new GenericSeqReader(a);
    } 
    else if(CoreLibConstants::WRITE_MSA_ACTOR == protoId ) {
        w = new MSAWriter(a);
    }
    else if(CoreLibConstants::WRITE_SEQ_ACTOR == protoId ) {
        w = new SeqWriter(a);
    } 
    else if (CoreLibConstants::WRITE_FASTQ_ACTOR == protoId ) {
        w = new FastQWriter(a);
    } else {
        assert(0);
    }
    return w;    
}

void DataWorkerFactory::init() {
    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_FASTA_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_GENBANK_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::READ_TEXT_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_TEXT_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::GENERIC_READ_SEQ_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::GENERIC_READ_MA_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_CLUSTAL_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_STOCKHOLM_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_MSA_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_SEQ_ACTOR));
    localDomain->registerEntry(new DataWorkerFactory(CoreLibConstants::WRITE_FASTQ_ACTOR));
}

} // Workflow namespace
} // GB2 namespace
