/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "BowtieConstants.h"
#include "BowtieTask.h"
#include "BowtieAdapter.h"
#include "BowtieContext.h"

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>
#include <core_api/AppResources.h>
#include <core_api/Log.h>
#include <core_api/StateLockableDataModel.h>
#include <core_api/DocumentModel.h>
#include <core_api/DocumentFormats.h>
#include <core_api/IOAdapter.h>
#include <core_api/Counter.h>
#include <core_api/ProjectModel.h>
#include <gobjects/DNASequenceObject.h>
#include <util_tasks/LoadDocumentTask.h>
#include <document_format/DocumentFormatUtils.h>

#include <QtCore/QMutexLocker>

namespace GB2 {

static LogCategory log(ULOG_CAT_BOWTIE);

QMutex BowtieBuildTask::mutex;

const QString BowtieTask::taskName(tr("Bowtie"));
const QString BowtieTask::OPTION_PREBUILT_INDEX = "prebuilt";
const QString BowtieTask::OPTION_N_MISMATCHES = "n-mismatches";
const QString BowtieTask::OPTION_V_MISMATCHES = "v-mismatches";
const QString BowtieTask::OPTION_MAQERR = "maqerr";
const QString BowtieTask::OPTION_SEED_LEN = "seedLen";
const QString BowtieTask::OPTION_NOFW = "nofw";
const QString BowtieTask::OPTION_NORC = "norc";
const QString BowtieTask::OPTION_MAXBTS = "maxbts";
const QString BowtieTask::OPTION_TRYHARD = "tryhard";
const QString BowtieTask::OPTION_CHUNKMBS = "chunkmbs";
const QString BowtieTask::OPTION_NOMAQROUND = "nomaqround";
const QString BowtieTask::OPTION_SEED = "seed";
const QString BowtieTask::OPTION_SORT_ALIGNMENT_BY_OFFSET = "sort";

BowtieTask::BowtieTask(const DnaAssemblyToRefTaskSettings & _config) : DnaAssemblyToReferenceTask(_config, TaskFlags_NR_FOSCOE)
{
    GCOUNTER( cvar, tvar, "BowtieTask" );
	tlsTask = NULL;
	buildTask = NULL;
    setMaxParallelSubtasks(1);
    fileSize = 0;
}


void BowtieTask::prepare()
{
	QString indexURL(settings.refSeqUrl.getURLString());

	if(settings.getCustomValue(BowtieTask::OPTION_PREBUILT_INDEX, false).toBool()) {
		QRegExp rx("(.+)(\\.rev)?\\.\\d\\.ebwt");
		if(rx.indexIn(indexURL) != -1) {
			indexPath = rx.cap(1);
		} else {
			indexPath = indexURL;
		}
		QString index1file = indexPath + ".1.ebwt";
		QFileInfo file(index1file);
		if(!file.exists()) {
			stateInfo.setError(BowtieBuildTask::tr("Reference index file \"%1\" not exists").arg(index1file));
			return;
		}
		fileSize = file.size();
	} else {
		//Build index
		QFileInfo file(indexURL);
		if(!file.exists()) {
			stateInfo.setError(BowtieBuildTask::tr("Reference sequence file \"%1\" not exists").arg(indexURL));
			return;
		}
		fileSize = file.size();
		buildTask = new BowtieBuildTask(indexURL, settings.resultFileName.dirPath() + "/" + settings.resultFileName.baseFileName()); 
		buildTask->setSubtaskProgressWeight(0.6);
		addSubTask(buildTask);
	}

	int shortReadAvgLen = settings.shortReads.first().length();
	int shortReadsCount = settings.shortReads.count();
	int step = shortReadsCount / 10;
	for(int i=step; i < step*10; i+=step) {
		shortReadAvgLen += settings.shortReads.at(i).length();
	}
	shortReadAvgLen /= 10;

	qint64 memUseMB = (fileSize *  4 + (shortReadAvgLen+200)*shortReadsCount) / 1024 / 1024 + 100;
	TaskResourceUsage memUsg(RESOURCE_MEMORY, memUseMB, true);
	taskResources.append(memUsg);

	tlsTask = new BowtieTLSTask();
	tlsTask->setSubtaskProgressWeight(0.4);
	addSubTask(tlsTask);	
}

QList<Task*> BowtieTask::onSubTaskFinished(Task* subTask) {
	Q_UNUSED(subTask);
	QList<Task*> res;
	if(subTask->hasErrors()) {
		return res;
	}
	if(subTask == buildTask)
		indexPath = static_cast<BowtieBuildTask*>(subTask)->getEbwtPath();
	return res;
}

Task::ReportResult BowtieTask::report() {
    return ReportResult_Finished;
}

/************************************************************************/
/* BowtieTLSTask                                                        */
/************************************************************************/

BowtieTLSTask::BowtieTLSTask() : TLSTask("Bowtie TLS Task", TaskFlags_RBSF_FOSCOE)
{
	tpm = Task::Progress_Manual;
	nThreads = AppContext::getAppSettings()->getAppResourcePool()->getIdealThreadCount();
}

void BowtieTLSTask::_run()
{
	BowtieTask* parent = static_cast<BowtieTask*>(getParentTask());
	BowtieContext* ctx = static_cast<BowtieContext*>(taskContext);
	//settings
	ctx->search.resetOptions();

	ctx->search.seedMms = parent->settings.getCustomValue(BowtieTask::OPTION_N_MISMATCHES, 2).toInt();
	int v_mismatches = parent->settings.getCustomValue(BowtieTask::OPTION_V_MISMATCHES, -1).toInt();
	if(v_mismatches != -1) {
		ctx->search.maqLike = 0;
		ctx->search.mismatches = v_mismatches;
	}
	ctx->search.qualThresh = parent->settings.getCustomValue(BowtieTask::OPTION_MAQERR, 70).toInt();
	ctx->search.seedLen = parent->settings.getCustomValue(BowtieTask::OPTION_SEED_LEN, 28).toInt();
	ctx->search.noMaqRound = parent->settings.getCustomValue(BowtieTask::OPTION_NOMAQROUND, false).toBool();
	ctx->search.nofw = parent->settings.getCustomValue(BowtieTask::OPTION_NOFW, false).toBool();
	ctx->search.norc = parent->settings.getCustomValue(BowtieTask::OPTION_NORC, false).toBool();
	int maxbts = parent->settings.getCustomValue(BowtieTask::OPTION_MAXBTS, -1).toInt();
	if(maxbts != -1) {
		ctx->search.maxBtsBetter = ctx->search.maxBts = maxbts; 
	}
	ctx->search.tryHard = parent->settings.getCustomValue(BowtieTask::OPTION_TRYHARD, false).toBool();
	ctx->search.chunkPoolMegabytes = parent->settings.getCustomValue(BowtieTask::OPTION_CHUNKMBS, 64).toInt();
	int seed = parent->settings.getCustomValue(BowtieTask::OPTION_SEED, -1).toInt();
	if(seed != -1) {
		ctx->search.seed = seed;
	}
	ctx->search.sortAlignment = parent->settings.getCustomValue(BowtieTask::OPTION_SORT_ALIGNMENT_BY_OFFSET, false).toBool();

	BowtieAdapter::doBowtie(parent->indexPath, parent->settings.shortReads, parent->result, stateInfo);
}

TLSContext* BowtieTLSTask::createContextInstance()
{
	return new BowtieContext(stateInfo, nThreads);
}

void BowtieTLSTask::prepare()
{
	TLSTask::prepare();
	for(int i=0;i<nThreads-1;i++) {
		addSubTask(new BowtieWorkerTask(i, taskContext));
	}
}

Task::ReportResult BowtieTLSTask::report()
{
	delete taskContext;
	taskContext = NULL;
	return ReportResult_Finished;
}
/************************************************************************/
/* BowtieBuildTask                                                      */
/************************************************************************/

BowtieBuildTask::BowtieBuildTask( QString _refPath, QString _outEbwtPath )
:TLSTask(tr("Bowtie Build")), refPath(_refPath), outEbwtPath(_outEbwtPath)
{
	tpm = Task::Progress_Manual;
	QFileInfo file(refPath);
	if(!file.exists()) {
		stateInfo.setError(BowtieBuildTask::tr("Reference file \"%1\" not exists").arg(refPath));
		return;
	}
	qint64 memUseMB = file.size() * 3 / 1024 / 1024 + 100;
	log.trace(QString("bowtie-build:Memory resourse %1").arg(memUseMB));
	TaskResourceUsage memUsg(RESOURCE_MEMORY, memUseMB);
	taskResources.append(memUsg);
}

TLSContext* BowtieBuildTask::createContextInstance() {
	return new BowtieContext(stateInfo, 1);
}

void BowtieBuildTask::_run()
{
	QMutexLocker lock(&mutex);
	BowtieAdapter::doBowtieBuild(this->refPath, this->outEbwtPath, stateInfo);
}

Task::ReportResult BowtieBuildTask::report()
{
	delete taskContext;
	taskContext = NULL;
	return ReportResult_Finished;
}


void BowtieWorkerTask::_run()
{
	BowtieAdapter::doBowtieWorker(id, stateInfo);
}

} //namespace
