/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <core_api/AppContext.h>
#include <core_api/SubstMatrixRegistry.h>
#include <core_api/L10n.h>

#include "uhmmer3BuildTests.h"
#include "uhmmer3SearchTests.h"
#include "uhmmer3PhmmerTests.h"

namespace GB2 {

/****************************************
 * GTest_UHMM3Phmmer
 ****************************************/

const QString GTest_UHMM3Phmmer::QUERY_FILENAME_TAG             = "query";
const QString GTest_UHMM3Phmmer::DB_FILENAME_TAG                = "db";
const QString GTest_UHMM3Phmmer::PHMMER_TASK_CTX_NAME_TAG       = "ctxName";

const QString GTest_UHMM3Phmmer::GAP_OPEN_PROBAB_OPTION_TAG     = "popen";
const QString GTest_UHMM3Phmmer::GAP_EXTEND_PROBAB_OPTION_TAG   = "pextend";
const QString GTest_UHMM3Phmmer::SUBST_MATR_NAME_OPTION_TAG     = "substMatr";

const double BAD_DOUBLE_OPTION = -1.0;

static void setDoubleOption( double & to, const QString & str, TaskStateInfo & ti ) {
    if( str.isEmpty() ) {
        return;
    }
    bool ok = false;
    to = str.toDouble( &ok );
    if( !ok ) {
        to = BAD_DOUBLE_OPTION;
        ti.setError( QString( "cannot_parse_double_from: %1" ).arg( str ) );
    }
}

static void setSubstMatrixOption( SMatrix& to, const QString & str, TaskStateInfo & ti ) {
    if( str.isEmpty() ) {
        to = SMatrix();
        return;
    }
    SubstMatrixRegistry * smr = AppContext::getSubstMatrixRegistry();
    assert( NULL != smr );
    
    if( smr->getMatrix( str ).isEmpty() ) {
        ti.setError( QString( "matrix %1 not registered" ).arg( str ) );
        return;
    }
    to = smr->getMatrix(str);
    assert(!to.isEmpty());
}

void GTest_UHMM3Phmmer::init( XMLTestFormat *tf, const QDomElement& el ) {
    Q_UNUSED( tf );
    
    phmmerTask      = NULL;
    queryFilename   = el.attribute( QUERY_FILENAME_TAG );
    dbFilename      = el.attribute( DB_FILENAME_TAG );
    taskCtxName     = el.attribute( PHMMER_TASK_CTX_NAME_TAG );
    
    UHMM3SearchSettings searchSettings;
    setDefaultUHMM3SearchSettings( &searchSettings );
    GTest_UHMM3Search::setSearchTaskSettings( searchSettings, el, settigsStateInfo );
    if( settigsStateInfo.hasErrors() ) { return; }
    settings.setSearchSettings( searchSettings );
    
    UHMM3BuildSettings buildSettings;
    setDefaultUHMM3BuildSettings( &buildSettings );
    GTest_UHMMER3Build::setBuildSettings( buildSettings, el, settigsStateInfo );
    settings.setBuildSettings( buildSettings );
    if( settigsStateInfo.hasErrors() ) { return; }
    
    setDoubleOption( settings.popen, el.attribute( GAP_OPEN_PROBAB_OPTION_TAG ), scoringSystemStateInfo );
    if( scoringSystemStateInfo.hasErrors() ) { return; }
    setDoubleOption( settings.pextend, el.attribute( GAP_EXTEND_PROBAB_OPTION_TAG ), scoringSystemStateInfo );
    if( scoringSystemStateInfo.hasErrors() ) { return; }
    setSubstMatrixOption( settings.substMatr, el.attribute( SUBST_MATR_NAME_OPTION_TAG ), scoringSystemStateInfo );
    assert( settings.substMatr.isEmpty() );
    if( scoringSystemStateInfo.hasErrors() ) { return; }
}

void GTest_UHMM3Phmmer::setAndCheckArgs() {
    assert( !stateInfo.hasErrors() );
    
    if( queryFilename.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( "query sequence filename" ) );
        return;
    }
    queryFilename = env->getVar( "COMMON_DATA_DIR" ) + "/" +queryFilename;
    
    if( dbFilename.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( "db sequence filename" ) );
        return;
    }
    dbFilename = env->getVar( "COMMON_DATA_DIR" ) + "/" + dbFilename;
    
    if( taskCtxName.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( "phmmer task context name" ) );
        return;
    }
    
    if( settigsStateInfo.hasErrors() ) {
        stateInfo.setError( settigsStateInfo.getError() );
        return;
    }
    if( scoringSystemStateInfo.hasErrors() ) {
        stateInfo.setError( scoringSystemStateInfo.getError() );
        return;
    }
}

void GTest_UHMM3Phmmer::prepare() {
    assert( !hasErrors() && NULL == phmmerTask );
    setAndCheckArgs();
    if( hasErrors() ) {
        return;
    }
    
    phmmerTask = new UHMM3PhmmerTask( queryFilename, dbFilename, settings );
    addSubTask( phmmerTask );
}

Task::ReportResult GTest_UHMM3Phmmer::report() {
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    assert( NULL != phmmerTask );
    
    if( !phmmerTask->isCanceled() && !phmmerTask->hasErrors() ) {
        addContext( taskCtxName, phmmerTask );
    }
    return ReportResult_Finished;
}

void GTest_UHMM3Phmmer::cleanup() {
    if( NULL != phmmerTask ) {
        removeContext( taskCtxName );
    }
}

GTest_UHMM3Phmmer::~GTest_UHMM3Phmmer() {
}

/****************************************
* GTest_UHMM3PhmmerCompare
****************************************/
const QString GTest_UHMM3PhmmerCompare::PHMMER_TASK_CTX_NAME_TAG    = "phmmerTask";
const QString GTest_UHMM3PhmmerCompare::TRUE_OUT_FILENAME_TAG       = "trueOut";

void GTest_UHMM3PhmmerCompare::init( XMLTestFormat *tf, const QDomElement& el ) {
    Q_UNUSED( tf );
    
    phmmerTaskCtxName = el.attribute( PHMMER_TASK_CTX_NAME_TAG );
    trueOutFilename   = el.attribute( TRUE_OUT_FILENAME_TAG );
}

void GTest_UHMM3PhmmerCompare::setAndCheckArgs() {
    if( phmmerTaskCtxName.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( "phmmer task context name" ) );
        return;
    }
    if( trueOutFilename.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( "true out filename" ) );
        return;
    }
    trueOutFilename = env->getVar( "COMMON_DATA_DIR" ) + "/" + trueOutFilename;
    
    phmmerTask = getContext< UHMM3PhmmerTask >( this, phmmerTaskCtxName );
    if( NULL == phmmerTask ) {
        stateInfo.setError( QString( "cannot find phmmer task %1 in context" ).arg( phmmerTaskCtxName ) );
        return;
    }
}

Task::ReportResult GTest_UHMM3PhmmerCompare::report() {
    assert( !hasErrors() );
    setAndCheckArgs();
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    
    UHMM3SearchResult trueRes;
    try {
        trueRes = GTest_UHMM3SearchCompare::getOriginalSearchResult( trueOutFilename );
    } catch( const QString& ex ) {
        stateInfo.setError( ex );
    } catch(...) {
        stateInfo.setError( "undefined_error_occurred" );
    }
    
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    
    assert( NULL != phmmerTask );
    GTest_UHMM3SearchCompare::generalCompareResults( phmmerTask->getResult(), trueRes, stateInfo );
    
    return ReportResult_Finished;
}

} // GB2
