/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "KalignTask.h"
#include "KalignAdapter.h"
#include "KalignConstants.h"

extern "C" {
#include "kalign2/kalign2_context.h"
}

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>
#include <core_api/AppResources.h>
#include <core_api/Log.h>
#include <core_api/StateLockableDataModel.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <core_api/Counter.h>
#include <gobjects/DNASequenceObject.h>
#include <util_tasks/LoadDocumentTask.h>

extern "C" kalign_context *getKalignContext() {
	GB2::KalignContext* ctx = static_cast<GB2::KalignContext*>(GB2::TLSUtils::current(KALIGN_CONTEXT_ID));
	assert(ctx->d != NULL);
	return ctx->d;
}

namespace GB2 {

static LogCategory log(ULOG_CAT_KALIGN);


void KalignTaskSettings::reset() {
	gapExtenstionPenalty = -1;
	gapOpenPenalty = -1;
	termGapPenalty = -1;
	secret = -1;
}

KalignTask::KalignTask(const MAlignment& ma, const KalignTaskSettings& _config) 
:TLSTask(tr("KALIGN alignment"), TaskFlags_FOSCOE), config(_config), inputMA(ma)
{
	GCOUNTER( cvar, tvar, "KalignTask" );
	inputSubMA = inputMA;
	resultSubMA.setAlphabet(inputSubMA.getAlphabet());
	tpm = Task::Progress_Manual;
	//TODO: add task resource usage
}

void KalignTask::_run() {
	assert(!hasErrors());
	doAlign(); 
	if (!hasErrors() && !isCanceled()) {
		assert(resultMA.getAlphabet()!=NULL);
	}
}

void KalignTask::doAlign() {
	assert(resultSubMA.isEmpty());
	KalignAdapter::align(inputSubMA, resultSubMA, stateInfo);
	if (hasErrors()) {
		return;
	}
	resultMA = resultSubMA;
}

Task::ReportResult KalignTask::report() {
	KalignContext* ctx = static_cast<KalignContext*>(taskContext);
	delete ctx->d;
	return ReportResult_Finished;
}

TLSContext* KalignTask::createContextInstance()
{
	kalign_context* ctx = new kalign_context;
	init_context(ctx, &stateInfo);
	if(config.gapOpenPenalty != -1) {
		ctx->gpo = config.gapOpenPenalty;
	}
	if(config.gapExtenstionPenalty != -1) {
		ctx->gpe = config.gapExtenstionPenalty;
	}
	if(config.termGapPenalty != -1) {
		ctx->tgpe = config.termGapPenalty;
	}
	if(config.secret != -1) {
		ctx->secret = config.secret;
	}
	return new KalignContext(ctx);
}
//////////////////////////////////////////////////////////////////////////
// KalignGObjectTask

KalignGObjectTask::KalignGObjectTask(MAlignmentObject* _obj, const KalignTaskSettings& _config) 
: Task("", TaskFlags_NR_FOSCOE), obj(_obj), lock(NULL), kalignTask(NULL), config(_config)
{
	QString aliName = obj->getDocument()->getName();
	QString tn;
	tn = tr("KALIGN align '%1'").arg(aliName);
	setTaskName(tn);
	setUseDescriptionFromSubtask(true);
	setVerboseLogMode(true);
}

KalignGObjectTask::~KalignGObjectTask() {
	assert(lock == NULL);
}

void KalignGObjectTask::prepare() {
	if (obj.isNull()) {
		stateInfo.setError(tr("object_removed"));
		return;
	}
	if (obj->isStateLocked()) {
		stateInfo.setError(tr("object_is_state_locked"));
		return;
	}

	lock = new StateLock("kalign_lock");
	obj->lockState(lock);
	kalignTask = new KalignTask(obj->getMAlignment(), config);

	addSubTask(kalignTask);
}

Task::ReportResult KalignGObjectTask::report() {
	if (lock!=NULL) {
		obj->unlockState(lock);
		delete lock;
		lock = NULL;
	}
	propagateSubtaskError();
	if (hasErrors() || isCanceled()) {
		return ReportResult_Finished;
	}
	assert(!obj.isNull());
	if (obj->isStateLocked()) {
		stateInfo.setError(tr("object_is_state_locked"));
		return ReportResult_Finished;
	}
	assert(kalignTask->inputMA.getNumRows() == kalignTask->resultMA.getNumRows());
	obj->setMAlignment(kalignTask->resultMA);    

	return ReportResult_Finished;
}

} //namespace
