/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SiteconIOWorkers.h"
#include "SiteconWorkers.h"
#include "SiteconSearchDialogController.h"
#include <workflow/Datatype.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>
#include <core_api/GUrlUtils.h>
#include <util_tasks/FailTask.h>
#include "SiteconIO.h"

/* TRANSLATOR GB2::SiteconIO */

namespace GB2 {
namespace LocalWorkflow {

const QString SiteconReader::ACTOR_ID("sitecon.read");
const QString SiteconWriter::ACTOR_ID("sitecon.write");

const QString SiteconWorkerFactory::SITECON_MODEL_TYPE_ID("sitecon.model");

DataTypePtr const SiteconWorkerFactory::SITECON_MODEL_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(SITECON_MODEL_TYPE_ID, SiteconIO::tr("Sitecon model"), "")));
        startup = false;
    }
    return dtr->getById(SITECON_MODEL_TYPE_ID);
}

const Descriptor SiteconWorkerFactory::SITECON_CATEGORY() {return Descriptor("hsitecon", SiteconIO::tr("SITECON"), "");}

static LogCategory log(ULOG_CAT_WD);

SiteconIOProto::SiteconIOProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: BusActorPrototype(_desc, _ports, _attrs) {
}

bool SiteconIOProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params, const QString & urlAttrId ) const {
    if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            QString url = urls.at(0).toLocalFile();
            QString ext = GUrlUtils::getUncompressedExtension(GUrl(url, GUrl_File));
            if (SiteconIO::SITECON_EXT == ext) {
                if (params) {
                    params->insert(urlAttrId, url);
                }
                return true;
            }
        }
    }
    return false;
}

ReadSiteconProto::ReadSiteconProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: SiteconIOProto( _desc, _ports, _attrs ) {
    
    attrs << new Attribute(CoreLibConstants::URL_IN_ATTR(), CoreDataTypes::STRING_TYPE(), true);
    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_IN_ATTR_ID] = new URLDelegate(SiteconIO::getFileFilter(), SiteconIO::SITECON_ID, true);
    setEditor(new DelegateEditor(delegateMap));
    setIconPath(":sitecon/images/sitecon.png");
}

bool ReadSiteconProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return SiteconIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_IN_ATTR_ID );
}

WriteSiteconProto::WriteSiteconProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: SiteconIOProto( _desc, _ports, _attrs ) {
    attrs << new Attribute(CoreLibConstants::URL_OUT_ATTR(), CoreDataTypes::STRING_TYPE(), true );
    attrs << new Attribute(BioActorLibrary::FILE_MODE_ATTR(), CoreDataTypes::NUM_TYPE(), false, SaveDoc_Roll);

    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_OUT_ATTR_ID] = new URLDelegate(SiteconIO::getFileFilter(), SiteconIO::SITECON_ID, false );
    delegateMap[BioActorLibrary::FILE_MODE_ATTR_ID] = new FileModeDelegate(false);
    
    setEditor(new DelegateEditor(delegateMap));
    setIconPath(":sitecon/images/sitecon.png");
    setValidator(new ScreenedParamValidator(CoreLibConstants::URL_OUT_ATTR_ID, ports.first()->getId(), CoreLibConstants::URL_SLOT_ID));
    setPortValidator(CoreLibConstants::IN_PORT_ID, new ScreenedSlotValidator(CoreLibConstants::URL_SLOT_ID));
}

bool WriteSiteconProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return SiteconIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_OUT_ATTR_ID );
}

void SiteconWorkerFactory::init() 
{
    WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
    assert(r);
    DataTypePtr dt = SITECON_MODEL_TYPE();
    {        
        QMap<Descriptor, DataTypePtr> m;
        Descriptor sd(CoreLibConstants::URL_SLOT_ID, SiteconIO::tr("Location"), SiteconIO::tr("Location hint for the target file."));
        m[sd] = CoreDataTypes::STRING_TYPE();
        m[*dt] = dt;
        DataTypePtr t(new MapDataType(Descriptor("write.sitecon.content"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(CoreLibConstants::IN_PORT_ID, SiteconIO::tr("Sitecon model"), SiteconIO::tr("Input Sitecon model"));
        p << new PortDescriptor(pd, t, true /*input*/);
        Descriptor desc(SiteconWriter::ACTOR_ID, SiteconIO::tr("Write Sitecon model"), SiteconIO::tr("Saves all input SITECON profiles to specified location."));
        BusActorPrototype* proto = new WriteSiteconProto(desc, p, a);
        proto->setPrompter(new SiteconWritePrompter());
        r->registerProto(SITECON_CATEGORY(), proto);
    }
    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(CoreLibConstants::OUT_PORT_ID, SiteconIO::tr("Sitecon model"), SiteconIO::tr("Loaded SITECON profile data."));
        p << new PortDescriptor(pd, dt, false /*input*/, true /*multi*/);
        Descriptor desc(SiteconReader::ACTOR_ID, SiteconIO::tr("Read Sitecon model"), SiteconIO::tr("Reads SITECON profiles from file(s). The files can be local or Internet URLs."));
        BusActorPrototype* proto = new ReadSiteconProto(desc, p, a);
        proto->setPrompter(new SiteconReadPrompter());
        r->registerProto(SITECON_CATEGORY(), proto);
    }

    SiteconBuildWorker::registerProto();
    SiteconSearchWorker::registerProto();

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconReader::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconWriter::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconSearchWorker::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconBuildWorker::ACTOR_ID));
}

Worker* SiteconWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
    if (SiteconReader::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconReader(a);
    } 
    else if (SiteconWriter::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconWriter(a);
    }
    else if (SiteconBuildWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconBuildWorker(a);
    }
    else if (SiteconSearchWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconSearchWorker(a);
    }

    return w;    
}

QString SiteconReadPrompter::composeRichDoc() {
    return tr("Read model(s) from <u>%1</u>").arg(getURL(CoreLibConstants::URL_IN_ATTR_ID));
}

QString SiteconWritePrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* producer = input->getProducer(SiteconWorkerFactory::SITECON_MODEL_TYPE_ID);
    if (!producer) {
        return getURL(CoreLibConstants::URL_OUT_ATTR_ID);
    }
    QString url = getScreenedURL(input, CoreLibConstants::URL_OUT_ATTR_ID, CoreLibConstants::URL_SLOT_ID); 
    QString doc = tr("Save the profile(s) from <u>%1</u> to %2.")
        .arg(producer->getLabel())
        .arg(url);
    return doc;
}

void SiteconReader::init() {
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
    urls = DesignerUtils::expandToUrls(actor->getParameter(CoreLibConstants::URL_IN_ATTR_ID)->getAttributeValue<QString>());
    mtype = SiteconWorkerFactory::SITECON_MODEL_TYPE();
}

Task* SiteconReader::tick() {
    Task* t = new SiteconReadTask(urls.takeFirst());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    tasks.append(t);
    return t;
}

void SiteconReader::sl_taskFinished() {
    SiteconReadTask* t = qobject_cast<SiteconReadTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    tasks.removeAll(t);
    if (output) {
        if (!t->hasErrors()) {
            QVariant v = qVariantFromValue<SiteconModel>(t->getResult());
            output->put(Message(mtype, v));
        }
        if (urls.isEmpty() && tasks.isEmpty()) {
            output->setEnded();
        }
        log.info(tr("Loaded SITECON model from %1").arg(t->getURL()));
    }
}

void SiteconWriter::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
}

Task* SiteconWriter::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    url = actor->getParameter(CoreLibConstants::URL_OUT_ATTR_ID)->getAttributeValue<QString>();
    fileMode = actor->getParameter(BioActorLibrary::FILE_MODE_ATTR_ID)->getAttributeValue<uint>();
    QVariantMap data = inputMessage.getData().toMap();
    SiteconModel model = data.value(SiteconWorkerFactory::SITECON_MODEL_TYPE_ID).value<SiteconModel>();
    
    QString anUrl = url;
    if (anUrl.isEmpty()) {
        anUrl = data.value(CoreLibConstants::URL_SLOT_ID).toString();
    }
    if (anUrl.isEmpty()) {
        QString err = tr("Unspecified URL for writing Sitecon");
        if (failFast) {
            return new FailTask(err);
        } else {
            log.error(err);
            return NULL;
        }
    }
    assert(!anUrl.isEmpty());
    int count = ++counter[anUrl];
    if (count != 1) {
        anUrl = GUrlUtils::prepareFileName(anUrl, count, QStringList("sitecon"));
    } else {
        anUrl = GUrlUtils::ensureFileExt( anUrl, QStringList("sitecon")).getURLString();
    }
    log.info(tr("Writing SITECON model to %1").arg(anUrl));
    return new SiteconWriteTask(anUrl, model, fileMode);
}

} //namespace LocalWorkflow
} //namespace GB2

