/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtGui/QMessageBox>

#include <core_api/AppContext.h>
#include <distributed_computing/RemoteMachineMonitor.h>
#include <distributed_computing/DistributedComputingUtil.h>
#include <util_gui/RemoteMachineMonitorDialogImpl.h>

#include <muscle_local_task/MuscleLocalTask.h>
#include "MuscleAlignDialogController.h"

/* TRANSLATOR GB2::MuscleAlignDialogController */    

namespace GB2 {

MuscleAlignDialogController::MuscleAlignDialogController(QWidget* w, const MAlignment& _ma, MuscleTaskSettings& _settings) 
: QDialog(w), ma(_ma), settings(_settings)
{
    setupUi(this);

    rangeEndSB->setMaximum(ma.getLength());
    rangeEndSB->setValue(ma.getLength());

    if (settings.alignRegion) {
        customRangeRB->setChecked(true);
        rangeStartSB->setValue(settings.regionToAlign.startPos);
        rangeEndSB->setValue(settings.regionToAlign.endPos());
    }
    connect(confBox, SIGNAL(currentIndexChanged(int)), SLOT(sl_onPresetChanged(int)));
    initPresets();
    foreach(const MuscleAlignPreset* p, presets.qlist) {
        confBox->addItem(p->name);
    }
    
    connect( remoteRunPushButton, SIGNAL( clicked() ), SLOT( sl_remoteRunButtonClicked() ) );
}

void MuscleAlignDialogController::accept() {
    int n = confBox->currentIndex();
    assert(n >=0 && n < presets.qlist.size());
    const MuscleAlignPreset* p = presets.qlist[n];
    p->apply(settings);
    settings.stableMode = stableCB->isChecked();

    if (wholeRangeRB->isChecked()) {
        settings.regionToAlign = LRegion(0, ma.getLength());
        settings.alignRegion = false;
    } else {
        int startPos = rangeStartSB->value() - 1;
        int endPos = rangeEndSB->value();
        if (endPos - startPos < 2) {
            QMessageBox::critical(NULL, tr("Error"), tr("Illegal alignment region"));
            rangeStartSB->setFocus();
            return;
        }
        settings.alignRegion = true;
        settings.regionToAlign = LRegion(startPos,  endPos - startPos);
    }

    if (maxItersCheckBox->isChecked()) {
        settings.maxIterations = maxItersSpinBox->value();
        assert(settings.maxIterations >= 2);
    }
    if (maxMinutesCheckBox->isChecked()) {
        settings.maxSecs = maxMinutesSpinBox->value() * 60; 
        assert(settings.maxSecs > 0);
    }
    QDialog::accept();
}

void MuscleAlignDialogController::sl_remoteRunButtonClicked() {
    int rc = 0;
    do{
        RemoteMachineMonitor * rmm = AppContext::getRemoteMachineMonitor();
        assert( NULL != rmm );
        RemoteMachineMonitorDialogImpl dlg( QApplication::activeWindow(), rmm->getRemoteMachineMonitorItems(), 
            MuscleLocalTaskFactory::ID );
        rc = dlg.exec();
        if( QDialog::Rejected == rc ) {
            return;
        }
        assert( QDialog::Accepted == rc );
        
        QList< RemoteMachineMonitorDialogItem > dlgModel = dlg.getModel();
        DistributedComputingUtil::applyChangesForRemoteMachineMonitor( rmm, dlgModel );
        QList< RemoteMachineSettings* > selectedMachines = rmm->getSelectedMachines();
        int howManyMachines = selectedMachines.size();
        if( 0 == howManyMachines ) {
            QMessageBox::critical( this, tr( "Selecting machines error!" ), tr( "You didn't select a machine to run remote task!" ) );
            continue;
        } else if( 1 != howManyMachines ) {
            QMessageBox::critical( this, tr( "Selecting machines error!" ), 
                tr( "Distributed run on many machines is not supported yet. Select 1 machine" ) );
            continue;
        }
        machines << selectedMachines.first();
        break;
    } while( QDialog::Accepted == rc );
    
    accept();
}

QList< RemoteMachineSettings* > MuscleAlignDialogController::getSelectedMachines() const {
    return machines;
}

void MuscleAlignDialogController::sl_onPresetChanged(int newPreset) {
    assert(newPreset>=0 && newPreset < presets.qlist.size());
    const MuscleAlignPreset* p = presets.qlist[newPreset];
    confEdit->setText(p->desc);
}


//////////////////////////////////////////////////////////////////////////
// presets
DefaultModePreset::DefaultModePreset() {
    name = MuscleAlignDialogController::tr("MUSCLE default");
    desc = MuscleAlignDialogController::tr("<p>The default settings are designed to give the best accuracy");
    desc+= MuscleAlignDialogController::tr("<p><b>Command line:</b> muscle <no-parameters>");
}

LargeModePreset::LargeModePreset() {
    name = MuscleAlignDialogController::tr("Large alignment");
    desc = MuscleAlignDialogController::tr("<p>If you have a large number of sequences (a few thousand), or they are very long, then the default settings may be too slow for practical use. A good compromise between speed and accuracy is to run just the first two iterations of the algorithm");
    desc+= MuscleAlignDialogController::tr("<p><b>Command line:</b> muscle <i>-maxiters 2</i>");
}

RefineModePreset::RefineModePreset() {
    name = MuscleAlignDialogController::tr("Refine only");
    desc = MuscleAlignDialogController::tr("<p>Improves existing alignment without complete realignment");
    desc+= MuscleAlignDialogController::tr("<p><b>Command line:</b> muscle <i>-refine</i>");
}


void MuscleAlignDialogController::initPresets() {
    presets.qlist.append(new DefaultModePreset());
    presets.qlist.append(new LargeModePreset());
    presets.qlist.append(new RefineModePreset());
}


}//namespace
