/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "LogDriver.h"

#include <core_api/Timer.h>
#include <core_api/Settings.h>
#include <core_api/Task.h>
#include <cmdline/CMDLineRegistry.h>
#include <cmdline/CMDLineHelpProvider.h>
#include "TaskStatusBar.h"

#ifdef Q_OS_WIN32
#include "windows.h"
#endif
#include <stdio.h>
//#include <conio.h>
#define LOG_SETTINGS_ROOT QString("log_settings/")

namespace GB2 {

const QString LogDriver::LOG_NO_SHOW_DATE_CMD_OPTION           = "log-no-show-date";
const QString LogDriver::LOG_NO_SHOW_LEVEL_CMD_OPTION          = "log-no-show-level";
const QString LogDriver::LOG_NO_SHOW_CATEGORY_CMD_OPTION       = "log-no-show-category";
const QString LogDriver::LOG_LEVEL_NONE_CMD_OPTION             = "log-level-none";
const QString LogDriver::LOG_LEVEL_ERROR_CMD_OPTION            = "log-level-error";
const QString LogDriver::LOG_LEVEL_INFO_CMD_OPTION             = "log-level-info";
const QString LogDriver::LOG_LEVEL_DETAILS_CMD_OPTION          = "log-level-details";
const QString LogDriver::LOG_LEVEL_TRACE_CMD_OPTION            = "log-level-trace";
const QString LogDriver::COLOR_OUTPUT_CMD_OPTION               = "log-color-output";
const QString LogDriver::TEAMCITY_OUTPUT_CMD_OPTION            = "log-teamcity-out";
const QString LogDriver::LOG_SETTINGS_ACTIVE_FLAG              = "activeFlagLevel";

bool LogDriver::helpRegistered = false;

LogDriver::LogDriver() : print_to_console(true) {
    connect(LogServer::getInstance(), SIGNAL(si_message(const LogMessage&)), SLOT(sl_onMessage(const LogMessage&)));
    
    if( !helpRegistered ) {
        setLogCmdlineHelp();
    }
    if( !AppContext::getSettings()->getValue( SETTINGS_FROM_FILE, false ).toBool() ) {
        setCmdLineSettings();
    }
    
    settings.reinitCategories();
    settings.reinitAll();
}

void LogDriver::setLogCmdlineHelp() {
    assert( !helpRegistered );
    helpRegistered = true;
    
    CMDLineRegistry * cmdLineRegistry = AppContext::getCMDLineRegistry();
    assert( NULL != cmdLineRegistry );
    
    CMDLineHelpProvider * teamcityOutputSection = new CMDLineHelpProvider( TEAMCITY_OUTPUT_CMD_OPTION,
        tr( "output test's messages for TeamCity system" ));
    CMDLineHelpProvider * colorOutputSection = new CMDLineHelpProvider( COLOR_OUTPUT_CMD_OPTION,
        tr( "colored output messages" ) );
    CMDLineHelpProvider * logLevelTraceSection = new CMDLineHelpProvider( LOG_LEVEL_TRACE_CMD_OPTION,
        tr( "show trace log messages" ) );
    CMDLineHelpProvider * logLevelDetailsSection = new CMDLineHelpProvider( LOG_LEVEL_DETAILS_CMD_OPTION,
        tr( "show details log messages" ) );
    CMDLineHelpProvider * logLevelInfoSection = new CMDLineHelpProvider( LOG_LEVEL_INFO_CMD_OPTION,
        tr( "show error and info log messages (default)" ) );
    CMDLineHelpProvider * logLevelErrorSection = new CMDLineHelpProvider( LOG_LEVEL_ERROR_CMD_OPTION,
        tr( "show only error log messages" ) );
    CMDLineHelpProvider * logLevelNoneSection = new CMDLineHelpProvider( LOG_LEVEL_NONE_CMD_OPTION,
        tr( "Don't show log messages" ) );
    CMDLineHelpProvider * logNoShowCatSection = new CMDLineHelpProvider( LOG_NO_SHOW_CATEGORY_CMD_OPTION,
        tr( "Don't show message category at log" ) );
    CMDLineHelpProvider * logNoShowLevelSection = new CMDLineHelpProvider( LOG_NO_SHOW_LEVEL_CMD_OPTION,
        tr( "Don't show message level at log" ) );
    CMDLineHelpProvider * logNoShowDateSection = new CMDLineHelpProvider( LOG_NO_SHOW_DATE_CMD_OPTION,
        tr( "Don't show date&time info at log" ) );
    
    cmdLineRegistry->registerCMDLineHelpProvider( teamcityOutputSection );
    cmdLineRegistry->registerCMDLineHelpProvider( colorOutputSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logLevelTraceSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logLevelDetailsSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logLevelInfoSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logLevelErrorSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logLevelNoneSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logNoShowCatSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logNoShowLevelSection );
    cmdLineRegistry->registerCMDLineHelpProvider( logNoShowDateSection );
}

void LogDriver::setCmdLineSettings() {
    CMDLineRegistry * cmdLineRegistry = AppContext::getCMDLineRegistry();
    assert( NULL != cmdLineRegistry );
    Settings * settings = AppContext::getSettings();
    assert( NULL != settings );

    settings->setValue( LOG_SETTINGS_ROOT + "showDate", !cmdLineRegistry->hasParameter( LOG_NO_SHOW_DATE_CMD_OPTION ) );
    settings->setValue( LOG_SETTINGS_ROOT + "showLevel", !cmdLineRegistry->hasParameter( LOG_NO_SHOW_LEVEL_CMD_OPTION ) );
    settings->setValue( LOG_SETTINGS_ROOT + "showCategory", !cmdLineRegistry->hasParameter( LOG_NO_SHOW_CATEGORY_CMD_OPTION ) );

    int i = 0;
    if( cmdLineRegistry->hasParameter( LOG_LEVEL_NONE_CMD_OPTION ) ) {
        for( i = 0; i < LogLevel_NumLevels; i++ ) {
            settings->setValue( LOG_SETTINGS_ROOT + LOG_SETTINGS_ACTIVE_FLAG + QString::number( i ), false );
        }
    } else if( cmdLineRegistry->hasParameter( LOG_LEVEL_ERROR_CMD_OPTION ) ) {
        for( i = 0; i < LogLevel_NumLevels; i++ ) {
            settings->setValue( LOG_SETTINGS_ROOT + LOG_SETTINGS_ACTIVE_FLAG + QString::number( i ), i >= LogLevel_ERROR );
        }
    } else if( cmdLineRegistry->hasParameter( LOG_LEVEL_INFO_CMD_OPTION ) ) {
        for( i = 0; i < LogLevel_NumLevels; i++ ) {
            settings->setValue( LOG_SETTINGS_ROOT + LOG_SETTINGS_ACTIVE_FLAG + QString::number( i ), i >= LogLevel_INFO );
        }
    } else if( cmdLineRegistry->hasParameter( LOG_LEVEL_DETAILS_CMD_OPTION ) ) {
        for( i = 0; i < LogLevel_NumLevels; i++ ) {
            settings->setValue( LOG_SETTINGS_ROOT + LOG_SETTINGS_ACTIVE_FLAG + QString::number( i ), i >= LogLevel_DETAILS );
        }
    } else if( cmdLineRegistry->hasParameter( LOG_LEVEL_TRACE_CMD_OPTION ) ) {
        for( i = 0;i < LogLevel_NumLevels; i++ ) {
            settings->setValue( LOG_SETTINGS_ROOT + LOG_SETTINGS_ACTIVE_FLAG + QString::number( i ), i >= LogLevel_TRACE );
        }
    } else {
        for( i = 0; i < LogLevel_NumLevels; i++ ) {
            settings->setValue( LOG_SETTINGS_ROOT + LOG_SETTINGS_ACTIVE_FLAG + QString::number( i ), i >= LogLevel_INFO );
        }
    }
    settings->setValue( LOG_SETTINGS_ROOT + "colorOut", cmdLineRegistry->hasParameter( COLOR_OUTPUT_CMD_OPTION ) );
    settings->setValue( LOG_SETTINGS_ROOT + "teamcityOut", cmdLineRegistry->hasParameter( TEAMCITY_OUTPUT_CMD_OPTION ) );
}

QString LogDriver::prepareText(const LogMessage& msg) const {
    QString date = settings.showDate ? "["+GTimer::createDateTime(msg.time).toString("hh:mm") + "]" : QString();
    QString category = settings.showCategory ? "["+ msg.category + "]" : QString();
    QString level = settings.showLevel ? "["+ LogCategories::getLocalizedLevelName(msg.level) +"] " : QString();
    QString spacing = date.isEmpty() && category.isEmpty() && level.isEmpty() ? QString() : QString(" ");
    QString text = date + category + level + spacing + msg.text;

    return text;
}


void LogDriver::sl_onMessage(const LogMessage& msg) {
    if (!print_to_console || !settings.activeLevelGlobalFlag[msg.level])
    {
        return;
    }
    LogCategorySettings s;
    for(int i = 0;i < LogLevel_NumLevels; i++ ) {
        QVariant v = AppContext::getSettings()->getValue(LOG_SETTINGS_ROOT + LOG_SETTINGS_ACTIVE_FLAG + QString::number( i ));
        s.activeLevelFlag[i] = v.toBool();
    }
    if (!s.activeLevelFlag[msg.level]) {
        return;
    }


    if (print_to_console == true){
        //this is Hard Code
        if (msg.text.contains(QString("##teamcity"))&&(!AppContext::getSettings()->getValue("log_settings/teamcityOut",true).toBool()))
            return;
        QByteArray ba = prepareText(msg).toLocal8Bit();
        char* buf = ba.data();
    #ifdef Q_OS_WIN32
        // a bit of magic to workaround Windows console encoding issues
        CharToOemA(buf,buf);
    #endif
        if(AppContext::getSettings()->getValue(TSB_SETTINGS_ROOT + "showTaskStatusBar", true).toBool()){
            printf("                                                                               \r");//80 spaces for remove TaskStatusBar
        }
        if(!AppContext::getSettings()->getValue(LOG_SETTINGS_ROOT + "colorOut", false).toBool()){
        printf("%s \n", buf);
        }else{
#ifdef Q_OS_WIN32
            if (msg.level==LogLevel_ERROR){
                SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE), FOREGROUND_INTENSITY | FOREGROUND_RED );
            }else if (msg.level==LogLevel_DETAILS){
                SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE), FOREGROUND_INTENSITY | FOREGROUND_GREEN );
            }else if (msg.level==LogLevel_TRACE){
                SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE), FOREGROUND_INTENSITY | FOREGROUND_BLUE );
            }
            printf("%s \n", buf);
            SetConsoleTextAttribute(GetStdHandle(STD_OUTPUT_HANDLE), 0x0007);
#else
            //TO DO: for MacOs release coloring function
            if (msg.level==LogLevel_ERROR){
                printf("\e[31m%s \e[0m\n", buf);
            }else if (msg.level==LogLevel_DETAILS){
                printf("\e[32m%s \e[0m\n", buf);
            }else if (msg.level==LogLevel_TRACE){
                printf("\e[34m%s \e[0m\n", buf);
            }
#endif
        }
        //if(!AppContext::getSettings()->getValue(LOG_SETTINGS_ROOT + "colorOut", false).toBool()){
    }
}

}//namespace

