/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_services/AppContextImpl.h>
#include <core_services/SettingsImpl.h>
#include <core_services/DocumentFormatRegistryImpl.h>
#include <core_services/IOAdapterRegistryImpl.h>

#include "plugin_support/PluginSupportImpl.h"
#include "plugin_support/ServiceRegistryImpl.h"
#include "plugin_viewer/PluginViewerImpl.h"
#include "project_support/ProjectLoaderImpl.h"
#include "main_window/MainWindowImpl.h"
#include "project_view/ProjectViewImpl.h"
#include "logview/LogCache.h"
#include "logview/LogView.h"
#include <task_scheduler/TaskSchedulerImpl.h>
#include "task_view/TaskViewController.h"
#include <core_services/AppSettingsImpl.h>
#include "app_settings/AppSettingsGUIImpl.h"
#include "shtirlitz/Shtirlitz.h"
#include <util_msa_consensus/MSAConsensusAlgorithmRegistry.h>
#include <util_weight_matrix/PWMConversionAlgorithmRegistry.h>

#include <core_api/Log.h>
#include <core_api/Timer.h>
#include <core_api/GUrlUtils.h>
#include <core_api/DNATranslation.h>
#include <core_api/ObjectViewModel.h>
#include <core_api/ResourceTracker.h>
#include <core_api/DocumentFormatConfigurators.h>
#include <core_api/DBXRefRegistry.h>
#include <core_api/UserApplicationsSettings.h>
#include <core_api/SecStructPredictAlgRegistry.h>
#include <core_api/CudaGpuRegistry.h>
#include <core_api/AtiStreamGpuRegistry.h>
#include <core_api/VirtualFileSystem.h>

#include <core_api/SubstMatrixRegistry.h>
#include <core_api/SWResultFilterRegistry.h>
#include <core_api/SmithWatermanTaskFactoryRegistry.h>
#include <core_api/DnaAssemblyAlgRegistry.h>
#include <core_api/DataBaseRegistry.h>
#include <core_api/RepeatFinderTaskFactoryRegistry.h>

#include <cmdline/CMDLineRegistry.h>
#include <cmdline/CMDLineUtils.h>
#include <cmdline/CMDLineCoreOptions.h>

#include <molecular_geometry/MolecularSurfaceFactoryRegistry.h>
#include <phyltree/PhyTreeGeneratorRegistry.h>

#include <document_format/DNAAlphabetRegistryImpl.h>
#include <gobjects/AnnotationSettings.h>
#include <gobjects/GObjectTypes.h>
#include <test_framework/GTestFrameworkComponents.h>
#include <util_gui/BaseDocumentFormatConfigurators.h>
#include <util_ov_msaedit/MSAColorScheme.h>

#include <workflow_support/WorkflowEnvImpl.h>
#include <workflow_library/BioActorLibrary.h>
#include <util_tasks/TaskStarter.h>
#include <util_tasks/LoadRemoteDocumentTask.h>
#include <util_index_support/UIndexSupport.h>
#include <util_dna_assembly/DnaAssemblyUtils.h>

#include <QtGui/QApplication>
#include <QtGui/QIcon>

#include <distributed_computing/DistributedComputingUtil.h>

#include <test_framework/xmltest/XMLTestFormat.h>
#include <project_support/ProjectTasksGui.h>

static GB2::LogCategory logCat(ULOG_CAT_USER_INTERFACE);

/* TRANSLATOR GB2::AppContextImpl */

#if defined(Q_WS_WIN) || defined(Q_WS_MAC)
QT_BEGIN_NAMESPACE
extern Q_GUI_EXPORT bool qt_use_native_dialogs;
QT_END_NAMESPACE
#endif

namespace GB2 {

static void registerCoreServices() {
    ServiceRegistry* sr = AppContext::getServiceRegistry();
    TaskScheduler* ts = AppContext::getTaskScheduler();
    ts->registerTopLevelTask(sr->registerServiceTask(new PluginViewerImpl()));
    ts->registerTopLevelTask(sr->registerServiceTask(new ProjectViewImpl()));
//    ts->registerTopLevelTask(sr->registerServiceTask(new ScriptRegistryService()));
}

}//namespace

using namespace GB2;

static void updateStaticTranslations() {
    GObjectTypes::initTypeTranslations();
    GObjectTypes::initTypeIcons();
}

static void setDataSearchPaths() {
    //set search paths for data files
    QStringList dataSearchPaths;
#if (defined(Q_OS_LINUX) || defined(Q_OS_UNIX)) && defined( UGENE_DATA_DIR )
    //using directory which is set during installation process on linux
    QString ugene_data_dir( UGENE_DATA_DIR );
    if( QDir(ugene_data_dir).exists() ) {
        dataSearchPaths.push_back( QString(UGENE_DATA_DIR) );
    }
#endif
    const static char * RELATIVE_DATA_DIR = "/data";
    const static char * RELATIVE_DEV_DATA_DIR = "/../../data";
    //on windows data is normally located in the application directory
    QString appDirPath = QCoreApplication::applicationDirPath();
    if( QDir(appDirPath+RELATIVE_DATA_DIR).exists() ) {
        dataSearchPaths.push_back( appDirPath+RELATIVE_DATA_DIR );
    } else if( QDir(appDirPath+RELATIVE_DEV_DATA_DIR).exists() ) {          //data location for developers
        dataSearchPaths.push_back( appDirPath+RELATIVE_DEV_DATA_DIR );
    }
    if( dataSearchPaths.empty() ) {
        dataSearchPaths.push_back("/");
    }
    QDir::setSearchPaths( PATH_PREFIX_DATA, dataSearchPaths );
    //now data files may be opened using QFile( "data:some_data_file" )
} 

static void setSearchPaths() {
    setDataSearchPaths();
}

int main(int argc, char **argv) 
{
    QT_REQUIRE_VERSION( argc, argv, QT_VERSION_STR );

    GTIMER(c1, t1, "main()->QApp::exec");

    QApplication app(argc, argv);
    QCoreApplication::addLibraryPath(QCoreApplication::applicationDirPath());

// add some extra paths used during development
#ifdef Q_OS_WIN
#ifdef _DEBUG
    QString devPluginsPath = QDir(QCoreApplication::applicationDirPath()+"/../../extras/windows/dotnet_style/_debug").absolutePath();
#else 
    QString devPluginsPath = QDir(QCoreApplication::applicationDirPath()+"/../../extras/windows/dotnet_style/_release").absolutePath();
#endif
    QCoreApplication::addLibraryPath(devPluginsPath); //dev version
#endif

    setSearchPaths();

    AppContextImpl* appContext = AppContextImpl::getApplicationContext();

    // parse all cmdline arguments
    CMDLineRegistry* cmdLineRegistry = new CMDLineRegistry(app.arguments()); 
    appContext->setCMDLineRegistry(cmdLineRegistry);
    
    //1 create settings
    SettingsImpl* globalSettings = new SettingsImpl(QSettings::SystemScope);
    appContext->setGlobalSettings(globalSettings);

    SettingsImpl* settings = new SettingsImpl(QSettings::UserScope);
    settings->setValue(GLOBAL_SETTINGS + "drawIcons", true);
    appContext->setSettings(settings);

    AppSettings* appSettings = new AppSettingsImpl();
    appContext->setAppSettings(appSettings);

    UserAppsSettings* userAppSettings = AppContext::getAppSettings()->getUserAppsSettings();
    
    // set translations
    QTranslator translator;
    QString transFile[] = {
        userAppSettings->getTranslationFile(),
        "transl_" + QLocale::system().name(),
        "transl_en"
    };
    bool trOK = false;
    for (int i = transFile[0].isEmpty() ? 1 : 0; i < 3; ++i) {
        if (!translator.load(transFile[i], QCoreApplication::applicationDirPath())) {
            fprintf(stderr, "Translation not found: %s\n", transFile[i].toAscii().constData());
        } else {
            trOK = true;
            break;
        }
    }
    if (!trOK) {
        fprintf(stderr, "No translations found, exiting\n");
        return 1;   
    }
    
    app.installTranslator(&translator);
    updateStaticTranslations();
    
    // 3 create functional components of ugene
    QStringList envList = QProcess::systemEnvironment();

    LogCache logsCache;
#ifdef _DEBUG
    logsCache.print_to_console=true;
#else
    logsCache.print_to_console = envList.indexOf("UGENE_PRINT_TO_CONSOLE=true") >= 0 || envList.indexOf("UGENE_PRINT_TO_CONSOLE=1") >= 0;
#endif

    logCat.details(AppContextImpl::tr("UGENE initialization started"));

#if defined(Q_WS_WIN) || defined(Q_WS_MAC)
    qt_use_native_dialogs = envList.indexOf("UGENE_USE_NATIVE_DIALOGS=false") < 0 && envList.indexOf("UGENE_USE_NATIVE_DIALOGS=0") < 0;
#endif

    QString style = userAppSettings->getVisualStyle();
    if (!style.isEmpty()) {
        QStyle* qstyle = QStyleFactory::create(style);
        if (qstyle!=NULL) {
            app.setStyle(qstyle);
        } else {
            logCat.details(AppContextImpl::tr("Style not available %1").arg(style));
        }
    } 
    
    ResourceTracker* resTrack = new ResourceTracker();
    appContext->setResourceTracker(resTrack);

    TaskSchedulerImpl* ts = new TaskSchedulerImpl(appSettings->getAppResourcePool());
    appContext->setTaskScheduler(ts);

    AnnotationSettingsRegistry* asr = new AnnotationSettingsRegistry();
    appContext->setAnnotationSettingsRegistry(asr);

    TestFramework* tf = new TestFramework();
    appContext->setTestFramework(tf);

    GTestFormatRegistry* tfr = AppContext::getTestFramework()->getTestFormatRegistry();
    XMLTestFormat *xmlTestFormat = qobject_cast<XMLTestFormat*>(tfr->findFormat("XML"));
    QList<XMLTestFactory*> fs = ProjectTests::createTestFactories();
    foreach(XMLTestFactory* f, fs) { 
        bool res = xmlTestFormat->registerTestFactory(f);
        assert(res); Q_UNUSED(res);
    }

    MainWindowImpl* mw = new MainWindowImpl();
    appContext->setMainWindow(mw);
    mw->show();

    AppSettingsGUI* appSettingsGUI = new AppSettingsGUIImpl();
    appContext->setAppSettingsGUI(appSettingsGUI);

    AppContext::getMainWindow()->getDockManager()->registerDock(MWDockArea_Bottom, new TaskViewDockWidget(), QKeySequence(Qt::ALT | Qt::Key_2));
    AppContext::getMainWindow()->getDockManager()->registerDock(MWDockArea_Bottom, new LogViewDockWidget(&logsCache), QKeySequence(Qt::ALT | Qt::Key_3));
    
    GObjectViewFactoryRegistry* ovfr = new GObjectViewFactoryRegistry();
    appContext->setObjectViewFactoryRegistry(ovfr);

    PluginSupportImpl* psp = new PluginSupportImpl();
    appContext->setPluginSupport(psp);

    ServiceRegistryImpl* sreg = new ServiceRegistryImpl() ;
    appContext->setServiceRegistry(sreg);

    DocumentFormatRegistryImpl* dfr = new DocumentFormatRegistryImpl();
    appContext->setDocumentFormatRegistry(dfr);

    DocumentFormatConfigurators* dfc = new DocumentFormatConfigurators();
    appContext->setDocumentFormatConfigurators(dfc);
    BaseDocumentFormatConfigurators::initBuiltInConfigurators();
    
    IOAdapterRegistryImpl* io = new IOAdapterRegistryImpl();
    appContext->setIOAdapterRegistry(io);

    DNATranslationRegistry* dtr = new DNATranslationRegistry();
    appContext->setDNATranslationRegistry(dtr);

    DNAAlphabetRegistry* dal = new DNAAlphabetRegistryImpl(dtr);
    appContext->setDNAAlphabetRegistry(dal);

    ProjectLoaderImpl* pli = new ProjectLoaderImpl();
    appContext->setProjectLoader(pli);

    DBXRefRegistry* dbxr = new DBXRefRegistry();
    appContext->setDBXRefRegistry(dbxr);

    MSAColorSchemeRegistry* mcsr = new MSAColorSchemeRegistry();
    appContext->setMSAColorSchemeRegistry(mcsr);

    MSAConsensusAlgorithmRegistry* msaConsReg = new MSAConsensusAlgorithmRegistry();
    appContext->setMSAConsensusAlgorithmRegistry(msaConsReg);

    PWMConversionAlgorithmRegistry* pwmConvReg = new PWMConversionAlgorithmRegistry();
    appContext->setPWMConversionAlgorithmRegistry(pwmConvReg);

    SubstMatrixRegistry* smr = new SubstMatrixRegistry();
    appContext->setSubstMatrixRegistry(smr);

    SmithWatermanTaskFactoryRegistry* swar = new SmithWatermanTaskFactoryRegistry();
    appContext->setSmithWatermanTaskFactoryRegistry(swar);

    RepeatFinderTaskFactoryRegistry* rfr = new RepeatFinderTaskFactoryRegistry();
    appContext->setRepeatFinderTaskFactoryRegistry(rfr);

    CMDLineUtils::init();
    
    PhyTreeGeneratorRegistry* genRegistry = new PhyTreeGeneratorRegistry();
    appContext->setPhyTreeGeneratorRegistry(genRegistry);

    MolecularSurfaceFactoryRegistry* msfr = new MolecularSurfaceFactoryRegistry();
    appContext->setMolecularSurfaceFactoryRegistry(msfr);

    SWResultFilterRegistry* swrfr = new SWResultFilterRegistry();
    appContext->setSWResultFilterRegistry(swrfr);

    SecStructPredictAlgRegistry* sspar = new SecStructPredictAlgRegistry();
    appContext->setSecStructPedictAlgRegistry(sspar);

    CudaGpuRegistry * cgr = new CudaGpuRegistry();
    appContext->setCudaGpuRegistry(cgr); 

    AtiStreamGpuRegistry * asgr = new AtiStreamGpuRegistry ();
    appContext->setAtiStreamGpuRegistry( asgr ); 

    RecentlyDownloadedCache* rdc = new RecentlyDownloadedCache();
    appContext->setRecentlyDownloadedCache(rdc);

    DistributedComputingUtil * dcu = new DistributedComputingUtil();
    
    VirtualFileSystemRegistry * vfsReg = new VirtualFileSystemRegistry();
    appContext->setVirtualFileSystemRegistry( vfsReg );
    
    Workflow::WorkflowEnv::init(new Workflow::WorkflowEnvImpl());
    Workflow::BioActorLibrary::init();
    
    UIndexSupport indexSupport;

    DnaAssemblyAlgRegistry* assemblyReg = new DnaAssemblyAlgRegistry();
    appContext->setDnaAssemblyAlgRegistry(assemblyReg);
    DnaAssemblySupport assemblySupport;

    DataBaseRegistry *dbr = new DataBaseRegistry();
    appContext->setDataBaseRegistry(dbr);

    QStringList urls = CMDLineRegistryUtils::getPureValues();
    if( !urls.isEmpty() ) {
        QList<GUrl> gurls;
        foreach( const QString & url, urls ) {
            gurls << GUrl(url);
        }
        Task * t = AppContext::getProjectLoader()->openProjectTask( gurls, false );
        // defer loading until all plugins/services loaded
        QObject::connect( AppContext::getPluginSupport(), SIGNAL( si_allStartUpPluginsLoaded() ), 
            new TaskStarter( t ), SLOT( registerTask() ) );
        
    } else if (AppContext::getAppSettings()->getUserAppsSettings()->openLastProjectAtStartup()) {
        QString lastProject = ProjectLoaderImpl::getLastProjectURL();
        if (!lastProject.isEmpty()) {
            Task* t = AppContext::getProjectLoader()->openProjectTask(lastProject, false);
            // defer project start until all plugins/services loaded
            QObject::connect(AppContext::getPluginSupport(), SIGNAL(si_allStartUpPluginsLoaded()), 
                new TaskStarter(t), SLOT(registerTask()));
        }
    }
    
    registerCoreServices();
    
    if ( !envList.contains(ENV_UGENE_DEV+QString("=1")) ) {
        Shtirlitz::wakeup();
    }

    //3 run QT GUI
    t1.stop();
    logCat.info(AppContextImpl::tr("UGENE started"));
    int rc = app.exec();

    //4 deallocate resources
    if ( !envList.contains(ENV_UGENE_DEV+QString("=1")) ) {
        Shtirlitz::saveGatheredInfo();
    }

    Workflow::WorkflowEnv::shutdown();
    
    delete dcu;
    
    appContext->setVirtualFileSystemRegistry( NULL );
    delete vfsReg;
    
    appContext->setRecentlyDownloadedCache(NULL);
    delete rdc;

    appContext->setSWResultFilterRegistry(NULL);
    delete swrfr;

    appContext->setSmithWatermanTaskFactoryRegistry(NULL);
    delete swar;

    appContext->setMolecularSurfaceFactoryRegistry(NULL);
    delete msfr;

    appContext->setSubstMatrixRegistry(NULL);
    delete smr;

    appContext->setMSAColorSchemeRegistry(NULL);
    delete mcsr;

    appContext->setDBXRefRegistry(NULL);
    delete dbxr;

    appContext->setProjectLoader(NULL);
    delete pli;

    appContext->setServiceRegistry(NULL);
    delete sreg;

    appContext->setPluginSupport(NULL);
    delete psp;

    appContext->setMainWindow(NULL);
    delete mw;

    appContext->setTestFramework(0);
    delete tf;

    appContext->setObjectViewFactoryRegistry(NULL);
    delete ovfr;

    appContext->setDNAAlphabetRegistry(NULL);
    delete dal;

    appContext->setDNATranslationRegistry(NULL);
    delete dtr;

    appContext->setIOAdapterRegistry(NULL);
    delete io;

    appContext->setDocumentFormatConfigurators(NULL);
    delete dfc;

    appContext->setDocumentFormatRegistry(NULL);
    delete dfr;

    appContext->setTaskScheduler(NULL);
    delete ts;

    appContext->setAnnotationSettingsRegistry(NULL);
    delete asr;

    appContext->setResourceTracker(NULL);
    delete resTrack;

    appContext->setCudaGpuRegistry(NULL);
    delete cgr;

    appContext->setAtiStreamGpuRegistry(NULL);
    delete asgr;

    appContext->setAppSettingsGUI(NULL);
    delete appSettingsGUI;

    appContext->setAppSettings(NULL);
    delete appSettings;

    appContext->setSettings(NULL);
    delete settings;

    appContext->setGlobalSettings(NULL);
    delete globalSettings;
    
    delete cmdLineRegistry;
    appContext->setCMDLineRegistry(NULL);

    appContext->setSecStructPedictAlgRegistry(NULL);
    delete sspar;

    appContext->setMSAConsensusAlgorithmRegistry(NULL);
    delete msaConsReg;

    appContext->setPWMConversionAlgorithmRegistry(NULL);
    delete pwmConvReg;

    appContext->setDnaAssemblyAlgRegistry(NULL);
    delete assemblyReg;
    
    appContext->setDataBaseRegistry(NULL);
    delete dbr;

    return rc;
}

