/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "LogSettingsGUIController.h"

#include "../../logview/LogView.h"

#include <core_api/AppContext.h>
#include <core_api/AppSettings.h>

#include <QtGui/QHeaderView>
#include <QtGui/QToolButton>
#include <QtGui/QColorDialog>

namespace GB2 {

LogSettingsPageState::LogSettingsPageState()
{
}

LogSettingsPageController::LogSettingsPageController(LogViewDockWidget* _view, QObject* p) 
: AppSettingsGUIPageController(tr("log_settings"), APP_SETTINGS_GUI_LOG, p), view(_view)
{
}


AppSettingsGUIPageState* LogSettingsPageController::getSavedState() {
    LogSettingsPageState* state = new LogSettingsPageState();
    state->settings = view->getSettings();
    return state;
}

void LogSettingsPageController::saveState(AppSettingsGUIPageState* s) {
    LogSettingsPageState* state = qobject_cast<LogSettingsPageState*>(s);
    view->setSettings(state->settings);
}

AppSettingsGUIPageWidget* LogSettingsPageController::createWidget(AppSettingsGUIPageState* data) {
    LogSettingsPageWidget* w = new LogSettingsPageWidget();
    w->setState(data);
    return w;
}

//////////////////////////////////////////////////////////////////////////
// widget

LogSettingsPageWidget::LogSettingsPageWidget() {
    setupUi( this );
    tableWidget->verticalHeader()->setVisible(false);
    connect(tableWidget, SIGNAL(currentCellChanged(int, int, int, int)), SLOT(sl_currentCellChanged(int, int, int, int)));
}

void LogSettingsPageWidget::setState(AppSettingsGUIPageState* s) {
    LogSettingsPageState* state = qobject_cast<LogSettingsPageState*>(s);
    LogSettings& settings = state->settings;
    
    tableWidget->clearContents();
    
    tableWidget->setRowCount(1 + settings.getLogCategorySettings().size());
    
    // set header
    tableWidget->setHorizontalHeaderItem(0, new QTableWidgetItem(tr("Category")));
    for (int i=0; i< LogLevel_NumLevels; i++) {
        QString catName = LogCategories::getLocalizedLevelName((LogLevel) i);
        tableWidget->setHorizontalHeaderItem(i + 1, new QTableWidgetItem(catName) );
    }

    // create global line
    QTableWidgetItem* item00 = new QTableWidgetItem(tr("<<all>>"));
    tableWidget->setItem(0, 0, item00);
    for (int i=0; i<LogLevel_NumLevels; i++) {
        QTableWidgetItem* levelItem = new QTableWidgetItem();
        tableWidget->setItem(0, i+1, levelItem);

        LogSettingsTopLineWidget* tw = new LogSettingsTopLineWidget(this, settings.levelColors[i], LogLevel(i));
        QHBoxLayout* l = new QHBoxLayout();

        int marginLeft = 6; //TODO: align with usual setCheckState boxes
        l->setContentsMargins(marginLeft, 1, 10, 1);
        l->setSpacing(0);
        
        tw->cb = new QCheckBox(tw);
        tw->cb->setChecked(settings.activeLevelGlobalFlag[i]);
        connect(tw->cb, SIGNAL(stateChanged (int)), SLOT(sl_levelStateChanged(int)));
        l->addWidget(tw->cb);

        QLabel* label = new QLabel(tw);
        updateColorLabel(label, settings.levelColors[i]);
        connect(label, SIGNAL(linkActivated(const QString&)), SLOT(sl_changeColor(const QString&)));
        l->addWidget(label);

        tw->setLayout(l);
        tableWidget->setCellWidget(0, i+1, tw);
    }

    // create category lines
    int row=1;
    QList<QString> categoryNames = settings.getLogCategorySettings().keys();
    qSort(categoryNames);
    foreach(const QString& categoryName, categoryNames) {
        const LogCategorySettings& cs = settings.getLogCategorySettings(categoryName);
        QTableWidgetItem* nameItem = new QTableWidgetItem(cs.categoryName);        
        tableWidget->setItem(row, 0, nameItem);
        for (int i=0; i<LogLevel_NumLevels; i++) {
            QTableWidgetItem* catItem = new QTableWidgetItem();
            catItem->setFlags(Qt::ItemIsUserCheckable | (settings.activeLevelGlobalFlag[i] ? Qt::ItemIsEnabled : Qt::NoItemFlags));
            catItem->setCheckState(cs.activeLevelFlag[i] ? Qt::Checked : Qt::Unchecked);
            tableWidget->setItem(row, i+1, catItem);
        }
        row++;
    }
    tableWidget->resizeRowsToContents();

    showCategoryCB->setChecked(settings.showCategory);
    showDateCB->setChecked(settings.showDate);
    showLevelCB->setChecked(settings.showLevel);
}

AppSettingsGUIPageState* LogSettingsPageWidget::getState(QString& err) const {
    Q_UNUSED(err);
    LogSettingsPageState* state = new LogSettingsPageState();
    LogSettings& settings = state->settings;
    
    //process global settings
    for (int i = 0; i < LogLevel_NumLevels; i++) {
        QWidget* w = tableWidget->cellWidget(0, i+1);
        LogSettingsTopLineWidget* tw = qobject_cast<LogSettingsTopLineWidget*>(w);
        settings.levelColors[i] = tw->color;
        settings.activeLevelGlobalFlag[i] = tw->cb->checkState() == Qt::Checked;
    }

    //process per-category settings
    for (int row=1 ; row < tableWidget->rowCount(); row++) {
        LogCategorySettings logCat;
        logCat.categoryName  = tableWidget->item(row, 0)->text();
        for (int i = 0; i < LogLevel_NumLevels; i++) {
            logCat.activeLevelFlag[i]= tableWidget->item(row, i+1)->checkState() == Qt::Checked;
        }
        settings.addCategory(logCat);
    }

    settings.showCategory = showCategoryCB->isChecked();
    settings.showDate = showDateCB->isChecked();
    settings.showLevel = showLevelCB->isChecked();
    
    return state;
}

void LogSettingsPageWidget::sl_currentCellChanged ( int currentRow, int currentColumn, int previousRow, int previousColumn) {
    Q_UNUSED(currentRow); Q_UNUSED(currentColumn); Q_UNUSED(previousRow); Q_UNUSED(previousColumn);
    tableWidget->setCurrentCell(0, 0, QItemSelectionModel::NoUpdate);
}

void LogSettingsPageWidget::sl_levelStateChanged(int state) {
    bool checked = state == Qt::Checked;
    QCheckBox* cb = qobject_cast<QCheckBox*>(sender());
    int column = 1 + qobject_cast<LogSettingsTopLineWidget*>(cb->parent())->level;
    assert(column >= 1 && column <=LogLevel_NumLevels);
    int nEqual = 0;
    for (int row = 1; row < tableWidget->rowCount(); row++) {
        QTableWidgetItem* catItem = tableWidget->item(row, column);
        assert(catItem!=NULL);
        catItem->setFlags(Qt::ItemIsUserCheckable | (checked ? Qt::ItemIsEnabled : Qt::NoItemFlags));   
        nEqual+=catItem->checkState() == cb->checkState() ? 1 : 0;
    }
    if (nEqual == 0) {
        for (int row = 1; row < tableWidget->rowCount(); row++) {
            QTableWidgetItem* catItem = tableWidget->item(row, column);
            catItem->setCheckState(cb->checkState());
        }
    }
}

void LogSettingsPageWidget::sl_changeColor(const QString& v) {
    Q_UNUSED(v);
    QLabel* label = qobject_cast<QLabel*>(sender());
    LogSettingsTopLineWidget* tw = qobject_cast<LogSettingsTopLineWidget*>(label->parent());
    QColor color = QColorDialog::getColor(tw->color, this);
    if (color.isValid()) {
        tw->color = color.name();
        updateColorLabel(label, tw->color);
    }
}

void LogSettingsPageWidget::updateColorLabel(QLabel* l, const QString& color) {
    QString style = "{color: "+color+"; text-decoration: none;};";
    l->setText("<style> a "+style+" a:visited "+style+" a:hover "+style+"</style>"
                + "<a href=\".\">" + tr("Sample text") + "</a>");
}

} //namespace
