/* Drawing.

   Copyright (C) 2000 Daiki Ueno <ueno@unixuser.org>

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 2000-05-16

   This file is part of UltraPoint.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include "canvas.h"

static const guint16 canvas_font_sizes[] = {
  72, 64, 56, 48, 40, 36, 32, 28, 26, 24, 22, 20, 18, 16, 14, 13,
  12, 11, 10, 9, 8
};

typedef struct _CanvasContext CanvasContext;

struct _CanvasContext {
  UptCanvas **tab;
  gint num;
  gint max;

  gint ref_count;
  GFreeFunc free;
};

static CanvasContext *canvas_context_new (gint max);
static CanvasContext *canvas_context = NULL;

static void canvas_context_push (CanvasContext *context, UptCanvas *canvas);
static UptCanvas *canvas_context_pop (CanvasContext *context);

static void canvas_set_aa_map (UptCanvas *canvas);
static void canvas_free (UptCanvas *canvas);
static void canvas_adjust_halign (UptCanvas *canvas, glong width);
static gint canvas_get_bg_pixel (UptCanvas *canvas, glong x, glong y,
				 ggi_pixel *pixel);

static void
canvas_context_push (context, canvas)
     CanvasContext *context;
     UptCanvas *canvas;
{
  if (context->num >= context->max)
    {
      context->max *= 2;
      context->tab =
	g_realloc (context->tab, context->max * sizeof(context->tab[0]));
      g_assert (context->tab != NULL);
    }
  context->tab[context->num] = canvas;
  upt_object_ref (canvas);

  context->num++;
}

static UptCanvas *
canvas_context_pop (context)
     CanvasContext *context;
{
  UptCanvas *canvas;

  g_assert (context->num > 0);
  canvas = context->tab[--context->num];
  upt_object_unref (canvas);

  return canvas;
}

static void
canvas_set_aa_map (canvas)
     UptCanvas *canvas;
{
  static const gdouble factor = 0.25;
  ggi_color col_fg, col_bg, aa_col;
  gint i;

  ggiUnmapPixel (canvas->visual, canvas->fg, &col_fg);
  ggiUnmapPixel (canvas->visual, canvas->bg, &col_bg);

  for (i = 0; i < 5; i++)
    {
      aa_col.r = col_bg.r + (col_fg.r - col_bg.r) * factor * i;
      aa_col.g = col_bg.g + (col_fg.g - col_bg.g) * factor * i;
      aa_col.b = col_bg.b + (col_fg.b - col_bg.b) * factor * i;
      canvas->aa_map[i] = ggiMapColor (canvas->visual, &aa_col);
    }
}  

static void
canvas_free (canvas)
     UptCanvas *canvas;
{
  upt_object_unref (canvas->bg_image);
  if (canvas->aa_map)
    g_free (canvas->aa_map);
  g_free (canvas);
}

UptCanvas *
upt_canvas_new (visual)
     ggi_visual_t visual;
{
  UptCanvas *canvas;

  canvas = g_new0 (UptCanvas, 1);
  canvas->visual = visual;
  ggiGetMode (visual, &canvas->mode);

  ggiGetGCForeground (visual, &canvas->fg);
  ggiGetGCBackground (visual, &canvas->bg);

  canvas->psize = (GT_SIZE (canvas->mode.graphtype) + 7) / 8;
  canvas->font_size_base = 
    MAX (0, - (canvas->mode.virt.x / 200) + 4);
  canvas->aa_map = g_malloc0 (sizeof(ggi_pixel) * 8);
  canvas_set_aa_map (canvas);
  canvas->free = (GFreeFunc)canvas_free;

  return canvas;
}

CanvasContext *
canvas_context_new (max)
     gint max;
{
  CanvasContext *context;

  context = g_new0 (CanvasContext, 1);
  context->tab = g_malloc0 (sizeof(UptCanvas) * max);
  context->max = max;
  context->free = g_free;

  return context;
}

void
upt_canvas_move (canvas, off_x, off_y)
     UptCanvas *canvas;
     gint off_x, off_y;
{
  canvas->off_x += off_x;
  canvas->off_y += off_y;
}

void
upt_canvas_set_position (canvas, off_x, off_y)
     UptCanvas *canvas;
     gint off_x, off_y;
{
  canvas->off_x = off_x;
  canvas->off_y = off_y;
}


static void
canvas_adjust_halign (canvas, width)
     UptCanvas *canvas;
     glong width;
{
  switch (canvas->align)
    {
    case ALIGN_LEFT:
      upt_canvas_set_position (canvas, 0, canvas->off_y);
      break;
    case ALIGN_CENTER:
      upt_canvas_set_position (canvas, (canvas->mode.virt.x - width) / 2,
			       canvas->off_y);
      break;
    case ALIGN_RIGHT:
      upt_canvas_set_position (canvas, canvas->mode.virt.x - width,
			       canvas->off_y);
      break;
    }
}

static gint
canvas_get_bg_pixel (canvas, x, y, pixel)
     UptCanvas *canvas;
     glong x, y;
     ggi_pixel *pixel;
{
  glong bg_width, bg_height;
  x = canvas->off_x + x;
  y = canvas->off_y + y;

  if (canvas->use_multiple_frames)
    bg_width = canvas->mode.virt.x,
      bg_height = canvas->mode.virt.y;
  else
    bg_width = canvas->bg_image->width,
      bg_height = canvas->bg_image->height;
    
  if (x > bg_width || y > bg_height)
    return -1;

  if (canvas->use_multiple_frames)
    ggiGetPixel (canvas->visual, x, y, pixel);
  else
    memcpy (pixel,
	    &canvas->bg_image->pbuf[(canvas->bg_image->width * y + x)
				   * canvas->psize],
	    canvas->psize);

  return 0;
}

gint
upt_canvas_get_aa_pixel (canvas, x, y, n_bits, pixel)
     UptCanvas *canvas;
     glong x, y;
     gint n_bits;
     ggi_pixel *pixel;
{
  ggi_color col_fg, col_bg, aa_col;
  ggi_pixel bg;

  if (!canvas->has_background_image)
    {
      *pixel = canvas->aa_map[n_bits];
      return 0;
    }

  if (canvas_get_bg_pixel (canvas, x, y, &bg) < 0)
    return -1;

  if (!n_bits)
    {
      *pixel = bg;
      return 0;
    }

  if (n_bits == 4)
    {
      *pixel = canvas->fg;
      return 0;
    }

  ggiUnmapPixel (canvas->visual, canvas->fg, &col_fg);
  ggiUnmapPixel (canvas->visual, bg, &col_bg);

  aa_col.r = col_bg.r + (col_fg.r - col_bg.r) * n_bits / 4.0;
  aa_col.g = col_bg.g + (col_fg.g - col_bg.g) * n_bits / 4.0;
  aa_col.b = col_bg.b + (col_fg.b - col_bg.b) * n_bits / 4.0;

  *pixel = ggiMapColor (canvas->visual, &aa_col);

  return 0;
}

void
upt_canvas_draw_page (canvas, page)
     UptCanvas *canvas;
     UptPage *page;
{
  upt_page_get_metric (page, &page->width, &page->height);
  canvas_adjust_halign (canvas, page->width);
  upt_page_instantiate (canvas, page);
  ggiPutBox (canvas->visual, canvas->off_x, canvas->off_y,
	     page->width, page->height, page->pbuf);
  upt_canvas_move (canvas, page->width, page->height);
}

void
upt_canvas_draw_hline (canvas, width, thickness)
     UptCanvas *canvas;
     gint width, thickness;
{
  gint i;

  for (i = 0; i<thickness; i++)
    ggiDrawHLine (canvas->visual, canvas->off_x, canvas->off_y + i, width);
  upt_canvas_move (canvas, width, i);
}

void
upt_canvas_set_align (canvas, align)
     UptCanvas *canvas;
     gint align;
{
  canvas->align = align;
}


void
upt_canvas_set_foreground (canvas, fg)
     UptCanvas *canvas;
     ggi_pixel fg;
{
  ggiSetGCForeground (canvas->visual, fg);
  canvas->fg = fg;
  if (!canvas->has_background_image)
    canvas_set_aa_map (canvas);
}

void
upt_canvas_set_foreground_rgb (canvas, r, g, b)
     UptCanvas *canvas;
     guint16 r, g, b;
{
  static ggi_color col;
  ggi_pixel fg;

  col.r = r;
  col.g = g;
  col.b = b;

  fg = ggiMapColor (canvas->visual, &col);
  upt_canvas_set_foreground (canvas, fg);
}

void
upt_canvas_set_undo_boundary (canvas)
     UptCanvas *canvas;
{
  UptCanvas *saved_canvas;

  if (!canvas_context)
    canvas_context = canvas_context_new (16);

  saved_canvas = g_memdup (canvas, sizeof(UptCanvas));
  canvas_context_push (canvas_context, saved_canvas);
}

void
upt_canvas_undo_horizontal (canvas)
     UptCanvas *canvas;
{
  UptCanvas *saved_canvas;

  saved_canvas = canvas_context_pop (canvas_context);
  upt_canvas_set_position (canvas, saved_canvas->off_x, canvas->off_y);
  upt_canvas_set_foreground (canvas, saved_canvas->fg);
  upt_canvas_set_align (canvas, saved_canvas->align);
}

void
upt_canvas_undo_vertical (canvas)
     UptCanvas *canvas;
{
  UptCanvas *saved_canvas;

  saved_canvas = canvas_context_pop (canvas_context);
  upt_canvas_set_position (canvas, canvas->off_x, saved_canvas->off_y);
  upt_canvas_set_foreground (canvas, saved_canvas->fg);
}

void
upt_canvas_clear (canvas)
     UptCanvas *canvas;
{
  if (canvas->has_background_image)
    if (canvas->use_multiple_frames)
      ggiCopyBox (canvas->visual, 0, 0,
		  canvas->mode.virt.x, canvas->mode.virt.y, 0, 0);
    else
      ggiPutBox (canvas->visual, 0, 0,
		 canvas->bg_image->width, canvas->bg_image->height,
		 canvas->bg_image->pbuf);
  else
    {
      ggiSetGCForeground (canvas->visual, canvas->bg);
      ggiFillscreen (canvas->visual);
      ggiSetGCForeground (canvas->visual, canvas->fg);
    }
  upt_canvas_set_align (canvas, ALIGN_DEFAULT);
  upt_canvas_set_position (canvas, 0, 0);
}

void
upt_canvas_flush (canvas)
     UptCanvas *canvas;
{
  ggiFlush (canvas->visual);
}

void
upt_canvas_draw_circle (canvas, radius)
     UptCanvas *canvas;
     gint radius;
{
  gint center_x = canvas->off_x + radius,
    center_y = canvas->off_y + radius;
  gint x, y, od, md, sd;

  x = radius / 2;
  y = md = 0;
  while (x >= y)
    {
      ggiDrawHLine (canvas->visual, center_x - y, center_y - x, y * 2);
      ggiDrawHLine (canvas->visual, center_x - x, center_y - y, x * 2);
      ggiDrawHLine (canvas->visual, center_x - y, center_y + x, y * 2);
      ggiDrawHLine (canvas->visual, center_x - x, center_y + y, x * 2);
      od = md + y + y + 1;
      sd = od - x - x - 1;
      y++;
      md = od;
      if (abs (sd) < abs (od))
	{
	  x--;
	  md = sd;
	}
    }

  upt_canvas_move (canvas, radius * 2, radius * 2);
}

guint16
upt_canvas_get_font_size (canvas, size_index)
     UptCanvas *canvas;
     gint size_index;
{
  size_index =
    MIN(canvas->font_size_base+size_index,
	G_N_ELEMENTS (canvas_font_sizes) - 1);

  return canvas_font_sizes[size_index];
}

#ifdef HAVE_LIBMAGICK
static void
image_free (image)
     UptImage *image;
{
  if (image->pbuf)
    g_free (image->pbuf);
  g_free (image);
}

UptImage *
upt_image_new (void)
{
  UptImage *image;

  image = g_malloc0 (sizeof(UptImage));
  image->free = image_free;

  return image;
}

UptImage *
upt_canvas_load_image (canvas, filename, width, height, mag_x, mag_y)
     UptCanvas *canvas;
     const gchar *filename;
     gint width, height;
     gdouble mag_x, mag_y;
{
  guint8 *pbuf;
  UptImage *upt_image;
  Image *image;
  ImageInfo image_info;
  gint i, j;
#if MagickLibVersion < 0x0520
  gint image_length;
  RunlengthPacket *image_data;
#else
  PixelPacket *image_data;
  ExceptionInfo exception;
  ViewInfo *image_view;

  GetExceptionInfo (&exception);
#endif
  GetImageInfo (&image_info);
  strncpy(image_info.filename, filename, MaxTextExtent);
  image_info.dither = 0;
  image_info.colorspace = RGBColorspace;

#if MagickLibVersion < 0x0520
  image = ReadImage(&image_info);
#else
  image = ReadImage(&image_info, &exception);
#endif
  if(!image)
    {
      g_warning (_("Cannot read image (%s)."), filename);
      return NULL;
    }

  if (width == -1)
    width = image->columns;
  if (height == -1)
    height = image->rows;
  if (mag_x != -1)
    width *= mag_x;
  if (mag_y != -1)
    height *= mag_y;

  if (width != image->columns || height != image->rows)
#if MagickLibVersion < 0x0520
    image = ScaleImage (image, width, height);
#else
    image = ScaleImage (image, width, height, &exception);
#endif
  if(!image)
    return NULL;

  pbuf = g_malloc0 (image->columns * image->rows * canvas->psize);
  g_assert (pbuf != NULL);
#if  MagickLibVersion < 0x0520
  image_length = image->columns * image->rows;
  image_data = image->pixels;
  for (i = 0; i < image_length; i++, image_data++)
    {
      ggi_color col;
      ggi_pixel fg;

      col.r = (image_data->red / 4) * 0x0410;
      col.g = (image_data->green / 4) * 0x0410;
      col.b = (image_data->blue / 4) * 0x0410;

      fg = ggiMapColor (canvas->visual, &col);
      for (j = 0; j < image_data->length; j++, i++)
	memcpy (&pbuf[i*canvas->psize], &fg, canvas->psize);
      memcpy (&pbuf[i*canvas->psize], &fg, canvas->psize);
    }
#else
  image_view = OpenCacheView (image);
  for (i = 0; i < image->rows; i++)
    {
      image_data = GetCacheView (image_view, 0, i, image->columns, 1);
      if (!image_data)
	return NULL;
      for (j = 0; j < image->columns; j++)
	{
	  ggi_color col;
	  ggi_pixel fg;

	  col.r = (image_data->red / 4) * 0x0410;
	  col.g = (image_data->green / 4) * 0x0410;
	  col.b = (image_data->blue / 4) * 0x0410;

	  fg = ggiMapColor (canvas->visual, &col);
	  memcpy (&pbuf[(image->columns*i+j)*canvas->psize],
		  &fg, canvas->psize);
	  image_data++;
	}
    }
#endif
  upt_image = upt_image_new ();
  upt_image->pbuf = pbuf;
  upt_image->width = image->columns;
  upt_image->height = image->rows;
  DestroyImage(image);

  return upt_image;
}

gboolean
upt_canvas_draw_image (canvas, filename, width, height, mag_x, mag_y)
     UptCanvas *canvas;
     const gchar *filename;
     gint width, height;
     gdouble mag_x, mag_y;
{
  UptImage *image;

  image = upt_canvas_load_image (canvas, filename,
				 width, height, mag_x, mag_y);
  if (!image)
    return FALSE;
  canvas_adjust_halign (canvas, image->width);
  ggiPutBox (canvas->visual, canvas->off_x, canvas->off_y,
	     image->width, image->height, image->pbuf);
  upt_canvas_move (canvas, image->width, image->height);
  upt_object_unref (image);

  return TRUE;
}

void
upt_canvas_load_bg_image (canvas, filename)
     UptCanvas *canvas;
     const gchar *filename;
{
  canvas->bg_image =
    upt_canvas_load_image (canvas, filename,
			   canvas->mode.virt.x, canvas->mode.virt.y,
			   -1, -1);
  if (!canvas->bg_image)
    return;

  canvas->has_background_image = TRUE;
  if (canvas->use_multiple_frames)
    {
      ggi_pixel bg, fg;
      sint32 frames;

      ggiGetGCBackground (canvas->visual, &bg);
      ggiGetGCForeground (canvas->visual, &fg);

      frames = canvas->mode.frames;
      if (frames < 2)
	canvas->mode.frames = 2;

      if (ggiSetMode (canvas->visual, &canvas->mode) < 0)
	canvas->mode.frames = frames;

      if (canvas->mode.frames < 2)
	{
	  canvas->use_multiple_frames = FALSE;
	  return;
	}

      ggiSetWriteFrame (canvas->visual, 1);
      ggiPutBox (canvas->visual, 0, 0,
		 canvas->bg_image->width, canvas->bg_image->height,
		 canvas->bg_image->pbuf);
      upt_object_unref (canvas->bg_image);
      upt_canvas_flush (canvas);
      ggiSetWriteFrame (canvas->visual, 0);
      ggiSetReadFrame (canvas->visual, 1);

      ggiSetGCBackground (canvas->visual, bg);
      ggiSetGCForeground (canvas->visual, fg);
    }
}

#endif
