/* Slide.

   Copyright (C) 2000 Daiki Ueno <ueno@unixuser.org>

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 2000-05-16

   This file is part of UltraPoint.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include "slide.h"

gdouble slide_min_glue_width = 0.5;
gdouble slide_max_glue_width = 0.75;
gdouble slide_agglutinative_min_glue_width = 0.1;
gdouble slide_agglutinative_max_glue_width = 0.1;
gdouble slide_base_lineskip = 0.25;

ggi_pixel slide_header_foreground_pixel;
ggi_pixel slide_itemize_foreground_pixel;
ggi_pixel slide_border_foreground_pixel;

gint slide_border_thickness = 8;

#define SLIDE_BLOCK_MAX_DEPTH 5

enum {
  SLIDE_FSPEC_TITLE = 0,
  SLIDE_FSPEC_SUBTITLE = 1,
  SLIDE_FSPEC_AA = 2,
  SLIDE_FSPEC_AN = 3,
  SLIDE_FSPEC_DATE = 4,
  SLIDE_FSPEC_HEAD = 5
};

static const UptFont slide_font_specs[] = {
  { 0, CHARSET_ISO_10646, FAMILY_DEFAULT, FACE_BOLDITALIC },
  { 3, CHARSET_ISO_10646, FAMILY_DEFAULT, FACE_DEFAULT },
  { 6, CHARSET_ISO_10646, FAMILY_DEFAULT, FACE_DEFAULT },
  { 5, CHARSET_ISO_10646, FAMILY_DEFAULT, FACE_DEFAULT },
  { 8, CHARSET_ISO_10646, FAMILY_DEFAULT, FACE_DEFAULT },
  { 4, CHARSET_ISO_10646, FAMILY_DEFAULT, FACE_BOLD },
};

static UptFont *slide_font_new (UptCanvas *canvas, gint spec_index);
static void slide_draw_coverpage (UptCanvas *canvas, UptXMLSlideShow *ss);
static void slide_draw_para (UptCanvas *canvas, gchar *para, gint depth);

static void slide_draw_ul_listmark (UptCanvas *canvas,
				    GNode *node, gint depth);
static void slide_draw_ol_listmark (UptCanvas *canvas,
				    GNode *node, gint depth);

typedef void (*UptListmarkRenderer) (UptCanvas *, GNode *, gint);
UptListmarkRenderer listmark_renderer[] = {
  NULL, slide_draw_ul_listmark, slide_draw_ol_listmark,
  slide_draw_ol_listmark, slide_draw_ol_listmark, 
};

static gboolean slide_block_traverse_func (GNode *node, gpointer data);
static void slide_draw (UptCanvas *canvas, UptXMLSlide *slide);

static UptFont *
slide_font_new (canvas, spec_index)
     UptCanvas *canvas;
     gint spec_index;
{
  const UptFont *spec;
  UptFont *font;

  spec = &slide_font_specs[spec_index];
  font = upt_font_new (upt_canvas_get_font_size (canvas, spec->pixel_size),
		       spec->family_id, spec->face_id);
  return font;
}

static void
slide_draw_coverpage (canvas, ss)
     UptCanvas *canvas;
     UptXMLSlideShow *ss;
{
  UptGlue *base_glue;
  UptFont *base_font;
  UptPage *page;

  upt_canvas_clear (canvas);
  upt_canvas_set_align (canvas, ALIGN_CENTER);

  base_glue =
    upt_glue_new (0, 0, 0, slide_min_glue_width, slide_max_glue_width);

  if (ss->title)
    {
      base_font = slide_font_new (canvas, SLIDE_FSPEC_TITLE);
      page = upt_page_new (canvas->mode.virt.x, canvas->mode.virt.y,
			   base_glue, slide_base_lineskip,
			   base_font, !canvas->disable_antialias);
      upt_page_add_text (page, ss->title);

      upt_canvas_move (canvas, 0, canvas->mode.virt.y/4);
      upt_canvas_draw_page (canvas, page);
      upt_object_unref (base_font);
      upt_object_unref (page);
    }

  if (ss->author)
    {
      base_font = slide_font_new (canvas, SLIDE_FSPEC_AN);
      page = upt_page_new (canvas->mode.virt.x, canvas->mode.virt.y,
			   base_glue, slide_base_lineskip,
			   base_font, !canvas->disable_antialias);
      upt_page_add_text (page, ss->author);

      upt_canvas_move (canvas, 0, (canvas->mode.virt.y / 5) * 2);
      upt_canvas_draw_page (canvas, page);
      upt_object_unref (base_font);
      upt_object_unref (page);
    }

  if (ss->affiliation)
    {
      base_font = slide_font_new (canvas, SLIDE_FSPEC_AA);
      page = upt_page_new (canvas->mode.virt.x, canvas->mode.virt.y,
			   base_glue, slide_base_lineskip,
			   base_font, !canvas->disable_antialias);
      upt_page_add_text (page, ss->affiliation);

      upt_canvas_move (canvas, 0, canvas->mode.virt.y / 24);
      upt_canvas_draw_page (canvas, page);
      upt_object_unref (base_font);
      upt_object_unref (page);
    }

  base_font = slide_font_new (canvas, SLIDE_FSPEC_DATE);
  page = upt_page_new (canvas->mode.virt.x, canvas->mode.virt.y,
		       base_glue, slide_base_lineskip,
		       base_font, !canvas->disable_antialias);
  {
    GDate *date;
    gchar date_string[32];

    date = g_date_new ();
    g_date_set_time (date, time(NULL));
    
    g_date_strftime (date_string, sizeof(date_string)-1,
		     "%D", date);
    upt_page_add_text (page, date_string);
  }

  upt_canvas_move (canvas, 0, canvas->mode.virt.y / 32);
  upt_canvas_draw_page (canvas, page);
  upt_object_unref (base_font);
  upt_object_unref (page);
  upt_canvas_flush (canvas);
}

static void
slide_draw_para (canvas, para, depth)
     UptCanvas *canvas;
     gchar *para;
     gint depth;
{
  UptGlue *base_glue;
  UptFont *base_font;
  UptPage *page;

  base_glue =
    upt_glue_new (0, 0, 0, slide_min_glue_width, slide_max_glue_width);
  base_font =
    upt_font_new (upt_canvas_get_font_size (canvas, depth),
		  FAMILY_DEFAULT, FACE_DEFAULT);
  page = upt_page_new (canvas->mode.virt.x - (canvas->off_x * 5 / 3),
		       canvas->mode.virt.y - canvas->off_y,
		       base_glue, slide_base_lineskip, base_font,
		       !canvas->disable_antialias);
  upt_page_add_text (page, para);

  upt_canvas_set_undo_boundary (canvas);
  upt_canvas_draw_page (canvas, page);
  upt_canvas_move (canvas, 0, canvas->mode.virt.y / 32);
  upt_canvas_undo_horizontal (canvas);

  upt_object_unref (base_font);
  upt_object_unref (page);
}

static void
slide_draw_ul_listmark (canvas, node, depth)
     UptCanvas *canvas;
     GNode *node;
     gint depth;
{
  guint16 font_size;

  font_size = upt_canvas_get_font_size (canvas, depth);
  upt_canvas_set_undo_boundary (canvas);
  upt_canvas_set_foreground (canvas, slide_itemize_foreground_pixel);
  upt_canvas_move (canvas, -font_size, 0);
  upt_canvas_draw_circle (canvas, font_size/2);
  upt_canvas_undo_vertical (canvas);
}

static void
slide_draw_ol_listmark (canvas, node, depth)
     UptCanvas *canvas;
     GNode *node;
     gint depth;
{
  UptGlue *base_glue;
  UptFont *base_font;
  UptPage *page;
  UptXMLTerm *term;
  gchar index[16];
  guint16 font_size;
  gint base;

  font_size = upt_canvas_get_font_size (canvas, depth);

  index[0] = '\0';
  base = g_node_child_position (node->parent->parent, node->parent);
  term = node->data;
  switch (term->list_type)
    {
    case LIST_ORDER_ARABIC:
      g_snprintf (index, sizeof(index), "(%d)", base + 1);
      break;
    case LIST_ORDER_UPPERALPHA:
      g_snprintf (index, sizeof(index), "(%c)", base + 0x41);
      break;
    case LIST_ORDER_LOWERALPHA:
      g_snprintf (index, sizeof(index), "(%c)", base + 0x61);
      break;
    }

  base_glue =
    upt_glue_new (0, 0, 0, slide_min_glue_width, slide_max_glue_width);
  base_font = upt_font_new (upt_canvas_get_font_size (canvas, depth),
			    FAMILY_DEFAULT, FACE_DEFAULT);
  page = upt_page_new (canvas->mode.virt.x, canvas->mode.virt.y,
		       base_glue, slide_base_lineskip, base_font,
		       !canvas->disable_antialias);
  upt_page_add_text (page, index);
  upt_canvas_set_undo_boundary (canvas);
  upt_canvas_move (canvas, -font_size/2*strlen(index), 0);
  upt_canvas_draw_page (canvas, page);
  upt_canvas_move (canvas, font_size/2, 0);
  upt_object_unref (base_font);
  upt_object_unref (page);
  upt_canvas_undo_vertical (canvas);
}

static gboolean
slide_block_traverse_func (node, data)
     GNode *node;
     gpointer data;
{
  gint depth;
  UptCanvas *canvas;
  UptXMLTerm *term;
  UptListmarkRenderer renderer;

  depth = (g_node_depth (node) / 2) % SLIDE_BLOCK_MAX_DEPTH;
  canvas = data;
  term = node->data;

  switch (term->type)
    {
    case TERM_TEXT:
      upt_canvas_set_undo_boundary (canvas);
      upt_canvas_move (canvas, canvas->mode.virt.x / 32 * depth, 0);

      depth += 3;
      renderer = listmark_renderer[term->list_type];
      if (renderer)
	(*renderer) (canvas, node, depth);

      slide_draw_para (canvas, term->data.text, depth);
      upt_canvas_undo_horizontal (canvas);
      break;
#ifdef HAVE_LIBMAGICK
    case TERM_IMAGE:
      upt_canvas_set_undo_boundary (canvas);
      upt_canvas_set_align (canvas, ALIGN_CENTER);
      upt_canvas_draw_image (canvas,
			     term->data.image.filename,
			     term->data.image.width,
			     term->data.image.height,
			     term->data.image.scale,
			     term->data.image.scale);
      upt_canvas_undo_horizontal (canvas);
      break;
#endif
    }
  return FALSE;
}

static void
slide_draw (canvas, slide)
     UptCanvas *canvas;
     UptXMLSlide *slide;
{
  UptGlue *base_glue;
  UptFont *base_font;
  UptPage *page;
  UptColor *color;

  upt_canvas_clear (canvas);
  upt_canvas_move (canvas, canvas->mode.virt.x/32, canvas->mode.virt.y/32);

  base_glue =
    upt_glue_new (0, 0, 0, slide_min_glue_width, slide_max_glue_width);
  base_font = slide_font_new (canvas, SLIDE_FSPEC_HEAD);
  page = upt_page_new (canvas->mode.virt.x, canvas->mode.virt.y,
		       base_glue, slide_base_lineskip, base_font,
		       !canvas->disable_antialias);
  upt_page_add_text (page, slide->head);

  upt_canvas_set_undo_boundary (canvas);
  upt_canvas_set_foreground (canvas, slide_header_foreground_pixel);
  upt_canvas_draw_page (canvas, page);

  upt_canvas_set_position (canvas, 0, canvas->off_y
			   + canvas->mode.virt.y / 64);
  upt_canvas_set_foreground (canvas, slide_border_foreground_pixel);
  upt_canvas_draw_hline (canvas, canvas->mode.virt.x,
			 slide_border_thickness / 2);

  upt_canvas_set_position (canvas, 0, canvas->off_y);
  color = upt_color_unmap (canvas->visual, slide_border_foreground_pixel);
  upt_color_to_hls (color);
  color->data.hls.lightness = MAX (0, color->data.hls.lightness - 0.3);
  upt_canvas_set_foreground (canvas, upt_color_map (canvas->visual, color));
  upt_canvas_draw_hline (canvas, canvas->mode.virt.x,
			 slide_border_thickness / 2);
  upt_canvas_undo_horizontal (canvas);

  upt_object_unref (base_font);
  upt_object_unref (page);

  upt_canvas_move (canvas, 0, canvas->mode.virt.y / 32);
  g_node_traverse (slide->body, G_PRE_ORDER, G_TRAVERSE_LEAFS, -1,
		   slide_block_traverse_func, (gpointer)canvas);
  upt_canvas_flush (canvas);
}

void
upt_slideshow_draw (canvas, ss)
     UptCanvas *canvas;
     UptXMLSlideShow *ss;
{
  if (!ss->current_slide)
    slide_draw_coverpage (canvas, ss);
  else
    slide_draw (canvas, ss->current_slide->data);
}

void
upt_slideshow_next (ss)
     UptXMLSlideShow *ss;
{
  if (!ss->current_slide)
    ss->current_slide = ss->slides;
  else if (g_list_next (ss->current_slide))
    ss->current_slide =
      g_list_next (ss->current_slide);
}

void
upt_slideshow_previous (ss)
     UptXMLSlideShow *ss;
{
  if (ss->current_slide)
    ss->current_slide = g_list_previous (ss->current_slide);
}
