# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unit tests for Unperish publish plugin."""


import os
import shutil
import tempfile
import unittest

import unperishlib


class PublishTests(unittest.TestCase):

    def dirpath(self, name):
        return os.path.join(self.dirname, name)
        
    def buildpath(self, name):
        return os.path.join(self.build_area, name)
        
    def resultpath(self, name):
        return os.path.join(self.resultname, name)

    def setUp(self):
        self.tempdir = tempfile.mkdtemp()
        self.dirname = os.path.join(self.tempdir, "directory")
        self.build_area = os.path.join(self.tempdir, "build_area")
        self.resultname = os.path.join(self.tempdir, "result")

        os.mkdir(self.dirname)
        os.mkdir(self.build_area)
        os.mkdir(self.resultname)

        for name in ["NEWS", "README"]:
            file(self.dirpath(name), "w").close()

        tarball = self.buildpath("pink-1.2.tar.gz")
        file(tarball, "w").close()

        changelog = self.buildpath("pink.changelog")
        file(changelog, "w").close()
        
        app = unperishlib.Application(plugins=["plugins"])
        options, _ = app.parse_args(["--directory", self.dirname,
                                     "--upstream-tarball", tarball,
                                     "--changelog", changelog,
                                     "--publish-to", self.resultname])
        oper = app.get_plugin_manager().get_operation("publish")
        oper.do_it(options)

    def tearDown(self):
        shutil.rmtree(self.tempdir)

    def testPublishesNEWS(self):
        self.failUnless(os.path.exists(self.resultpath("NEWS")))

    def testPublishesREADME(self):
        self.failUnless(os.path.exists(self.resultpath("README")))

    def testPublishesTarball(self):
        self.failUnless(os.path.exists(self.resultpath("pink-1.2.tar.gz")))

    def testPublishesChangelog(self):
        self.failUnless(os.path.exists(self.resultpath("pink.changelog")))
