/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef PLANE_H
#define PLANE_H

#include <visu_data.h>
#include <openGLFunctions/view.h>
#include <coreTools/toolColor.h>

G_BEGIN_DECLS

/**
 * PLANE_TYPE:
 *
 * return the type of #Plane.
 */
#define PLANE_TYPE	     (plane_get_type ())
/**
 * PLANE:
 * @obj: a #GObject to cast.
 *
 * Cast the given @obj into #Plane type.
 */
#define PLANE(obj)	     (G_TYPE_CHECK_INSTANCE_CAST(obj, PLANE_TYPE, Plane))
/**
 * PLANE_CLASS:
 * @klass: a #GClassObject to cast.
 *
 * Cast the given @klass into #PlaneClass.
 */
#define PLANE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST(klass, PLANE_TYPE, PlaneClass))
/**
 * IS_PLANE_TYPE:
 * @obj: a #GObject to test.
 *
 * Test if the given @ogj is of the type of #Plane object.
 */
#define IS_PLANE_TYPE(obj)    (G_TYPE_CHECK_INSTANCE_TYPE(obj, PLANE_TYPE))
/**
 * IS_PLANE_CLASS:
 * @klass: a #GClassObject to test.
 *
 * Test if the given @klass is of the type of #PlaneClass class.
 */
#define IS_PLANE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE(klass, PLANE_TYPE))
/**
 * PLANE_GET_CLASS:
 * @obj: a #GObject to get the class of.
 *
 * It returns the class of the given @obj.
 */
#define PLANE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS(obj, PLANE_TYPE, PlaneClass))

/**
 * PlaneClass:
 *
 * A short way to identify #PlaneClass_struct structure.
 */
typedef struct PlaneClass_struct PlaneClass;

/**
 * Plane:
 *
 * A short wy to address #Plane_struct objects.
 */
typedef struct Plane_struct Plane;

/**
 * plane_get_type:
 *
 * This method returns the type of #Plane, use PLANE_TYPE instead.
 *
 * Returns : the type of #Plane.
 */
GType plane_get_type(void);


/**
 * Plane_hidingMode:
 * @plane_hideUnion: element are masked if one plane at least mask it ;
 * @plane_hideInter: element are masked if all planes mask it ;
 * @plane_nbHidingMode: number of masking possibilities.
 *
 * Enum used to address different hiding modes. See planeSet_hidingMode() for
 * further details.
 */
typedef enum
  {
    plane_hideUnion,
    plane_hideInter,
    plane_nbHidingMode
  } Plane_hidingMode;

/**
 * PLANE_SIDE_PLUS
 *
 * This is a key that defines which side is hidden by the plane. For
 * this value, the side is the one pointed by the normal vector.
 */
#define PLANE_SIDE_PLUS +1
/**
 * PLANE_SIDE_MINUS
 *
 * This is a key that defines which side is hidden by the plane. For
 * this value, the side is the one at the opposite of the one pointed
 * by the normal vector.
 */
#define PLANE_SIDE_MINUS -1
/**
 * PLANE_SIDE_NONE
 *
 * This is a key that defines which side is hidden by the plane. For
 * this value, no node is hidden.
 */
#define PLANE_SIDE_NONE 0
/**
 * planeNew:
 * @view: a #OpenGLView object that describes the rendering box ;
 * @vect: three values defining the normal vector (unitary or not) ;
 * @dist: the distance between origin and intersection of the plane and the
 * line made by origin and normal vector ;
 * @color: a #Color.
 *
 * Create a plane with the specified attributes.
 *
 * Returns: a newly allocated #Plane structure.
 */
Plane* planeNew(OpenGLView *view, float vect[3], float dist, Color *color);
/**
 * planeSet_normalVector:
 * @plane: a #Plane object ;
 * @vect: three values defining the normal vector (unitary or not).
 *
 * Change the normal vector defining the orientation of the plane.
 *
 * Returns: 1 if the intersections should be recalculated by
 *          a call to planeComputeInter(), 0 if not. Or -1 if there is
 *          an error.
 */
int planeSet_normalVector(Plane *plane, float vect[3]);
/**
 * planeSet_distanceFromOrigin:
 * @plane: a #Plane object ;
 * @dist: the distance between origin and intersection of the plane and the
 * line made by origin and normal vector.
 *
 * Change the position of the plane.
 *
 * Returns: 1 if the intersections should be recalculated by
 *          a call to planeComputeInter(), 0 if not. Or -1 if there is
 *          an error.
 */
int planeSet_distanceFromOrigin(Plane *plane, float dist);
/**
 * planeSet_color:
 * @plane: a #Plane object ;
 * @color: a #Color.
 *
 * Change the color of the plane.
 *
 * Returns: 0 if everything went right.
 */
int planeSet_color(Plane *plane, Color *color);
/**
 * planeComputeInter:
 * @plane: a Plane ;
 * @view: a #OpenGLView object that describes the rendering box.
 *
 * This method is used to compute the intersections between the bounding
 * box and the plane.
 *
 * Returns: 0 if everything went right.
 */
int planeComputeInter(Plane* plane, OpenGLView *view);
/**
 * planeShowHide:
 * @plane: a #Plane ;
 * @visuData: a #VisuData which show or hide nodes to;
 * @hide: a boolean, TRUE to hide elements, FALSE to return them to visibility ;
 * @side: give a positive number to hide elements in the same side that the normal vector,
 * a negative number will hide the other nodes.
 *
 * A call to this method change the visibility flag of all nodes of the given #VisuData.
 * Notice that this method makes a call to visuData_createAllNodes if necessary.
 * 
 * Returns: 1 if the calling method should ask for redraw, 0 otherwise.
 */
int planeShowHide(Plane *plane, VisuData *visuData, gboolean hide, float side);
/**
 * planeDraw:
 * @plane: a #Plane ;
 * @view: a #OpenGLView object that describes the rendering box.
 *
 * Call OpenGL primitives to draw the plane.
 */
void planeDraw(Plane* plane, OpenGLView *view);

/**
 * planeGet_barycentre:
 * @plane: a #Plane ;
 * @point: an already alloacted (size 3) float array.
 *
 * Stores the coordinates of barycentre of the plane in @point. The planeComputeInter()
 * should have been called before.
 */
void planeGet_barycentre(Plane *plane, float *point);
/**
 * planeGet_intersection:
 * @plane: a #Plane.
 *
 * The list of intersection between the plane and the box is made of
 * float[3]. The planeComputeInter() should have been called before.
 *
 * Returns: a list of float[3] elements. This list is owned by V_Sim.
 */
GList* planeGet_intersection(Plane *plane);
/**
 * planeGet_nVectUser:
 * @plane: a #Plane.
 * @vect: an already alloacted (size 3) float array.
 *
 * Stores the coordinates of the normal vector in @vec of the plane. It returns the values
 * given by the user, not the normalized vaues.
 */
void planeGet_nVectUser(Plane *plane, float *vect);
/**
 * planeGet_distanceFromOrigin:
 * @plane: a #Plane ;
 * @dist: a float location to store the value.
 *
 * Stores the distance of the plane to the origin.
 */
void planeGet_distanceFromOrigin(Plane *plane, float *dist);
/**
 * planeGet_color:
 * @plane: a #Plane ;
 * @color: a #Color pointer location to store the value.
 *
 * Stores the color of the plane.
 */
void planeGet_color(Plane *plane, Color **color);
/**
 * planeSet_hiddenState:
 * @plane: a #Plane ;
 * @side: a key, #PLANE_SIDE_NONE, #PLANE_SIDE_PLUS or #PLANE_SIDE_MINUS.
 *
 * The plane can hide the nodes on one of its side.
 * The @side argument can be PLANE_SIDE_PLUS or PLANE_SIDE_MINUS or
 * PLANE_SIDE_NONE. It codes the side of the plane which hides the nodes.
 * If PLANE_SIDE_NONE is selected all nodes are rendered.
 * 
 * Returns: 1 if planeShowHide_all() should be called.
 */
int planeSet_hiddenState(Plane *plane, int side);
/**
 * planeGet_hiddenState:
 * @plane: a #Plane.
 *
 * The plane can hide the nodes on one of its side. this
 * method get the status for the given @plane.
 * 
 * Returns: the state, defined by PLANE_SIDE_PLUS or PLANE_SIDE_MINUS or
 * PLANE_SIDE_NONE.
 */
int planeGet_hiddenState(Plane *plane);
/**
 * planeSet_rendered:
 * @plane: a #Plane ;
 * @rendered: TRUE to make the plane drawable.
 *
 * Change the visibility of the plane.
 *
 * Returns:  1 if planeShowHide_all() should be called.
 */
int planeSet_rendered(Plane *plane, gboolean rendered);
/**
 * planeGet_rendered:
 * @plane: a #Plane.
 *
 * Get the visibility of a plane.
 */
gboolean planeGet_rendered(Plane *plane);




/* No object method. */

/**
 * planesComputeInter_list:
 * @view: a #OpenGLView object that describes the rendering box ;
 * @list: a GList of #Plane.
 *
 * This method is used to compute the intersections between the bounding
 * box and all the planes of the list. This is a wrapper around planeComputeInter().
 *
 * Returns: 0 if everything went right.
 */
int planesComputeInter_list(OpenGLView *view, GList *list);
/**
 * planesDraw_list:
 * @view: a #OpenGLView object that describes the rendering box ;
 * @list: a GList of #Plane.
 *
 * Call OpenGL primitives to draw each plane of the given list. It's
 * a convenient wrapper around planeDraw() for a list.
 */
void planesDraw_list(OpenGLView *view, GList *list);
/**
 * planeShowHide_all:
 * @visuData: a #VisuData which show or hide nodes to;
 * @listOfPlanes: a GList of #Plane.
 *
 * This method test for each node if it is hidden or not by the
 * combination of all the given planes.
 *
 * Returns: 1 if visuData_createAllNodes() should be called.
 */
int planeShowHide_all(VisuData *visuData, GList *listOfPlanes);
/**
 * planeSet_hidingMode:
 * @mode: a value related to the hiding mode (look at the enum #Plane_hidingMode).
 *
 * This method is used to set the hiding mode flag. In union mode, elements
 * are not drwn if they are hidden by one plane at least. In intersection mode,
 * elements are only hidden if masked by all planes. This flag has no
 * relevence if planeShowHide() is used, but only if planeShowHide_all() is
 * called.
 *
 * Returns: 1 if planeShowHide_all() should be called.
 */
int planeSet_hidingMode(Plane_hidingMode mode);

/**
 * planesParse_XMLFile:
 * @filename: the file to parse ;
 * @list: a pointer to store the parsed list ;
 * @error: a pointer to store the error (can be NULL).
 *
 * Read the given file (syntax in XML) and create a list of planes.
 *
 * Returns: TRUE if everything goes right, if not and @error (if not NULL)
 *          is set and contains the message of the error.
 */
gboolean planesParse_XMLFile(gchar* filename, GList **list, GError **error);
/**
 * planesExport_XMLFile:
 * @filename: the file to export to ;
 * @list: the list to export ;
 * @error: a pointer to store the error (can be NULL).
 *
 * Export in XML format the given list of planes to the given file.
 *
 * Returns: TRUE if everything goes right, if not and @error (if not NULL)
 *          is set and contains the message of the error.
 */
gboolean planesExport_XMLFile(gchar* filename, GList *list, GError **error);

G_END_DECLS


#endif
