/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "light.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <glib.h>

#include <visu_tools.h>

struct lightEnvironnement_struct
{
  GList* list;
  /* Current size of the list. */
  gint nbStoredLights;
  /* Number of lights called by glEnable() when last applied. */
  gint nbEnabledLights;
};

/* Local methods. */
void lighting_set(gpointer data, gpointer user_data);


LightEnvironnement* lightEnvironnement_new()
{
  LightEnvironnement *env;

  env = g_malloc(sizeof(LightEnvironnement));
  env->list = (GList*)0;
  env->nbStoredLights = 0;
  env->nbEnabledLights = 0;

  return env;
}
void lightEnvironnementFree(LightEnvironnement *env)
{
  g_return_if_fail(env);

  lightEnvironnementEmpty_lightList(env);
  /* Need to apply to disable all previously enbaled lights before
     freeing the object. */
  lightEnvironnementApply(env);
  g_free(env);
}
gboolean lightEnvironnementAdd_light(LightEnvironnement *env, Light *light)
{
  g_return_val_if_fail(env && light, FALSE);
  g_return_val_if_fail(env->nbStoredLights < GL_MAX_LIGHTS, FALSE);

  DBG_fprintf(stderr, "Light : add a new light (%p).\n", (gpointer)light);
  env->list = g_list_append(env->list, light);
  env->nbStoredLights += 1;

  return TRUE;
}
gboolean lightEnvironnementDelete_light(LightEnvironnement *env, Light *light)
{
  g_return_val_if_fail(env && light, FALSE);
  
  DBG_fprintf(stderr, "Light : remove a light (%p).\n", (gpointer)light);
  env->list = g_list_remove(env->list, light);
  g_free(light);
  env->nbStoredLights -= 1;

  return TRUE;
}
GList* lightEnvironnementGet_lightList(LightEnvironnement *env)
{
  g_return_val_if_fail(env, (GList*)0);

  return env->list;
}
gboolean lightEnvironnementEmpty_lightList(LightEnvironnement *env)
{
  GList *list;
  int n;

  g_return_val_if_fail(env, FALSE);

  if (!env->list)
    return FALSE;

  DBG_fprintf(stderr, "Light : emptying list of stored lights of"
	      " environnement %p.\n", (gpointer)env);
  list = env->list;
  n = 0;
  while (list)
    {
      DBG_fprintf(stderr, " | removing light %p\n", list->data);
      g_free(list->data);
      n += 1;
      list = g_list_next(list);
    }
  g_list_free(env->list);
  env->list = (GList*)0;
  env->nbStoredLights = 0;

  return TRUE;
}
gboolean lightEnvironnementApply(LightEnvironnement *env)
{
  gint nb;
  float lmodel_ambient[4] = {0.2f, 0.2f, 0.2f, 1.0f};

  g_return_val_if_fail(env, FALSE);

  if (env->nbEnabledLights == 0 && !env->list)
    return FALSE;

  glPushMatrix();
  glLoadIdentity();

  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
  glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE);
  glEnable(GL_LIGHTING);

  for (nb = 0; nb < env->nbEnabledLights; nb++)
    glDisable(GL_LIGHT0 + nb);
    
  g_list_foreach(env->list, lighting_set, (gpointer)env);

  env->nbEnabledLights = env->nbStoredLights;
   
  glPopMatrix();

  return TRUE;
}

Light* light_newDefault()
{
  float params[16] =  {1.0f, 1.0f, 1.0f, 1.0f,
		       1.0f, 1.0f, 1.0f, 1.0f,
		       1.0f, 1.0f, 1.0f, 1.0f,
		       3.0f, 2.0f, 1.7f, 0.0f};
  Light *light;
  int i;

  light = g_malloc(sizeof(Light));
  light->enabled = TRUE;
  light->multiplier = 1.;
  for(i = 0; i < 4; i++)
    {
      light->ambient[i] = params[i];
      light->diffuse[i] = params[4 + i];
      light->specular[i] = params[8 + i];
      light->position[i] = params[12 + i];
    }
  return light;
}

void lighting_set(gpointer data, gpointer user_data)
{
  Light *light0 = data;
  int n;
  float values[4];
  int i;
  LightEnvironnement *env;

  g_return_if_fail(user_data);
  env = (LightEnvironnement*)user_data;

  n = g_list_index(env->list, light0);
  if(light0->enabled == FALSE)
    {
      /* From glLightfv man page : "It is always the case that GL_LIGHTi = GL_LIGHT0 + i." */
      glDisable(GL_LIGHT0 + n);
      return;
    }

  glEnable(GL_LIGHT0 + n);
  for (i = 0; i < 4; i++) values[i] = light0->ambient[i] * light0->multiplier;
  glLightfv(GL_LIGHT0 + n, GL_AMBIENT,  values);
  for (i = 0; i < 4; i++) values[i] = light0->diffuse[i] * light0->multiplier;
  glLightfv(GL_LIGHT0 + n, GL_DIFFUSE, values);
  for (i = 0; i < 4; i++) values[i] = light0->specular[i] * 1.;
  glLightfv(GL_LIGHT0 + n, GL_SPECULAR, values);
  glLightfv(GL_LIGHT0 + n, GL_POSITION, light0->position);
  DBG_fprintf(stderr, "Light : set light [%f][%f][%f][%f]\n", light0->position[0],
	      light0->position[1], light0->position[2], light0->multiplier);
}
