/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "atomic_xyz.h"
#include "renderingAtomic_ascii.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <visu_rendering.h>


static gboolean loadXyz(VisuData *data, const gchar* filename,
			FileFormat *format, GError **error);
static int read_xyz_file(VisuData *data, FILE *flux, GError **error);

/******************************************************************************/

RenderingFormatLoad* atomicXyzInit()
{
  char *typeXYZ[] = {"*.xyz", (char*)0};
  char *descrXYZ = _("'Element x y z' format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "XYZ format (very loose).";
  meth->fmt = fileFormatNew(descrXYZ, typeXYZ);
  if (!meth->fmt)
    g_error("Can't initialize the rendering atomic method, aborting...\n");
  meth->priority = 100;
  meth->load = loadXyz;

  return meth;
}

static gboolean loadXyz(VisuData *data, const gchar* filename,
			FileFormat *format, GError **error)
{
  int res, nbLine;
  FILE *readFrom;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  readFrom = fopen(filename, "r");
  if (!readFrom)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("WARNING : impossible to open this file.\n"));
      return FALSE;
    }

  res = read_xyz_file(data, readFrom, error);
  fclose(readFrom);

  if (res < 0)
    /* The file is not a D3 file. */
    return FALSE;
  else if (res > 0)
    /* The file is a D3 file but some errors occured. */
    return TRUE;
  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}

static int read_xyz_file(VisuData *data, FILE *flux, GError **error)
{
  char line[MAX_LINE_LENGTH], info[MAX_LINE_LENGTH];
  char *ptChar;
  double xu, yu, zu;
  int i, res, nbLine;
  char nomloc[MAX_LINE_LENGTH];
  VisuElement *type;
  VisuElement **types;
  int *nattyp;
  int ntype;
  GHashTable *elements;
  struct dataAscii *infos;
  double xMin, yMin, zMin, xMax, yMax, zMax;
  float boxGeometry[6];
  float translation[3];
  gchar *infoUTF8;

  /* We read every line that corresponds to this schema : "%s %f %f %f" */
  DBG_fprintf(stderr, "Atomic xyz : reading file as an xyz file.\n");

  info[0] = '\0';
  /* 1st pass to count ntype */
  nbLine = 0;
  ntype = 0;
  elements = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, freeDataAscii);
  if (!elements)
    {
      allocationProblems();
      exit(1);
    }
  while(!feof(flux))
    {
      (void)fgets(line, MAX_LINE_LENGTH, flux);
      nbLine += 1;
      if(feof(flux))
	{
	  (void)rewind(flux);
	  break;
	}
      if(sscanf(line, "%s %lf %lf %lf", nomloc, &xu, &yu, &zu) != 4)
	{
	  if (nbLine == 2)
	    {
	      strcpy(info, line);
	      ptChar = strchr(info, '\n');
	      if (ptChar)
		*ptChar = '\0';
	    }
	  continue;
	}
      /* Don't read the line when the first element doesn't begin with a letter. */
      if ( (*nomloc < 'a' || *nomloc > 'z') && (*nomloc < 'A' || *nomloc > 'Z'))
	continue;

      nomloc[8] = '\0';
      /* adding nomloc to the hashtable */
      type = visuElementGet_fromName(nomloc);
      if (!type)
	{
	  type = visuElementNewWithName(nomloc);
	  if (!type)
	    {
	      g_warning("Cannot create a new"
			" type for '%s'\n", nomloc);
	      return 1;
	    }
	  res = visuElementAdd(type);
	  if (res)
	    {
	      g_warning("Cannot add new element, maximum has been reached.");
	      return 1;
	    }
	}
      infos = (struct dataAscii*)g_hash_table_lookup(elements, (gconstpointer)type);
      if (!infos)
	{
	  infos = malloc(sizeof(struct dataAscii));
	  if (!infos)
	    {
	      allocationProblems();
	      exit(1);
	    }
	  infos->ele = type;
	  infos->pos = ntype;
	  infos->nbNodes = 1;
	  g_hash_table_insert(elements, (gpointer)type, (gpointer)infos);
	  ntype++;
	}
      else
	infos->nbNodes += 1;
    }

  /* Allocate the space for the nodes. */
  types = malloc(sizeof(VisuElement*) * ntype);
  if (!types)
    {
      allocationProblems();
      exit(1);
    }
  nattyp = malloc(sizeof(int) * ntype);
  if(!nattyp)
    {
      allocationProblems();
      exit(1);
    }
  g_hash_table_foreach(elements, (GHFunc)putValTypeInAsciiType, (gpointer)&types);
  g_hash_table_foreach(elements, (GHFunc)putValNbInAsciiType, (gpointer)&nattyp);
  g_hash_table_destroy(elements);
  if (ntype <= 0)
    {
      free(nattyp);
      free(types);
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("WARNING : the file contains no atom coordinates.\n"));
      return -1;
    }

  res = visuDataSet_population(data, ntype, nattyp, types);
  if (!res)
    g_error("Can't store the nodes in the VisuData object.");
  DBG_fprintf(stderr, "Atomic xyz : there are %d types in this file.\n", ntype);
  if (DEBUG)
    {
      for (i = 0; i < ntype; i++)
	DBG_fprintf(stderr, " | %d atom(s) for type %d.\n", nattyp[i], i);
    }
  free(nattyp);
  free(types);
         
  /* reread the file to store the coordinates */
  xMin = 1e5;
  yMin = 1e5;
  zMin = 1e5;
  xMax = -1e5;
  yMax = -1e5;
  zMax = -1e5;
  
  i = 0;
  for(;;)
    {
      (void)fgets(line, MAX_LINE_LENGTH, flux);
      if(feof(flux))
	break;
      
      if(sscanf(line, "%s %lf %lf %lf", nomloc, &xu, &yu, &zu) != 4)
	continue;
      /* Don't read the line when the first element doesn't begin with a letter. */
      if ( (*nomloc < 'a' || *nomloc > 'z') && (*nomloc < 'A' || *nomloc > 'Z'))
	continue;

      nomloc[8] = '\0';
      type = visuElementGet_fromName(nomloc);
      if(!type)
	{
	  g_warning("INTERNAL ERROR! The input file must"
		    " have been modified when loading since"
		    " at the second reading the element '%s'"
		    " seems to be new.\n", nomloc);
	  return 1;
	}
      visuDataAdd_VisuElement(data, type, xu, yu, zu, i);
      i++;
      if (xu < xMin)
	xMin = xu;
      if (yu < yMin)
	yMin = yu;
      if (zu < zMin)
	zMin = zu;
      if (xu > xMax)
	xMax = xu;
      if (yu > yMax)
	yMax = yu;
      if (zu > zMax)
	zMax = zu;
    }

  /* Set the commentary. */
/*   ptChar = strchr(info, '\n'); */
/*   if (ptChar) */
/*     *ptChar = '\0'; */
  if (info[0])
    {
      info[MAX_LINE_LENGTH - 1] = '\0';
      DBG_fprintf(stderr, "Atomic xyz : set the commentary to '%s'.\n", info);
      if (info[0] == '#')
	infoUTF8 = getStringInUTF8(info + 1);
      else
	infoUTF8 = getStringInUTF8(info);
      visuDataSet_fileCommentary(data, infoUTF8);
      g_free(infoUTF8);
    }

  DBG_fprintf(stderr, "Atomic xyz : the elements are in [%f, %f]x[%f, %f]x[%f, %f].\n",
	      xMin, xMax, yMin, yMax, zMin, zMax);
  boxGeometry[0] = xMax - xMin;
  boxGeometry[1] = 0.;
  boxGeometry[2] = yMax - yMin;
  boxGeometry[3] = 0.;
  boxGeometry[4] = 0.;
  boxGeometry[5] = zMax - zMin;
  visuDataSet_boxGeometry(data, boxGeometry, FALSE);

  translation[0] = -xMin;
  translation[1] = -yMin;
  translation[2] = -zMin;
  visuDataSet_XYZtranslation(data, translation);

  return 0;
}

