/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "renderingAtomic.h"
#include "renderingAtomic_ascii.h"
#include "renderingAtomic_d3.h"
#include "atomic_xyz.h"
#include <visu_elements.h>
#include <visu_configFile.h>
#include <renderingBackend/visu_windowInterface.h>
#include <openGLFunctions/view.h>
#include <openGLFunctions/objectList.h>
#include <opengl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <GL/gl.h>
#include <GL/glu.h> 


/* The OpenGL identifier to store the glObjectLists that
   describe the spheres. */
/* int identifierSpheres; */

#define RADIUS_DEFAULT 1.
#define SHAPE_DEFAULT shapeSphere
#define RATIO_DEFAULT 2.
#define PHI_DEFAULT  0.
#define THETA_DEFAULT 90.
/* Dealing with the radius resource. */
#define FLAG_RESOURCE_RADIUS_SHAPE "atomic_radius_shape"
#define DESC_RESOURCE_RADIUS_SHAPE "The radius of the element and its shape, a real > 0. & [Sphere Cube Elipsoid]"
gboolean renderingAtomicRead_atomicRadiusShape(gchar **lines, int nbLines,
					       int position, GString *errorMessage);
/* These functions write all the element list to export there associated resources. */
gboolean renderingAtomicExport_atomicRadiusShape(GString *data, int *nbLinesWritten,
						 VisuData* dataObj);

const char* shapeName[nbAtomicShapes + 1] = {"Sphere", "Cube", "Elipsoid", "Point", (const char*)0};
const char* shapeNameI18n[nbAtomicShapes + 1];
struct atomicResources_struct
{
  /* One of this structure is associated to each element. */

  /* The radius of the element. */
  float radius;
  /* The ratio long axis, short axis when the shape is an elipsoid. */
  float ratio;
  /* The angles positioning of the long axis when the shape is
     an elipsoid. */
  float phi, theta;

  /* The shape used. */
  int shape;

  /* An id of the list associated with the form. */
  int openGLIdentifier;
};


/* To build a list of rendering loading methods. */
/* This list contains all the method than can load an atomic file. */
static GList *allAtomicLoadingMethods;


void freeAtomicStruct(gpointer data);

RenderingMethod* initAtomic()
{
  RenderingMethod *atom;
  char *name = _("Atom visualisation");
  char *descr = _("It draws spheres at specified positions to represent atoms."
		  " The radius of the sphere can vary.");
  gchar *iconPath;
  GList *formatList;
  VisuConfigFileEntry *resourceEntry;
  GList *tmpLst;
  RenderingFormatLoad *meth;

  DBG_fprintf(stderr,"Initialising the atomic rendering method...\n");
  formatList = (GList*)0;
  allAtomicLoadingMethods = (GList*)0;

  meth = atomicD3Init();
  if (meth)
    allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  meth = atomicAsciiInit();
  if (meth)
    allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  meth = atomicXyzInit();
  if (meth)
    allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  allAtomicLoadingMethods = g_list_sort(allAtomicLoadingMethods, renderingMethodCompare_priority);
  tmpLst = allAtomicLoadingMethods;
  while (tmpLst)
    {
      formatList = g_list_append(formatList, ((RenderingFormatLoad*)tmpLst->data)->fmt);
      tmpLst = g_list_next(tmpLst);
    }

  atom = renderingMethod_new("Atom visualisation", name, descr, 1, renderingAtomicLoad);
  renderingMethodSet_fileType(atom, formatList, 0, _("Position files"));
  iconPath = g_build_filename(v_sim_pixmaps_dir, "stock-atomic.png", NULL);
  renderingMethodSet_icon(atom, iconPath);
  g_free(iconPath);
  setOpenGLMethods(atom, renderingAtomic_createShape,
		   renderingAtomic_positionShape,
		   renderingAtomicGet_radius);

  /* Dealing with config files. */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_RADIUS_SHAPE,
					  DESC_RESOURCE_RADIUS_SHAPE,
					  1,
					  renderingAtomicRead_atomicRadiusShape);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   renderingAtomicExport_atomicRadiusShape);
					  

  /* Initialise the OpenGl part. */
/*   identifierSpheres = openGLObjectList_new(NMAX_TP); */

  /* Set the general pointer to retrieve easily this method. */
  pointerOnRenderingAtomicMethod = atom;

  shapeNameI18n[0] = _("Sphere");
  shapeNameI18n[1] = _("Cube");
  shapeNameI18n[2] = _("Elipsoid");
  shapeNameI18n[3] = _("Point");
  shapeNameI18n[4] = (const char*)0;

  return atom;
}

gboolean renderingAtomicLoad(VisuData *data, FileFormat *format, GError **error)
{
  gchar *file;
  gboolean loadOk;
  GList *tmpLst;
  FileFormat *fmt;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);

  if (!data)
    return 0;

  file = visuDataGet_file(data, 0, &fmt);
  if (!file)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("No file name available."));
      return FALSE;
    }
  
  if (g_file_test(file, G_FILE_TEST_IS_DIR))
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("WARNING! The specified file is"
			     " a directory indeed. Can't load"
			     " a directory!\n"));
      return FALSE;
    }
  
  loadOk = FALSE;
  tmpLst = allAtomicLoadingMethods;
  while (tmpLst && !loadOk)
    {
      /* Each load may set error even if the format is not recognise
	 and loadOK is FALSE, then we need to free the error. */
      if (*error)
	g_error_free(*error);
      *error = (GError*)0;
      if (!fmt || ((RenderingFormatLoad*)tmpLst->data)->fmt == fmt)
	{
	  DBG_fprintf(stderr,"Rendering Atomic : testing '%s' with format : %s.\n",
		      file, ((RenderingFormatLoad*)tmpLst->data)->name);
	  loadOk = ((RenderingFormatLoad*)tmpLst->data)->load
	    (data, file, ((RenderingFormatLoad*)tmpLst->data)->fmt, error);
	}
      tmpLst = g_list_next(tmpLst);
    }

  if (!loadOk)
    {
      if (!*error)
	*error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			     _("WARNING : impossible to load this file.\n"));
      return FALSE;
    }

  return TRUE;
}

void freeAtomicStruct(gpointer data)
{
  if (data)
    free(data);
}

struct atomicResources_struct* createDefaultAtomicStruct()
{
  struct atomicResources_struct *str;

  str = malloc(sizeof(struct atomicResources_struct));
  if (!str)
    {
      allocationProblems();
      exit(1);
    }
  str->radius = RADIUS_DEFAULT;
  str->shape  = SHAPE_DEFAULT;
  str->ratio  = RATIO_DEFAULT;
  str->phi    = PHI_DEFAULT;
  str->theta  = THETA_DEFAULT;

  str->openGLIdentifier = openGLObjectList_new(1);

  return str;
}

float renderingAtomicGet_radius(VisuElement *ele)
{
  struct atomicResources_struct *str;

  if (!ele)
    return RADIUS_DEFAULT;

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  /* g_hash_table_lookup(shapeAndRadiusResource, (gpointer)ele); */
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
/*       g_hash_table_insert(shapeAndRadiusResource, (gpointer)ele, (gpointer)str); */
    }
  return str->radius;
}
float renderingAtomicGet_radiusDefault()
{
  return RADIUS_DEFAULT;
}
gboolean renderingAtomicSet_radius(VisuElement* ele, float value)
{
  struct atomicResources_struct *str;

  if (!ele)
    return FALSE;
  
  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
    }
  if (value <= 0.)
    {
      fprintf(stderr, "WARNING! Trying to set a negative value (%f)"
	      " for the radius of VisuElement '%s'.\n", value, ele->name);
      str->radius = 1.;
    }
  else if (value == str->radius)
    return FALSE;
  else
    str->radius = value;

  /* Ask for reDraw if needed. */
  return TRUE;
}

int renderingAtomicGet_shape(VisuElement *ele)
{
  struct atomicResources_struct *str;

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
/*   str = g_hash_table_lookup(shapeAndRadiusResource, (gpointer)ele); */
  if (!str)
    return -1;
  else
    return str->shape;
}
const char* renderingAtomicGet_shapeName(int shape)
{
  if (shape < 0 || shape >= nbAtomicShapes)
    return (char*)0;
  else
    return shapeName[shape];
}
const char* renderingAtomicGet_shapeNameDefault()
{
  return shapeName[SHAPE_DEFAULT];
}
int renderingAtomicGet_shapeDefault()
{
  return SHAPE_DEFAULT;
}
int renderingAtomicSet_shape(VisuElement *ele, int shape)
{
  struct atomicResources_struct *str;

  if (!ele || shape < 0 || shape >= nbAtomicShapes)
    {
      fprintf(stderr, "WARNING! Wrong parameters to the call of renderingAtomicSet_shape.\n");
      return 0;
    }
  
  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
/*   str = g_hash_table_lookup(shapeAndRadiusResource, (gpointer)ele); */
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
/*       g_hash_table_insert(shapeAndRadiusResource, (gpointer)ele, (gpointer)str); */
    }
  if (shape == str->shape)
    return 0;
  else
    str->shape = shape;

  return 1;
}

int renderingAtomicSet_shapeFromName(VisuElement *ele, char* shape)
{
  int res, i;

  res = -1;
  for (i = 0; shapeName[i] && res < 0; i++)
    {
      if (!strcmp(shapeName[i], shape))
	res = i;
    }
  if (res < 0)
    {
      fprintf(stderr, "WARNING! Unknown shape name in the call of renderingAtomicSet_shapeFromName.\n");
      return 0;
    }
  else
    return renderingAtomicSet_shape(ele, res);
}
const char** renderingAtomicGet_allShapes()
{
  return shapeName;
}
const char** renderingAtomicGet_allShapesI18n()
{
  return shapeNameI18n;
}
gboolean renderingAtomicSet_elipsoidParameters(VisuElement *ele, float ratio,
					       float phi, float theta)
{
  struct atomicResources_struct *str;
  gboolean refresh;

  g_return_val_if_fail(ele && (ratio >= 1.), 0);
  
  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
    }
  refresh = FALSE;
  if (ratio != str->ratio)
    {
      str->ratio = ratio;
      refresh = TRUE;
    }
  if (phi != str->phi)
    {
      str->phi = phi;
      refresh = TRUE;
    }
  if (theta != str->theta)
    {
      str->theta = theta;
      refresh = TRUE;
    }

  return refresh && (str->shape == shapeElipsoid);
}
gboolean renderingAtomicSet_elipsoidRatio(VisuElement *ele, float ratio)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele && (ratio >= 1.), FALSE);
  
  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
    }
  if (ratio != str->ratio)
    {
      str->ratio = ratio;
      return (str->shape == shapeElipsoid);
    }
  return FALSE;
}
gboolean renderingAtomicSet_elipsoidPhi(VisuElement *ele, float phi)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, FALSE);
  
  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
    }
  if (phi != str->phi)
    {
      str->phi = phi;
      return (str->shape == shapeElipsoid);
    }
  return FALSE;
}
gboolean renderingAtomicSet_elipsoidTheta(VisuElement *ele, float theta)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, FALSE);
  
  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
    }
  if (theta != str->theta)
    {
      str->theta = theta;
      return (str->shape == shapeElipsoid);
    }
  return FALSE;
}
float renderingAtomicGet_elipsoidRatio(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, RATIO_DEFAULT);

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (str)
    return str->ratio;
  else
    return RATIO_DEFAULT;
}
float renderingAtomicGet_elipsoidPhi(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, PHI_DEFAULT);

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (str)
    return str->phi;
  else
    return PHI_DEFAULT;
}
float renderingAtomicGet_elipsoidTheta(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, THETA_DEFAULT);

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (str)
    return str->theta;
  else
    return THETA_DEFAULT;
}


gboolean renderingAtomicRead_atomicRadiusShape(gchar **lines, int nbLines,
					       int position, GString *errorMessage)
{
  VisuElement* ele;
  char *cursor;
  int res, shapeNum, errorFlag, i;
  float radius;
  char shape[MAX_LINE_LENGTH];
  GenericRenderingWindow window;

  cursor = lines[0];
  ele = visuElementGet_fromValueLine(&cursor, -1, &errorFlag);
  if (!ele)
    {
      if (errorMessage)
	visuElementPrint_errorFromValueLine(errorMessage, position, errorFlag);
      return FALSE;
    }
  res = sscanf(cursor, "%f %s", &radius, shape);
  if (res != 2 || radius <=0.)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d,"
					       " 1 floating point value (v > 0.) and a shape name "
					       "must appear after the %s markup.\n"),
			       position, FLAG_RESOURCE_RADIUS_SHAPE);
      return FALSE;
    }
  shapeNum = -1;
  for (i = 0; shapeName[i] && shapeNum < 0; i++)
    {
      if (!strcmp(shapeName[i], shape))
	shapeNum = i;
    }
  if (shapeNum < 0)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d,"
					       " the shape '%s' is unknown.\n"),
			       position, shape);
      return 1;
    }
  res = renderingAtomicSet_radius(ele, radius);
  res = renderingAtomicSet_shape(ele, shapeNum) || res;

  window = visuRenderingWindowGet_current();
  if (res && window && visuRenderingWindowGet_visuData(window))
    renderingAtomic_createShape(visuRenderingWindowGet_visuData(window),
				ele);
  
  return TRUE;
}

/* These functions write all the element list to export there associated resources. */
gboolean renderingAtomicExport_atomicRadiusShape(GString *data, int *nbLinesWritten,
						 VisuData *dataObj)
{
  GList *pos, *eleList;
  int i;
  struct atomicResources_struct *str;

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_RADIUS_SHAPE);
  *nbLinesWritten = 1;
  /* We create a list of elements, or get the whole list. */
  eleList = (GList*)0;
  if (dataObj)
    {
      for (i = 0; i < dataObj->ntype; i++)
	eleList = g_list_prepend(eleList, (gpointer)dataObj->fromIntToVisuElement[i]);
      pos = eleList;
    }
  else
    pos = visuElementGet_allElements();
  while(pos)
    {
      str = (struct atomicResources_struct*)
	visuElementGet_property((VisuElement*)pos->data, "radiusAndShape");
      if (str)
	{
	  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_RADIUS_SHAPE);
	  g_string_append_printf(data, "    %s %10.3f %s\n",
				 ((VisuElement*)pos->data)->name,
				 str->radius, shapeName[str->shape]);
	  *nbLinesWritten += 2;
	}
      pos = g_list_next(pos);
    }
  g_string_append_printf(data, "\n");
  *nbLinesWritten += 1;
  if (eleList)
    g_list_free(eleList);

  return TRUE;
}


/***************/
/* OpenGL part */
/***************/

void renderingAtomic_positionShape(VisuData *visuData, VisuNode *node, VisuElement* ele)
{
  float rgba[4];
  float xyz[3];
  OpenGLView *view;

  visuDataGet_nodePosition(visuData, node, xyz);
  view = visuDataGet_openGLView(visuData);
  g_return_if_fail(view);

  glPushMatrix();
  glTranslated(xyz[0] - view->box->dxxs2,
	       xyz[1] - view->box->dyys2,
	       xyz[2] - view->box->dzzs2);
  if (visuData->setColor)
    {
      visuData->setColor(visuData, rgba, ele, node);
      setOpenGlMaterial(ele->material, rgba);
    }
  glCallList(ele->openGLIdentifier);
  glPopMatrix();
}

int renderingAtomic_createShape(VisuData *visuData, VisuElement* ele)
{
  struct atomicResources_struct *str;
  OpenGLView *view;
  int nlat;

  g_return_val_if_fail(visuData && ele, -1);

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (!str)
    {
      str = createDefaultAtomicStruct();
      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
    }

  view = visuDataGet_openGLView(visuData);
  nlat = OpenGLViewGet_numberOfFacettes(view, str->radius);

  DBG_fprintf(stderr, "Rendering Atomic : creating '%s' for %s (%d - OpenGL id"
	      " %d - fac %d)\n", shapeName[str->shape],
	      ele->name, ele->typeNumber, str->openGLIdentifier,
	      nlat);
  if (nlat < 0)
    return -1;
   
  glNewList(str->openGLIdentifier, GL_COMPILE);
  switch (str->shape)
    {
    case shapeSphere:
      gluSphere(obj, (double)str->radius, nlat, nlat);
      break;
    case shapeElipsoid:
      glPushMatrix();
      glRotatef(str->phi, 0., 0., 1.);
      glRotatef(str->theta, 0., 1., 0.);
      glScalef(1.0, 1.0, str->ratio);
      gluSphere(obj, (double)str->radius, nlat, nlat);
      glPopMatrix();
      break;
    case shapePoint:
      glPushMatrix();
      glPointSize(MAX(1, (int)(str->radius * view->camera->gross * 5.)));
      glBegin(GL_POINTS);
      glVertex3f(0., 0., 0.);
      glEnd();
      glPopMatrix();
      break;
    case shapeCube:
      glBegin(GL_QUADS);

      glNormal3f(0., 0., 1.);
      glVertex3f(str->radius / 2., str->radius / 2., str->radius / 2.);
      glVertex3f(-str->radius / 2., str->radius / 2., str->radius / 2.);
      glVertex3f(-str->radius / 2., -str->radius / 2., str->radius / 2.);
      glVertex3f(str->radius / 2., -str->radius / 2., str->radius / 2.);

      glNormal3f(0., 0., -1.);
      glVertex3f(str->radius / 2.,str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,-str->radius / 2.);

      glNormal3f(1., 0., 0.);
      glVertex3f(str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,str->radius / 2.,-str->radius / 2.);

      glNormal3f(-1., 0., 0.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,str->radius / 2.);

      glNormal3f(0., 1., 0.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,str->radius / 2.,-str->radius / 2.);

      glNormal3f(0., -1., 0.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,str->radius / 2.);

      glEnd();
      break;
    }
  glEndList();
   
/*   ele->openGLIdentifier = identifierSpheres + ele->typeNumber; */
  return str->openGLIdentifier;
}

void renderingAtomicAdd_loadMethod(RenderingFormatLoad* meth)
{
  g_return_if_fail(meth && meth->load);

  DBG_fprintf(stderr, "Rendering Atomic : adding a new loading method '%s'.\n", meth->name);
  allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  allAtomicLoadingMethods = g_list_sort(allAtomicLoadingMethods, renderingMethodCompare_priority);
  if (meth->fmt)
    renderingMethodAdd_fileFormat(pointerOnRenderingAtomicMethod, meth->fmt, 0);
}
