/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_DATA_H
#define VISU_DATA_H

#include <glib.h>
#include <glib-object.h>

#include "visu_elements.h"
#include "visu_nodes.h"
#include "renderingBackend/visu_windowInterface_def.h"
#include "openGLFunctions/view.h"
#include "coreTools/toolFileFormat.h"

G_BEGIN_DECLS

/**
 * VISU_DATA_TYPE:
 *
 * return the type of #VisuData.
 */
#define VISU_DATA_TYPE	     (visu_data_get_type ())
/**
 * VISU_DATA:
 * @obj: a #GObject to cast.
 *
 * Cast the given @obj into #VisuData type.
 */
#define VISU_DATA(obj)	     (G_TYPE_CHECK_INSTANCE_CAST(obj, VISU_DATA_TYPE, VisuData))
/**
 * VISU_DATA_CLASS:
 * @klass: a #GClassObject to cast.
 *
 * Cast the given @klass into #VisuDataClass.
 */
#define VISU_DATA_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST(klass, VISU_DATA_TYPE, VisuDataClass))
/**
 * IS_VISU_DATA_TYPE:
 * @obj: a #GObject to test.
 *
 * Test if the given @ogj is of the type of #VisuData object.
 */
#define IS_VISU_DATA_TYPE(obj)    (G_TYPE_CHECK_INSTANCE_TYPE(obj, VISU_DATA_TYPE))
/**
 * IS_VISU_DATA_CLASS:
 * @klass: a #GClassObject to test.
 *
 * Test if the given @klass is of the type of #VisuDataClass class.
 */
#define IS_VISU_DATA_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE(klass, VISU_DATA_TYPE))
/**
 * VISU_DATA_GET_CLASS:
 * @obj: a #GObject to get the class of.
 *
 * It returns the class of the given @obj.
 */
#define VISU_DATA_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS(obj, VISU_DATA_TYPE, VisuDataClass))

/**
 * VisuData:
 *
 * A short way to identify #_VisuData structure.
 */
typedef struct _VisuData VisuData;
/**
 * VisuData_private:
 *
 * Private fields for #ViuData objects.
 */
typedef struct VisuData_private_struct VisuData_private;
/**
 * VisuDataClass:
 *
 * A short way to identify #_VisuDataClass structure.
 */
typedef struct _VisuDataClass VisuDataClass;

/**
 * setColorFunc:
 * @visuData: a pointer to the calling object ;
 * @ele: a #VisuElement ;
 * @node: a #VisuNode ;
 * @rgba: an 4 allocated float area to store the return values.
 *
 * This prototype is used to specify an optional method
 * to associate a color with external values to each node.
 */
typedef void (*setColorFunc)(VisuData *visuData, float rgba[4], VisuElement *ele, VisuNode* node);

/**
 * VisuData:
 * @parent: an object to inherit from (NULL here).
 * @privateDt: a pointer to the private data.
 * @ntype: number of #VisuElement used in this object.
 * @fromVisuElementToInt: give the number of the #VisuElement when
 *                        its pointer is known.
 * @fromIntToVisuElement: give the pointer of a #ViosuElement when
 *                        its number is known.
 * @fromNumberToVisuNode: give the #VisuNode knowing its element number and
 *                        its own position.
 * @nbOfAllStoredNodes: number of registered #VisuNode in this object.
 * @numberOfNodes: give the number of allocated #VisuNode for each
 *                 #VisuElement.
 * @numberOfStoredNodes: give the number of stored #VisuNode for
 *                       each #VisuElement.
 * @nodes: array that stores all the nodes.
 * @setColor: method called before drawing each node (can be NULL).
 *
 * This structure describes a #VisuData object.
 */
struct _VisuData
{
  GObject parent;

  /********************/
  /* Nodes attributes */
  /********************/
  /* Number of VisuElements used by these data. */
  int ntype;
  /* This hashtable gives access to the good array  of VisuNodes
     when one has the VisuElement. */
  GHashTable *fromVisuElementToInt;
  /* This is a corresponding table from the nodes array indice
     to the associated VisuElement. */
  VisuElement **fromIntToVisuElement;

  /* This array gives access to the good VisuNode
     when one has its number. */
  VisuNode **fromNumberToVisuNode;
  /* The total of stored VisuNodes. */
  int nbOfAllStoredNodes;

  /* Number of nodes allocated (size of the nodes array) per VisuElement. */
  int* numberOfNodes;
  /* Number of nodes physically present in the array per VisuElement. */
  int* numberOfStoredNodes;
  /* Coordinates of all these nodes. */
  VisuNode **nodes;

  /* Define a method to set the color of each node.
     If this method is NULL, the color of the element is used. */
  setColorFunc setColor;

  VisuData_private *privateDt;
};

/**
 * visu_data_get_type:
 *
 * This method returns the type of #VisuData, use VISU_DATA_TYPE instead.
 *
 * Returns : the type of #VisuData.
 */
GType visu_data_get_type(void);

/**
 * visuDataNew:
 *
 * This creates an empty #VisuData object.
 *
 * Returns: a newly created #VisuData object (its ref count is set to 1).
 */
VisuData* visuDataNew();
/**
 * visuDataNew_withOpenGLView:
 * @view: an #OpenGLView object.
 *
 * This creates an empty #VisuData object with values for its veiw taken
 * from the given @view argument (the argument is indeed copied).
 *
 * Returns: a newly created #VisuData object (its ref count is set to 1).
 */
VisuData* visuDataNew_withOpenGLView(OpenGLView *view);






/**
 * visuDataGet_objectList:
 * @data: a valid #VisuObject ;
 *
 * All the drawned nodes are stored in an OpenGL list. The identifier of
 * this list can be access with the present method.
 *
 * Returns: an integer that identifies the OpenGL list of all drawn nodes.
 */
int visuDataGet_objectList(VisuData *data);





enum visuData_boxVector
  {
    VISU_DATA_BOX_DXX,
    VISU_DATA_BOX_DYX,
    VISU_DATA_BOX_DYY,
    VISU_DATA_BOX_DZX,
    VISU_DATA_BOX_DZY,
    VISU_DATA_BOX_DZZ
  };

/**
 * visuDataSet_population:
 * @data: a #VisuData object ;
 * @nbOfTypes: number of #VisuElement ;
 * @nbOfNodesPerVisuElement: number of #VisuNode per VisuElement ;
 * @visuElementUsed: pointers to #VisuElement in the same order that
 *                   @nbOfNodesPerVisuElement.
 *
 * This method allocates the storing part of the given #VisuData structure and
 * store all the #VisuNodes.
 *
 * Returns: 1 if everything goes right.
 */
int visuDataSet_population(VisuData *data, int nbOfTypes, int* nbOfNodesPerVisuElement,
			   VisuElement **visuElementUsed);
/**
 * visuDataFree_population:
 * @data: a VisuData to be freed.
 *
 * This method frees only the allocated memory that deals with
 * the nodes (i.e. everything except the data of the files,
 * the properties and the setColor method.
 */
void visuDataFree_population(VisuData *data);
/**
 * visuDataAdd_file:
 * @data: a #VisuData object ;
 * @file: a string that points to a file ;
 * @kind: an integer to qualify the file to add ;
 * @format: a file format.
 *
 * This method is used to add files
 * of type @kind to the @data. The @file
 * attribute is copied. The @format argument can be null.
 */
void visuDataAdd_file(VisuData *data, gchar* file, int kind, FileFormat *format);
/**
 * visuDataRemove_allFiles:
 * @data: a #VisuData object.
 *
 * This method is used to empty the list of
 * known file from the given @data.
 */
void visuDataRemove_allFiles(VisuData *data);
/**
 * visuDataGet_file:
 * @data: a #VisuData object.
 * @kind: an integer to qualify the required file ;
 * @format: a location for a file format (can be NULL).
 *
 * This prototype is used to retrieve stored
 * files identify by their @kind.
 *
 * Returns: the name of a file (it should not be deleted).
 */
gchar* visuDataGet_file(VisuData *data, int kind, FileFormat **format);
/**
 * visuDataSet_fileCommentary:
 * @data: a #VisuData object ;
 * @commentary: the message to be stored (null terminated).
 *
 * This method is used to store a description of the given @data. This
 * string is copied and @commentary can be freed.
 */
void visuDataSet_fileCommentary(VisuData *data, gchar* commentary);
/**
 * visuDataGet_fileCommentary:
 * @data: a #VisuData object.
 *
 * Get the commentary associated to the given @data.
 *
 * Returns: a string description or NULL. This string is own by V_Sim
 *          and should not be freed.
 */
gchar* visuDataGet_fileCommentary(VisuData *data);

/**
 * visuData_compareElements:
 * @data1: a #VisuData object ;
 * @data2: an other #VisuData object.
 *
 * This method is used to compare the composition of the given two #VisuData objects.
 * The test is only done on #VisuElement lists.
 *
 * Returns: TRUE if the two objects contains exactly the same #VisuElemet objects (not
 *          one more or one less or one different), FALSE otherwise.
 */
gboolean visuData_compareElements(VisuData *data1, VisuData *data2);
/**
 * visuDataSet_changeElementFlag:
 * @data: a #VisuData object ;
 * @changeElement: a boolean.
 *
 * This method is mainly used by internal gears to set a flag. This flag control
 * if the @data object has the same #VisuElement objects than the previously rendered one.
 */
void visuDataSet_changeElementFlag(VisuData *data, gboolean changeElement);
/**
 * visuDataGet_changeElementFlag:
 * @data: a #VisuData object.
 *
 * V_Sim can use a flag set on @data object to know if @data has exactly the same
 * #VisuElement list than the previously rendered one.
 *
 * Returns: TRUE if the previously rendered #VisuData object has had the same
 *          #VisuElement list than the given one, FALSE otherwise.
 */
gboolean visuDataGet_changeElementFlag(VisuData *data);

/**
 * visuDataAdd_VisuElement:
 * @data: the #VisuData where to add the new #VisuNode ;
 * @ele: the #VisuElement kind of the new #VisuNode ;
 * @x: its x coordinate ;
 * @y: its y coordinate ;
 * @z: its z coordinate ;
 * @number: its position in the input file (or anything else that can uniqualy
 *          identify this new #VisuNode).
 *
 * This method adds a new #VisuNode to the specified #VisuData.
 *
 * Returns: 0 if everything is allright.
 */
int visuDataAdd_VisuElement(VisuData *data, VisuElement *ele,
			    double x, double y, double z, int number);
/**
 * visuDataAdd_int:
 * @data: the #VisuData where to add the new #VisuNode ;
 * @position: a integer corresponding to the position of
 *            a #VisuElement in the array **nodes in the structure;
 * @x: its x coordinate ;
 * @y: its y coordinate ;
 * @z: its z coordinate ;
 * @number: its position in the input file (or anything else that can uniqualy
 *          identify this new #VisuNode).
 *
 * This method adds a new #VisuNode to the specified #VisuData. Position must be
 * chosen between 0 and (ntype - 1) and corresponds to the position of the array
 * of #VisuNodes of a #VisuElement.
 *
 * Returns: 0 if everything is alright.
 */
int visuDataAdd_int(VisuData *data, int position,
		    double x, double y, double z, int number);
/**
 * visuDataGet_nodeFromNumber:
 * @data: a %VisuData structure which stores the nodes.
 * @number: an integer.
 *
 * This methods retrieves the #VisuNode identified by the integer @number.
 * The number must be strictly positive. No eror is raised if no node corresponds
 * to the given number.
 *
 * Returns: the found #VisuNode or NULL if none corresponds to number.
 */
VisuNode* visuDataGet_nodeFromNumber(VisuData *data, int number);
/**
 * visuDataMatch_nodeFromId:
 * @data: a #VisuData object ;
 * @eleId: a positive integer ;
 * @posId: a positive integer.
 *
 * This method try to match a #VisuNode of the given @data that has the given
 * position, where @eleId describes the number of the element and @posId the position
 * of the node in the node array of this element. If no node matchs, no error is raised,
 * but the return value is NULL.
 *
 * Returns: the #VisuNode corresponding to the given id, if exists, NULL otherwise.
 */
VisuNode* visuDataGet_nodeFromId(VisuData *data, int eleId, int posId);

/**
 * visuDataSet_nodeProperty:
 * @data: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @key: a string ;
 * @value: an allocated area to store data.
 *
 * This method is used to store some values associated with
 * the given @node of the given @data. These values are anything allocated and will be
 * freed (with free()) when the node is deleted. These values
 * are described by the @key, and can be retrieved with the
 * #visuDataGet_nodeProperty method. The @key is used by the #VisuData
 * and should not be freed. Moreover it is not freed when the node
 * is destroyed.
 */
void visuDataSet_nodeProperty(VisuData* data, VisuNode* node, char* key, gpointer value);
/**
 * visuDataGet_nodeProperty:
 * @data: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @key: a string.
 *
 * This method is used to retrieve some data associated to
 * the specified @node, stored in the given @data. These return data
 * should not be freed after used.
 *
 * Returns: some data associated to the key.
 */
gpointer visuDataGet_nodeProperty(VisuData* data, VisuNode* node, char* key);
/**
 * visuDataAdd_nodeProperty:
 * @data: a #VisuData object ;
 * @key: a string ;
 * @freeFunc: a method to free each token.
 * 
 * This method adds and allocates a new area to store nodes associated data that
 * could be retrieve with the @key. When the property is removed with the
 * #visuDataRemove_nodePropertry the area is freed and @freeFunc is called for
 * each token.
 */
void visuDataAdd_nodeProperty(VisuData* data, char* key, GDestroyNotify freeFunc);
/**
 * visuDataRemove_nodeProperty:
 * @data: a #VisuData object ;
 * @key: a string.
 * 
 * This method remove all the data associated with the string @key, of all nodes.
 * They are freed.
 */
void visuDataRemove_nodeProperty(VisuData* data, char* key);

/**
 * visuDataSet_property:
 * @data: a #VisuData object ;
 * @key: a string ;
 * @value: an allocated area to store data.
 *
 * This method is used to store some values associated with
 * the given @data. These values are anything allocated and will be
 * freed (with free()) when the @data is deleted. These values
 * are described by the @key, and can be retrieved with the
 * #visuDataGet_property() method. The @key is used by the #VisuData
 * and should not be freed. Moreover it is not freed when the node
 * is destroyed.
 */
void visuDataSet_property(VisuData* data, char* key, gpointer value);
/**
 * visuDataSet_propertyWithDestroyFunc:
 * @data: a #VisuData object ;
 * @key: a string ;
 * @value: an allocated area to store data ;
 * @freeFunc: a method to free each token.
 *
 * Same method as visuDataSet_property() but giving a free method when
 * the data is removed.
 */
void visuDataSet_propertyWithDestroyFunc(VisuData* data, char* key, gpointer value,
					 GDestroyNotify freeFunc);
/**
 * visuDataGet_property:
 * @data: a #VisuData object ;
 * @key: a string.
 *
 * This method is used to retrieve some data associated to
 * the specified @data. These return data should not be freed after used.
 *
 * Returns: some data associated to the key.
 */
gpointer visuDataGet_property(VisuData* data, char* key);
/**
 * visuDataRemove_property:
 * @data: a #VisuData object ;
 * @key: a string.
 * 
 * This method remove the data associated with the string @key.
 * They are freed.
 */
void visuDataRemove_property(VisuData* data, char* key);

/**
 * visuDataSet_ColorFunc:
 * @data: a #VisuData object ;
 * @func: a method that colorize the nodes.
 *
 * This is a little trick to colorized the nodes. It should not be used since it
 * will probably be different in future release.
 */
void visuDataSet_ColorFunc(VisuData *data, setColorFunc func);

/**
 * visuDataGet_XYZtranslation:
 * @data: a #VisuData object.
 *
 * The nodes are rendered at thier coordinates plus a translation. This method
 * allows to retrieve that translation.
 *
 * Returns: a newly allocated array of 3 floats. It should be freed with a call
 *          to free() after use.
 */
float* visuDataGet_XYZtranslation(VisuData* data);
/**
 * visuDataSet_XYZtranslation:
 * @data: a #VisuData object ;
 * @xyz: an array of floating point values.
 *
 * This set the translations of the specified #VisuData whatever previous values.
 * The translation is done in the orthonormal referential, not the referential of
 * the box.
 *
 * Returns: if returns 1, visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
int visuDataSet_XYZtranslation(VisuData* data, float xyz[3]);
/**
 * visuDataConvert_XYZtoBoxCoordinates:
 * @data: a #VisuData object ;
 * @boxCoord: an array of floating point values to store the result ;
 * @xyz: an array of floating point values describing coordinates in cartesian.
 *
 * Use this method to transform cartesian coordinates to the box coordinates.
 */
void visuDataConvert_XYZtoBoxCoordinates(VisuData *data, float boxCoord[3], float xyz[3]);
/**
 * visuDataConvert_boxCoordinatestoXYZ:
 * @data: a #VisuData object ;
 * @xyz: an array of floating point values to store the result ;
 * @boxCoord: an array of floating point values that describes the box coordinates.
 *
 * Use this method to transform box coordinates into cartesian.
 */
void visuDataConvert_boxCoordinatestoXYZ(VisuData *data, float xyz[3], float boxCoord[3]);
/**
 * visuDataGet_boxMatrix:
 * @data: a #VisuData object ;
 * @matrix: an area to store the matrix.
 *
 * This method is used when the box matrix is required. This matrix can transform
 * a vector given in box coordinates into a cartesian vector. If a simple vector
 * multication is required, then the use of visuDataConvert_boxCoordinatestoXYZ()
 * should be prefered.
 */
void visuDataGet_boxMatrix(VisuData *data, float matrix[3][3]);
/**
 * visuDataSet_boxGeometry:
 * @data: a #VisuData object ;
 * @geometry: a 6 floating point array ;
 * @periodic: a boolean.
 *
 * This methods set the size of the box that defines the viewport. If the @periodic
 * argument is TRUE, the box is also the limit of a periodic system.
 */
void visuDataSet_boxGeometry(VisuData *data, float geometry[6], gboolean periodic);
/**
 * visuDataGet_boxGeometry:
 * @data: a #VisuData object ;
 * @vector: an int corresponding to a vector of the box.
 *
 * Retrieve the value of a vector defining the bounding box. The vector
 * is chosen with an int, see the #visuData_boxVector enum for more
 * details.
 *
 * Returns: the value of the required vector (always a positive value),
 *          a negative value if something goes wrong.
 */
float visuDataGet_boxGeometry(VisuData *data, int vector);
/**
 * visuDataGet_periodic:
 * @data: a #VisuData object.
 * 
 * The bounding box can be just useful to set the OpenGl viewport,
 * or it can define a limit for periodic conditions.
 *
 * Returns: if the box is a periodic limit or not.
 */
gboolean visuDataGet_periodic(VisuData *data);
/**
 * visuData_constrainedElementInTheBox:
 * @data: a #VisuData object ;
 * @element: a #VisuElement object.
 *
 * Check all the nodes of the specified @element and change their coordinates if they are out
 * of the bounding box. The position of each node is the result of the
 * sum of their own position and of the box translation.
 *
 * Returns: 1 if visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
int visuData_constrainedElementInTheBox(VisuData *data, VisuElement *element);
/**
 * visuData_constrainedInTheBox:
 * @data: a #VisuData object.
 *
 * It does the same things that visuData_constrainedElementInTheBox() but for all
 * the #VisuElement of the given @data. I.e. it checks all the nodes and changes
 * their coordinates if they are out of the bounding box.
 * The position of each node is the result of the
 * sum of their own position and of the box translation.
 *
 * Returns: 1 if visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
int visuData_constrainedInTheBox(VisuData *data);
/**
 * visuData_constrainedFree:
 * @data: a #VisuData object.
 *
 * Return all the nodes to their original position, except for the
 * global translation.
 *
 * Returns: 1 if visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
int visuData_constrainedFree(VisuData *data);
/**
 * visuDataGet_nodePosition:
 * @data: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @coord: an array of 3 floating point values to store the position.
 *
 * Position of nodes are subject to various translations and different transformations.
 * Their coordinates should not be access directly through node.[xyz]. This method
 * is used to retrieve the given node position.
 */
void visuDataGet_nodePosition(VisuData *data, VisuNode *node, float coord[3]);


/***************/
/* OpenGL Part */
/***************/

/**
 * visuData_createNodes:
 * @data: a #VisuData object ;
 * @ele: a #VisuElement object.
 *
 * This create the glObjectList registered at (%identifierAllNodes + the position
 * of the @ele in the @data object + 1) that contains all the nodes of the given #VisuElement.
 */
void visuData_createNodes(VisuData *data, VisuElement *ele);
/**
 * visuData_createNode:
 * @data: a #VisuData object ;
 * @node: a #VisuNode object.
 *
 * This method call the createNode method of the current rendering method
 * for the given node. It also calls the list of the material associated
 * to the given node.
 */
void visuData_createNode(VisuData *data, VisuNode *node);
/**
 * visuData_createAllNodes:
 * @data: a #VisuData object.
 *
 * This create the glObjectList registered at %identifierAllNodes
 * that contains all the nodes. This glObjectList is made of all
 * nodes of all element that has a flag %rendered at true translated
 * to their own positions.
 *
 * This method doesn't ask for redraw and signal %OpenGLAskForReDraw
 * should be emitted manually.
 */
void visuData_createAllNodes(VisuData *data);
/**
 * visuData_createAllElements:
 * @data: a #VisuData object.
 *
 * This method will call the createOpenGLElementFunc() method of the current #RenderingMethod
 * on all the nodes of the given #VisuData.
 */
void visuData_createAllElements(VisuData *data);

/**
 * visuDataEmit_askForShowHideNodes:
 * @data: a valid #VisuData object ;
 * @redraw: a pointer to a location to store if redraw
 *          is needed after all listeners have modified the
 *          nodes of the given @data.
 *
 * This methods is used to emit the 'NodeAskForShowHide' signal. This
 * signal asks all modules that may hide nodes to recompute their
 * hiding scheme and put in @redraw if they have changed something. Since
 * all listeners write in @redraw, they should modify it with an union.
 * @redraw is initialised at FALSE before the signal is emitted.
 */
void visuDataEmit_askForShowHideNodes(VisuData *data, gboolean *redraw);
/**
 * visuDataEmit_nodePositionChanged:
 * @data: a valid #VisuData object.
 *
 * This method is used to emit the 'NodePositionChanged' signal. This signal asks
 * all modules that are dependent of the nodes' positions to recompute their
 * OpenGL lists.
 */
void visuDataEmit_nodePositionChanged(VisuData *data);
/**
 * visuDataEmit_elementRenderedChange:
 * @data: a valid #VisuData object ;
 * @element: a valid #VisuElement object.
 *
 * This method is used to emit the 'ElementRenderedChanged' signal. This signal asks
 * all modules that are dependent of the given @element visibility to recompute their
 * OpenGL lists.
 */
void visuDataEmit_elementRenderedChange(VisuData *data, VisuElement *element);
/**
 * visuDataEmit_nodeRenderedChange:
 * @data: a valid #VisuData object.
 *
 * This method is used to emit the 'NodeRenderedChanged' signal. This signal asks
 * all modules that are dependent of the nodes' visibility to recompute their
 * OpenGL lists. This signal is usualy emitted after a call to
 * visuDataEmit_askForShowHideNodes().
 */
void visuDataEmit_nodeRenderedChange(VisuData *data);
/**
 * visuDataEmit_observeMovement:
 * @data: a valid #VisuData object ;
 * @start: TRUE to signify a starting movement, FALSE for an ending one.
 *
 * This method is used to emit the 'OpenGLObserveMovement' signal. This signal
 * signify that an observe movement has been initiated or has just finished.
 */
void visuDataEmit_observeMovement(VisuData *data, gboolean start);

/**
 * visuDataAdd_timeout:
 * @data: a valid #VisuData object ;
 * @time: the period of call in milliseconds ;
 * @func: the callback function to be called ;
 * @user_data: a pointer to some user defined informations.
 *
 * This method is used to add the @func method to be called regularly at the period
 * @time. This methos calls in fact g_timeout_add() with the given arguments. But
 * the source id is stored internaly and the timeout function is removed automatically
 * when the object @data is destroyed. It is convienient to add a method working
 * on the #VisuData object that is called periodically during the life of te object.
 *
 * Returns: the source id if the calling method need to work with it. To remove
 *          the callback, don't use g_source_remove() but visuDataRemove_timeout()
 *          to inform the #VisuData object that this source has been removed and
 *          not to remove it when the object will be destroyed.
 */
guint visuDataAdd_timeout(VisuData *data, guint time, GSourceFunc func, gpointer user_data);
/**
 * visuDataRemove_timeout:
 * @data: a valid #VisuData object ;
 * @timeoutId: a source id.
 *
 * This method is used to remove a timeout that has been associated to the given
 * @data (see visuDataAdd_timeout()).
 *
 * Returns: TRUE if the source has been found and removed.
 */
gboolean visuDataRemove_timeout(VisuData *data, guint timeoutId);

/**
 * visuDataGet_renderingWindow:
 * @data: a #VisuData object.
 *
 * The #VisuData objects can be rendered into a window. They are attached
 * to one using visuRenderingWindowSet_visuData() method. The actual method is
 * then used to retrieve the window the argument @data is attached to.
 * 
 * Returns: the window the @data argument is attached to, or NULL
 *          if the object has not yet been attached or if the rendering
 *          is done off-screen.
 */
GenericRenderingWindow visuDataGet_renderingWindow(VisuData *data);
/**
 * visuDataSet_renderingWindow:
 * @data: a #VisuData object ;
 * @window: a pointer to a rendering window (can be NULL).
 *
 * This method is used to tell the @data object which
 * window is used to render it. Normally, one has not to use
 * this method, using visuRenderingWindowSet_visuData() calls it
 * already.
 */
void visuDataSet_renderingWindow(VisuData *data, GenericRenderingWindow window);

/**
 * visuDataGet_openGLView:
 * @data: a #VisuData object.
 *
 * Once the object @data has been initialised, an #OpenGLView object
 * is automattically attached and this method can be used to retrieve
 * it.
 *
 * Returns: the #OpenGLView attached to the given @data.
 */
OpenGLView* visuDataGet_openGLView(VisuData *data);
/**
 * visuDataSet_angleOfView:
 * @data: a #VisuData object ;
 * @valueTheta: a floatinf point value in degrees ;
 * @valuePhi: a floating point value in degrees ;
 * @valueOmega: a floating point value in degrees ;
 * @mask: to specified what values will be changed.
 *
 * This method is used to change the camera orientation for the given @data.
 * If necessary, this method will emit the 'OpenGLThetaPhiOmega' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_angleOfView(VisuData *data, float valueTheta,
			    float valuePhi, float valueOmega, int mask);
/**
 * visuDataSet_positionOfView:
 * @data: a #VisuData object ;
 * @valueX: a floatinf point value in the bounding box scale
 *          (1 is the size of the bounding box) ;
 * @valueY: a floating point value in bounding box scale ;
 * @mask: to specified what values will be changed.
 *
 * This method is used to change the camera position for the given @data.
 * If necessary, this method will emit the 'OpenGLXsYs' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_positionOfView(VisuData *data, float valueX, float valueY, int mask);
/**
 * visuDataSet_zoomOfView:
 * @data: a #VisuData object ;
 * @value: a positive floating point value.
 *
 * This method is used to change the camera zoom for the given @data.
 * If necessary, this method will emit the 'OpenGLGross' signal and
 * the 'OpenGLFacetteChanged' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_zoomOfView(VisuData *data, float value);
/**
 * visuDataSet_perspectiveOfView:
 * @data: a #VisuData object ;
 * @value: a positive floating point value (> 1.1).
 *
 * This method is used to change the camera perspective for the given @data.
 * If necessary, this method will emit the 'OpenGLPersp' signal and
 * the 'OpenGLFacetteChanged' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_perspectiveOfView(VisuData *data, float value);
/**
 * visuDataSet_sizeOfView:
 * @data: a valid #VisuData object ;
 * @width: the new horizontal size ;
 * @height: the new vertical size.
 *
 * It changes the size of the OpenGl area and reccompute the OpenGL viewport.
 * Warning : it doesn't change the size of the window.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_sizeOfView(VisuData *data, guint width, guint height);
/**
 * visuDataEmit_facettesChanged:
 * @data: a valid #VisuData object.
 *
 * Emit the 'OpenGLFacetteChanged' signal.
 */
void visuDataEmit_facettesChanged(VisuData *data);



/**
 * visuDataGet_allObjects:
 *
 * This methods is used to retrieve all #VisuObject currently allocated
 * in V_Sim. It is usefull to apply some changes on all objects (resources
 * for example).
 *
 * Returns: a list of V_Sim own #VisuData objects.
 */
GList* visuDataGet_allObjects();




/* Return the maxium size occupied by a VisuElement that has
   at least one VisuNode. */
/* float getMaxExtens(); */
/* Store a new value in maxDrawingSizeOfVisuElements that controls
   the maximum size occupied by a VisuElement. This function emits
   a signal () that is tarpped by opengl.c. Value must be positive or
   nul. */
/* void setMaxExtends(float value); */

G_END_DECLS

#endif
