/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "panelMap.h"
#include "panelPlanes.h"
#include "panelSurfaces.h"

#include <GL/gl.h>

#include <visu_object.h>
#include <visu_extension.h>
#include <coreTools/toolMatrix.h>
#include <extraFunctions/plane.h>
#include <extraFunctions/scalarFields.h>
#include <extraGtkFunctions/gtk_toolPanelWidget.h>
#include <extraGtkFunctions/gtk_shadeComboBoxWidget.h>
#include <openGLFunctions/objectList.h>
#include <openGLFunctions/view.h>
#include <openGLFunctions/text.h>

/**
 * SECTION:panelMap
 * @short_description: The widget to create coloured map.
 *
 * <para>This is the user interface for the coloured maps. For a
 * plane, a scalar field and a shade, it is possible to draw one
 * coloured plane. The available planes are taken from the #panelPlane
 * subpanel and the scalar field for the #panelSurfaces.</para>
 */

/* Local variables. */
static GtkWidget *panelMap;
static int identifierMap;
static int identifierLegend;
static GtkWidget *comboPlane;
static GtkWidget *comboField;
static GtkWidget *comboShade;
static GtkWidget *spinPrecision;
static GtkWidget *checkLogScale;
static GtkWidget *spinIsoLines;
static GtkWidget *removeButton, *buildButton;
static OpenGLExtension *extensionMap;
static OpenGLExtension *extensionLegend;
static float drawnMinMax[2];

/* Local methods. */
static void panelMapCreate_gtkInterface(ToolPanel *panel);
static void panelMapRebuild_listMap(VisuData *dataObj);
static void panelMapRebuild_listLegend(VisuData *dataObj);
static gboolean panelMapGet_elements(Plane **plane, ScalarField **field, Shade **shade);

/* Local callbacks. */
static void onBuildClicked(GtkButton *button, gpointer data);
static void onRemoveClicked(GtkButton *button, gpointer data);
static void onPrecisionChanged(GtkSpinButton *spin, gpointer data);
static gboolean onIdleRebuild(gpointer data);
static void onComboChanged(GtkComboBox *combo, gpointer data);
static void onComboPlaneChanged(GtkComboBox *combo, gpointer data);
static void onPlaneMoved(Plane *plane, gpointer data);
static void onDataNew(VisuObject *visu, VisuData *dataObj, gpointer data);
static void onViewportChanged(VisuData *dataObj, OpenGLView *view, gpointer data);
static void onZoomChanged(VisuData *dataObj, gpointer data);
static void onSizeChanged(VisuData *dataObj, gpointer data);
static void onIsoChanged(GtkSpinButton *button, gpointer data);
static void onLogChanged(GtkToggleButton *button, gpointer data);

static void panelMapCreate_gtkInterface(ToolPanel *panel)
{
  GtkWidget *vbox, *hbox, *label, *combo, *align;
  GtkListStore *list;
  GtkCellRenderer *renderer;

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(panel), vbox);
  
  label = gtk_label_new(_("<b>Map sources</b>"));
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 3);
  
  /* The plane selector. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Cutting plane:"));
  gtk_misc_set_padding(GTK_MISC(label), 5, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  label = gtk_label_new(_("<span size='small'>"
			  "<i>Create elements in the 'planes' tab</i></span>"));
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_misc_set_padding(GTK_MISC(label), 5, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  
  align = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 15, 15);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 2);

  list = panelPlanesGet_listStore();
  combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(list));
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, FALSE);
  g_object_set(G_OBJECT(renderer), "xpad", 10, NULL);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(combo), renderer,
				"pixbuf", COLUMN_PLANE_COLOR_PIXBUF);
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, FALSE);
  g_object_set(G_OBJECT(renderer), "xalign", 1.0, NULL);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(combo), renderer,
				"markup", COLUMN_PLANE_LABEL);
  gtk_container_add(GTK_CONTAINER(align), combo);
  comboPlane = combo;
  g_signal_connect(G_OBJECT(combo), "changed",
		   G_CALLBACK(onComboPlaneChanged), (gpointer)0);

  /* The scalar field selector. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Scalar field:"));
  gtk_misc_set_padding(GTK_MISC(label), 5, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  label = gtk_label_new(_("<span size='small'>"
			  "<i>Import fields in the 'isosurfaces' tab</i></span>"));
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_misc_set_padding(GTK_MISC(label), 5, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  
  align = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 15, 15);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 2);

  list = panelIsosurfacesGet_listStore();
  combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(list));
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, FALSE);
  g_object_set(G_OBJECT(renderer), "xalign", 1.0, NULL);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(combo), renderer,
				"markup", COLUMN_FIELD_LABEL);
  gtk_container_add(GTK_CONTAINER(align), combo);
  comboField = combo;
  g_signal_connect(G_OBJECT(combo), "changed",
		   G_CALLBACK(onComboChanged), (gpointer)0);

  /* The shade selector. */
  label = gtk_label_new(_("Shade:"));
  gtk_misc_set_padding(GTK_MISC(label), 5, 0);
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  
  align = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 0, 15, 15);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 2);

  combo = shadeComboBox_new(TRUE);
  gtk_container_add(GTK_CONTAINER(align), combo);
  comboShade = combo;
  g_signal_connect(G_OBJECT(combo), "changed",
		   G_CALLBACK(onComboChanged), (gpointer)0);

  /* Options. */
  label = gtk_label_new(_("<b>Options</b>"));
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 3);
  
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new(_("Precision:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
  spinPrecision = gtk_spin_button_new_with_range(10, 500, 10);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spinPrecision), 0);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinPrecision), 100);
  gtk_box_pack_start(GTK_BOX(hbox), spinPrecision, FALSE, FALSE, 0);
  label = gtk_label_new("%");
  gtk_misc_set_padding(GTK_MISC(label), 5, 0);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(spinPrecision), "value-changed",
		   G_CALLBACK(onPrecisionChanged), (gpointer)0);

  checkLogScale = gtk_check_button_new_with_mnemonic(_("_Logarithmic scale"));
  gtk_box_pack_end(GTK_BOX(hbox), checkLogScale, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(checkLogScale), "toggled",
		   G_CALLBACK(onLogChanged), (gpointer)0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new(_("Number of isolines:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
  spinIsoLines = gtk_spin_button_new_with_range(0, 20, 1);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spinIsoLines), 0);
  gtk_box_pack_start(GTK_BOX(hbox), spinIsoLines, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(spinIsoLines), "value-changed",
		   G_CALLBACK(onIsoChanged), (gpointer)0);

  /* The action buttons. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_end(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  removeButton = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
  gtk_widget_set_sensitive(removeButton, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox), removeButton, FALSE, FALSE, 15);
  g_signal_connect(G_OBJECT(removeButton), "clicked",
		   G_CALLBACK(onRemoveClicked), (gpointer)0);

  buildButton = gtk_button_new_with_mnemonic(_("_Build"));
  gtk_box_pack_end(GTK_BOX(hbox), buildButton, FALSE, FALSE, 15);
  g_signal_connect(G_OBJECT(buildButton), "clicked",
		   G_CALLBACK(onBuildClicked), (gpointer)0);

  gtk_widget_show_all(vbox);
}

void panelMapSet(guint planeId, guint scalarFieldId, guint shadeId)
{
  VisuData *dataObj;

  extensionMap->used = FALSE;
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboPlane), planeId);
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboField), scalarFieldId);
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboShade), shadeId);
  extensionMap->used = TRUE;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelMap));
  panelMapRebuild_listMap(dataObj);
  panelMapRebuild_listLegend(dataObj);

  /* Update the interface. */
  gtk_widget_set_sensitive(removeButton, TRUE);
}

static void onBuildClicked(GtkButton *button _U_, gpointer data _U_)
{
  /* Add as an idle function the graphic update. */
  g_idle_add(onIdleRebuild, (gpointer)0);
}

static void onPrecisionChanged(GtkSpinButton *spin _U_, gpointer data _U_)
{
  if (extensionMap->used)
    /* Add as an idle function the graphic update. */
    g_idle_add(onIdleRebuild, (gpointer)0);
}

static void onComboChanged(GtkComboBox *combo _U_, gpointer data _U_)
{
  if (extensionMap->used)
    /* Add as an idle function the graphic update. */
    g_idle_add(onIdleRebuild, (gpointer)0);
}
static void onComboPlaneChanged(GtkComboBox *combo, gpointer data _U_)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gboolean valid;
  Plane *plane;

  /* We get the plane to listen to its signals. */
  model = gtk_combo_box_get_model(combo);
  valid = gtk_combo_box_get_active_iter(combo, &iter);
  if (!model || !valid)
    return;
  gtk_tree_model_get(model, &iter, COLUMN_PLANE_POINTER, &plane, -1);
  /* Because the get action increase the counter... */
  g_object_unref(G_OBJECT(plane));

  /* We listen to the moved signal. */
  g_signal_connect(G_OBJECT(plane), "moved",
		   G_CALLBACK(onPlaneMoved), (gpointer)0);

  if (extensionMap->used)
    /* Add as an idle function the graphic update. */
    g_idle_add(onIdleRebuild, (gpointer)0);
}

static void onPlaneMoved(Plane *plane _U_, gpointer data _U_)
{
  if (extensionMap->used)
    /* Add as an idle function the graphic update. */
    g_idle_add(onIdleRebuild, (gpointer)0);
}

static void onRemoveClicked(GtkButton *button _U_, gpointer data _U_)
{
  gboolean valid;
  Plane *plane;
  Shade *shade;
  ScalarField *field;

  /* Return the visibility status of the plane to TRUE. */
  valid = panelMapGet_elements(&plane, &field, &shade);
  if (plane)
    panelPlanesSet_planeRendered(plane, TRUE);

  /* Update the interface. */
  gtk_widget_set_sensitive(removeButton, FALSE);

  if (extensionMap->used == 0)
    return;
  extensionMap->used = 0;
  extensionLegend->used = 0;
  g_idle_add(visuObjectRedraw, (gpointer)0);
}

static void onDataNew(VisuObject *visu _U_, VisuData *dataObj, gpointer data _U_)
{
  extensionMap->used = 0;
  extensionLegend->used = 0;

  DBG_fprintf(stderr, "Panel Map: get the 'dataNew' signal.\n");
  DBG_fprintf(stderr, " | add signals...\n");
  g_signal_connect(G_OBJECT(dataObj), "OpenGLWidthHeight",
                   G_CALLBACK(onViewportChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLFacetteChanged",
                   G_CALLBACK(onZoomChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLNearFar",
                   G_CALLBACK(onSizeChanged), (gpointer)0);
}

static void onViewportChanged(VisuData *dataObj, OpenGLView *view _U_,
			      gpointer data _U_)
{
  DBG_fprintf(stderr, "Panel Map: get the 'openglWidthHeight' signal.\n");
  if (extensionMap->used)
    panelMapRebuild_listLegend(dataObj);
}
static void onZoomChanged(VisuData *dataObj, gpointer data _U_)
{
  DBG_fprintf(stderr, "Panel Map: get the 'OpenGLFacetteChanged' signal.\n");
  if (extensionMap->used)
    panelMapRebuild_listMap(dataObj);
}
static void onSizeChanged(VisuData *dataObj, gpointer data _U_)
{
  DBG_fprintf(stderr, "Panel Map: get the 'OpenGLNearFar' signal.\n");
  if (extensionMap->used)
    panelMapRebuild_listMap(dataObj);
}

static gboolean onIdleRebuild(gpointer data _U_)
{
  VisuData *dataObj;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelMap));
  panelMapRebuild_listMap(dataObj);
  panelMapRebuild_listLegend(dataObj);

  if (extensionMap->used)
    {
      /* Update the interface. */
      gtk_widget_set_sensitive(removeButton, TRUE);

      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
  return FALSE;
}

static void onLogChanged(GtkToggleButton *button _U_, gpointer data _U_)
{
  if (extensionMap->used)
    /* Add as an idle function the graphic update. */
    g_idle_add(onIdleRebuild, (gpointer)0);
}

static void onIsoChanged(GtkSpinButton *button _U_, gpointer data _U_)
{
  if (extensionMap->used)
    /* Add as an idle function the graphic update. */
    g_idle_add(onIdleRebuild, (gpointer)0);
}


static gboolean panelMapGet_elements(Plane **plane, ScalarField **field, Shade **shade)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gboolean valid;

  model = gtk_combo_box_get_model(GTK_COMBO_BOX(comboPlane));
  valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(comboPlane), &iter);
  if (!model || !valid)
    return FALSE;
  gtk_tree_model_get(model, &iter, COLUMN_PLANE_POINTER, plane, -1);
  /* Because the get action increase the counter... */
  g_object_unref(G_OBJECT(*plane));

  model = gtk_combo_box_get_model(GTK_COMBO_BOX(comboField));
  valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(comboField), &iter);
  if (!model || !valid)
    return FALSE;
  gtk_tree_model_get(model, &iter, COLUMN_FIELD_POINTER, field, -1);

  *shade = shadeComboBoxGet_selectedShade(SHADE_COMBOX(comboShade));
  if (!*shade)
    return FALSE;

  return TRUE;
}

static void panelMapRebuild_listMap(VisuData *dataObj)
{
  gboolean valid;
  Plane *plane;
  Shade *shade;
  ScalarField *field;

  DBG_fprintf(stderr, "Panel Map : rebuilding object list for map.\n");

  valid = panelMapGet_elements(&plane, &field, &shade);
  if (!valid)
    {
      DBG_fprintf(stderr, "Panel Map : aborting, not all elements are available.\n");
      extensionMap->used = 0;
      return;
    }
  /* Change the visibility status of the plane to FALSE. */
  panelPlanesSet_planeRendered(plane, FALSE);
  extensionMap->used = 1;
  
  glNewList(identifierMap, GL_COMPILE);
  scalarFieldDraw_map(field, plane, shade, visuDataGet_openGLView(dataObj),
                      gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinPrecision)),
                      drawnMinMax, gtk_toggle_button_get_active
		      (GTK_TOGGLE_BUTTON(checkLogScale)),
		      gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinIsoLines)));
  glEndList();
}

static void panelMapRebuild_listLegend(VisuData *dataObj _U_)
{
  Shade *shade;
  ScalarField *field;
  Plane *plane;
  gboolean valid;
  double minMax[2];
  float fMinMax[2], smin;

  DBG_fprintf(stderr, "Panel Map : rebuilding object list for legend.\n");

  valid = panelMapGet_elements(&plane, &field, &shade);
  if (!valid)
    {
      DBG_fprintf(stderr, "Panel Map : aborting, not all elements are available.\n");
      extensionLegend->used = 0;
      return;
    }
  extensionLegend->used = 1;

  scalarFieldGet_minMax(field, minMax);
  fMinMax[0] = minMax[0];
  fMinMax[1] = minMax[1];
  smin       = minMax[0];
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkLogScale)))
    smin = scalarFieldGet_secondaryMin(field);
  
  openGLText_rebuildFontList();

  glNewList(identifierLegend, GL_COMPILE);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_ALPHA_TEST);
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkLogScale)))
    shadeDraw_legend(shade, .05, -1., fMinMax, drawnMinMax, 2, &smin);
  else
    shadeDraw_legend(shade, .05, -1., fMinMax, drawnMinMax, 2, (float*)0);

  glEndList();
}
void panelMapSet_logScale(gboolean logScale)
{
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkLogScale), logScale);
}
void panelMapSet_nIsoLines(guint nIsoLines)
{
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinIsoLines), nIsoLines);
}


ToolPanel* panelMapInit()
{
  /* Long description */
  char *cl = _("Map projections");
  /* Short description */
  char *tl = _("Maps");

  panelMap = toolPanelNew_withIconFromPath("Panel_map", cl, tl, "stock-map_20.png");
  g_return_val_if_fail(panelMap, (ToolPanel*)0);

  toolPanelSet_dockable(TOOL_PANEL(panelMap), TRUE);

  /* Create the widgets. */
  panelMapCreate_gtkInterface(TOOL_PANEL(panelMap));
  
  /* Extension to draw values. */
  identifierMap = openGLObjectList_new(1);
  extensionMap = OpenGLExtension_new("Map", _("Map"),
				     _("Drawing extension for maps."),
				     identifierMap, panelMapRebuild_listMap);
  extensionMap->used = 0;
  OpenGLExtensionSet_priority(extensionMap, OPENGL_EXTENSION_PRIORITY_FIRST);
  OpenGLExtensionSet_sensitiveToRenderingMode(extensionMap, FALSE);
  OpenGLExtensionRegister(extensionMap);

  /* Extension to draw the legend. */
  identifierLegend = openGLObjectList_new(1);
  extensionLegend = OpenGLExtension_new("MapLegend", _("Map legend"),
					_("Responsible for drawing"
					  " the legend of the color map."),
					identifierLegend, panelMapRebuild_listLegend);
  extensionLegend->used = 0;
  OpenGLExtensionSet_priority(extensionLegend, OPENGL_EXTENSION_PRIORITY_LAST);
  OpenGLExtensionSet_saveOpenGLState(extensionLegend, TRUE);
  OpenGLExtensionRegister(extensionLegend);

  g_signal_connect(VISU_INSTANCE, "dataNew",
                   G_CALLBACK(onDataNew), (gpointer)0);

  return TOOL_PANEL(panelMap);
}
