/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "gtkSpin.h"

#include <visu_gtk.h>
#include <visu_elements.h>
#include <renderingMethods/renderingAtomic.h>
#include <renderingMethods/renderingSpin.h>
#include <openGLFunctions/view.h>
#include <coreTools/toolMatrix.h>
#include "gtkAtomic.h"
#include "panelElements.h"
#include "visuConfig.h"
#include "panelMethod.h"

#include <assert.h>
#include <string.h>
#include <math.h>

/**
 * SECTION:gtkSpin
 * @short_description: The gtk interface elements that can interfere
 * with renderingSpin parameters.
 *
 * <para>This is the gtk interface for all #RenderingSpin module
 * parameters. They are split in two parts. The first part is placed
 * under the config tab and allows the user to modify the "global
 * properties" of the renderingSpin module such as cone's axe
 * orientation, and color repartition. The second part is under the
 * element panel and allows to modify each element resource. Size of
 * the arrows, and their shape can be modified.</para>
 */

#define LABEL_ARROW_SIZE      _("<i>Shape size and color properties:</i>")
#define LABEL_ARROW_SIZE_MULT _("<i>Shape size and color properties <span style=\"italic\" size=\"smaller\" foreground=\"red\">\303\227 mult.</span>:</i>")
#define LABEL_SPIN_POLICY     _("Drawing policy for spins with null modulus:")

void set_view(GtkWidget* button, gpointer data);

/* A widget to select a shape for the arrows */
static GtkWidget *gtkw_element_shape_number = NULL;
static GtkWidget *heightResourceSpin = NULL;
static GtkWidget *rheightResourceSpin = NULL;
static GtkWidget *lengthResourceSpin = NULL;
static GtkWidget *rlengthResourceSpin = NULL;
static GtkWidget *ratioElipsoidSpin = NULL;
static GtkWidget *lengthElipsoidSpin = NULL;
static GtkWidget *gtkw_cone_theta_angle = NULL;
static GtkWidget *gtkw_cone_phi_angle = NULL;
static GtkWidget *gtkw_color_wheel_angle = NULL;
static GtkWidget *gtkw_use_element_color = NULL;
static GtkWidget *gtkw_use_element_color_hat = NULL;
static GtkWidget *useElementForElipsoid;
static GtkWidget *gtkw_x = NULL, *gtkw_y = NULL, *gtkw_z = NULL; 
static GtkWidget *gtkw_set_view;
static GtkWidget *labelSizeProp;
static GtkWidget *vboxArrowShape, *vboxElipsoidShape;
static GtkWidget *radioPerTypeMod, *radioGlobalMod;

static guint element_callback_ids[10];

static int need_redraw = 1;

static GList* currentListElement;

static gulong configResourcesLoadedId;

struct FileChooserSpin_struct
{
  GtkLabel* label[2];
  GtkRadioButton* radio[2];
  gchar* files[2];
  GtkFileChooserDialog *fileChooserDialog;
  int fileKind;

  VisuGtkPreview preview;
};
/* A special widget to select two kind of files at a time. */
static gboolean gtkSpinCreate_fileChooser(VisuData *data, GtkWindow *parent);
/* Its callbacks. */
static void fileSelectedForSpinMethod(GtkButton *button, gpointer data);
static void fileKindChangedForPosition(GtkToggleButton *button, gpointer data);
static void fileKindChangedForSpin(GtkToggleButton *button, gpointer data);
static void fileUpdatePreview(GtkFileChooser *chooser,
			      struct FileChooserSpin_struct *fileChooserSpin);

/* Callbacks */
void global_resource_callback(GtkWidget* widget, gpointer data);
void element_resource_callback(GtkWidget* widget, gpointer data);
void changeSomethingSpin(GList *eleList);
void onSpinBoundsChangeArrowSize(GObject *obj, gfloat mult, gpointer data);
GtkWidget* createGtkInterfaceForSpinMethod();
static void onPolicyChanged(GtkToggleButton *toggle, gpointer data);
static void onPolicyAtomicChanged(GtkToggleButton *toggle, gpointer data);
static void onPolicyModulusChanged(GtkToggleButton *toggle, gpointer data);

GtkWidget* createGtkConfigForSpinMethod(void);
void destroyGtkConfigForSpinMethod();

void redraw(int i);

static void sync_global_resources(GObject *object, VisuData *dataObj, gpointer data);
static void sync_local_resources(GObject *trash, gpointer data);


/***************/
/* Public part */
/***************/

/* See h file for more info */
void panelElementSpinInit_gtkPanel()
{
  panelElements_setInterfaceMethods(renderingSpinGet_static(),
				    changeSomethingSpin,
				    createGtkInterfaceForSpinMethod);
  currentListElement = (GList*)0;
}

/* See h file for more info */
void panelElementSpinInit_gtkConfig()
{
  panelMethodSet_renderingInterface(renderingSpinGet_static(),
				    createGtkConfigForSpinMethod,
				    destroyGtkConfigForSpinMethod);
}

/* Creates the hbox containing x,y,z spin buttons */
static GtkWidget *gtk_spin_create_direction(float *cartesian_values,
					    float *spherical_values)
{
  GtkWidget *table;
  GtkWidget* label_x = gtk_label_new(_("x:"));
  GtkWidget* label_y = gtk_label_new(_("y:"));
  GtkWidget* label_z = gtk_label_new(_("z:"));
  GtkWidget* label_theta = gtk_label_new("\316\270:");
  GtkWidget* label_phi = gtk_label_new("\317\206:");
  GtkTooltips *tooltips;

  table = gtk_table_new(2, 6, FALSE);

  gtk_misc_set_alignment(GTK_MISC(label_x), 1., 0.5);
  gtk_misc_set_alignment(GTK_MISC(label_y), 1., 0.5);
  gtk_misc_set_alignment(GTK_MISC(label_z), 1., 0.5);

  gtkw_x = gtk_spin_button_new_with_range(-99.99, 99.99, 0.05);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_x), cartesian_values[0]);
  g_signal_connect(gtkw_x, "value-changed", 
		   G_CALLBACK(global_resource_callback), GINT_TO_POINTER(-1));   

  gtkw_y = gtk_spin_button_new_with_range(-99.99, 99.99, 0.05);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_y), cartesian_values[1]);
  g_signal_connect(gtkw_y, "value-changed", 
		   G_CALLBACK(global_resource_callback), GINT_TO_POINTER(-2));   

  gtkw_z = gtk_spin_button_new_with_range(-99.99, 99.99, 0.05);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_z), cartesian_values[2]);
  g_signal_connect(gtkw_z, "value-changed", 
		   G_CALLBACK(global_resource_callback), GINT_TO_POINTER(-3));   

  gtk_table_attach(GTK_TABLE(table), label_x, 0, 1, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 2);
  gtk_table_attach(GTK_TABLE(table), label_y, 2, 3, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 2);
  gtk_table_attach(GTK_TABLE(table), label_z, 4, 5, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 2);

  gtk_table_attach(GTK_TABLE(table), gtkw_x, 1, 2, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 1, 2);
  gtk_table_attach(GTK_TABLE(table), gtkw_y, 3, 4, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 1, 2);
  gtk_table_attach(GTK_TABLE(table), gtkw_z, 5, 6, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 1, 2);

  tooltips = gtk_tooltips_new();

  gtk_misc_set_alignment(GTK_MISC(label_theta), 1., 0.5);
  gtk_misc_set_alignment(GTK_MISC(label_phi), 1., 0.5);

  gtkw_cone_theta_angle = gtk_spin_button_new_with_range(0, 180., 3);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_theta_angle), spherical_values[1]);
  g_signal_connect(gtkw_cone_theta_angle, "value-changed",
		   G_CALLBACK(global_resource_callback), GINT_TO_POINTER(spin_globalConeTheta));

  gtkw_cone_phi_angle = gtk_spin_button_new_with_range(0, 360., 3);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_phi_angle), spherical_values[2]);
  g_signal_connect(gtkw_cone_phi_angle, "value-changed",
		   G_CALLBACK(global_resource_callback), GINT_TO_POINTER(spin_globalConePhi));
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(gtkw_cone_phi_angle), TRUE);

  gtkw_set_view = gtk_button_new_with_label (_("Set ortho."));
  gtk_tooltips_set_tip(tooltips, gtkw_set_view,
		       _("Set the cone orientation to be orthogonal to the screen."), NULL);
  g_signal_connect(gtkw_set_view, "clicked",
		   G_CALLBACK(set_view), NULL);

  gtk_table_attach(GTK_TABLE(table), label_theta, 0, 1, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 2);
  gtk_table_attach(GTK_TABLE(table), label_phi,   2, 3, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 2);

  gtk_table_attach(GTK_TABLE(table), gtkw_cone_theta_angle, 1, 2, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 1, 2);
  gtk_table_attach(GTK_TABLE(table), gtkw_cone_phi_angle, 3, 4, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_SHRINK, 1, 2);

  gtk_table_attach(GTK_TABLE(table), gtkw_set_view, 4, 6, 1, 2,
		   GTK_EXPAND, GTK_SHRINK, 1, 2);

  return table;
}

/* Creates the hbox containing the color_wheel_angle button */
static GtkWidget *gtk_spin_create_color_wheel(float color_wheel_angle)
{
  GtkWidget* hbox = gtk_hbox_new(FALSE, 0);
  GtkWidget* label = gtk_label_new(_("Rotate color wheel:"));

  gtk_misc_set_alignment(GTK_MISC(label), 0.,0.5);

  gtkw_color_wheel_angle = gtk_spin_button_new_with_range(0, 360., 3);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_color_wheel_angle), color_wheel_angle);
  g_signal_connect(gtkw_color_wheel_angle, "value-changed",
		   G_CALLBACK(global_resource_callback), GINT_TO_POINTER(spin_globalColorWheel));
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(gtkw_color_wheel_angle), TRUE);

  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(hbox), gtkw_color_wheel_angle, FALSE, FALSE, 1);

  /* Degrees. */
  label = gtk_label_new(_("deg."));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 1);

  return hbox;
}

GtkWidget* createHidingModeRadioWidgets()
{
  GtkWidget *radioAlwaysSpin, *radioEmptySpin, *radioAtomicSpin;
  GtkWidget *hbox;
  GSList *radiobutton_group;

  hbox = gtk_hbox_new(FALSE, 0);
  radioAlwaysSpin = gtk_radio_button_new_with_label(NULL, _("always"));
  gtk_radio_button_set_group(GTK_RADIO_BUTTON(radioAlwaysSpin), (GSList*)0);
  radiobutton_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radioAlwaysSpin));
  gtk_box_pack_start(GTK_BOX(hbox), radioAlwaysSpin, TRUE, TRUE, 1);
  g_signal_connect(G_OBJECT(radioAlwaysSpin), "toggled",
		   G_CALLBACK(onPolicyChanged), GINT_TO_POINTER(policyAlwaysSpin));
/*   image1 = create_pixmap(mainWindow, "stock-go-around.png"); */
/*   gtk_widget_show (image1); */
/*   gtk_container_add(GTK_CONTAINER(radioGoAround), image1); */
  radioEmptySpin = gtk_radio_button_new_with_label(NULL, _("never"));
  gtk_radio_button_set_group(GTK_RADIO_BUTTON(radioEmptySpin), radiobutton_group);
  radiobutton_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radioEmptySpin));
  gtk_box_pack_start(GTK_BOX(hbox), radioEmptySpin, TRUE, TRUE, 1);
  g_signal_connect(G_OBJECT(radioEmptySpin), "toggled",
		   G_CALLBACK(onPolicyChanged), GINT_TO_POINTER(policyHideNullSpin));
/*   image1 = create_pixmap(mainWindow, "stock-go-once.png"); */
/*   gtk_widget_show (image1); */
/*   gtk_container_add(GTK_CONTAINER(radioGoOnce), image1); */
  radioAtomicSpin = gtk_radio_button_new_with_label(NULL, _("atomic"));
  gtk_radio_button_set_group(GTK_RADIO_BUTTON(radioAtomicSpin), radiobutton_group);
  radiobutton_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radioAtomicSpin));
  gtk_box_pack_start(GTK_BOX(hbox), radioAtomicSpin, TRUE, TRUE, 1);
  g_signal_connect(G_OBJECT(radioAtomicSpin), "toggled",
		   G_CALLBACK(onPolicyChanged), GINT_TO_POINTER(policyAtomicNullSpin));
/*   image1 = create_pixmap(mainWindow, "stock-go-and-back.png"); */
/*   gtk_widget_show (image1); */
/*   gtk_container_add(GTK_CONTAINER(radioGoAndBack), image1); */
  switch (rspin_getGlobalResource_uint(spin_globalHidingMode))
    {
    case policyAlwaysSpin:
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioAlwaysSpin), TRUE);
      break;
    case policyHideNullSpin:
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioEmptySpin), TRUE);
      break;
    case policyAtomicNullSpin:
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioAtomicSpin), TRUE);
      break;
    default:
      g_warning("Wrong 'hiding-mode'.");
    }

  gtk_widget_show_all(hbox);
  return hbox;
}
  
/* Creates the vbox displayed in the config panel */
GtkWidget* createGtkConfigForSpinMethod()
{
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  GtkWidget *label2 = gtk_label_new(_("Color cone orientation:"));
  float cartesian[3];
  float spherical[3];
  GtkWidget *labelPolicy;
  GtkWidget *hbox, *check, *label, *align;
  GSList *radiobutton_group;
  SpinModulusPolicy modPol;

  DBG_fprintf(stderr, "Gtk Spin : building specific spin rendering method config widget.\n");

  spherical[0] = 1;
  spherical[1] = rspin_getGlobalResource_float(spin_globalConeTheta);
  spherical[2] = rspin_getGlobalResource_float(spin_globalConePhi);

  matrix_sphericalToCartesian(cartesian, spherical);

  /* Use modulus. */
  modPol = rspin_getGlobalResource_uint(spin_globalModulus);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Spin lenght is proportional to modulus: "));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_widget_set_name(label, "label_head_2");
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 3);
  check = gtk_check_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), check, FALSE, FALSE, 3);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check),
			       (modPol != policyNoneModulus));
  g_signal_connect(G_OBJECT(check), "toggled",
		   G_CALLBACK(onPolicyModulusChanged),
		   GINT_TO_POINTER(policyNoneModulus));

  hbox = gtk_hbox_new(FALSE, 0);
  radioPerTypeMod = gtk_radio_button_new_with_label(NULL, _("per node type"));
  gtk_radio_button_set_group(GTK_RADIO_BUTTON(radioPerTypeMod), (GSList*)0);
  radiobutton_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radioPerTypeMod));
  gtk_box_pack_start(GTK_BOX(hbox), radioPerTypeMod, TRUE, TRUE, 1);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioPerTypeMod),
			       modPol == policyPerTypeModulus);
  gtk_widget_set_sensitive(radioPerTypeMod, (modPol != policyNoneModulus));
  g_signal_connect(G_OBJECT(radioPerTypeMod), "toggled",
		   G_CALLBACK(onPolicyModulusChanged),
		   GINT_TO_POINTER(policyPerTypeModulus));
  radioGlobalMod = gtk_radio_button_new_with_label(NULL, _("globaly"));
  gtk_radio_button_set_group(GTK_RADIO_BUTTON(radioGlobalMod), radiobutton_group);
  radiobutton_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radioGlobalMod));
  gtk_box_pack_start(GTK_BOX(hbox), radioGlobalMod, TRUE, TRUE, 1);
  g_signal_connect(G_OBJECT(radioGlobalMod), "toggled",
		   G_CALLBACK(onPolicyModulusChanged),
		   GINT_TO_POINTER(policyGlobalModulus));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radioGlobalMod),
			       modPol == policyGlobalModulus);
  gtk_widget_set_sensitive(radioGlobalMod, (modPol != policyNoneModulus));
  gtk_widget_show_all(hbox);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  /* Drawing atomic. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Use atomic rendering in addition to spin: "));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_widget_set_name(label, "label_head_2");
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 3);
  check = gtk_check_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), check, FALSE, FALSE, 3);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check),
			       rspin_getGlobalResource_boolean(spin_globalAtomic));
  g_signal_connect(G_OBJECT(check), "toggled",
		   G_CALLBACK(onPolicyAtomicChanged), (gpointer)0);

  /* Drawing policy. */
  labelPolicy = gtk_label_new(LABEL_SPIN_POLICY);
  gtk_misc_set_alignment(GTK_MISC(labelPolicy), 0., 0.5);
  gtk_misc_set_padding(GTK_MISC(labelPolicy), 3, 0);
  gtk_widget_set_name(labelPolicy, "label_head_2");
  gtk_box_pack_start(GTK_BOX(vbox), labelPolicy, FALSE, FALSE, 3);

  hbox = createHidingModeRadioWidgets();
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(label), _("Color distribution options:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_misc_set_padding(GTK_MISC(label), 3, 0);
  gtk_widget_set_name(label, "label_head_2");
  align = gtk_alignment_new(0.5, 0.5, 1., 1.);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 5, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 3);
  gtk_container_add(GTK_CONTAINER(align), label);

  gtk_box_pack_start(GTK_BOX(vbox),
		     gtk_spin_create_color_wheel(rspin_getGlobalResource_float
						 (spin_globalColorWheel)),
		     FALSE, FALSE, 3);

  gtk_box_pack_start(GTK_BOX(vbox), label2, FALSE, FALSE, 3);
  gtk_misc_set_alignment(GTK_MISC(label2), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), gtk_spin_create_direction(cartesian, spherical), FALSE, FALSE, 3);

  gtk_widget_show_all(vbox);

  configResourcesLoadedId = g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
					     G_CALLBACK(sync_global_resources), NULL);

  return vbox;
}
void destroyGtkConfigForSpinMethod()
{
  DBG_fprintf(stderr, "Gtk Spin : remove signal from config rendering widget.\n");
  g_signal_handler_disconnect(VISU_INSTANCE, configResourcesLoadedId);
}
static void onPolicyChanged(GtkToggleButton *toggle, gpointer data)
{
  if (!gtk_toggle_button_get_active(toggle))
    return;

  if (rspin_setGlobalResource_uint(spin_globalHidingMode, (guint)GPOINTER_TO_INT(data)))
    redraw(2);
}
static void onPolicyAtomicChanged(GtkToggleButton *toggle, gpointer data _U_)
{
  VisuData *dataObj;

  if (rspin_setGlobalResource_boolean(spin_globalAtomic,
				      gtk_toggle_button_get_active(toggle)))
    {
      dataObj = toolPanelGet_visuData(TOOL_PANEL(PANEL_ELEMENTS));
      if (!dataObj)
	return;

      visuData_createAllElements(dataObj);
      redraw(2);
    }
}
static void onPolicyModulusChanged(GtkToggleButton *toggle, gpointer data)
{
  VisuData *dataObj;
  SpinModulusPolicy modPol;

  if ((SpinModulusPolicy)data == policyNoneModulus)
    {
      if (!gtk_toggle_button_get_active(toggle))
	modPol = policyNoneModulus;
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioPerTypeMod)))
	modPol = policyPerTypeModulus;
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioGlobalMod)))
	modPol = policyGlobalModulus;
      else
	modPol = policyNoneModulus;
    }
  else
    modPol = (SpinModulusPolicy)data;
  gtk_widget_set_sensitive(radioGlobalMod, gtk_toggle_button_get_active(toggle));
  gtk_widget_set_sensitive(radioPerTypeMod, gtk_toggle_button_get_active(toggle));
  if (rspin_setGlobalResource_uint(spin_globalModulus, modPol))
    {
      dataObj = toolPanelGet_visuData(TOOL_PANEL(PANEL_ELEMENTS));
      if (!dataObj)
	return;

      visuData_createAllElements(dataObj);
      redraw(2);
    }
}

/* Initialise the gtk methods associated with
   the spin rendering method. */
void panelElementSpinInit_gtkMain()
{
  gtkMainClassSet_renderingSpecificMethods(renderingSpinGet_static(),
					   gtkSpinCreate_fileChooser);
}


/* Creates a custom file chooser which allows the user to select a spin and a position file */
static gboolean gtkSpinCreate_fileChooser(VisuData *data, GtkWindow *parent)
{
  GtkWidget *fileSelection;
  GList *tmpLst, *tmpLst2;
  GtkWidget *vbox;
  gchar* directory;
  GtkFileFilter *filterPos, *filterSpin, *filterAll;
  GtkWidget *table, *hbox, *align;
  GtkWidget *label, *button;
  GtkWidget *labelPosition, *labelSpin;
  GtkWidget *radioPosition, *radioSpin;
  GtkWidget *hiding_mode;
  struct FileChooserSpin_struct fileChooserSpin;
  gint status;


  if (!data)
    return 0;

  fileSelection = gtk_file_chooser_dialog_new(_("Load session"),
					      GTK_WINDOW(parent),
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_OK,
					      NULL);
  directory = gtkMainClassGet_lastOpenDirectory();
  if (directory)
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(fileSelection), directory);

  gtk_widget_set_name(fileSelection, "filesel");
  gtk_window_set_position(GTK_WINDOW(fileSelection), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_modal(GTK_WINDOW(fileSelection), TRUE);

  tmpLst = renderingMethodGet_fileType(renderingAtomicGet_static(),
				       FILE_KIND_POSITION);
  filterPos = gtk_file_filter_new ();
  while(tmpLst)
    {
      tmpLst2 = ((FileFormat*)tmpLst->data)->fileType;
      while (tmpLst2)
	{
	  gtk_file_filter_add_pattern (filterPos, (char*)tmpLst2->data);
	  tmpLst2 = g_list_next(tmpLst2);
	}
      tmpLst = g_list_next(tmpLst);
    }
  gtk_file_filter_set_name(filterPos,
			   renderingMethodGet_fileTypeName(renderingSpinGet_static(),
							   FILE_KIND_POSITION));

  tmpLst = renderingMethodGet_fileType(renderingSpinGet_static(), FILE_KIND_SPIN);
  filterSpin = gtk_file_filter_new ();
  while(tmpLst)
    {
      tmpLst2 = ((FileFormat*)tmpLst->data)->fileType;
      while (tmpLst2)
	{
	  gtk_file_filter_add_pattern (filterSpin, (char*)tmpLst2->data);
	  tmpLst2 = g_list_next(tmpLst2);
	}
      tmpLst = g_list_next(tmpLst);
    }
  gtk_file_filter_set_name(filterSpin,
			   renderingMethodGet_fileTypeName(renderingSpinGet_static(),
							   FILE_KIND_SPIN));
  filterAll = gtk_file_filter_new ();
  gtk_file_filter_add_pattern (filterAll, "*");
  gtk_file_filter_set_name(filterAll, _("All files"));
  
/*   g_signal_connect(G_OBJECT(fileSelection), "file-activated", */
/* 		   G_CALLBACK(fileSelectedForSpinMethod), (gpointer)&fileChooserSpin); */

  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(fileSelection), filterPos);
  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(fileSelection), filterSpin);
  gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(fileSelection), filterAll);
  gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(fileSelection), filterPos);

  /* The preview widget. */
  visuGtkPreview_add(&fileChooserSpin.preview, GTK_FILE_CHOOSER(fileSelection));
  g_signal_connect(GTK_FILE_CHOOSER(fileSelection), "update-preview",
		   G_CALLBACK(fileUpdatePreview), &fileChooserSpin);

  /* The additional widgets. */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(fileSelection), vbox);
  
  label = gtk_label_new(_("Current files:"));
  gtk_widget_set_name(label, "label_head_2");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 2);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);

  /* The table for the two file types selections (spin & position). */
  table = gtk_table_new(2, 3, FALSE);
  gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 2);
  
  /* Position button. */
  radioPosition = gtk_radio_button_new_with_label(NULL, _("position"));
  gtk_table_attach(GTK_TABLE(table), radioPosition, 0, 1, 0, 1,
		   GTK_SHRINK, GTK_SHRINK, 2, 0);
  g_signal_connect(G_OBJECT(radioPosition), "toggled",
		   G_CALLBACK(fileKindChangedForPosition),
		   (gpointer)&fileChooserSpin);

  /* Spin button */
  radioSpin = gtk_radio_button_new_with_label_from_widget
    (GTK_RADIO_BUTTON(radioPosition), _("spin"));
  gtk_table_attach(GTK_TABLE(table), radioSpin, 0, 1, 1, 2,
		   GTK_SHRINK, GTK_SHRINK, 2, 0);
  g_signal_connect(G_OBJECT(radioSpin), "toggled",
		   G_CALLBACK(fileKindChangedForSpin),
		   (gpointer)&fileChooserSpin);

  /* Layout labels */
  label = gtk_label_new(":");
  gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1, GTK_SHRINK, GTK_SHRINK, 5, 0);

  label = gtk_label_new(":");
  gtk_table_attach(GTK_TABLE(table), label, 1, 2, 1, 2, GTK_SHRINK, GTK_SHRINK, 5, 0);

  labelPosition = gtk_label_new(_("None"));
  gtk_misc_set_alignment(GTK_MISC(labelPosition), 0., 0.5);
  gtk_table_attach(GTK_TABLE(table), labelPosition, 2, 3, 0, 1,
		   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);

  labelSpin = gtk_label_new(_("None"));
  gtk_misc_set_alignment(GTK_MISC(labelSpin), 0., 0.5);
  gtk_table_attach(GTK_TABLE(table), labelSpin, 2, 3, 1, 2,
		   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);

  /* Right aligned apply button. */
  align = gtk_alignment_new(0.5, 1.0, 0., 0.);
  gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 2);
  button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
  gtk_container_add(GTK_CONTAINER(align), button);
  g_signal_connect(G_OBJECT(button), "clicked",
		   G_CALLBACK(fileSelectedForSpinMethod),
		   (gpointer)&fileChooserSpin);

  /* Hiding mode button */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);
  label = gtk_label_new(LABEL_SPIN_POLICY);
  gtk_widget_set_name(label, "label_head_2");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  hiding_mode = createHidingModeRadioWidgets();
  gtk_box_pack_start(GTK_BOX(hbox), hiding_mode, FALSE, FALSE, 2);

  gtk_widget_show_all(vbox);
  
/*   frame = gtk_frame_new(NULL); */
/*   gtk_widget_show (frame); */
/*   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), frame, TRUE, TRUE, 0); */

/*   label = gtk_label_new(_("The spin method uses 2 files: one stores the position and the other is used for the spin orientation. These two files are required to render a spin configurations. Either position or spin file can be kept between two views if they have the same number of nodes.")); */
/*   gtk_widget_show(label); */
/*   gtk_container_add(GTK_CONTAINER(frame), label); */
/*   gtk_label_set_line_wrap(GTK_LABEL(label), TRUE); */
/*   gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5); */
/*   gtk_misc_set_padding(GTK_MISC(label), 5, 5); */

/*   hbox = gtk_hbox_new(FALSE, 0); */
/*   gtk_widget_show(hbox); */
/*   gtk_frame_set_label_widget(GTK_FRAME(frame), hbox); */
/*   image = gtk_image_new_from_stock(GTK_STOCK_HELP, GTK_ICON_SIZE_SMALL_TOOLBAR); */
/*   gtk_widget_show(image); */
/*   gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0); */
/*   label = gtk_label_new (_("Help")); */
/*   gtk_widget_show(label); */
/*   gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0); */
  
  fileChooserSpin.label[FILE_KIND_POSITION] = GTK_LABEL(labelPosition);
  fileChooserSpin.label[FILE_KIND_SPIN] = GTK_LABEL(labelSpin);
  fileChooserSpin.radio[FILE_KIND_POSITION] = GTK_RADIO_BUTTON(radioPosition);
  fileChooserSpin.radio[FILE_KIND_SPIN] = GTK_RADIO_BUTTON(radioSpin);
  fileChooserSpin.fileChooserDialog = GTK_FILE_CHOOSER_DIALOG(fileSelection);
  fileChooserSpin.fileKind = FILE_KIND_POSITION;
  fileChooserSpin.files[FILE_KIND_POSITION] = (gchar*)0;
  fileChooserSpin.files[FILE_KIND_SPIN] = (gchar*)0;

  do
    {
      status = gtk_dialog_run(GTK_DIALOG(fileSelection));
      if (status != GTK_RESPONSE_OK)
	{
	  if (fileChooserSpin.files[FILE_KIND_POSITION])
	    g_free(fileChooserSpin.files[FILE_KIND_POSITION]);
	  if (fileChooserSpin.files[FILE_KIND_SPIN])
	    g_free(fileChooserSpin.files[FILE_KIND_SPIN]);

	  gtk_widget_destroy(fileSelection);
	  
	  return FALSE;
	}
      else
	{
	  fileSelectedForSpinMethod((GtkButton*)0, &fileChooserSpin);
	  if (fileChooserSpin.files[FILE_KIND_POSITION] &&
	      fileChooserSpin.files[FILE_KIND_SPIN])
	    {
	      visuDataAdd_file(data, fileChooserSpin.files[FILE_KIND_POSITION],
			       FILE_KIND_POSITION, (FileFormat*)0);
	      visuDataAdd_file(data, fileChooserSpin.files[FILE_KIND_SPIN],
			       FILE_KIND_SPIN, (FileFormat*)0);
	      g_free(fileChooserSpin.files[FILE_KIND_POSITION]);
	      g_free(fileChooserSpin.files[FILE_KIND_SPIN]);

	      directory = gtk_file_chooser_get_current_folder
		(GTK_FILE_CHOOSER(fileSelection));
	      gtkMainClassSet_lastOpenDirectory((char*)directory);
	      g_free(directory);
	      
	      gtk_widget_destroy(fileSelection);
	  
	      return TRUE;
	    }
	}
    } while (TRUE);
  return FALSE;
}

static void fileSelectedForSpinMethod(GtkButton *button _U_, gpointer data)
{
  gchar *filename = NULL, *filenameUTF8;
  struct FileChooserSpin_struct *values;
  int res;
  GString *message;

  values = (struct FileChooserSpin_struct*)data;
  filename = gtk_file_chooser_get_filename
    (GTK_FILE_CHOOSER(values->fileChooserDialog));
  if (!g_file_test(filename, G_FILE_TEST_IS_DIR) && filename != NULL)
    {
      DBG_fprintf(stderr, "Gtk Spin : select '%s' as file %d\n", filename, values->fileKind);
      res = 0;
      values->files[values->fileKind] = filename;
      filenameUTF8 = g_filename_to_utf8(filename, -1, NULL, NULL, NULL);
      g_return_if_fail(filenameUTF8);

      message = g_string_new("");
      if (g_utf8_strlen(filenameUTF8, 51) > 50)
	g_string_printf(message, "(...)%s",
			 g_utf8_offset_to_pointer(filenameUTF8,
						  g_utf8_strlen(filenameUTF8, -1) - 50));
      else
	g_string_printf(message, "%s", filenameUTF8);
      gtk_label_set_text(values->label[values->fileKind], message->str);
      g_string_free(message, TRUE);
      g_free(filenameUTF8);
      if (values->fileKind == FILE_KIND_POSITION)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(values->radio[FILE_KIND_SPIN]), TRUE);
      else if (values->fileKind == FILE_KIND_SPIN)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(values->radio[FILE_KIND_POSITION]), TRUE);
      
      if (values->files[FILE_KIND_POSITION] && values->files[FILE_KIND_SPIN])
	gtk_dialog_response(GTK_DIALOG(values->fileChooserDialog), GTK_RESPONSE_OK);
    }
}

static void fileUpdatePreview(GtkFileChooser *chooser,
			      struct FileChooserSpin_struct *fileChooserSpin)
{
  const char *filenames[3];

  if (fileChooserSpin->files[FILE_KIND_POSITION])
    filenames[0] = g_strdup(fileChooserSpin->files[FILE_KIND_POSITION]);
  else
    filenames[0] = gtk_file_chooser_get_preview_filename(chooser);
  if (fileChooserSpin->files[FILE_KIND_POSITION] &&
      !fileChooserSpin->files[FILE_KIND_SPIN])
    filenames[1] = gtk_file_chooser_get_preview_filename(chooser);
  else if (fileChooserSpin->files[FILE_KIND_SPIN])
    filenames[1] = g_strdup(fileChooserSpin->files[FILE_KIND_SPIN]);
  else
    filenames[1] = (char*)0;
  filenames[2] = (char*)0;

  /* We test if the selected filename is a directory or not. */
  if (filenames[0] && !g_file_test(filenames[0], G_FILE_TEST_IS_DIR) &&
      filenames[1] && !g_file_test(filenames[1], G_FILE_TEST_IS_DIR))
    visuGtkPreview_update(&fileChooserSpin->preview, filenames);
  if (filenames[0])
    g_free((gchar*)filenames[0]);
  if (filenames[1])
    g_free((gchar*)filenames[1]);
}

static void fileKindChangedForPosition(GtkToggleButton *button, gpointer data)
{
  struct FileChooserSpin_struct* param = data;
  GSList *tmpLst;

  if (!gtk_toggle_button_get_active(button))
    return;

  tmpLst = gtk_file_chooser_list_filters(GTK_FILE_CHOOSER(param->fileChooserDialog));
  gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(param->fileChooserDialog),
			      (GtkFileFilter*)(tmpLst->data));
  param->fileKind = FILE_KIND_POSITION;

  if (param->files[0])
    gtk_file_chooser_select_filename(GTK_FILE_CHOOSER(param->fileChooserDialog),
				     param->files[0]);
}

static void fileKindChangedForSpin(GtkToggleButton *button, gpointer data)
{
  struct FileChooserSpin_struct* param = data;
  GSList *tmpLst;

  if (!gtk_toggle_button_get_active(button))
    return;

  tmpLst = gtk_file_chooser_list_filters(GTK_FILE_CHOOSER(param->fileChooserDialog));
  gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(param->fileChooserDialog),
			      (GtkFileFilter*)(tmpLst->next->data));
  param->fileKind = FILE_KIND_SPIN;

  if (param->files[1])
    gtk_file_chooser_select_filename(GTK_FILE_CHOOSER(param->fileChooserDialog),
				     param->files[1]);
}

/* Create the gtk widgets (a hbox with a spin with
   positive values) and return it. */
GtkWidget* createGtkInterfaceForSpinMethod()
{
  GtkWidget* hbox[7], *vbox, *label, *expand, *vboxAtomic;
  GtkWidget* shapeLabel, *hLabel, *rhLabel, *lLabel, *rlLabel, *ratioLabel, *lengthLabel;
  int i, j = 0;

  vbox = gtk_vbox_new(FALSE, 0);

  for(i=0; i<7; i++)
    hbox[i] = gtk_hbox_new(FALSE, 0);
  vboxArrowShape = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(vboxArrowShape);
  for(i=2; i<5; i++)
    {
      gtk_box_pack_start(GTK_BOX(vboxArrowShape), hbox[i], FALSE, FALSE, 2);
      gtk_widget_show(hbox[i]);
    }
  vboxElipsoidShape = gtk_vbox_new(FALSE, 0);
  for(i=5; i<7; i++)
    {
      gtk_box_pack_start(GTK_BOX(vboxElipsoidShape), hbox[i], FALSE, FALSE, 2);
      gtk_widget_show(hbox[i]);
    }
      
  /* Labels creation */

  shapeLabel = gtk_label_new("");
  gtk_label_set_markup(GTK_LABEL(shapeLabel), _("<i>Spin shape:</i>"));
  gtk_misc_set_alignment(GTK_MISC(shapeLabel), 0., 0.5);
  gtk_misc_set_padding(GTK_MISC(shapeLabel), 10, 0);
  gtk_widget_show(shapeLabel);

  labelSizeProp = gtk_label_new("");
  if (panelConfigGet_spinBoundsValue() == 1.)
    gtk_label_set_markup(GTK_LABEL(labelSizeProp), LABEL_ARROW_SIZE);
  else
    gtk_label_set_markup(GTK_LABEL(labelSizeProp), LABEL_ARROW_SIZE_MULT);
  gtk_misc_set_alignment(GTK_MISC(labelSizeProp), 0., 0.5);
  gtk_misc_set_padding(GTK_MISC(labelSizeProp), 10, 0);
  gtk_widget_show(labelSizeProp);

  /* Size labels. */
  hLabel = gtk_label_new(_("Hat length:"));
  gtk_misc_set_alignment(GTK_MISC(hLabel), 1.0, 0.5);
  gtk_widget_show(hLabel);

  rhLabel = gtk_label_new(_("Tail length:"));
  gtk_misc_set_alignment(GTK_MISC(rhLabel), 1.0, 0.5);
  gtk_widget_show(rhLabel);

  lLabel = gtk_label_new(_("Hat radius:"));
  gtk_misc_set_alignment(GTK_MISC(lLabel), 1.0, 0.5);
  gtk_widget_show(lLabel);

  rlLabel = gtk_label_new(_("Tail radius:"));
  gtk_misc_set_alignment(GTK_MISC(rlLabel), 1.0, 0.5);
  gtk_widget_show(rlLabel);

  /* Spin buttons creation */

  heightResourceSpin = gtk_spin_button_new_with_range(0, 9, 0.05);
  element_callback_ids[0] = g_signal_connect((gpointer)heightResourceSpin, "value-changed",
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementHatRadius));
  gtk_widget_show(heightResourceSpin);

  rheightResourceSpin = gtk_spin_button_new_with_range(0, 9, 0.05);
  element_callback_ids[1] = g_signal_connect((gpointer)rheightResourceSpin, "value-changed",
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementTailRadius));
  gtk_widget_show(rheightResourceSpin);

  lengthResourceSpin = gtk_spin_button_new_with_range(0, 9, 0.05);
  element_callback_ids[2] = g_signal_connect((gpointer)lengthResourceSpin, "value-changed",
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementHatLength));
  gtk_widget_show(lengthResourceSpin);

  rlengthResourceSpin = gtk_spin_button_new_with_range(0, 9, 0.05);
  element_callback_ids[3] = g_signal_connect((gpointer)rlengthResourceSpin, "value-changed",
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementTailLength));
  gtk_widget_show(rlengthResourceSpin);

  /* This callback is for multiplier */
  g_signal_connect(G_OBJECT(gtkMainClassGet_currentPanel()), "spinBoundsChanged",
		   G_CALLBACK(onSpinBoundsChangeArrowSize), (gpointer)0);


  /* Check box creation */
  label = gtk_label_new(_("Use element color on:"));
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_widget_show(label);

  gtkw_use_element_color = gtk_check_button_new_with_label(_(" tail"));
  element_callback_ids[4] = g_signal_connect(gtkw_use_element_color, "toggled", 
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementTailColor));
  gtk_widget_show(gtkw_use_element_color);

  gtkw_use_element_color_hat = gtk_check_button_new_with_label(_(" hat"));
  element_callback_ids[5] = g_signal_connect(gtkw_use_element_color_hat, "toggled", 
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementHatColor));
  gtk_widget_show(gtkw_use_element_color_hat);

  /* Combo boxes creation */
  gtkw_element_shape_number = gtk_combo_box_new_text();
  for (i=0; i<rspin_get_number_of_shapes(); i++)
    gtk_combo_box_append_text(GTK_COMBO_BOX(gtkw_element_shape_number),
			      rspin_shape_number_to_translated_name(i));
  /* set callback for the combo button. */
  element_callback_ids[6] = g_signal_connect(gtkw_element_shape_number, "changed",
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementShape));
  gtk_widget_show(gtkw_element_shape_number);

  /* Set the spins for the elipsoid shape. */
  ratioElipsoidSpin = gtk_spin_button_new_with_range(0, 9, 0.05);
  element_callback_ids[7] = g_signal_connect(G_OBJECT(ratioElipsoidSpin), "value-changed",
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementBAxis));
  gtk_widget_show(ratioElipsoidSpin);

  lengthElipsoidSpin = gtk_spin_button_new_with_range(0, 9, 0.05);
  element_callback_ids[8] = g_signal_connect(G_OBJECT(lengthElipsoidSpin), "value-changed",
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementAAxis));
  gtk_widget_show(lengthElipsoidSpin);
  ratioLabel = gtk_label_new(_("B axis: "));
  gtk_misc_set_alignment(GTK_MISC(ratioLabel), 1.0, 0.5);
  gtk_widget_show(ratioLabel);
  lengthLabel = gtk_label_new(_("A axis: "));
  gtk_misc_set_alignment(GTK_MISC(lengthLabel), 1.0, 0.5);
  gtk_widget_show(lengthLabel);
  useElementForElipsoid = gtk_check_button_new_with_label(_("Use element color"));
  element_callback_ids[9] = g_signal_connect(G_OBJECT(useElementForElipsoid), "toggled", 
					     G_CALLBACK(element_resource_callback),
					     GINT_TO_POINTER(spin_elementElipsoidColor));
  gtk_widget_show(useElementForElipsoid);

  /* Atomic options. */
  expand = gtk_expander_new(_("Atomic rendering options"));
  gtk_expander_set_expanded(GTK_EXPANDER(expand), FALSE);
  gtk_widget_show(expand);
  vboxAtomic = panelElementAtomicCreate_gtkInterface();
  gtk_container_add(GTK_CONTAINER(expand), vboxAtomic);

  /* Displaying every widget created so far */

  gtk_box_pack_start(GTK_BOX(hbox[j]), shapeLabel, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j]), gtkw_element_shape_number, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(vbox), hbox[j], FALSE, FALSE, 2);
  gtk_widget_show(hbox[j++]);

  gtk_box_pack_start(GTK_BOX(hbox[j]), labelSizeProp, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(vbox), hbox[j], FALSE, FALSE, 2);
  gtk_widget_show(hbox[j++]);

  gtk_box_pack_start(GTK_BOX(vbox), vboxArrowShape, FALSE, FALSE, 2);
  gtk_box_pack_start(GTK_BOX(hbox[j]), hLabel, FALSE, FALSE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j]), heightResourceSpin, FALSE,FALSE, 3);
  gtk_box_pack_start(GTK_BOX(hbox[j]), rhLabel, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j++]), rheightResourceSpin, FALSE,FALSE, 3);

  gtk_box_pack_start(GTK_BOX(hbox[j]), lLabel, FALSE, FALSE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j]), lengthResourceSpin, FALSE,FALSE, 3);
  gtk_box_pack_start(GTK_BOX(hbox[j]), rlLabel, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j++]), rlengthResourceSpin, FALSE,FALSE, 3);

  gtk_box_pack_start(GTK_BOX(hbox[j]), label, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j]), gtkw_use_element_color_hat, FALSE, FALSE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j++]), gtkw_use_element_color, FALSE, FALSE, 1);

  gtk_box_pack_start(GTK_BOX(vbox), vboxElipsoidShape, FALSE, FALSE, 2);
  gtk_box_pack_start(GTK_BOX(hbox[j]), lengthLabel, FALSE, FALSE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j]), lengthElipsoidSpin, FALSE,FALSE, 3);
  gtk_box_pack_start(GTK_BOX(hbox[j]), ratioLabel, TRUE, TRUE, 1);
  gtk_box_pack_start(GTK_BOX(hbox[j++]), ratioElipsoidSpin, FALSE,FALSE, 3);

  gtk_box_pack_start(GTK_BOX(hbox[j++]), useElementForElipsoid, FALSE, FALSE, 1);

  gtk_box_pack_start(GTK_BOX(vbox), expand, FALSE, FALSE, 0);

  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
		   G_CALLBACK(sync_local_resources), NULL);

  return vbox;
}

/* This function is called whenever the current element is changed in the gtk panel. */
void changeSomethingSpin(GList *eleList)
{
  float mult;
  VisuElement *ele;

  DBG_fprintf(stderr, "Gtk Spin : changing values on element change.\n");

  /* We copy the list of given elements. */
  if (currentListElement)
    g_list_free(currentListElement);
  currentListElement = g_list_copy(eleList);

  DBG_fprintf(stderr, "Gtk Spin : give the element change to the atomic part.\n");
  panelElementAtomicCallback_onElementChange(eleList);

  /* If the list has only one element, we continue
     and update the values on the widgets. */
  if (g_list_next(eleList))
    return;

  ele = (VisuElement*)eleList->data;
  need_redraw = 0;

  mult = panelConfigGet_spinBoundsValue();
  
  g_signal_handler_block (heightResourceSpin, element_callback_ids[0] );  
  g_signal_handler_block (rheightResourceSpin, element_callback_ids[1] );  
  g_signal_handler_block (lengthResourceSpin, element_callback_ids[2] );  
  g_signal_handler_block (rlengthResourceSpin, element_callback_ids[3] );  
  g_signal_handler_block (gtkw_use_element_color, element_callback_ids[4] );  
  g_signal_handler_block (gtkw_use_element_color_hat, element_callback_ids[5] );  
/*   g_signal_handler_block (gtkw_element_shape_number, element_callback_ids[6] );   */
  g_signal_handler_block (ratioElipsoidSpin, element_callback_ids[7] );  
  g_signal_handler_block (lengthElipsoidSpin, element_callback_ids[8] );  
  g_signal_handler_block (useElementForElipsoid, element_callback_ids[9] );  

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(heightResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementHatRadius) / mult);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(rheightResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementTailRadius) / mult);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lengthResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementHatLength) / mult);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(rlengthResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementTailLength) / mult);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lengthElipsoidSpin),
			    rspin_getElementResource_float(ele, spin_elementAAxis) / mult);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(ratioElipsoidSpin),
			    rspin_getElementResource_float(ele, spin_elementBAxis) / mult);
  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtkw_use_element_color),
			       rspin_getElementResource_boolean(ele, spin_elementTailColor));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtkw_use_element_color_hat),
			       rspin_getElementResource_boolean(ele, spin_elementHatColor));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(useElementForElipsoid),
			       rspin_getElementResource_boolean(ele, spin_elementElipsoidColor));

  gtk_combo_box_set_active(GTK_COMBO_BOX(gtkw_element_shape_number),
			   rspin_getElementResource_uint(ele, spin_elementShape));

  g_signal_handler_unblock (heightResourceSpin, element_callback_ids[0] );  
  g_signal_handler_unblock (rheightResourceSpin, element_callback_ids[1] );  
  g_signal_handler_unblock (lengthResourceSpin, element_callback_ids[2] );  
  g_signal_handler_unblock (rlengthResourceSpin, element_callback_ids[3] );  
  g_signal_handler_unblock (gtkw_use_element_color, element_callback_ids[4] );  
  g_signal_handler_unblock (gtkw_use_element_color_hat, element_callback_ids[5] );  
/*   g_signal_handler_unblock (gtkw_element_shape_number, element_callback_ids[6] );   */
  g_signal_handler_unblock (ratioElipsoidSpin, element_callback_ids[7] );  
  g_signal_handler_unblock (lengthElipsoidSpin, element_callback_ids[8] );  
  g_signal_handler_unblock (useElementForElipsoid, element_callback_ids[9] );  

  need_redraw = 1;
}

void onSpinBoundsChangeArrowSize(GObject *obj _U_, gfloat mult, gpointer data _U_)
{
  unsigned int i, j;
  float *val;
  GType type;
  SpinElementResources props[6] = {spin_elementHatLength, spin_elementHatRadius,
				   spin_elementTailLength, spin_elementTailRadius,
				   spin_elementAAxis, spin_elementBAxis};
  VisuData *dataObj;
  int id;

  if (getRenderingMethodInUse() != renderingSpinGet_static())
    return;

  if (panelConfigGet_spinBoundsValue() == 1.)
    gtk_label_set_text(GTK_LABEL(labelSizeProp), LABEL_ARROW_SIZE);
  else
    {
      gtk_label_set_text(GTK_LABEL(labelSizeProp), LABEL_ARROW_SIZE_MULT);
      gtk_label_set_use_markup(GTK_LABEL(labelSizeProp), TRUE);
    }

  dataObj = toolPanelGet_visuData(TOOL_PANEL(PANEL_ELEMENTS));
  for (i = 0; i < dataObj->ntype; i++)
    {
      for (j = 0; j < 6; j++)
	{
	  val = (float*)rspin_getElementResource(dataObj->fromIntToVisuElement[i], props[j], &type);
	  *val = *val * mult;
	}
      id = rspin_createShapeSpin(dataObj, dataObj->fromIntToVisuElement[i]);
      dataObj->fromIntToVisuElement[i]->openGLIdentifier = id;
    }
}



/* This is the callback func for each element resource. */
void element_resource_callback(GtkWidget* widget, gpointer data)
{
  gboolean refresh;
  GList *tmpLst;
  VisuElement *ele;
  SpinElementResources property;
  gboolean new_bool;
  guint new_int;
  float new_float, mult;

  property = GPOINTER_TO_INT(data);
  g_return_if_fail(property < spin_nbElementResources);

  refresh = FALSE;
  tmpLst = currentListElement;
  while (tmpLst)
    {
      ele = (VisuElement*)tmpLst->data;
      switch (property)
	{
	case spin_elementHatColor:
	case spin_elementTailColor:
	case spin_elementElipsoidColor:
	  new_bool = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

	  DBG_fprintf(stderr, "GtkSpin : Set element '%d' property to %d\n",
		      property, new_bool);
	  refresh = rspin_setElementResource_boolean(ele, property, new_bool) || refresh;
	  break;
	case spin_elementShape:
	  new_int = (guint)gtk_combo_box_get_active(GTK_COMBO_BOX(widget));

	  DBG_fprintf(stderr, "GtkSpin : Set element '%d' property to %d\n",
		      property, new_int);
	  refresh = rspin_setElementResource_uint(ele, property, new_int) || refresh;
	  if (new_int == 2)
	    {
	      gtk_widget_show(vboxElipsoidShape);
	      gtk_widget_hide(vboxArrowShape);
	    }
	  else
	    {
	      gtk_widget_hide(vboxElipsoidShape);
	      gtk_widget_show(vboxArrowShape);
	    }
	  break;
	case spin_elementHatLength:
	case spin_elementHatRadius:
	case spin_elementTailLength:
	case spin_elementTailRadius:
	case spin_elementAAxis:
	case spin_elementBAxis:
	  mult = panelConfigGet_spinBoundsValue();
	  new_float = gtk_spin_button_get_value(GTK_SPIN_BUTTON(widget));

	  DBG_fprintf(stderr, "GtkSpin : Set element '%d' property to %f\n",
		      property, new_float * mult);
	  refresh = rspin_setElementResource_float(ele, property, new_float * mult) || refresh;

	  break;
	default:
	  g_warning("Unknown property '%d' in callback for spin element values.", property);
	  return;
	}
      tmpLst = g_list_next(tmpLst);
    }

  if(need_redraw && refresh)
    redraw(1);
}

/* This is the callback func for each global resource. */
void global_resource_callback(GtkWidget* button, gpointer data)
{
  SpinGlobalResources property;
  int value;
  gboolean refresh;

  value = GPOINTER_TO_INT(data);
  if (value >= 0)
    {
      property = (SpinGlobalResources)value;
      g_return_if_fail(property < spin_nbGlobalResources);
      switch (property)
	{
	case spin_globalConeTheta:
	case spin_globalConePhi:
	case spin_globalColorWheel:
	  refresh = rspin_setGlobalResource_float
	    (property, gtk_spin_button_get_value(GTK_SPIN_BUTTON(button)));

	  if(need_redraw && refresh)
	    {
	      float spherical[3];
	      float cartesian[3];
	      int i;

	      need_redraw = 0;
	  
	      spherical[0] = 1;
	      spherical[1] = rspin_getGlobalResource_float(spin_globalConeTheta);
	      spherical[2] = rspin_getGlobalResource_float(spin_globalConePhi);

	      matrix_sphericalToCartesian(cartesian, spherical);

	      for(i=0; i<3; i++)
		if(cartesian[i] < 0.01 && cartesian [i] > -0.01)
		  cartesian[i] = 0;
	  
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_x), cartesian[0]);
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_y), cartesian[1]);
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_z), cartesian[2]);

	      need_redraw = 1;
	    }
	  break;
	default:
	  g_warning("Unknown property '%d' in callback for spin global values.", property);
	  return;
	}
    }
  else
    {
      float cartesian[3];
      float spherical[3];
      float x = (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gtkw_x));
      float y = (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gtkw_y));
      float z = (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(gtkw_z));
      
      if(need_redraw)
	{
	  need_redraw = 0;

	  cartesian[0] = x;
	  cartesian[1] = y;
	  cartesian[2] = z;
	  
	  matrix_cartesianToSpherical(spherical, cartesian);

	  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_theta_angle), spherical[1]);
	  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_phi_angle), spherical[2]);

	  need_redraw = 1;
	}
    }

  if(need_redraw == 1)
    redraw(2);
}

void set_view(GtkWidget* button _U_, gpointer data _U_)
{
  float theta;
  float phi;
  VisuData *dataObj;
  OpenGLView *view;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(PANEL_ELEMENTS));
  if (!dataObj)
    return;
  view = visuDataGet_openGLView(dataObj);

  theta = fModulo(view->camera->theta, 360);
  if(theta > 180)
      theta = 360 - theta;
  phi = fModulo(view->camera->phi, 360);

  need_redraw = 0;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_theta_angle), theta);

  need_redraw = 1;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_phi_angle), phi);
}

/* i = 1 if called by a function which modified an element resource
   i = 2 if called by a function which modified a global resource */
void redraw(int i)
{
  VisuData *dataObj;
  int id;
  VisuElement *ele;
  GList *tmpLst;

  if(currentListElement == NULL)
    return;

  DBG_fprintf(stderr, "Gtk Spin : Updating rendering area.\n");

  dataObj = toolPanelGet_visuData(TOOL_PANEL(PANEL_ELEMENTS));
  if (!dataObj)
    return;

  if(i == 1)
    {
      tmpLst = currentListElement;
      while (tmpLst)
	{
	  ele = (VisuElement*)tmpLst->data;
	  id = rspin_createShapeSpin(dataObj, ele);
	  ele->openGLIdentifier = id;
	  tmpLst = g_list_next(tmpLst);
	}
    }

  visuData_createAllNodes(dataObj);  
  
  if (i == 2)
    g_signal_emit(VISU_INSTANCE, VISU_GET_CLASS(VISU_INSTANCE)->SpinAxes_signal_id,
		  0 , dataObj, NULL);
  g_idle_add(visuObjectRedraw, (gpointer)0);
}

static void sync_global_resources(GObject *object _U_, VisuData *dataObj _U_,
				  gpointer data _U_)
{
  float spherical[3];
  float cartesian[3];
  int i;

  DBG_fprintf(stderr, "Gtk Spin : caught the 'resourcesLoaded' signal.\n");

  if(gtkw_cone_theta_angle == NULL)
    return;

  spherical[0] = 1;
  spherical[1] = rspin_getGlobalResource_float(spin_globalConeTheta);
  spherical[2] = rspin_getGlobalResource_float(spin_globalConePhi);

  matrix_sphericalToCartesian(cartesian, spherical);

  for(i=0; i<3; i++)
    if(cartesian[i] < 0.01 && cartesian [i] > -0.01)
      cartesian[i] = 0;

  need_redraw = 0;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_x), cartesian[0]);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_y), cartesian[1]);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_z), cartesian[2]);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_theta_angle), spherical[1]);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_cone_phi_angle), spherical[2]); 
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtkw_color_wheel_angle),
			    rspin_getGlobalResource_float(spin_globalColorWheel)); 

  need_redraw = 1;  
}      

static void sync_local_resources(GObject *trash _U_, gpointer data _U_)
{
  VisuElement *ele;

  if(heightResourceSpin == NULL || currentListElement == NULL)
    return;
  
  ele = (VisuElement*)currentListElement->data;
  need_redraw = 0;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(heightResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementHatRadius));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(rheightResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementTailRadius));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lengthResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementHatLength));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(rlengthResourceSpin),
			    rspin_getElementResource_float(ele, spin_elementTailLength));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lengthElipsoidSpin),
			    rspin_getElementResource_float(ele, spin_elementAAxis));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(ratioElipsoidSpin),
			    rspin_getElementResource_float(ele, spin_elementBAxis));
  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtkw_use_element_color),
			       rspin_getElementResource_boolean(ele, spin_elementTailColor));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gtkw_use_element_color_hat),
			       rspin_getElementResource_boolean(ele, spin_elementHatColor));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(useElementForElipsoid),
			       rspin_getElementResource_boolean(ele, spin_elementElipsoidColor));

  gtk_combo_box_set_active(GTK_COMBO_BOX(gtkw_element_shape_number),
			   rspin_getElementResource_uint(ele, spin_elementShape));

  need_redraw = 1;
}
