/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "toolShade.h"
#include "toolColor.h"

#include <string.h>
#include <math.h>
#include <GL/gl.h>
#include <GL/glu.h> 

#include <visu_tools.h>
#include <visu_basic.h>
#include <openGLFunctions/text.h>

/**
 * SECTION:toolShade
 * @short_description: Shades are color gradients.
 *
 * <para>This module allow V_Sim to deal with color gradients. Such a
 * gradient is defined by a linear transformation of color space. This
 * space can be RBG or HSV (see the enum #ShadeColorMode). This linear
 * transformation can be written [resulting color vector] = [vectB] +
 * lambda.[vectA], where lambda denotes the input variable of the
 * gradient (ranging from 0 to 1). Resulting color vector are clamped
 * to [0;1] if needed.</para>
 * <para>Use shadeNew() to create a new shade, giving the arguments as
 * defined above. A shade can be linked to an image to represent it,
 * use shadeSet_pathToImage() to do it.</para>
 * <!--<para>In near future, the code that can discretize the
 * resulting color gradient will be imported here from dataFile.c and
 * could then be shared between other modules. At present time, each
 * module needs to write it own transformation code after having
 * retrieved the transformation coefficients with
 * shadeGet_colorTransformation().</para>-->
 * <para>To share color gradients between modules in V_Sim, you can
 * add new shade to the global list of stored shades using
 * toolShadeListAppend() and get this list with a call to
 * toolShadeListGet().</para>
 */


#define SHADE_LEGEND_WIDTH 20
#define SHADE_LEGEND_HEIGHT 150
#define SHADE_LEGEND_N_QUADS 20

struct Shade_struct
{
  gchar* labelUTF8;
  ShadeColorMode colorMode;
  ShadeMode mode;

  /* The linear storage. */
  float vectA[3], vectB[3];
  /* The array storage. */
  float *vectCh[3];
  int nVals;
};

static GList *toolShadeList = (GList*)0;

Shade* shadeNew(const gchar* labelUTF8, float vectA[3], float vectB[3],
		ShadeColorMode colorMode)
{
  Shade *shade;
  int i;

  g_return_val_if_fail(labelUTF8 && vectA && vectB &&
		       colorMode < shade_nb_colorMode, (Shade*)0);

  DBG_fprintf(stderr, "Tool Shade: create a new Shade object (linear):");
  shade = g_malloc(sizeof(Shade));
  shade->labelUTF8 = g_strdup(labelUTF8);
  for (i = 0; i < 3; i++)
    {
      shade->vectA[i] = vectA[i];
      shade->vectB[i] = vectB[i];
    }
  shade->vectCh[0] = (float*)0;
  shade->vectCh[1] = (float*)0;
  shade->vectCh[2] = (float*)0;
  shade->colorMode = colorMode;
  shade->mode = shade_modeLinear;

  DBG_fprintf(stderr, " %p.\n", (gpointer)shade);
  return shade;
}
Shade* shadeNew_fromData(const gchar* labelUTF8, guint len, float *vectCh1,
			 float *vectCh2, float *vectCh3, ShadeColorMode colorMode)
{
  Shade *shade;

  g_return_val_if_fail(labelUTF8 && vectCh1 && vectCh2 && vectCh3 &&
		       colorMode < shade_nb_colorMode && len > 0, (Shade*)0);

  DBG_fprintf(stderr, "Tool Shade: create a new Shade object (array):");
  shade = g_malloc(sizeof(Shade));
  shade->labelUTF8 = g_strdup(labelUTF8);
  shade->nVals     = len;
  shade->vectCh[0] = g_malloc(sizeof(float) * (len + 1));
  memcpy(shade->vectCh[0], vectCh1, sizeof(float) * len);
  shade->vectCh[0][len] = vectCh1[len - 1];
  shade->vectCh[1] = g_malloc(sizeof(float) * (len + 1));
  memcpy(shade->vectCh[1], vectCh2, sizeof(float) * len);
  shade->vectCh[1][len] = vectCh2[len - 1];
  shade->vectCh[2] = g_malloc(sizeof(float) * (len + 1));
  memcpy(shade->vectCh[2], vectCh3, sizeof(float) * len);
  shade->vectCh[2][len] = vectCh3[len - 1];
  shade->colorMode = colorMode;
  shade->mode = shade_modeArray;

  DBG_fprintf(stderr, " %p.\n", (gpointer)shade);
  return shade;
}
void shadeFree(Shade *shade)
{
  if (shade)
    {
      g_free(shade->labelUTF8);
      g_free(shade->vectCh[0]);
      g_free(shade->vectCh[1]);
      g_free(shade->vectCh[2]);
      g_free(shade);
    }
}
Shade* shadeCopy(Shade *shade)
{
  Shade *out;

  g_return_val_if_fail(shade, (Shade*)0);

  out = g_malloc(sizeof(Shade));
  out->labelUTF8 = g_strdup(shade->labelUTF8);
  out->colorMode = shade->colorMode;
  out->mode      = shade->mode;
  out->nVals     = shade->nVals;
  out->vectA[0]  = shade->vectA[0];
  out->vectA[1]  = shade->vectA[1];
  out->vectA[2]  = shade->vectA[2];
  out->vectB[0]  = shade->vectB[0];
  out->vectB[1]  = shade->vectB[1];
  out->vectB[2]  = shade->vectB[2];
  out->vectCh[0] = g_memdup(shade->vectCh[0], sizeof(float) * out->nVals);
  out->vectCh[1] = g_memdup(shade->vectCh[1], sizeof(float) * out->nVals);
  out->vectCh[2] = g_memdup(shade->vectCh[2], sizeof(float) * out->nVals);

  return out;
}
gboolean shadeCompare(Shade* sh1, Shade *sh2)
{
  int i;
  gboolean equal;

  g_return_val_if_fail(sh1 && sh2, FALSE);

  if (sh1->mode != sh2->mode)
    return FALSE;
  if (sh1->colorMode != sh2->colorMode)
    return FALSE;

  if (sh1->mode == shade_modeLinear)
    return (sh1->vectA[0] == sh2->vectA[0] &&
	    sh1->vectA[1] == sh2->vectA[1] &&
	    sh1->vectA[2] == sh2->vectA[2] &&
	    sh1->vectB[0] == sh2->vectB[0] &&
	    sh1->vectB[1] == sh2->vectB[1] &&
	    sh1->vectB[2] == sh2->vectB[2]);
  else
    {
      if (sh1->nVals != sh2->nVals)
	return FALSE;
      equal = TRUE;
      for (i = 0; i < sh1->nVals && equal; i++)
	equal = equal && (sh1->vectCh[0][i] == sh2->vectCh[0][i]) &&
	  (sh1->vectCh[1][i] == sh2->vectCh[1][i]) &&
	  (sh1->vectCh[2][i] == sh2->vectCh[2][i]);
      return equal;
    }
}

gchar* shadeGet_label(Shade *shade)
{
  g_return_val_if_fail(shade, (gchar*)0);
  return shade->labelUTF8;
}
ShadeColorMode shadeGet_colorMode(Shade *shade)
{
  g_return_val_if_fail(shade, (int)0);
  return shade->colorMode;
}
ShadeMode shadeGet_mode(Shade *shade)
{
  g_return_val_if_fail(shade, (int)0);
  return shade->mode;
}
gboolean shadeSet_colorMode(Shade *shade, ShadeColorMode mode)
{
  g_return_val_if_fail(shade, FALSE);

  if (shade->colorMode == mode)
    return FALSE;
  
  shade->colorMode = mode;
  return TRUE;
}
gboolean shadeGet_linearCoeff(Shade *shade, float **vectA, float **vectB)
{
  g_return_val_if_fail(shade, FALSE);
  g_return_val_if_fail(shade->mode == shade_modeLinear, FALSE);
  g_return_val_if_fail(vectA && vectB, FALSE);
  
  *vectA = shade->vectA;
  *vectB = shade->vectB;
  return TRUE;
}
gboolean shadeSet_linearCoeff(Shade *shade, float coeff, int channel, int order)
{
  float *pt;

  g_return_val_if_fail(shade, FALSE);
  g_return_val_if_fail(channel >= 0 && channel < 3 && order >= 0 && order < 2, FALSE);

  if (order == 0)
    pt = shade->vectB + channel;
  else
    pt = shade->vectA + channel;

  DBG_fprintf(stderr, "Tool Shade: set the %d value of vect[%d]"
	      " to %f (previuosly %f).\n", channel, order, coeff, *pt);

  if (*pt == coeff)
    return FALSE;

  *pt = coeff;
  return TRUE;
}
void shadeGet_RGBFromValues(Shade *shade, float rgba[4], float values[3])
{
  int i;
  float f;

  g_return_if_fail(shade);

  if (shade->mode == shade_modeLinear)
    {
      rgba[0] = CLAMP(shade->vectA[0] * values[0] + shade->vectB[0], 0.f, 1.f);
      rgba[1] = CLAMP(shade->vectA[1] * values[1] + shade->vectB[1], 0.f, 1.f);
      rgba[2] = CLAMP(shade->vectA[2] * values[2] + shade->vectB[2], 0.f, 1.f);
    }
  else
    {
      f = values[0] * (shade->nVals - 1);
      i = (int)f;
      rgba[0] = CLAMP(shade->vectCh[0][i] +
		      (shade->vectCh[0][i + 1] - shade->vectCh[0][i]) *
		      (f - (float)i), 0.f, 1.f);
      f = values[1] * (shade->nVals - 1);
      i = (int)f;
      rgba[1] = CLAMP(shade->vectCh[1][i] +
		      (shade->vectCh[1][i + 1] - shade->vectCh[1][i]) *
		      (f - (float)i), 0.f, 1.f);
      f = values[2] * (shade->nVals - 1);
      i = (int)f;
      rgba[2] = CLAMP(shade->vectCh[2][i] +
		      (shade->vectCh[2][i + 1] - shade->vectCh[2][i]) *
		      (f - (float)i), 0.f, 1.f);
    }
  /* Don't use alpha channel at present time. */
  rgba[3] = 1.;
  /* Transform if required. */
  if (shade->colorMode == shade_colorModeHSV)
    color_HSVtoRGB(rgba, rgba);
}
void shadeGet_valueTransformedInRGB(Shade *shade, float rgba[4], float value)
{
  float vals[3];

  vals[0] = value;
  vals[1] = value;
  vals[2] = value;
  shadeGet_RGBFromValues(shade, rgba, vals);
}

/****************/
/* OpenGL part. */
/****************/
void shadeDraw_legend(Shade *shade, float widthVal, float heightVal,
		      float minMax[2], float *marks, int nbMarks,
		      matrix_scaleflag scale, float *smin)
{
  int viewport[4];
  guint width, height, wTotal, hTotal;
  int i;
  float yStep, logval;
  float rgba[4];
  char value[16];
  matrixGet_scaleVal get_val, get_inv;
  double minmax[2];

  g_return_if_fail(shade);
  g_return_if_fail(widthVal != 0. && heightVal != 0.);

  /* We get the size of the current viewport. */
  glGetIntegerv(GL_VIEWPORT, viewport);

  /* We change the viewport. */
  if (widthVal < 0.)
    width = SHADE_LEGEND_WIDTH;
  else if (widthVal < 1.)
    width = (guint)((float)viewport[2] * widthVal);
  else
    width = (guint)widthVal;
  if (heightVal < 0.)
    height = SHADE_LEGEND_HEIGHT;
  else if (heightVal < 1.)
    height = (guint)((float)viewport[3] * heightVal);
  else
    height = (guint)heightVal;
  wTotal = width + 5 * 3 + 12 * 7;
  hTotal = height + 5 * 2;
  glViewport(10, 10, wTotal, hTotal);

  glDisable(GL_FOG);
  glDisable(GL_LIGHTING);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  /* We change the projection for a 2D one. */
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, (float)wTotal, 0., (float)hTotal);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  /* We draw a big transparent square to do the back. */
  glColor4f(1., 1., 1., 0.4);
  glRecti(0, 0, wTotal, hTotal);

  /* We draw the colored bar. */
  shadeGet_valueTransformedInRGB(shade, rgba, 0.);
  glColor4fv(rgba);
  yStep = (float)height / (float)SHADE_LEGEND_N_QUADS;
  glBegin(GL_QUAD_STRIP);
/*   minlog = log((float)SHADE_LEGEND_N_QUADS); */
  for (i = 0; i <= SHADE_LEGEND_N_QUADS; i++)
    {
      glVertex2f(5., 5. + (float)i * yStep);
      glVertex2f(5. + (float)width, 5. + (float)i * yStep);
/*       if (logScale) */
/* 	shadeGet_valueTransformedInRGB(shade, rgba, */
/* 				       (i == 0)?0.f:log((float)i) / minlog); */
/*       else */
	shadeGet_valueTransformedInRGB(shade, rgba,
				       (float)i / (float)SHADE_LEGEND_N_QUADS);
      glColor4fv(rgba);
    }
  glEnd();

  minmax[0] = minMax[0];
  minmax[1] = minMax[1];
  switch (scale)
    {
    case linear:
      logval = 0.;
      get_val = matrixGet_linear;
      get_inv = matrixGet_invLinear;
      break;
    case logarithm:
      logval = log10((*smin - minMax[0]) /
		     (minMax[1] - minMax[0]));
      get_val = matrixGet_logarithm;
      get_inv = matrixGet_invLogarithm;
      break;
    case zeroCentredLog:
      logval = 0;
      get_val = matrixGet_zeroCenteredLog;
      get_inv = matrixGet_invZeroCenteredLog;
      break;
    default:
      logval = 0.;
      get_val = (matrixGet_scaleVal)0;
      get_inv = (matrixGet_scaleVal)0;
      break;
    }
  g_return_if_fail(get_val);

  /* We draw some marks. */
  if (marks)
    {
      glLineWidth(2);
      glBegin(GL_LINES);
      for (i = 0; i < nbMarks; i++)
	{
	  yStep = (float)get_val((double)marks[i], minmax, logval);
	  shadeGet_valueTransformedInRGB(shade, rgba, yStep);
	  rgba[0] = 1. - rgba[0];
	  rgba[1] = 1. - rgba[1];
	  rgba[2] = 1. - rgba[2];
	  glColor4fv(rgba);
	  yStep *= (float)height;
	  glVertex2f(5. + 3., 5. + yStep);
	  glVertex2f(5. + (float)width - 3., 5. + yStep);
	}
      glEnd();
    }

  /* We draw the frame around. */
  glColor3f(0., 0., 0.);
  glLineWidth(1);
  glBegin(GL_LINE_STRIP);
  glVertex2i(5, 5);
  glVertex2i(5 + width, 5);
  glVertex2i(5 + width, 5 + height);
  glVertex2i(5, 5 + height);
  glVertex2i(5, 5);
  glEnd();

  /* We draw the tics. */
  glColor3f(0., 0., 0.);
  glLineWidth(1);
  glBegin(GL_LINES);
  glVertex2i(5 + width    , 5);
  glVertex2i(5 + width + 3, 5);
  glVertex2i(5 + width    , 5 + height / 3);
  glVertex2i(5 + width + 3, 5 + height / 3);
  glVertex2i(5 + width    , 5 + 2 * height / 3);
  glVertex2i(5 + width + 3, 5 + 2 * height / 3);
  glVertex2i(5 + width    , 5 + height);
  glVertex2i(5 + width + 3, 5 + height);
  glEnd();
  
  /* We print the labels. */
  sprintf(value, "%.3g", get_inv(0., minmax, logval));
  glRasterPos2i(5 + width + 5, 5);
  openGLText_drawChars(value, TEXT_NORMAL); 
  sprintf(value, "%.3g", get_inv(0.33333, minmax, logval));
  glRasterPos2i(5 + width + 5, 5 + height / 3 - 6);
  openGLText_drawChars(value, TEXT_NORMAL); 
  sprintf(value, "%.3g", get_inv(0.66667, minmax, logval));
  glRasterPos2i(5 + width + 5, 5 + 2 * height / 3 - 6);
  openGLText_drawChars(value, TEXT_NORMAL);
  sprintf(value, "%.3g", get_inv(1., minmax, logval));
  glRasterPos2i(5 + width + 5, 5 + height - 12);
  openGLText_drawChars(value, TEXT_NORMAL);

  /* We change the projection back. */
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  /* We set viewport back. */
  glViewport(0, 0,viewport[2], viewport[3]);
}


/*********************************************/
/* Methods to deal with internal shade list. */
/*********************************************/
GList* toolShadeListGet()
{
  return toolShadeList;
}
GList* toolShadeListAppend(Shade *shade)
{
  g_return_val_if_fail(shade, (GList*)0);

  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);
  return toolShadeList;
}
GList* toolShadeBuild_presetList()
{
  Shade *shade;
  float vectA[3], vectB[3];

  /* Zero Centred Coloured. */
  #define zccLn 3
  float zccH[zccLn] = {0.f, 0.333f, 0.667f};
  float zccS[zccLn] = {1.0f, 0.0f, 1.0f};
  float zccV[zccLn] = {1.f, 1.f, 1.f};

  /* Zero Centred Light. */
  #define zclLn 3
  float zclR[zclLn] = {1.f, 1.f, 0.f};
  float zclG[zclLn] = {0.0f, 1.0f, 0.0f};
  float zclB[zclLn] = {0.f, 1.f, 1.f};

  /* Create a blue to red color range. */
  vectA[0] = -0.66667;
  vectA[1] = 0.;
  vectA[2] = 0.;
  vectB[0] = 0.66667;
  vectB[1] = 1.;
  vectB[2] = 1.;
  shade = shadeNew(_("blue to red"), vectA, vectB, shade_colorModeHSV);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a black to white (through yellow and red) color range. */
  vectA[0] = 2.66667;
  vectA[1] = 2.66667;
  vectA[2] = 4.;
  vectB[0] = 0.;
  vectB[1] = -1.;
  vectB[2] = -3.;
  shade = shadeNew(_("hot color"), vectA, vectB, shade_colorModeRGB);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a blue to yellow (through dark purple) color range. */
  vectA[0] = 1.33333;
  vectA[1] = 2.;
  vectA[2] = -2;
  vectB[0] = 0.;
  vectB[1] = -1.;
  vectB[2] = 1.;
  shade = shadeNew(_("blue to yellow"), vectA, vectB, shade_colorModeRGB);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a blue and red shade with black zero centred. */
  vectA[0] = -2.;
  vectA[1] = 0.;
  vectA[2] = 2.;
  vectB[0] = 1.;
  vectB[1] = 0.;
  vectB[2] = -1.;
  shade = shadeNew(_("zero centred dark"), vectA, vectB, shade_colorModeRGB);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a blue and red shade with withe zero centred. */
  shade = shadeNew_fromData(_("zero centred light"), zclLn,
			    zclR, zclG, zclB, shade_colorModeRGB);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  shade = shadeNew_fromData(_("zero centred coloured"), zccLn,
			    zccH, zccS, zccV, shade_colorModeHSV);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a green to red color range. */
  vectA[0] = -.3333;
  vectA[1] = 0.;
  vectA[2] = 0.;
  vectB[0] = 0.3333;
  vectB[1] = 1.;
  vectB[2] = 1.;
  shade = shadeNew(_("green to red"), vectA, vectB, shade_colorModeHSV);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a white to red color range. */
  vectA[0] = -.3333;
  vectA[1] = 0.8;
  vectA[2] = 0.;
  vectB[0] = 0.3333;
  vectB[1] = 0.1;
  vectB[2] = 1.;
  shade = shadeNew(_("light green to red"), vectA, vectB, shade_colorModeHSV);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a black to white color range. */
  vectA[0] = 0.;
  vectA[1] = 0.;
  vectA[2] = 1.;
  vectB[0] = 0.;
  vectB[1] = 0.;
  vectB[2] = 0.;
  shade = shadeNew(_("black to white"), vectA, vectB, shade_colorModeHSV);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a black to white color range. */
  vectA[0] = 0.;
  vectA[1] = 0.;
  vectA[2] = -1.;
  vectB[0] = 0.;
  vectB[1] = 0.;
  vectB[2] = 1.;
  shade = shadeNew(_("white to black"), vectA, vectB, shade_colorModeHSV);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);
  
  /* Create a black to red (through purple) color range. */
  vectA[0] = 1.;
  vectA[1] = 1.;
  vectA[2] = 1.;
  vectB[0] = 0.;
  vectB[1] = 0.;
  vectB[2] = 0.;
  shade = shadeNew(_("purple color"), vectA, vectB, shade_colorModeHSV);
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  return toolShadeList;
}
