/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Damien
	CALISTE, laboratoire L_Sim, (2001-2009)
  
	Adresse ml :
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Damien
	CALISTE, laboratoire L_Sim, (2001-2009)

	E-mail address:
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at COPYING.
*/
#include "legend.h"

#include <visu_object.h>
#include <visu_configFile.h>
#include <openGLFunctions/objectList.h>
#include <openGLFunctions/text.h>
#include <coreTools/toolConfigFile.h>

static OpenGLExtension* extLegend;
static gboolean extLegendIsBuilt;

#define FLAG_RESOURCE_LEGEND_USED   "legend_is_on"
#define DESC_RESOURCE_LEGEND_USED   "Control if the legend is drawn ; boolean (0 or 1)"

/* Local callbacks. */
static void onDataNewSignal(GObject *visu, VisuData *dataObj, gpointer data);
static void onDataReadySignal(GObject *obj, VisuData *dataObj, gpointer data);
static void rebuildLegendOnResources(GObject *obj, VisuData *dataObj, gpointer data);
static void onLegendParametersChange(VisuData *dataObj, OpenGLView *view,
				     gpointer data);
static void onNodePopulationChanged(VisuData *dataObj, int *nodes, gpointer data);

/* Local routines. */
static void rebuildLegend(VisuData *dataObj);
static gboolean readLegendIsOn(gchar **lines, int nbLines,
			       int position, VisuData *dataObj, GError **error);
static void exportResources(GString *data, VisuData *dataObj);

OpenGLExtension* extLegendInit()
{
  char *name = EXT_LEGEND_ID;
  char *description = _("Draw the name and the shape of available elements on screen.");
  int listLegend;
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr,"Ext Legend: initialising the legend OpenGL extension...\n");

  listLegend = openGLObjectList_new(1);
  extLegend  = OpenGLExtension_new(name, _(name), description,
				   listLegend, rebuildLegend);
  OpenGLExtensionSet_saveOpenGLState(extLegend, TRUE);
  OpenGLExtensionSet_priority(extLegend, OPENGL_EXTENSION_PRIORITY_LAST);

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_LEGEND_USED,
					  DESC_RESOURCE_LEGEND_USED,
					  1, readLegendIsOn);
  visuConfigFileSet_version(resourceEntry, 3.5f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResources);

  /* Initialisation des valeurs par dfaut. */
  extLegendIsBuilt = FALSE;
  extLegend->used  = FALSE;

  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onDataNewSignal), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReadySignal), (gpointer)0);

  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
 		   G_CALLBACK(rebuildLegendOnResources), (gpointer)0);

  return extLegend;
}

gboolean extLegendSet_isOn(gboolean value)
{
  if (value == extLegend->used)
    return FALSE;

  extLegend->used = value;
  return (value && !extLegendIsBuilt);
}
gboolean extLegendGet_isOn()
{
  return extLegend->used;
}

static void onDataNewSignal(GObject *visu _U_, VisuData *dataObj,
			    gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "OpenGLWidthHeight",
		   G_CALLBACK(onLegendParametersChange), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationIncrease",
		   G_CALLBACK(onNodePopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
		   G_CALLBACK(onNodePopulationChanged), (gpointer)0);
}
static void onDataReadySignal(GObject *obj _U_, VisuData *dataObj, gpointer data _U_)
{
  DBG_fprintf(stderr, "Ext Legend: catch 'dataReadyForRendering' signal.\n");
  if (dataObj)
    {
      if (visuDataGet_changeElementFlag(dataObj))
	{
	  DBG_fprintf(stderr,"Ext Legend: elements not changed, keep it.\n");
	  return;
	}
      extLegendIsBuilt = FALSE;
      extLegendDraw(dataObj);
    }
}
static void rebuildLegendOnResources(GObject *obj _U_, VisuData *dataObj,
				     gpointer data _U_)
{
  if (!dataObj)
    return;

  extLegendIsBuilt = FALSE;
  extLegendDraw(dataObj);
}
static void onLegendParametersChange(VisuData *dataObj, OpenGLView *view _U_,
				     gpointer data _U_)
{
  extLegendIsBuilt = FALSE;
  extLegendDraw(dataObj);
}
static void onNodePopulationChanged(VisuData *dataObj, int *nodes _U_,
				    gpointer data _U_)
{
  extLegendIsBuilt = FALSE;
  extLegendDraw(dataObj);
}

static void rebuildLegend(VisuData *dataObj)
{
  extLegendIsBuilt = FALSE;
  extLegendDraw(dataObj);
}

void extLegendDraw(VisuData *dataObj)
{
  int viewport[4];
  guint width, height, dw;
#define LEGEND_HEIGHT 30
#define ELEMENT_SIZE 25
#define LEGEND_PADDING 10
  guint i;
  VisuElement *ele;
  float scale;
  GString *str;
  VisuNodeArray *array;

  if (extLegendIsBuilt || !dataObj)
    return;

  DBG_fprintf(stderr, "Ext Legend: drawing the legend.\n");
  extLegendIsBuilt = TRUE;

  openGLText_initFontList();

  glNewList(extLegend->objectListId, GL_COMPILE);

  /* We get the size of the current viewport. */
  glGetIntegerv(GL_VIEWPORT, viewport);

  /* We change the viewport. */
  width = (guint)(viewport[2] - 2 * LEGEND_PADDING);
  height = (guint)LEGEND_HEIGHT;

  glViewport(LEGEND_PADDING, viewport[3] - LEGEND_HEIGHT - LEGEND_PADDING,
	     width, height);

  glDisable(GL_FOG);
  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  /* We change the projection for a 2D one. */
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0.0, (float)width, 0., (float)height, -50., +50.);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  /* We draw a big transparent square to do the back. */
  glColor4f(1., 1., 1., 0.4);
  glRecti(0, 0, width, height);

  /* We draw the legend. */
  str = g_string_new("");
  dw = MAX(width / dataObj->ntype, ELEMENT_SIZE + 10 + 60);
  scale = 0.5f * ELEMENT_SIZE / visuDataGet_allElementExtens(dataObj);
  array = visuDataGet_nodeArray(dataObj);
  for (i = 0; i < dataObj->ntype; i++)
    {
      ele = dataObj->fromIntToVisuElement[i];

      /* The element. */
      glEnable(GL_LIGHTING);
      glCallList(visuElementGet_identifierMaterial(ele));
      glPushMatrix();
      glTranslated(i * dw + 5 + ELEMENT_SIZE / 2,
		   LEGEND_HEIGHT / 2, 0.f);
      glRotated(45., 0, 0, 1);  
      glRotated(60., 0, 1, 0); 
      glScalef(scale, scale, scale);
      glCallList(ele->openGLIdentifier);
      glPopMatrix();
      glDisable(GL_LIGHTING);

      /* The label. */
      glColor3f(0., 0., 0.);
      g_string_printf(str, "%s (%d)", ele->name, array->numberOfStoredNodes[i]);
      glRasterPos2i(i * dw + 5 + ELEMENT_SIZE + 5, (LEGEND_HEIGHT - 20) / 2 + 5);
      openGLText_drawChars(str->str, TEXT_SMALL);
    }
  g_string_free(str, TRUE);

  /* We change the projection back. */
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  /* We set viewport back. */
  glViewport(0, 0,viewport[2], viewport[3]);

  glEndList();
}


static gboolean readLegendIsOn(gchar **lines, int nbLines,
			       int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  extLegendSet_isOn(val);
  return TRUE;
}
static void exportResources(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_LEGEND_USED);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_LEGEND_USED);
  g_string_append_printf(data, "    %d\n", extLegend->used);
}
