/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_extension.h"
#include "visu_configFile.h"
#include "coreTools/toolConfigFile.h"
#include "extensions/externalOpenGLExtensions.h"

#include <stdlib.h>
#include <string.h>
#include <GL/gl.h>
#include <GL/glu.h> 

#include "visu_tools.h"

/**
 * SECTION:visu_extension
 * @short_description: All objects drawn by V_Sim are defined in by a
 * #OpenGLExtension object
 *
 * <para>All objects that are drawn by V_Sim are handled by a
 * #OpenGLExtension object. Such an object has an OpenGL list. This
 * list is only COMPILED. When V_Sim receives the 'OpenGLAskForReDraw'
 * or the 'OpenGLForceReDraw' signals, each list of all known
 * #OpenGLExtension are excecuted. This excecution can be canceled if
 * the used flag of the #OpenGLExtension object is set to FALSE. The
 * order in which the lists are called depends on the priority of the
 * #OpenGLExtension object. This priority is set to
 * #OPENGL_EXTENSION_PRIORITY_NORMAL as default value, but it can be
 * tune by a call to OpenGLExtensionSet_priority(). This priority is
 * an integer, the lower it is, the sooner the list is
 * excecuted.</para>
 *
 * <para>The method registerOpenGLExtension() is used to declare to
 * V_Sim that there is a new #OpenGLExtension object available. This
 * allows to create extension when V_Sim is already
 * running. Nevertheless, an extension must be initialized in the
 * initialisation process, it is better to add an
 * #initOpenGLExtensionFunc method in the #listInitExtensionFunc array
 * declared in extensions/externalOpenGLExtensions.h.</para>
 *
 * <para>Once again, the OpenGL list corresponding to an OpenGL
 * extension is COMPILE only. Then, OpenGL methods like glIsenabled()
 * are totally unusefull because it is called when the list is
 * compiled not when the list is called. If the extension needs to
 * alter some OpenGL state, such as desable GL_LIGHTING, it needs to
 * set a flag for the extension. With this flag, V_Sim will save the
 * OpenGL states and restore it when the list is called. Use
 * OpenGLExtensionSet_saveOpenGLState() to set this flag.</para>
 */

#define FLAG_PARAMETER_MODE "extension_render"
#define DESC_PARAMETER_MODE "Rules the way OpenGl draws extensions (see opengl_render); name (string) value (string)"
static gboolean readExtensionRendering(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);
static void exportParametersRendering(GString *data, VisuData *dataObj);

/* A GHashTable to store all the available OpenGL extensions
   in the system. The keys are the name of each extension. */
static GList *availableOpenGLExtensions;

/* This flag is TRUE when some priorities are new or have changed, and 
   the list should be reordered. */
static gboolean OpenGLExtension_reorderingNeeded;
/* Method used to compare the priority of two extensions. */
static gint compareExtensionPriority(gconstpointer a, gconstpointer b);

static void callList(OpenGLExtension *ext, RenderingModeId *renderingMode,
		     RenderingModeId globalRenderingMode);
static void drawExtension(OpenGLExtension *extension, RenderingModeId renderingMode);

/***************/
/* Public part */
/***************/


/* Method used to create such structures. */
OpenGLExtension* OpenGLExtension_new(const gchar* name, const gchar *nameI18n,
				     const gchar* description,
				     int objectListId, rebuildObjectListFunc rebuild)
{
  OpenGLExtension *extension;

  extension = g_malloc(sizeof(OpenGLExtension));
  extension->name = g_strdup(name);
  extension->nameI18n = g_strdup(nameI18n);
  if (description)
    extension->description = g_strdup(description);
  else
    extension->description = (char*)0;

  extension->objectListId = objectListId;
  extension->rebuild = rebuild;
  extension->priority = OPENGL_EXTENSION_PRIORITY_NORMAL;
  extension->saveState = FALSE;
  extension->isSensitiveToRenderingMode = FALSE;
  extension->preferedRenderingMode = followGeneralSetting;

  extension->used = 0;

  return extension;
}

/* Free all the allocated attributes of the specified extension. */
void OpenGLExtension_free(OpenGLExtension* extension)
{
  if (!extension)
    return;
  if (extension->name)
    g_free(extension->name);
  if (extension->nameI18n)
    g_free(extension->nameI18n);
  if (extension->description)
    g_free(extension->description);
  g_free(extension);
}
/* Get if the extension is used or not. */
int OpenGLExtensionGet_active(OpenGLExtension* extension)
{
  if (extension)
    return extension->used;
  else
    return 0;
}
/* Set if an extension is actually used or not. */
void OpenGLExtensionSet_active(OpenGLExtension* extension, int value)
{
  if (!extension || extension->used == value)
    return;

  extension->used = value;
}
GList* OpenGLExtensionGet_list()
{
  if (OpenGLExtension_reorderingNeeded)
    {
      DBG_fprintf(stderr, "Visu Extension : sorting known extension depending on their priority.\n");
      availableOpenGLExtensions = g_list_sort(availableOpenGLExtensions, compareExtensionPriority);
      OpenGLExtension_reorderingNeeded = FALSE;
    }
  return availableOpenGLExtensions;
}
OpenGLExtension* OpenGLExtensionGet_fromName(const gchar* name)
{
  GList *pnt;
  OpenGLExtension *ext;

  DBG_fprintf(stderr, "Visu Extension: get '%s' from list.\n", name);

  pnt = availableOpenGLExtensions;
  while (pnt)
    {
      ext = (OpenGLExtension*)pnt->data;
      if (!strcmp(ext->name, name))
	return ext;
      pnt = g_list_next(pnt);
    }
  return (OpenGLExtension*)0;
}

void OpenGLExtensionSet_priority(OpenGLExtension* extension, int priority)
{
  g_return_if_fail(extension);
  extension->priority = priority;
  OpenGLExtension_reorderingNeeded = TRUE;
}

void OpenGLExtensionSet_saveOpenGLState(OpenGLExtension *extension, gboolean saveState)
{
  g_return_if_fail(extension);
  extension->saveState = saveState;
}

void OpenGLExtensionSet_sensitiveToRenderingMode(OpenGLExtension* extension,
						 gboolean status)
{
  g_return_if_fail(extension);
  extension->isSensitiveToRenderingMode = status;
}
gboolean OpenGLExtensionSet_preferedRenderingMode(OpenGLExtension* extension,
						  RenderingModeId value)
{
  g_return_val_if_fail(extension, FALSE);
  g_return_val_if_fail(value < nb_renderingModes ||
		       value == followGeneralSetting, FALSE);

  if (extension->preferedRenderingMode == value)
    return FALSE;

  extension->preferedRenderingMode = value;

  return TRUE;
}



/* A method used by user to registered a new extension. */
void OpenGLExtensionRegister(OpenGLExtension *extension)
{
  DBG_fprintf(stderr, "Visu Extension : registering a new OpenGL extension ... ");
  g_return_if_fail(extension && extension->name && extension->name[0]);

  availableOpenGLExtensions = g_list_append(availableOpenGLExtensions,
					    (gpointer)extension);

  OpenGLExtension_reorderingNeeded = TRUE;
  DBG_fprintf(stderr, "'%s' (%p).\n", extension->name, (gpointer)extension);
}

/* A method used by user to remove a previously registered extension. */
void OpenGLExtensionRemove(OpenGLExtension *extension)
{
  DBG_fprintf(stderr, "Visu Extension : removing a registered OpenGL"
	      " extension (%p).\n", (gpointer)extension);
  g_return_if_fail(extension);

  availableOpenGLExtensions = g_list_remove(availableOpenGLExtensions,
					    (gpointer)extension);
}

void OpenGLExtensionCall_list(const char *name, gboolean lastOnly)
{
  OpenGLExtension *ext;
  RenderingModeId renderingMode, globalRenderingMode;

  DBG_fprintf(stderr, "Visu Extension: call '%s' list.\n", name);

  globalRenderingMode = openGLGet_globalRenderingOption();
  renderingMode = globalRenderingMode;
  ext = OpenGLExtensionGet_fromName(name);
  g_return_if_fail(ext);

  if (ext->used && ((lastOnly && ext->priority == OPENGL_EXTENSION_PRIORITY_LAST) ||
		    (!lastOnly && ext->priority < OPENGL_EXTENSION_PRIORITY_LAST)))
    {
      callList(ext, &renderingMode, globalRenderingMode);
      if (renderingMode != globalRenderingMode)
	/* Return the rendering mode to normal. */
	openGLApply_renderingMode(globalRenderingMode);
    }
}
/* For each extension that has a valid list (id > 1000) a glCallList is raised. */
void OpenGLExtensionCall_allLists()
{
  GList *pnt;
  RenderingModeId renderingMode, globalRenderingMode;
  OpenGLExtension *ext;

  if (OpenGLExtension_reorderingNeeded)
    {
      DBG_fprintf(stderr, "Visu Extension: sorting known extension depending on their priority.\n");
      availableOpenGLExtensions = g_list_sort(availableOpenGLExtensions, compareExtensionPriority);
      OpenGLExtension_reorderingNeeded = FALSE;
    }
  globalRenderingMode = openGLGet_globalRenderingOption();
  renderingMode = globalRenderingMode;
  for (pnt = availableOpenGLExtensions; pnt; pnt = g_list_next(pnt))
    {
      ext = (OpenGLExtension*)pnt->data;
      if (ext->priority < OPENGL_EXTENSION_PRIORITY_LAST)
	callList((OpenGLExtension*)pnt->data, &renderingMode, globalRenderingMode);
    }
  if (renderingMode != globalRenderingMode)
    /* Return the rendering mode to normal. */
    openGLApply_renderingMode(globalRenderingMode);
}
void OpenGLExtensionCall_lastLists()
{
  GList *pnt;
  RenderingModeId renderingMode, globalRenderingMode;
  OpenGLExtension *ext;

  DBG_fprintf(stderr, "Visu Extension: draw PRIORITY_LAST only.\n");
  globalRenderingMode = openGLGet_globalRenderingOption();
  renderingMode = globalRenderingMode;
  for (pnt = availableOpenGLExtensions; pnt; pnt = g_list_next(pnt))
    {
      ext = (OpenGLExtension*)pnt->data;
      if (ext->priority == OPENGL_EXTENSION_PRIORITY_LAST)
	callList(ext, &renderingMode, globalRenderingMode);
    }
  if (renderingMode != globalRenderingMode)
    /* Return the rendering mode to normal. */
    openGLApply_renderingMode(globalRenderingMode);
}
void rebuildAllExtensionsLists(VisuData *dataObj)
{
  GList *pnt;

  DBG_fprintf(stderr, "Visu Extension: Rebuilding lists...\n");
  if (OpenGLExtension_reorderingNeeded)
    {
      DBG_fprintf(stderr, "Visu Extension: sorting known extension depending on their priority.\n");
      availableOpenGLExtensions = g_list_sort(availableOpenGLExtensions, compareExtensionPriority);
      OpenGLExtension_reorderingNeeded = FALSE;
    }
  pnt = availableOpenGLExtensions;
  while (pnt)
    {
      if (((OpenGLExtension*)pnt->data)->used &&
	  ((OpenGLExtension*)pnt->data)->rebuild)
	{
	  DBG_fprintf(stderr, "Visu Extension: rebuild extension %s (list %d).\n",
		      ((OpenGLExtension*)pnt->data)->name,
		      ((OpenGLExtension*)pnt->data)->objectListId);
	  ((OpenGLExtension*)pnt->data)->rebuild(dataObj);
	}
      pnt = g_list_next(pnt);
    }
}
void OpenGLExtensionRebuild_list(VisuData *dataObj, const char *name)
{
  OpenGLExtension *ext;

  DBG_fprintf(stderr, "Visu Extension: rebuilding '%s' list.\n", name);
  ext = OpenGLExtensionGet_fromName(name);
  g_return_if_fail(ext);

  if (ext->used && ext->rebuild)
    ext->rebuild(dataObj);
}


/****************/
/* Private area */
/****************/

/* Initialise all the variable of this part. */
int initOpenGLExtensions()
{
  VisuConfigFileEntry *confEntry;

  availableOpenGLExtensions = (GList*)0;
  OpenGLExtension_reorderingNeeded = FALSE;

  confEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
				      FLAG_PARAMETER_MODE,
				      DESC_PARAMETER_MODE,
				      1, readExtensionRendering);
  visuConfigFileSet_version(confEntry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersRendering);
  return 1;
}
void loadExtensions()
{
  OpenGLExtension *extension;
  int i, res;

  res = 1;
  for (i = 0; listInitExtensionFunc[i]; i++)
    {
      extension = listInitExtensionFunc[i]();
      if (!extension)
	res = 0;
      OpenGLExtensionRegister(extension);
    }
  
  if (!res)
    g_warning("Some OpenGL extensions can't initialse.\n");
}

static void callList(OpenGLExtension *ext, RenderingModeId *renderingMode,
		     RenderingModeId globalRenderingMode)
{
  if (ext->used &&
      ext->objectListId > 1000)
    {
      /* The extension needs its own rendering mode. */
      if (ext->isSensitiveToRenderingMode &&
	  ext->preferedRenderingMode < nb_renderingModes)
	{
	  if (ext->preferedRenderingMode != *renderingMode)
	    {
	      openGLApply_renderingMode(ext->preferedRenderingMode);
	      *renderingMode = ext->preferedRenderingMode;
	    }
	}
      else
	{
	  if (*renderingMode != globalRenderingMode)
	    {
	      openGLApply_renderingMode(globalRenderingMode);
	      *renderingMode = globalRenderingMode;
	    }
	}
      /* Save OpenGL state if necessary. */
      if (ext->saveState)
	{
	  DBG_fprintf(stderr, "Visu Extension: save state.\n");
	  glPushAttrib(GL_ENABLE_BIT);
	}

      drawExtension(ext, *renderingMode);

      if (ext->saveState)
	{
	  DBG_fprintf(stderr, "Visu Extension: restore state.\n");
	  glPopAttrib();
	}
    }
}
static void drawExtension(OpenGLExtension *extension, RenderingModeId renderingMode)
{
/*   float mat[5] = {0., 0., 0., 0., 0.}; */
/*   float rgba[4] = {0., 0., 0., 0.}; */

  /* Add a wireframe draw if renderingMode is SmoothAndEdge. */
  if (extension->isSensitiveToRenderingMode && renderingMode == SmoothAndEdge)
    {
      glPushAttrib(GL_ENABLE_BIT);
      glEnable(GL_POLYGON_OFFSET_FILL);
      glPolygonOffset(1.0, 1.0);
    }

  /* Call the compiled list. */
  DBG_fprintf(stderr, "Visu Extension: call list %d (%s).\n",
	      extension->objectListId, extension->name);
  glCallList(extension->objectListId);

  /* Add a wireframe draw if renderingMode is SmoothAndEdge. */
  if (extension->isSensitiveToRenderingMode && renderingMode == SmoothAndEdge)
    {
      glDisable(GL_POLYGON_OFFSET_FILL);
      glDisable(GL_LIGHTING);
      glColor3f (0.0, 0.0, 0.0);
      glLineWidth(1);
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glCallList(extension->objectListId);
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      glPopAttrib();
    }
}

static gint compareExtensionPriority(gconstpointer a, gconstpointer b)
{
  if (((OpenGLExtension*)a)->priority < ((OpenGLExtension*)b)->priority)
    return (gint)-1;
  else if (((OpenGLExtension*)a)->priority > ((OpenGLExtension*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}

static gboolean readExtensionRendering(gchar **lines, int nbLines, int position,
				       VisuData *dataObj _U_, GError **error)
{
  gchar **val;
  OpenGLExtension *ext;
  RenderingModeId id;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_string(lines[0], position, &val, 2, FALSE, error))
    return FALSE;
  ext = OpenGLExtensionGet_fromName(val[0]);
  if (!ext)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: the extension"
			     " '%s' is unknown.\n"), position, val[0]);
      g_strfreev(val);
      return FALSE;
    }
  if (!openGLGet_renderingFromName(val[1], &id))
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: the rendering mode"
			     " '%s' is unknown.\n"), position, val[1]);
      g_strfreev(val);
      return FALSE;
    }
  g_strfreev(val);
  OpenGLExtensionSet_preferedRenderingMode(ext, id);

  return TRUE;
}
static void exportParametersRendering(GString *data, VisuData *dataObj _U_)
{
  GList *tmp;
  OpenGLExtension *ext;
  const char **names;

  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_MODE);

  names = openGLGet_allRenderingModes();
  tmp = availableOpenGLExtensions;
  while (tmp)
    {
      ext = (OpenGLExtension*)tmp->data;
      if (ext->isSensitiveToRenderingMode &&
	  ext->preferedRenderingMode < nb_renderingModes)
	g_string_append_printf(data, "%s: %s %s\n", FLAG_PARAMETER_MODE,
			       ext->name, names[ext->preferedRenderingMode]);
      tmp = g_list_next(tmp);
    }
  g_string_append_printf(data, "\n");
}
