/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "panelAxes.h"

#include <string.h>

#include <support.h>
#include <visu_object.h>
#include <openGLFunctions/view.h>
#include <coreTools/toolColor.h>
#include <extraGtkFunctions/gtk_toolPanelWidget.h>
#include <extraGtkFunctions/gtk_lineObjectWidget.h>
#include <extensions/box.h>
#include <extensions/axes.h>
#include <extensions/scale.h>
#include <extensions/legend.h>

/**
 * SECTION: panelAxes
 * @short_description: The tab where axes, box, scale and legend are setup.
 */

/* Sensitive widget in this subpanel. */
static GtkWidget *panelAxes;
static GtkWidget *lineBox;
static GtkWidget *lineAxes;
static GtkWidget *lineScales;
static GtkWidget *spinScaleLength;
static GtkWidget *entryScale;
static GtkWidget *originespin[3];
static GtkWidget *orientationspin[3];
static GtkWidget *checkLegend;
static GtkWidget *checkLengths, *checkBasis;
static GtkWidget *lblScales;

static gboolean disableCallbacks;

/* Private functions. */
static GtkWidget *createInteriorAxes();

/* Local callbacks. */
static void onBoxChecked(LineObject *line, gboolean used, gpointer data);
static void onBoxRGBValueChanged(LineObject *line, float *rgb, gpointer data);
static void onBoxLineWidthChanged(LineObject *line, gint width, gpointer data);
static void onBoxStippleChanged(LineObject *line, gint stipple, gpointer data);
static void onBoxLengthsToggled(GtkToggleButton *toggle, gpointer data);
static void onAxesChecked(LineObject *line, gboolean used, gpointer data);
static void onAxesWidthChanged(LineObject *line, gint width, gpointer user_data);
static void onAxesRGBChanged(LineObject *line, float *rgb, gpointer data);
static void onAxesStippleChanged(LineObject *line, gint stipple, gpointer data);
static void onEnter(ToolPanel *toolPanel, gpointer data);
static void onResources(GObject *object, VisuData *dataObj, gpointer data);
static void onScalesChecked(LineObject *line, gboolean used, gpointer data);
static void onScalesWidthChanged(LineObject *line, gint width, gpointer user_data);
static void onScalesRGBChanged(LineObject *line, float *rgb, gpointer data);
static void onScalesStippleChanged(LineObject *line, gint stipple, gpointer data);
static void scaleDistanceChanged(GtkSpinButton *spin,gpointer data);
static void scaleOriginChanged(GtkSpinButton *spin,gpointer data);
static void scaleOrientationChanged(GtkSpinButton *spin,gpointer data);
static void onLegendChanged(GtkEntry *entry, gpointer data);
static void onUseLegendChanged(GtkToggleButton *toggle, gpointer data);

ToolPanel* panelAxes_init()
{
  panelAxes = toolPanelNew_withIconFromPath("Panel_axes", _("Box, axes and labels"),
					    _("Frames/labels"), "stock-axes_20.png");
  if (!panelAxes)
    return (ToolPanel*)0;

  /* Create the widgets required before interior creation. */
  checkLengths = gtk_check_button_new_with_mnemonic(_("Show box _lengths"));
  checkBasis = gtk_check_button_new();

  gtk_container_add(GTK_CONTAINER(panelAxes), createInteriorAxes());
  toolPanelSet_dockable(TOOL_PANEL(panelAxes), TRUE);

  /* Create the callbacks of all the sensitive widgets. */
  g_signal_connect(G_OBJECT(panelAxes), "page-entered",
		   G_CALLBACK(onEnter), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
		   G_CALLBACK(onResources), (gpointer)0);

  /* Private parameters. */
  disableCallbacks = FALSE;

  return TOOL_PANEL(panelAxes);
}

static GtkWidget *createInteriorAxes()
{
  GtkWidget *vbox, *hbox, *vbox2;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *align;
  float *xyz, *orientation, len;
  VisuGlExtScale *scale;
  gchar *lblXYZ[3] = {"X", "Y", "Z"};
  float xyzDefault[3] = {0.f, 0.f, 0.f};
  float orientationDefault[3] = {1.f, 0.f, 0.f};
  int i;
  GList *scales;
#if GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();
#endif

  vbox = gtk_vbox_new(FALSE, 0);

  /*********************/
  /* The Bounding box. */
  /*********************/
  align = gtk_alignment_new(0.5, 0.5, 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 15, 0, 0);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);
  lineBox = lineObjectNew(_("Bounding box"));
  gtk_container_add(GTK_CONTAINER(align), lineBox);
  vbox2 = lineObjectGet_optionVBox(LINE_OBJECT(lineBox));
  gtk_box_pack_start(GTK_BOX(vbox2), checkLengths, FALSE, FALSE, 0);
/*   label = gtk_label_new(_("Draw basis set:")); */
/*   gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5); */
/*   gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0); */
/*   gtk_box_pack_start(GTK_BOX(hbox), checkBasis, FALSE, FALSE, 0); */
  g_signal_connect(G_OBJECT(lineBox), "use-changed",
		   G_CALLBACK(onBoxChecked), (gpointer)0);
  g_signal_connect(G_OBJECT(lineBox), "width-changed",
		   G_CALLBACK(onBoxLineWidthChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineBox), "color-changed",
		   G_CALLBACK(onBoxRGBValueChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineBox), "stipple-changed",
		   G_CALLBACK(onBoxStippleChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(checkLengths), "toggled",
		   G_CALLBACK(onBoxLengthsToggled), (gpointer)0);

  /*************/
  /* The Axes. */
  /*************/
  lineAxes = lineObjectNew(_("Basis set"));
  align = gtk_alignment_new(0.5, 0., 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 15, 0, 0);
  gtk_container_add(GTK_CONTAINER(align), lineAxes);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(lineAxes), "use-changed",
		   G_CALLBACK(onAxesChecked), (gpointer)0);
  g_signal_connect(G_OBJECT(lineAxes), "width-changed",
		   G_CALLBACK(onAxesWidthChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineAxes), "color-changed",
		   G_CALLBACK(onAxesRGBChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineAxes), "stipple-changed",
		   G_CALLBACK(onAxesStippleChanged), (gpointer)0);

  /**************/
  /* The legend */
  /**************/
  hbox = gtk_hbox_new(FALSE, 0);
  align = gtk_alignment_new(0.5, 0., 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 15, 0, 0);
  gtk_container_add(GTK_CONTAINER(align), hbox);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);

  /* The drawn checkbox. */
  checkLegend = gtk_check_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), checkLegend, FALSE, FALSE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkLegend),
			       visu_glExt_legend_getOn());
  g_signal_connect(G_OBJECT(checkLegend), "toggled",
                   G_CALLBACK(onUseLegendChanged), (gpointer)0);

  /* The label. */
  label = gtk_label_new(_("<b>Legend</b>"));
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_widget_set_name(label, "label_head");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);

  /*************************/
  /* Adding scale widgets. */
  /*************************/
  lineScales = lineObjectNew(_("Label"));
  align = gtk_alignment_new(0.5, 0., 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 15, 0, 0);
  gtk_container_add(GTK_CONTAINER(align), lineScales);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);
  vbox2 = lineObjectGet_optionVBox(LINE_OBJECT(lineScales));
  g_signal_connect(G_OBJECT(lineScales), "use-changed",
		   G_CALLBACK(onScalesChecked), (gpointer)0);
  g_signal_connect(G_OBJECT(lineScales), "width-changed",
		   G_CALLBACK(onScalesWidthChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineScales), "color-changed",
		   G_CALLBACK(onScalesRGBChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineScales), "stipple-changed",
		   G_CALLBACK(onScalesStippleChanged), (gpointer)0);

  /* To be removed. */
  lblScales = gtk_label_new(_("<i><b>Several scales are defined from resource files,\n"
                              "but only one is editable.</b></i>"));
  gtk_label_set_use_markup(GTK_LABEL(lblScales), TRUE);
  gtk_box_pack_end(GTK_BOX(vbox2), lblScales, FALSE, FALSE, 10);
  gtk_widget_set_no_show_all(lblScales, TRUE);
  if (visu_glExt_scale_getAll())
    {
      scales = visu_glExt_scale_getAll();
      scale = VISU_GLEXT_SCALE(scales->data);
      len = visu_glExt_scale_getLength(scale);
      xyz = visu_glExt_scale_getOrigin(scale);
      orientation = visu_glExt_scale_getOrientation(scale);
      if (scales->next)
        gtk_widget_show(lblScales);
    }
  else
    {
      len = 5.;
      xyz = xyzDefault;
      orientation = orientationDefault;
    }

  /* code yoann*/
  hbox = gtk_hbox_new(FALSE,0);
  label = gtk_label_new(_("Legend:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox),label, FALSE, FALSE, 0); 
  entryScale = gtk_entry_new();
  gtk_widget_set_tooltip_text(entryScale,
		       _("Use blank legend to print the default"
			 " value with the distance."));
  gtk_box_pack_start(GTK_BOX(hbox), entryScale, TRUE, TRUE, 3);
  g_signal_connect(G_OBJECT(entryScale), "activate",
		   G_CALLBACK(onLegendChanged), (gpointer)0);
  
  label = gtk_label_new(_("Length:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
 
  spinScaleLength = gtk_spin_button_new_with_range(1., 1000., 1.);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinScaleLength), len);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spinScaleLength), 2);
  g_signal_connect((gpointer)spinScaleLength, "value-changed",
                   G_CALLBACK(scaleDistanceChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hbox),spinScaleLength, FALSE, FALSE, 3); 
  gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,5);

  table = gtk_table_new(3, 4, FALSE); /* je declare une nouvelle table*/
  gtk_box_pack_start(GTK_BOX(vbox2), table, FALSE, FALSE, 0); 

  label = gtk_label_new(_("Origin")); /* je cree le premier labelle*/
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);/*fonction pour placer le labelle a droite*/
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, /*je positionne le label */
                   GTK_FILL | GTK_EXPAND, (GtkAttachOptions)0, 5, 0);/*j'indique qu il peut prendre toute la place qu'il a besoins*/
  label = gtk_label_new(_("Orientation")); /* je crée le second label*/
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,/* je le positionne*/
                   GTK_FILL | GTK_EXPAND, (GtkAttachOptions)0, 5, 0);/* j'indique qui peut s etendre */

  for (i = 0; i < 3; i++)
    {
      gtk_table_attach(GTK_TABLE(table), gtk_label_new(lblXYZ[i]), 1 + i, 2 + i, 0, 1,
		       (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 0);

      originespin[i] = gtk_spin_button_new_with_range(-10., 999., 1.);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(originespin[i]), xyz[i]);
      gtk_spin_button_set_digits(GTK_SPIN_BUTTON(originespin[i]), 2);
      g_signal_connect(G_OBJECT(originespin[i]), "value-changed",
		       G_CALLBACK(scaleOriginChanged), GINT_TO_POINTER(i));
      gtk_table_attach(GTK_TABLE(table), originespin[i], 1 + i, 2 + i, 1, 2,
		       (GtkAttachOptions)0, (GtkAttachOptions)0, 3, 0);
      orientationspin[i] = gtk_spin_button_new_with_range(-10., 999., 1.);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(orientationspin[i]), orientation[i]);
      gtk_spin_button_set_digits(GTK_SPIN_BUTTON(orientationspin[i]), 2);
      g_signal_connect(G_OBJECT(orientationspin[i]), "value-changed",
		       G_CALLBACK(scaleOrientationChanged), GINT_TO_POINTER(i));
      gtk_table_attach(GTK_TABLE(table), orientationspin[i], 1 + i, 2 + i, 2, 3,
		       (GtkAttachOptions)0, (GtkAttachOptions)0, 3, 0);
    }

  gtk_widget_show_all(vbox); /* fonctions qui permet de tous dessiner */

  return vbox;
}


/*************/
/* Callbacks */
/*************/
static void axesRedraw()
{
  VisuData *dataObj;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelAxes));
  if (!dataObj) 
    return;
  visu_glExt_axes_draw(dataObj);
  VISU_ADD_REDRAW;
}
static void onAxesChecked(LineObject *line _U_, gboolean used, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  visu_glExt_axes_setOn(used);
  axesRedraw();
}
static void onAxesWidthChanged(LineObject *line _U_, gint width, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (visu_glExt_axes_setLineWidth((float)width))
    axesRedraw();
}
static void onAxesRGBChanged(LineObject *line _U_, float *rgb, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (visu_glExt_axes_setRGBValues(rgb, TOOL_COLOR_MASK_RGBA))
    axesRedraw();
}
static void onAxesStippleChanged(LineObject *line _U_, gint stipple, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (visu_glExt_axes_setLineStipple(stipple))
    axesRedraw();
}
static void onResources(GObject *object _U_, VisuData *dataObj _U_, gpointer data)
{
  DBG_fprintf(stderr, "Panel Axes: catch the 'resourcesChanged'"
	      " signal, updating.\n");

  onEnter(TOOL_PANEL(panelAxes), data);
}
static void onEnter(ToolPanel *toolPanel _U_, gpointer data _U_)
{
  float *xyz, *orientation;
  VisuGlExtScale *scale;
  float xyzDefault[3] = {0.f, 0.f, 0.f};
  float orientationDefault[3] = {1.f, 0.f, 0.f};
  gchar *legend;
  GList *scales;

  disableCallbacks = TRUE;

  /* Set the box values. */
  lineObjectSet_use(LINE_OBJECT(lineBox), visu_glExt_box_getOn());
  lineObjectSet_width(LINE_OBJECT(lineBox), visu_glExt_box_getLineWidth());
  lineObjectSet_color(LINE_OBJECT(lineBox), visu_glExt_box_getRGBValues());
  lineObjectSet_stipple(LINE_OBJECT(lineBox), visu_glExt_box_getLineStipple());
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkLengths),
			       visu_glExt_box_getShowLegend());

  /* Set the axes values. */
  lineObjectSet_use(LINE_OBJECT(lineAxes), visu_glExt_axes_getOn());
  lineObjectSet_width(LINE_OBJECT(lineAxes), visu_glExt_axes_getLineWidth());
  lineObjectSet_color(LINE_OBJECT(lineAxes), visu_glExt_axes_getRGBValues());
  lineObjectSet_stipple(LINE_OBJECT(lineAxes), visu_glExt_axes_getLineStipple());

  /* Set the scale class values. */
  lineObjectSet_use(LINE_OBJECT(lineScales), visu_glExt_scale_getOn());
  lineObjectSet_width(LINE_OBJECT(lineScales), visu_glExt_scale_getDefaultLineWidth());
  lineObjectSet_color(LINE_OBJECT(lineScales), visu_glExt_scale_getDefaultRGBValues());
  lineObjectSet_stipple(LINE_OBJECT(lineScales), visu_glExt_scale_getDefaultStipple());

  /* To be removed. */
  gtk_widget_hide(lblScales);
  scales = visu_glExt_scale_getAll();
  if (scales)
    {
      scale = VISU_GLEXT_SCALE(scales->data);
      DBG_fprintf(stderr, "Panel Axes: set scales values from %p.\n",
		  (gpointer)scale);
      xyz = visu_glExt_scale_getOrigin(scale);
      orientation = visu_glExt_scale_getOrientation(scale);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinScaleLength),
				visu_glExt_scale_getLength(scale));
      legend = (gchar*)visu_glExt_scale_getLegend(scale);
      if (!legend)
	legend = "";
      if (scales->next)
        gtk_widget_show(lblScales);
    }
  else
    {
      xyz = xyzDefault;
      orientation = orientationDefault;
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinScaleLength), 5.);
      legend = "";
    }
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(originespin[0]), xyz[0]);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(originespin[1]), xyz[1]);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(originespin[2]), xyz[2]);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(orientationspin[0]), orientation[0]);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(orientationspin[1]), orientation[1]);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(orientationspin[2]), orientation[2]);

  gtk_entry_set_text(GTK_ENTRY(entryScale), legend);
      
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkLegend),
			       visu_glExt_legend_getOn());

  disableCallbacks = FALSE;
}


static void scaleRedraw()
{
  VisuData *dataObj;
  VisuOpenGLView *view;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelAxes));
  if (!dataObj) 
    return;
  view = visu_data_getOpenGLView(dataObj);
  visu_glExt_scale_drawAll(view->camera, MIN(view->window->width, view->window->height));
  VISU_ADD_REDRAW;
}
static void onScalesChecked(LineObject *line _U_, gboolean used, gpointer data _U_)
{
  visu_glExt_scale_setOn(used);
  scaleRedraw();
}
static void scaleDistanceChanged(GtkSpinButton *spin, gpointer data _U_)
{
  gboolean res;
  VisuGlExtScale *scale;
  float xyz[3], orientation[3], len;

  if (disableCallbacks)
    return;

  /* To be removed. */
  if (visu_glExt_scale_getAll())
    scale = VISU_GLEXT_SCALE(visu_glExt_scale_getAll()->data);
  else
    {
      xyz[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[0]));
      xyz[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[1]));
      xyz[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[2]));
      orientation[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[0]));
      orientation[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[1]));
      orientation[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[2]));
      len = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)) / 2.f;
      scale = visu_glExt_scale_new(xyz, orientation, len, (const gchar*)0);
    }

  DBG_fprintf(stderr, "Panel Box&axes: scale distance changed.\n");
  res = visu_glExt_scale_setLength(scale, (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)));
  if (*gtk_entry_get_text(GTK_ENTRY(entryScale)) == '\0')
    visu_glExt_scale_setLegend(scale, (const gchar*)0);
  else
    visu_glExt_scale_setLegend(scale, gtk_entry_get_text(GTK_ENTRY(entryScale)));

  if (res)
    scaleRedraw();
}

static void scaleOriginChanged(GtkSpinButton *spin, gpointer data)
{
  gboolean res;
  int xyzMask[3] = {VISU_GLEXT_XYZ_MASK_X, VISU_GLEXT_XYZ_MASK_Y, VISU_GLEXT_XYZ_MASK_Z};
  int val;
  VisuGlExtScale *scale;
  float xyz[3], orientation[3];

  val = GPOINTER_TO_INT(data);
  g_return_if_fail(val >= 0 && val < 3);

  if (disableCallbacks)
    return;

  /* To be removed. */
  if (visu_glExt_scale_getAll())
    scale = VISU_GLEXT_SCALE(visu_glExt_scale_getAll()->data);
  else
    {
      xyz[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[0]));
      xyz[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[1]));
      xyz[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[2]));
      orientation[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[0]));
      orientation[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[1]));
      orientation[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[2]));
      xyz[val] += 1.f;
      scale = visu_glExt_scale_new(xyz, orientation,
		       gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinScaleLength)),
		       (const gchar*)0);
    }

  DBG_fprintf(stderr, "Panel Box&axes: scale origine %d change.\n", val);
  xyz[val] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  res = visu_glExt_scale_setOrigin(scale, xyz, xyzMask[val]);

  if (res)
    scaleRedraw();
}

static void scaleOrientationChanged(GtkSpinButton *spin, gpointer data)
{
  gboolean res;
  int xyzMask[3] = {VISU_GLEXT_XYZ_MASK_X, VISU_GLEXT_XYZ_MASK_Y, VISU_GLEXT_XYZ_MASK_Z};
  int val;
  VisuGlExtScale *scale;
  float xyz[3], orientation[3];

  val = GPOINTER_TO_INT(data);
  g_return_if_fail(val >= 0 && val < 3);

  if (disableCallbacks)
    return;

  /* To be removed. */
  if (visu_glExt_scale_getAll())
    scale = VISU_GLEXT_SCALE(visu_glExt_scale_getAll()->data);
  else
    {
      xyz[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[0]));
      xyz[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[1]));
      xyz[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[2]));
      orientation[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[0]));
      orientation[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[1]));
      orientation[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[2]));
      orientation[val] += 999.f;
      scale = visu_glExt_scale_new(xyz, orientation,
		       gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinScaleLength)),
		       (const gchar*)0);
    }

  DBG_fprintf(stderr, "Panel Box&axes: scale orientation %d changed.\n", val);
  xyz[val] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin));
  res = visu_glExt_scale_setOrientation(scale, xyz, xyzMask[val]);

  if (res)
    scaleRedraw();
}
static void onScalesWidthChanged(LineObject *line _U_, gint width, gpointer data _U_)
{
  if (visu_glExt_scale_setDefaultLineWidth(width))
    scaleRedraw();
}
static void onScalesRGBChanged(LineObject *line _U_, float *rgb, gpointer data _U_)
{
  float rgba[4];

  memcpy(rgba, rgb, sizeof(float) * 3);
  rgba[3] = 1.f;
  if (visu_glExt_scale_setDefaultRGBValues(rgba, TOOL_COLOR_MASK_RGBA))
    scaleRedraw();
}
static void onScalesStippleChanged(LineObject *line _U_, gint stipple, gpointer data _U_)
{
  if (visu_glExt_scale_setDefaultStipple(stipple))
    scaleRedraw();
}
static void onLegendChanged(GtkEntry *entry, gpointer data _U_)
{
  gboolean res;
  VisuGlExtScale *scale;
  float xyz[3], orientation[3];

  if (disableCallbacks)
    return;

  /* To be removed. */
  if (visu_glExt_scale_getAll())
    scale = VISU_GLEXT_SCALE(visu_glExt_scale_getAll()->data);
  else
    {
      xyz[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[0]));
      xyz[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[1]));
      xyz[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(originespin[2]));
      orientation[0] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[0]));
      orientation[1] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[1]));
      orientation[2] = gtk_spin_button_get_value(GTK_SPIN_BUTTON(orientationspin[2]));
      scale = visu_glExt_scale_new(xyz, orientation,
		       gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinScaleLength)),
		       (const gchar*)0);
    }

  DBG_fprintf(stderr, "Panel Box&axes: scale legend changed.\n");
  res = visu_glExt_scale_setLegend(scale, gtk_entry_get_text(entry));

  if (res)
    scaleRedraw();
}
static void onUseLegendChanged(GtkToggleButton *toggle, gpointer data _U_)
{
  VisuData *dataObj;

  if (disableCallbacks)
    return;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelAxes));

  if (visu_glExt_legend_setOn(gtk_toggle_button_get_active(toggle)))
    {
      if (!dataObj) 
	return;

      visu_glExt_legend_draw(dataObj);
    }
  VISU_ADD_REDRAW;
}
static void boxRedraw()
{
  VisuData *dataObj;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelAxes));
  if (!dataObj) 
    return;
  visu_glExt_box_draw(dataObj);
  visu_glExt_box_draw_legend(dataObj);
  VISU_ADD_REDRAW;
}
static void onBoxLengthsToggled(GtkToggleButton *toggle, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  visu_glExt_box_setShowLegend(gtk_toggle_button_get_active(toggle));
  boxRedraw();
}
static void onBoxLineWidthChanged(LineObject *line _U_, gint width, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (visu_glExt_box_setLineWidth((float)width))
    boxRedraw();
}
static void onBoxStippleChanged(LineObject *line _U_, gint stipple, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (visu_glExt_box_setLineStipple(stipple))
    boxRedraw();
}
static void onBoxRGBValueChanged(LineObject *line _U_, float *rgb, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (visu_glExt_box_setRGBValues(rgb, TOOL_COLOR_MASK_RGBA))
    boxRedraw();
}
static void onBoxChecked(LineObject *line _U_, gboolean used, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  visu_glExt_box_setOn(used);
  boxRedraw();
}
