/* hashmap.vala
 *
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1997-2000  GLib Team and others
 * Copyright (C) 2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "hashmap.h"

typedef struct _GeeHashMapKeySet GeeHashMapKeySet;
typedef struct _GeeHashMapKeySetClass GeeHashMapKeySetClass;
typedef struct _GeeHashMapKeySetPrivate GeeHashMapKeySetPrivate;
typedef struct _GeeHashMapKeyIterator GeeHashMapKeyIterator;
typedef struct _GeeHashMapKeyIteratorClass GeeHashMapKeyIteratorClass;
typedef struct _GeeHashMapKeyIteratorPrivate GeeHashMapKeyIteratorPrivate;
typedef struct _GeeHashMapValueCollection GeeHashMapValueCollection;
typedef struct _GeeHashMapValueCollectionClass GeeHashMapValueCollectionClass;
typedef struct _GeeHashMapValueCollectionPrivate GeeHashMapValueCollectionPrivate;
typedef struct _GeeHashMapValueIterator GeeHashMapValueIterator;
typedef struct _GeeHashMapValueIteratorClass GeeHashMapValueIteratorClass;
typedef struct _GeeHashMapValueIteratorPrivate GeeHashMapValueIteratorPrivate;
typedef struct _GeeHashMapNode GeeHashMapNode;
struct _GeeHashMapPrivate {
	gint _array_size;
	gint _nnodes;
	GeeHashMapNode** _nodes;
	gint _nodes_length1;
	gint _stamp;
	GHashFunc _key_hash_func;
	GEqualFunc _key_equal_func;
	GEqualFunc _value_equal_func;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};
#define GEE_HASH_MAP_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_HASH_MAP, GeeHashMapPrivate))
enum  {
	GEE_HASH_MAP_DUMMY_PROPERTY,
	GEE_HASH_MAP_SIZE,
	GEE_HASH_MAP_KEY_HASH_FUNC,
	GEE_HASH_MAP_KEY_EQUAL_FUNC,
	GEE_HASH_MAP_VALUE_EQUAL_FUNC,
	GEE_HASH_MAP_K_DUP_FUNC,
	GEE_HASH_MAP_K_DESTROY_FUNC,
	GEE_HASH_MAP_V_DUP_FUNC,
	GEE_HASH_MAP_V_DESTROY_FUNC
};
static const gint GEE_HASH_MAP_MIN_SIZE = 11;
static const gint GEE_HASH_MAP_MAX_SIZE = 13845163;
static GeeSet* gee_hash_map_real_get_keys (GeeMap* base);
static GeeCollection* gee_hash_map_real_get_values (GeeMap* base);
static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gpointer key);
static gboolean gee_hash_map_real_contains (GeeMap* base, gpointer key);
static gpointer gee_hash_map_real_get (GeeMap* base, gpointer key);
static void gee_hash_map_real_set (GeeMap* base, gpointer key, gpointer value);
static gboolean gee_hash_map_real_remove (GeeMap* base, gpointer key);
static void gee_hash_map_real_clear (GeeMap* base);
static void gee_hash_map_resize (GeeHashMap* self);
static GObject * gee_hash_map_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);

#define GEE_HASH_MAP_TYPE_KEY_SET (gee_hash_map_key_set_get_type ())
#define GEE_HASH_MAP_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySet))
#define GEE_HASH_MAP_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))
#define GEE_HASH_MAP_IS_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_IS_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_KEY_SET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))

struct _GeeHashMapKeySet {
	GObject parent;
	GeeHashMapKeySetPrivate * priv;
};
struct _GeeHashMapKeySetClass {
	GObjectClass parent;
};
struct _GeeHashMapKeySetPrivate {
	GeeHashMap* _map;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};
#define GEE_HASH_MAP_KEY_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetPrivate))
enum  {
	GEE_HASH_MAP_KEY_SET_DUMMY_PROPERTY,
	GEE_HASH_MAP_KEY_SET_MAP,
	GEE_HASH_MAP_KEY_SET_SIZE,
	GEE_HASH_MAP_KEY_SET_K_DUP_FUNC,
	GEE_HASH_MAP_KEY_SET_K_DESTROY_FUNC,
	GEE_HASH_MAP_KEY_SET_V_DUP_FUNC,
	GEE_HASH_MAP_KEY_SET_V_DESTROY_FUNC
};
static GeeHashMapKeySet* gee_hash_map_key_set_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeIterator* gee_hash_map_key_set_real_iterator (GeeIterable* base);
static gboolean gee_hash_map_key_set_real_add (GeeCollection* base, gpointer key);
static void gee_hash_map_key_set_real_clear (GeeCollection* base);
static gboolean gee_hash_map_key_set_real_remove (GeeCollection* base, gpointer key);
static gboolean gee_hash_map_key_set_real_contains (GeeCollection* base, gpointer key);
static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value);
static gpointer gee_hash_map_key_set_parent_class = NULL;
static GeeIterableIface* gee_hash_map_key_set_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_map_key_set_gee_collection_parent_iface = NULL;
static GeeSetIface* gee_hash_map_key_set_gee_set_parent_iface = NULL;
static void gee_hash_map_key_set_dispose (GObject * obj);
GType gee_hash_map_key_set_get_type (void);

#define GEE_HASH_MAP_TYPE_KEY_ITERATOR (gee_hash_map_key_iterator_get_type ())
#define GEE_HASH_MAP_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIterator))
#define GEE_HASH_MAP_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))
#define GEE_HASH_MAP_IS_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_IS_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_KEY_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))

struct _GeeHashMapKeyIterator {
	GObject parent;
	GeeHashMapKeyIteratorPrivate * priv;
};
struct _GeeHashMapKeyIteratorClass {
	GObjectClass parent;
};
struct _GeeHashMapKeyIteratorPrivate {
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};
#define GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorPrivate))
enum  {
	GEE_HASH_MAP_KEY_ITERATOR_DUMMY_PROPERTY,
	GEE_HASH_MAP_KEY_ITERATOR_MAP,
	GEE_HASH_MAP_KEY_ITERATOR_K_DUP_FUNC,
	GEE_HASH_MAP_KEY_ITERATOR_K_DESTROY_FUNC,
	GEE_HASH_MAP_KEY_ITERATOR_V_DUP_FUNC,
	GEE_HASH_MAP_KEY_ITERATOR_V_DESTROY_FUNC
};
static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base);
static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value);
static gpointer gee_hash_map_key_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_key_iterator_gee_iterator_parent_iface = NULL;
static void gee_hash_map_key_iterator_dispose (GObject * obj);
GType gee_hash_map_key_iterator_get_type (void);

#define GEE_HASH_MAP_TYPE_VALUE_COLLECTION (gee_hash_map_value_collection_get_type ())
#define GEE_HASH_MAP_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollection))
#define GEE_HASH_MAP_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_VALUE_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))

struct _GeeHashMapValueCollection {
	GObject parent;
	GeeHashMapValueCollectionPrivate * priv;
};
struct _GeeHashMapValueCollectionClass {
	GObjectClass parent;
};
struct _GeeHashMapValueCollectionPrivate {
	GeeHashMap* _map;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};
#define GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionPrivate))
enum  {
	GEE_HASH_MAP_VALUE_COLLECTION_DUMMY_PROPERTY,
	GEE_HASH_MAP_VALUE_COLLECTION_MAP,
	GEE_HASH_MAP_VALUE_COLLECTION_SIZE,
	GEE_HASH_MAP_VALUE_COLLECTION_K_DUP_FUNC,
	GEE_HASH_MAP_VALUE_COLLECTION_K_DESTROY_FUNC,
	GEE_HASH_MAP_VALUE_COLLECTION_V_DUP_FUNC,
	GEE_HASH_MAP_VALUE_COLLECTION_V_DESTROY_FUNC
};
static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeIterable* base);
static gboolean gee_hash_map_value_collection_real_add (GeeCollection* base, gpointer value);
static void gee_hash_map_value_collection_real_clear (GeeCollection* base);
static gboolean gee_hash_map_value_collection_real_remove (GeeCollection* base, gpointer value);
static gboolean gee_hash_map_value_collection_real_contains (GeeCollection* base, gpointer value);
static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value);
static gpointer gee_hash_map_value_collection_parent_class = NULL;
static GeeIterableIface* gee_hash_map_value_collection_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_map_value_collection_gee_collection_parent_iface = NULL;
static void gee_hash_map_value_collection_dispose (GObject * obj);
GType gee_hash_map_value_collection_get_type (void);

#define GEE_HASH_MAP_TYPE_VALUE_ITERATOR (gee_hash_map_value_iterator_get_type ())
#define GEE_HASH_MAP_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIterator))
#define GEE_HASH_MAP_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_VALUE_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))

struct _GeeHashMapValueIterator {
	GObject parent;
	GeeHashMapValueIteratorPrivate * priv;
};
struct _GeeHashMapValueIteratorClass {
	GObjectClass parent;
};
struct _GeeHashMapValueIteratorPrivate {
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};
#define GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorPrivate))
enum  {
	GEE_HASH_MAP_VALUE_ITERATOR_DUMMY_PROPERTY,
	GEE_HASH_MAP_VALUE_ITERATOR_MAP,
	GEE_HASH_MAP_VALUE_ITERATOR_K_DUP_FUNC,
	GEE_HASH_MAP_VALUE_ITERATOR_K_DESTROY_FUNC,
	GEE_HASH_MAP_VALUE_ITERATOR_V_DUP_FUNC,
	GEE_HASH_MAP_VALUE_ITERATOR_V_DESTROY_FUNC
};
static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base);
static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value);
static gpointer gee_hash_map_value_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_value_iterator_gee_iterator_parent_iface = NULL;
static void gee_hash_map_value_iterator_dispose (GObject * obj);
GType gee_hash_map_value_iterator_get_type (void);
struct _GeeHashMapNode {
	gpointer key;
	gpointer value;
	GeeHashMapNode* next;
	guint key_hash;
};
static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash);
static void gee_hash_map_node_free (GeeHashMapNode* self);
static void gee_hash_map_node_free (GeeHashMapNode* self)
{
	NULL;
	NULL;
	(self->next == NULL ? NULL : (self->next = (gee_hash_map_node_free (self->next), NULL)));
	g_slice_free (GeeHashMapNode, self);
}


static gpointer gee_hash_map_parent_class = NULL;
static GeeMapIface* gee_hash_map_gee_map_parent_iface = NULL;
static void gee_hash_map_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);


GeeHashMap* gee_hash_map_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func)
{
	GeeHashMap * self;
	self = g_object_newv (GEE_TYPE_HASH_MAP, 0, NULL);
	gee_hash_map_set_key_hash_func (self, key_hash_func);
	gee_hash_map_set_key_equal_func (self, key_equal_func);
	gee_hash_map_set_value_equal_func (self, value_equal_func);
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	return self;
}


static GeeSet* gee_hash_map_real_get_keys (GeeMap* base)
{
	GeeHashMap * self;
	self = ((GeeHashMap*) base);
	return GEE_SET (gee_hash_map_key_set_new (self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_dup_func, self->priv->v_destroy_func, self));
}


static GeeCollection* gee_hash_map_real_get_values (GeeMap* base)
{
	GeeHashMap * self;
	self = ((GeeHashMap*) base);
	return GEE_COLLECTION (gee_hash_map_value_collection_new (self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_dup_func, self->priv->v_destroy_func, self));
}


static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gpointer key)
{
	guint hash_value;
	GeeHashMapNode** node;
	g_return_val_if_fail (GEE_IS_HASH_MAP (self), NULL);
	hash_value = self->priv->_key_hash_func (key);
	node = &self->priv->_nodes[hash_value % self->priv->_array_size];
	while ((*node) != NULL && (hash_value != (*node)->key_hash || !self->priv->_key_equal_func ((*node)->key, key))) {
		node = &((*node)->next);
	}
	return node;
}


static gboolean gee_hash_map_real_contains (GeeMap* base, gpointer key)
{
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = ((GeeHashMap*) base);
	node = gee_hash_map_lookup_node (self, key);
	return (*node != NULL);
}


static gpointer gee_hash_map_real_get (GeeMap* base, gpointer key)
{
	GeeHashMap * self;
	GeeHashMapNode* node;
	self = ((GeeHashMap*) base);
	node = *gee_hash_map_lookup_node (self, key);
	if (node != NULL) {
		gpointer __temp4;
		return (__temp4 = node->value, (__temp4 == NULL || self->priv->v_dup_func == NULL ? __temp4 : self->priv->v_dup_func (__temp4)));
	} else {
		return NULL;
	}
}


static void gee_hash_map_real_set (GeeMap* base, gpointer key, gpointer value)
{
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = ((GeeHashMap*) base);
	node = gee_hash_map_lookup_node (self, key);
	if (*node != NULL) {
		gpointer __temp8;
		gpointer __temp7;
		(*node)->value = (__temp8 = (__temp7 = value, (__temp7 == NULL || self->priv->v_dup_func == NULL ? __temp7 : self->priv->v_dup_func (__temp7))), ((*node)->value == NULL || self->priv->v_destroy_func == NULL ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL))), __temp8);
	} else {
		guint hash_value;
		gpointer __temp10;
		gpointer __temp9;
		hash_value = self->priv->_key_hash_func (key);
		*node = gee_hash_map_node_new ((__temp9 = key, (__temp9 == NULL || self->priv->k_dup_func == NULL ? __temp9 : self->priv->k_dup_func (__temp9))), (__temp10 = value, (__temp10 == NULL || self->priv->v_dup_func == NULL ? __temp10 : self->priv->v_dup_func (__temp10))), hash_value);
		self->priv->_nnodes++;
		gee_hash_map_resize (self);
	}
	self->priv->_stamp++;
}


static gboolean gee_hash_map_real_remove (GeeMap* base, gpointer key)
{
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = ((GeeHashMap*) base);
	node = gee_hash_map_lookup_node (self, key);
	if (*node != NULL) {
		gpointer __temp11;
		gpointer __temp12;
		(*node)->key = (__temp11 = NULL, ((*node)->key == NULL || self->priv->k_destroy_func == NULL ? NULL : ((*node)->key = (self->priv->k_destroy_func ((*node)->key), NULL))), __temp11);
		(*node)->value = (__temp12 = NULL, ((*node)->value == NULL || self->priv->v_destroy_func == NULL ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL))), __temp12);
		*node = (*node)->next;
		self->priv->_nnodes--;
		gee_hash_map_resize (self);
		self->priv->_stamp++;
		return TRUE;
	}
	return FALSE;
}


static void gee_hash_map_real_clear (GeeMap* base)
{
	GeeHashMap * self;
	self = ((GeeHashMap*) base);
	{
		gint i;
		i = 0;
		for (i = 0; i < self->priv->_array_size; i++) {
			GeeHashMapNode* __temp15;
			GeeHashMapNode* node;
			__temp15 = NULL;
			node = (__temp15 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, __temp15);
			while (node != NULL) {
				GeeHashMapNode* __temp16;
				GeeHashMapNode* next;
				gpointer __temp17;
				gpointer __temp18;
				GeeHashMapNode* __temp20;
				GeeHashMapNode* __temp19;
				__temp16 = NULL;
				next = (__temp16 = node->next, node->next = NULL, __temp16);
				node->key = (__temp17 = NULL, (node->key == NULL || self->priv->k_destroy_func == NULL ? NULL : (node->key = (self->priv->k_destroy_func (node->key), NULL))), __temp17);
				node->value = (__temp18 = NULL, (node->value == NULL || self->priv->v_destroy_func == NULL ? NULL : (node->value = (self->priv->v_destroy_func (node->value), NULL))), __temp18);
				__temp20 = NULL;
				__temp19 = NULL;
				node = (__temp20 = (__temp19 = next, next = NULL, __temp19), (node == NULL ? NULL : (node = (gee_hash_map_node_free (node), NULL))), __temp20);
				(next == NULL ? NULL : (next = (gee_hash_map_node_free (next), NULL)));
			}
			(node == NULL ? NULL : (node = (gee_hash_map_node_free (node), NULL)));
		}
	}
	self->priv->_nnodes = 0;
	gee_hash_map_resize (self);
}


static void gee_hash_map_resize (GeeHashMap* self)
{
	g_return_if_fail (GEE_IS_HASH_MAP (self));
	if ((self->priv->_array_size >= 3 * self->priv->_nnodes && self->priv->_array_size >= GEE_HASH_MAP_MIN_SIZE) || (3 * self->priv->_array_size <= self->priv->_nnodes && self->priv->_array_size < GEE_HASH_MAP_MAX_SIZE)) {
		gint new_array_size;
		GeeHashMapNode** __temp21;
		gint new_nodes_length1;
		GeeHashMapNode** new_nodes;
		GeeHashMapNode** __temp33;
		GeeHashMapNode** __temp32;
		new_array_size = g_spaced_primes_closest (self->priv->_nnodes);
		new_array_size = CLAMP (new_array_size, GEE_HASH_MAP_MIN_SIZE, GEE_HASH_MAP_MAX_SIZE);
		__temp21 = NULL;
		new_nodes = (__temp21 = g_new0 (GeeHashMapNode*, new_array_size), new_nodes_length1 = new_array_size, __temp21);
		{
			gint i;
			i = 0;
			for (i = 0; i < self->priv->_array_size; i++) {
				GeeHashMapNode* node;
				GeeHashMapNode* next;
				GeeHashMapNode* __temp25;
				GeeHashMapNode* __temp24;
				GeeHashMapNode* __temp23;
				GeeHashMapNode* __temp22;
				node = NULL;
				next = NULL;
				__temp25 = NULL;
				__temp24 = NULL;
				__temp23 = NULL;
				__temp22 = NULL;
				for (node = (__temp23 = (__temp22 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, __temp22), (node == NULL ? NULL : (node = (gee_hash_map_node_free (node), NULL))), __temp23); node != NULL; node = (__temp25 = (__temp24 = next, next = NULL, __temp24), (node == NULL ? NULL : (node = (gee_hash_map_node_free (node), NULL))), __temp25)) {
					GeeHashMapNode* __temp27;
					GeeHashMapNode* __temp26;
					guint hash_val;
					GeeHashMapNode* __temp29;
					GeeHashMapNode* __temp28;
					GeeHashMapNode* __temp31;
					GeeHashMapNode* __temp30;
					__temp27 = NULL;
					__temp26 = NULL;
					next = (__temp27 = (__temp26 = node->next, node->next = NULL, __temp26), (next == NULL ? NULL : (next = (gee_hash_map_node_free (next), NULL))), __temp27);
					hash_val = node->key_hash % new_array_size;
					__temp29 = NULL;
					__temp28 = NULL;
					node->next = (__temp29 = (__temp28 = new_nodes[hash_val], new_nodes[hash_val] = NULL, __temp28), (node->next == NULL ? NULL : (node->next = (gee_hash_map_node_free (node->next), NULL))), __temp29);
					__temp31 = NULL;
					__temp30 = NULL;
					new_nodes[hash_val] = (__temp31 = (__temp30 = node, node = NULL, __temp30), (new_nodes[hash_val] == NULL ? NULL : (new_nodes[hash_val] = (gee_hash_map_node_free (new_nodes[hash_val]), NULL))), __temp31);
				}
				(node == NULL ? NULL : (node = (gee_hash_map_node_free (node), NULL)));
				(next == NULL ? NULL : (next = (gee_hash_map_node_free (next), NULL)));
			}
		}
		__temp33 = NULL;
		__temp32 = NULL;
		self->priv->_nodes = (__temp33 = (__temp32 = new_nodes, new_nodes = NULL, __temp32), (self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, ((GDestroyNotify) gee_hash_map_node_free)), NULL)), self->priv->_nodes_length1 = new_nodes_length1, __temp33);
		self->priv->_array_size = new_array_size;
		(new_nodes = (_vala_array_free (new_nodes, new_nodes_length1, ((GDestroyNotify) gee_hash_map_node_free)), NULL));
	}
}


static gint gee_hash_map_real_get_size (GeeHashMap* self)
{
	g_return_val_if_fail (GEE_IS_HASH_MAP (self), 0);
	return self->priv->_nnodes;
}


void gee_hash_map_set_key_hash_func (GeeHashMap* self, GHashFunc value)
{
	g_return_if_fail (GEE_IS_HASH_MAP (self));
	self->priv->_key_hash_func = value;
}


void gee_hash_map_set_key_equal_func (GeeHashMap* self, GEqualFunc value)
{
	g_return_if_fail (GEE_IS_HASH_MAP (self));
	self->priv->_key_equal_func = value;
}


void gee_hash_map_set_value_equal_func (GeeHashMap* self, GEqualFunc value)
{
	g_return_if_fail (GEE_IS_HASH_MAP (self));
	self->priv->_value_equal_func = value;
}


static GObject * gee_hash_map_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties)
{
	GObject * obj;
	GeeHashMapClass * klass;
	GObjectClass * parent_class;
	GeeHashMap * self;
	klass = GEE_HASH_MAP_CLASS (g_type_class_peek (GEE_TYPE_HASH_MAP));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GEE_HASH_MAP (obj);
	{
		GeeHashMapNode** __temp35;
		self->priv->_array_size = GEE_HASH_MAP_MIN_SIZE;
		__temp35 = NULL;
		self->priv->_nodes = (__temp35 = g_new0 (GeeHashMapNode*, self->priv->_array_size), (self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, ((GDestroyNotify) gee_hash_map_node_free)), NULL)), self->priv->_nodes_length1 = self->priv->_array_size, __temp35);
	}
	return obj;
}


static GeeHashMapKeySet* gee_hash_map_key_set_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map)
{
	GeeHashMapKeySet * self;
	g_return_val_if_fail (GEE_IS_HASH_MAP (map), NULL);
	self = g_object_newv (GEE_HASH_MAP_TYPE_KEY_SET, 0, NULL);
	gee_hash_map_key_set_set_map (self, map);
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	return self;
}


static GeeIterator* gee_hash_map_key_set_real_iterator (GeeIterable* base)
{
	GeeHashMapKeySet * self;
	self = ((GeeHashMapKeySet*) base);
	return GEE_ITERATOR (gee_hash_map_key_iterator_new (self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map));
}


static gboolean gee_hash_map_key_set_real_add (GeeCollection* base, gpointer key)
{
	GeeHashMapKeySet * self;
	self = ((GeeHashMapKeySet*) base);
	g_assert_not_reached ();
	return FALSE;
}


static void gee_hash_map_key_set_real_clear (GeeCollection* base)
{
	GeeHashMapKeySet * self;
	self = ((GeeHashMapKeySet*) base);
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_remove (GeeCollection* base, gpointer key)
{
	GeeHashMapKeySet * self;
	self = ((GeeHashMapKeySet*) base);
	g_assert_not_reached ();
	return FALSE;
}


static gboolean gee_hash_map_key_set_real_contains (GeeCollection* base, gpointer key)
{
	GeeHashMapKeySet * self;
	self = ((GeeHashMapKeySet*) base);
	return gee_map_contains (GEE_MAP (self->priv->_map), key);
}


static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value)
{
	GeeHashMap* __temp41;
	GeeHashMap* __temp40;
	g_return_if_fail (GEE_HASH_MAP_IS_KEY_SET (self));
	__temp41 = NULL;
	__temp40 = NULL;
	self->priv->_map = (__temp41 = (__temp40 = value, (__temp40 == NULL ? NULL : g_object_ref (__temp40))), (self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL))), __temp41);
}


static gint gee_hash_map_key_set_real_get_size (GeeHashMapKeySet* self)
{
	g_return_val_if_fail (GEE_HASH_MAP_IS_KEY_SET (self), 0);
	return gee_map_get_size (((GeeMap*) self->priv->_map));
}


static void gee_hash_map_key_set_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	GeeHashMapKeySet * self;
	self = GEE_HASH_MAP_KEY_SET (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_SET_SIZE:
		g_value_set_int (value, gee_hash_map_key_set_real_get_size (self));
		break;
	}
}


static void gee_hash_map_key_set_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	GeeHashMapKeySet * self;
	self = GEE_HASH_MAP_KEY_SET (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_SET_MAP:
		gee_hash_map_key_set_set_map (self, g_value_get_object (value));
		break;
		case GEE_HASH_MAP_KEY_SET_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_SET_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_hash_map_key_set_class_init (GeeHashMapKeySetClass * klass)
{
	gee_hash_map_key_set_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapKeySetPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_key_set_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_key_set_set_property;
	G_OBJECT_CLASS (klass)->dispose = gee_hash_map_key_set_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_MAP, g_param_spec_object ("map", "foo", "bar", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_SET_SIZE, "size");
}


static void gee_hash_map_key_set_gee_iterable_interface_init (GeeIterableIface * iface)
{
	gee_hash_map_key_set_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->iterator = gee_hash_map_key_set_real_iterator;
}


static void gee_hash_map_key_set_gee_collection_interface_init (GeeCollectionIface * iface)
{
	gee_hash_map_key_set_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->add = gee_hash_map_key_set_real_add;
	iface->clear = gee_hash_map_key_set_real_clear;
	iface->remove = gee_hash_map_key_set_real_remove;
	iface->contains = gee_hash_map_key_set_real_contains;
}


static void gee_hash_map_key_set_gee_set_interface_init (GeeSetIface * iface)
{
	gee_hash_map_key_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_hash_map_key_set_init (GeeHashMapKeySet * self)
{
	self->priv = GEE_HASH_MAP_KEY_SET_GET_PRIVATE (self);
}


static void gee_hash_map_key_set_dispose (GObject * obj)
{
	GeeHashMapKeySet * self;
	GeeHashMapKeySetClass * klass;
	GObjectClass * parent_class;
	self = GEE_HASH_MAP_KEY_SET (obj);
	(self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)));
	klass = GEE_HASH_MAP_KEY_SET_CLASS (g_type_class_peek (GEE_HASH_MAP_TYPE_KEY_SET));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType gee_hash_map_key_set_get_type (void)
{
	static GType gee_hash_map_key_set_type_id = 0;
	if (G_UNLIKELY (gee_hash_map_key_set_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeySetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeySet), 0, (GInstanceInitFunc) gee_hash_map_key_set_init };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_set_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashMapKeySet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_hash_map_key_set_type_id;
}


static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map)
{
	GeeHashMapKeyIterator * self;
	g_return_val_if_fail (GEE_IS_HASH_MAP (map), NULL);
	self = g_object_newv (GEE_HASH_MAP_TYPE_KEY_ITERATOR, 0, NULL);
	gee_hash_map_key_iterator_set_map (self, map);
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	return self;
}


static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base)
{
	GeeHashMapKeyIterator * self;
	self = ((GeeHashMapKeyIterator*) base);
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (self->priv->_node == NULL && self->priv->_index + 1 < self->priv->_map->priv->_array_size) {
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	return (self->priv->_node != NULL);
}


static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base)
{
	GeeHashMapKeyIterator * self;
	gpointer __temp44;
	self = ((GeeHashMapKeyIterator*) base);
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	return (__temp44 = self->priv->_node->key, (__temp44 == NULL || self->priv->k_dup_func == NULL ? __temp44 : self->priv->k_dup_func (__temp44)));
}


static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value)
{
	GeeHashMap* __temp47;
	GeeHashMap* __temp46;
	g_return_if_fail (GEE_HASH_MAP_IS_KEY_ITERATOR (self));
	__temp47 = NULL;
	__temp46 = NULL;
	self->priv->_map = (__temp47 = (__temp46 = value, (__temp46 == NULL ? NULL : g_object_ref (__temp46))), (self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL))), __temp47);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
}


static void gee_hash_map_key_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	GeeHashMapKeyIterator * self;
	self = GEE_HASH_MAP_KEY_ITERATOR (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_ITERATOR_MAP:
		gee_hash_map_key_iterator_set_map (self, g_value_get_object (value));
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_KEY_ITERATOR_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_hash_map_key_iterator_class_init (GeeHashMapKeyIteratorClass * klass)
{
	gee_hash_map_key_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapKeyIteratorPrivate));
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_key_iterator_set_property;
	G_OBJECT_CLASS (klass)->dispose = gee_hash_map_key_iterator_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_ITERATOR_MAP, g_param_spec_object ("map", "foo", "bar", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
}


static void gee_hash_map_key_iterator_gee_iterator_interface_init (GeeIteratorIface * iface)
{
	gee_hash_map_key_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_key_iterator_real_next;
	iface->get = gee_hash_map_key_iterator_real_get;
}


static void gee_hash_map_key_iterator_init (GeeHashMapKeyIterator * self)
{
	self->priv = GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_key_iterator_dispose (GObject * obj)
{
	GeeHashMapKeyIterator * self;
	GeeHashMapKeyIteratorClass * klass;
	GObjectClass * parent_class;
	self = GEE_HASH_MAP_KEY_ITERATOR (obj);
	(self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)));
	klass = GEE_HASH_MAP_KEY_ITERATOR_CLASS (g_type_class_peek (GEE_HASH_MAP_TYPE_KEY_ITERATOR));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType gee_hash_map_key_iterator_get_type (void)
{
	static GType gee_hash_map_key_iterator_type_id = 0;
	if (G_UNLIKELY (gee_hash_map_key_iterator_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeyIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeyIterator), 0, (GInstanceInitFunc) gee_hash_map_key_iterator_init };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_key_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashMapKeyIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_key_iterator_type_id;
}


static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map)
{
	GeeHashMapValueCollection * self;
	g_return_val_if_fail (GEE_IS_HASH_MAP (map), NULL);
	self = g_object_newv (GEE_HASH_MAP_TYPE_VALUE_COLLECTION, 0, NULL);
	gee_hash_map_value_collection_set_map (self, map);
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	return self;
}


static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeIterable* base)
{
	GeeHashMapValueCollection * self;
	self = ((GeeHashMapValueCollection*) base);
	return GEE_ITERATOR (gee_hash_map_value_iterator_new (self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map));
}


static gboolean gee_hash_map_value_collection_real_add (GeeCollection* base, gpointer value)
{
	GeeHashMapValueCollection * self;
	self = ((GeeHashMapValueCollection*) base);
	g_assert_not_reached ();
}


static void gee_hash_map_value_collection_real_clear (GeeCollection* base)
{
	GeeHashMapValueCollection * self;
	self = ((GeeHashMapValueCollection*) base);
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_remove (GeeCollection* base, gpointer value)
{
	GeeHashMapValueCollection * self;
	self = ((GeeHashMapValueCollection*) base);
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_contains (GeeCollection* base, gpointer value)
{
	GeeHashMapValueCollection * self;
	GeeIterator* it;
	gboolean __temp52;
	self = ((GeeHashMapValueCollection*) base);
	it = gee_iterable_iterator (GEE_ITERABLE (self));
	while (gee_iterator_next (it)) {
		gpointer __temp49;
		gboolean __temp50;
		if ((__temp50 = self->priv->_map->priv->_value_equal_func ((__temp49 = gee_iterator_get (it)), value), (__temp49 == NULL || self->priv->v_destroy_func == NULL ? NULL : (__temp49 = (self->priv->v_destroy_func (__temp49), NULL))), __temp50)) {
			gboolean __temp51;
			return (__temp51 = TRUE, (it == NULL ? NULL : (it = (g_object_unref (it), NULL))), __temp51);
		}
	}
	return (__temp52 = FALSE, (it == NULL ? NULL : (it = (g_object_unref (it), NULL))), __temp52);
	(it == NULL ? NULL : (it = (g_object_unref (it), NULL)));
}


static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value)
{
	GeeHashMap* __temp54;
	GeeHashMap* __temp53;
	g_return_if_fail (GEE_HASH_MAP_IS_VALUE_COLLECTION (self));
	__temp54 = NULL;
	__temp53 = NULL;
	self->priv->_map = (__temp54 = (__temp53 = value, (__temp53 == NULL ? NULL : g_object_ref (__temp53))), (self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL))), __temp54);
}


static gint gee_hash_map_value_collection_real_get_size (GeeHashMapValueCollection* self)
{
	g_return_val_if_fail (GEE_HASH_MAP_IS_VALUE_COLLECTION (self), 0);
	return gee_map_get_size (((GeeMap*) self->priv->_map));
}


static void gee_hash_map_value_collection_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	GeeHashMapValueCollection * self;
	self = GEE_HASH_MAP_VALUE_COLLECTION (object);
	switch (property_id) {
		case GEE_HASH_MAP_VALUE_COLLECTION_SIZE:
		g_value_set_int (value, gee_hash_map_value_collection_real_get_size (self));
		break;
	}
}


static void gee_hash_map_value_collection_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	GeeHashMapValueCollection * self;
	self = GEE_HASH_MAP_VALUE_COLLECTION (object);
	switch (property_id) {
		case GEE_HASH_MAP_VALUE_COLLECTION_MAP:
		gee_hash_map_value_collection_set_map (self, g_value_get_object (value));
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_COLLECTION_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_hash_map_value_collection_class_init (GeeHashMapValueCollectionClass * klass)
{
	gee_hash_map_value_collection_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapValueCollectionPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_value_collection_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_value_collection_set_property;
	G_OBJECT_CLASS (klass)->dispose = gee_hash_map_value_collection_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_MAP, g_param_spec_object ("map", "foo", "bar", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_COLLECTION_SIZE, "size");
}


static void gee_hash_map_value_collection_gee_iterable_interface_init (GeeIterableIface * iface)
{
	gee_hash_map_value_collection_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->iterator = gee_hash_map_value_collection_real_iterator;
}


static void gee_hash_map_value_collection_gee_collection_interface_init (GeeCollectionIface * iface)
{
	gee_hash_map_value_collection_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->add = gee_hash_map_value_collection_real_add;
	iface->clear = gee_hash_map_value_collection_real_clear;
	iface->remove = gee_hash_map_value_collection_real_remove;
	iface->contains = gee_hash_map_value_collection_real_contains;
}


static void gee_hash_map_value_collection_init (GeeHashMapValueCollection * self)
{
	self->priv = GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE (self);
}


static void gee_hash_map_value_collection_dispose (GObject * obj)
{
	GeeHashMapValueCollection * self;
	GeeHashMapValueCollectionClass * klass;
	GObjectClass * parent_class;
	self = GEE_HASH_MAP_VALUE_COLLECTION (obj);
	(self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)));
	klass = GEE_HASH_MAP_VALUE_COLLECTION_CLASS (g_type_class_peek (GEE_HASH_MAP_TYPE_VALUE_COLLECTION));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType gee_hash_map_value_collection_get_type (void)
{
	static GType gee_hash_map_value_collection_type_id = 0;
	if (G_UNLIKELY (gee_hash_map_value_collection_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueCollectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_collection_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueCollection), 0, (GInstanceInitFunc) gee_hash_map_value_collection_init };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_map_value_collection_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_map_value_collection_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_value_collection_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashMapValueCollection", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_value_collection_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_map_value_collection_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
	}
	return gee_hash_map_value_collection_type_id;
}


static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map)
{
	GeeHashMapValueIterator * self;
	g_return_val_if_fail (GEE_IS_HASH_MAP (map), NULL);
	self = g_object_newv (GEE_HASH_MAP_TYPE_VALUE_ITERATOR, 0, NULL);
	gee_hash_map_value_iterator_set_map (self, map);
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	return self;
}


static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base)
{
	GeeHashMapValueIterator * self;
	self = ((GeeHashMapValueIterator*) base);
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (self->priv->_node == NULL && self->priv->_index + 1 < self->priv->_map->priv->_array_size) {
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	return (self->priv->_node != NULL);
}


static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base)
{
	GeeHashMapValueIterator * self;
	gpointer __temp57;
	self = ((GeeHashMapValueIterator*) base);
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	return (__temp57 = self->priv->_node->value, (__temp57 == NULL || self->priv->v_dup_func == NULL ? __temp57 : self->priv->v_dup_func (__temp57)));
}


static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value)
{
	GeeHashMap* __temp60;
	GeeHashMap* __temp59;
	g_return_if_fail (GEE_HASH_MAP_IS_VALUE_ITERATOR (self));
	__temp60 = NULL;
	__temp59 = NULL;
	self->priv->_map = (__temp60 = (__temp59 = value, (__temp59 == NULL ? NULL : g_object_ref (__temp59))), (self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL))), __temp60);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
}


static void gee_hash_map_value_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	GeeHashMapValueIterator * self;
	self = GEE_HASH_MAP_VALUE_ITERATOR (object);
	switch (property_id) {
		case GEE_HASH_MAP_VALUE_ITERATOR_MAP:
		gee_hash_map_value_iterator_set_map (self, g_value_get_object (value));
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_VALUE_ITERATOR_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_hash_map_value_iterator_class_init (GeeHashMapValueIteratorClass * klass)
{
	gee_hash_map_value_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapValueIteratorPrivate));
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_value_iterator_set_property;
	G_OBJECT_CLASS (klass)->dispose = gee_hash_map_value_iterator_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_ITERATOR_MAP, g_param_spec_object ("map", "foo", "bar", GEE_TYPE_HASH_MAP, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
}


static void gee_hash_map_value_iterator_gee_iterator_interface_init (GeeIteratorIface * iface)
{
	gee_hash_map_value_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_value_iterator_real_next;
	iface->get = gee_hash_map_value_iterator_real_get;
}


static void gee_hash_map_value_iterator_init (GeeHashMapValueIterator * self)
{
	self->priv = GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_value_iterator_dispose (GObject * obj)
{
	GeeHashMapValueIterator * self;
	GeeHashMapValueIteratorClass * klass;
	GObjectClass * parent_class;
	self = GEE_HASH_MAP_VALUE_ITERATOR (obj);
	(self->priv->_map == NULL ? NULL : (self->priv->_map = (g_object_unref (self->priv->_map), NULL)));
	klass = GEE_HASH_MAP_VALUE_ITERATOR_CLASS (g_type_class_peek (GEE_HASH_MAP_TYPE_VALUE_ITERATOR));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType gee_hash_map_value_iterator_get_type (void)
{
	static GType gee_hash_map_value_iterator_type_id = 0;
	if (G_UNLIKELY (gee_hash_map_value_iterator_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueIterator), 0, (GInstanceInitFunc) gee_hash_map_value_iterator_init };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_value_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_value_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashMapValueIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_value_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_value_iterator_type_id;
}


static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash)
{
	GeeHashMapNode* self;
	gpointer __temp62;
	gpointer __temp61;
	gpointer __temp64;
	gpointer __temp63;
	self = g_slice_new0 (GeeHashMapNode);
	self->key = (__temp62 = (__temp61 = k, k = NULL, __temp61), NULL, __temp62);
	self->value = (__temp64 = (__temp63 = v, v = NULL, __temp63), NULL, __temp64);
	self->key_hash = hash;
	return self;
}


static void gee_hash_map_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	GeeHashMap * self;
	self = GEE_HASH_MAP (object);
	switch (property_id) {
		case GEE_HASH_MAP_SIZE:
		g_value_set_int (value, gee_hash_map_real_get_size (self));
		break;
	}
}


static void gee_hash_map_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	GeeHashMap * self;
	self = GEE_HASH_MAP (object);
	switch (property_id) {
		case GEE_HASH_MAP_KEY_HASH_FUNC:
		gee_hash_map_set_key_hash_func (self, g_value_get_pointer (value));
		break;
		case GEE_HASH_MAP_KEY_EQUAL_FUNC:
		gee_hash_map_set_key_equal_func (self, g_value_get_pointer (value));
		break;
		case GEE_HASH_MAP_VALUE_EQUAL_FUNC:
		gee_hash_map_set_value_equal_func (self, g_value_get_pointer (value));
		break;
		case GEE_HASH_MAP_K_DESTROY_FUNC:
		self->priv->k_destroy_func = g_value_get_pointer (value);
		break;
		case GEE_HASH_MAP_V_DESTROY_FUNC:
		self->priv->v_destroy_func = g_value_get_pointer (value);
		break;
	}
}


static void gee_hash_map_class_init (GeeHashMapClass * klass)
{
	gee_hash_map_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeHashMapPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_hash_map_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_hash_map_set_property;
	G_OBJECT_CLASS (klass)->constructor = gee_hash_map_constructor;
	G_OBJECT_CLASS (klass)->dispose = gee_hash_map_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_K_DUP_FUNC, g_param_spec_pointer ("k-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_K_DESTROY_FUNC, g_param_spec_pointer ("k-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_V_DUP_FUNC, g_param_spec_pointer ("v-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_V_DESTROY_FUNC, g_param_spec_pointer ("v-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_SIZE, "size");
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_HASH_FUNC, g_param_spec_pointer ("key-hash-func", "foo", "bar", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_KEY_EQUAL_FUNC, g_param_spec_pointer ("key-equal-func", "foo", "bar", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_HASH_MAP_VALUE_EQUAL_FUNC, g_param_spec_pointer ("value-equal-func", "foo", "bar", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
}


static void gee_hash_map_gee_map_interface_init (GeeMapIface * iface)
{
	gee_hash_map_gee_map_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_keys = gee_hash_map_real_get_keys;
	iface->get_values = gee_hash_map_real_get_values;
	iface->contains = gee_hash_map_real_contains;
	iface->get = gee_hash_map_real_get;
	iface->set = gee_hash_map_real_set;
	iface->remove = gee_hash_map_real_remove;
	iface->clear = gee_hash_map_real_clear;
}


static void gee_hash_map_init (GeeHashMap * self)
{
	self->priv = GEE_HASH_MAP_GET_PRIVATE (self);
	self->priv->_stamp = 0;
}


static void gee_hash_map_dispose (GObject * obj)
{
	GeeHashMap * self;
	GeeHashMapClass * klass;
	GObjectClass * parent_class;
	self = GEE_HASH_MAP (obj);
	{
		gee_map_clear (GEE_MAP (self));
	}
	(self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, ((GDestroyNotify) gee_hash_map_node_free)), NULL));
	klass = GEE_HASH_MAP_CLASS (g_type_class_peek (GEE_TYPE_HASH_MAP));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType gee_hash_map_get_type (void)
{
	static GType gee_hash_map_type_id = 0;
	if (G_UNLIKELY (gee_hash_map_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMap), 0, (GInstanceInitFunc) gee_hash_map_init };
		static const GInterfaceInfo gee_map_info = { (GInterfaceInitFunc) gee_hash_map_gee_map_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeHashMap", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_type_id, GEE_TYPE_MAP, &gee_map_info);
	}
	return gee_hash_map_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func)
{
	if (array != NULL && destroy_func != NULL) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}




