/* valagidlparser.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include "valagidlparser.h"
#include <vala/valasourcereference.h>
#include <vala/valadatatype.h>
#include <gee/map.h>
#include <gee/set.h>
#include <gee/hashmap.h>
#include <glib/gstdio.h>
#include <vala/valareport.h>
#include <gidlparser.h>
#include <stdio.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valacodenode.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valaformalparameter.h>
#include <vala/valaenumvalue.h>
#include <gee/hashset.h>
#include <gee/collection.h>
#include <gmetadata.h>
#include <vala/valacreationmethod.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valaexpression.h>

struct _ValaGIdlParserPrivate {
	ValaCodeContext* context;
	ValaSourceFile* current_source_file;
	ValaSourceReference* current_source_reference;
	ValaDataType* current_data_type;
	GeeMap* codenode_attributes_map;
	GeeSet* current_type_symbol_set;
};
#define VALA_GIDL_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_PARSER, ValaGIdlParserPrivate))
enum  {
	VALA_GIDL_PARSER_DUMMY_PROPERTY
};
static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file);
static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, GIdlModule* module);
static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, GIdlModule* module);
static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module);
static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node);
static ValaStruct* vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node);
static ValaStruct* vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node);
static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node);
static ValaClass* vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node);
static ValaInterface* vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node);
static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node);
static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name);
static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n);
static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual, gboolean is_interface);
static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name);
static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node);
static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node);
static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node);
static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode);
static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s);
static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node);
static gpointer vala_gidl_parser_parent_class = NULL;
static void vala_gidl_parser_dispose (GObject * obj);


/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context)
{
	ValaCodeContext* __temp0;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	__temp0 = NULL;
	self->priv->context = (__temp0 = g_object_ref (context), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), __temp0);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file)
{
	ValaGIdlParser * self;
	self = ((ValaGIdlParser*) base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gidl")) {
		vala_gidl_parser_parse_file (self, source_file);
	}
}


static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file)
{
	GError * inner_error;
	char* __temp1;
	char* __temp2;
	char* metadata_filename;
	ValaSourceFile* __temp3;
	GeeMap* __temp4;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	inner_error = NULL;
	__temp1 = NULL;
	__temp2 = NULL;
	metadata_filename = (__temp2 = g_strdup_printf ("%s.metadata", (__temp1 = g_strndup (vala_source_file_get_filename (source_file), strlen (vala_source_file_get_filename (source_file)) - strlen (".gidl")))), (__temp1 = (g_free (__temp1), NULL)), __temp2);
	__temp3 = NULL;
	self->priv->current_source_file = (__temp3 = g_object_ref (source_file), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), __temp3);
	__temp4 = NULL;
	self->priv->codenode_attributes_map = (__temp4 = GEE_MAP (gee_hash_map_new (g_strdup, g_free, g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal)), (self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_object_unref (self->priv->codenode_attributes_map), NULL))), __temp4);
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			gulong metadata_len;
			metadata = NULL;
			metadata_len = 0UL;
			g_file_get_contents (metadata_filename, &metadata, &metadata_len, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** line_collection;
				char** line_it;
				line_collection = g_strsplit (metadata, "\n", 0);
				for (line_it = line_collection; *line_it != NULL; line_it = line_it + 1) {
					const char* __temp6;
					char* line;
					__temp6 = NULL;
					line = (__temp6 = *line_it, (__temp6 == NULL ? NULL : g_strdup (__temp6)));
					{
						char** __temp5;
						gint line_parts_length1;
						char** line_parts;
						__temp5 = NULL;
						line_parts = (__temp5 = g_strsplit (line, " ", 2), line_parts_length1 = -1, __temp5);
						if (line_parts[0] == NULL) {
							(line = (g_free (line), NULL));
							(line_parts = (g_strfreev (line_parts), NULL));
							continue;
						}
						gee_map_set (self->priv->codenode_attributes_map, line_parts[0], line_parts[1]);
						(line = (g_free (line), NULL));
						(line_parts = (g_strfreev (line_parts), NULL));
					}
				}
				(line_collection = (g_strfreev (line_collection), NULL));
			}
			(metadata = (g_free (metadata), NULL));
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* __temp7;
				__temp7 = NULL;
				vala_report_error (NULL, (__temp7 = g_strdup_printf ("Unable to read metadata file: %s", e->message)));
				(__temp7 = (g_free (__temp7), NULL));
			}
		}
		__finally0:
		;
	}
	{
		GList* modules;
		ValaSourceReference* __temp8;
		modules = g_idl_parse_file (vala_source_file_get_filename (source_file), &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_MARKUP_ERROR) {
				goto __catch1_g_markup_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		__temp8 = NULL;
		self->priv->current_source_reference = (__temp8 = vala_source_reference_new (source_file, 0, 0, 0, 0), (self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL))), __temp8);
		{
			GList* module_collection;
			GList* module_it;
			module_collection = modules;
			for (module_it = module_collection; module_it != NULL; module_it = module_it->next) {
				GIdlModule* module;
				module = module_it->data;
				{
					ValaNamespace* ns;
					ns = vala_gidl_parser_parse_module (self, module);
					if (ns != NULL) {
						vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
					}
					(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
				}
			}
		}
		(modules == NULL ? NULL : (modules = (g_list_foreach (modules, (GFunc) g_free, NULL), g_list_free (modules), NULL)));
	}
	goto __finally1;
	__catch1_g_markup_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "error parsing GIDL file: %s\n", e->message);
		}
	}
	__finally1:
	;
	(metadata_filename = (g_free (metadata_filename), NULL));
}


static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, GIdlModule* module)
{
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_name != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	if (g_str_has_prefix (type_name, module->name)) {
		const char* __temp9;
		__temp9 = NULL;
		return (__temp9 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (module->name, -1)), (__temp9 == NULL ? NULL : g_strdup (__temp9)));
	} else {
		if (g_utf8_collate (module->name, "GLib") == 0 && g_str_has_prefix (type_name, "G")) {
			const char* __temp11;
			__temp11 = NULL;
			return (__temp11 = g_utf8_offset_to_pointer (type_name, 1), (__temp11 == NULL ? NULL : g_strdup (__temp11)));
		}
	}
	return g_strdup (type_name);
}


static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, GIdlModule* module)
{
	char* __temp15;
	char* __temp14;
	gboolean __temp16;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_name != NULL, NULL);
	g_return_val_if_fail (module != NULL, NULL);
	__temp15 = NULL;
	__temp14 = NULL;
	if ((__temp16 = g_str_has_prefix (const_name, (__temp15 = g_strconcat ((__temp14 = g_utf8_strup (module->name, -1)), "_", NULL))), (__temp15 = (g_free (__temp15), NULL)), (__temp14 = (g_free (__temp14), NULL)), __temp16)) {
		const char* __temp17;
		__temp17 = NULL;
		return (__temp17 = g_utf8_offset_to_pointer (const_name, g_utf8_strlen (module->name, -1) + 1), (__temp17 == NULL ? NULL : g_strdup (__temp17)));
	} else {
		if (g_utf8_collate (module->name, "GLib") == 0 && g_str_has_prefix (const_name, "G_")) {
			const char* __temp19;
			__temp19 = NULL;
			return (__temp19 = g_utf8_offset_to_pointer (const_name, 2), (__temp19 == NULL ? NULL : g_strdup (__temp19)));
		}
	}
	return g_strdup (const_name);
}


static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module)
{
	ValaSymbol* sym;
	ValaNamespace* ns;
	char** __temp25;
	gint attributes_length1;
	char** attributes;
	ValaNamespace* __temp40;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (module != NULL, NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope (((ValaSymbol*) vala_code_context_get_root (self->priv->context))), module->name);
	ns = NULL;
	if (VALA_IS_NAMESPACE (sym)) {
		ValaNamespace* __temp23;
		ValaNamespace* __temp22;
		__temp23 = NULL;
		__temp22 = NULL;
		ns = (__temp23 = (__temp22 = VALA_NAMESPACE (sym), (__temp22 == NULL ? NULL : g_object_ref (__temp22))), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), __temp23);
		vala_namespace_set_pkg (ns, FALSE);
	} else {
		ValaNamespace* __temp24;
		__temp24 = NULL;
		ns = (__temp24 = vala_namespace_new (module->name, self->priv->current_source_reference), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), __temp24);
	}
	__temp25 = NULL;
	attributes = (__temp25 = vala_gidl_parser_get_attributes (self, vala_symbol_get_name (((ValaSymbol*) ns))), attributes_length1 = -1, __temp25);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp30;
				char* attr;
				__temp30 = NULL;
				attr = (__temp30 = attr_collection[attr_it], (__temp30 == NULL ? NULL : g_strdup (__temp30)));
				{
					char** __temp26;
					gint nv_length1;
					char** nv;
					__temp26 = NULL;
					nv = (__temp26 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp26);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp27;
						__temp27 = NULL;
						vala_namespace_set_cheader_filename (ns, (__temp27 = vala_gidl_parser_eval (self, nv[1])));
						(__temp27 = (g_free (__temp27), NULL));
					} else {
						if (g_utf8_collate (nv[0], "cprefix") == 0) {
							char* __temp28;
							__temp28 = NULL;
							vala_namespace_set_cprefix (ns, (__temp28 = vala_gidl_parser_eval (self, nv[1])));
							(__temp28 = (g_free (__temp28), NULL));
						} else {
							if (g_utf8_collate (nv[0], "lower_case_cprefix") == 0) {
								char* __temp29;
								__temp29 = NULL;
								vala_namespace_set_lower_case_cprefix (ns, (__temp29 = vala_gidl_parser_eval (self, nv[1])));
								(__temp29 = (g_free (__temp29), NULL));
							}
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	{
		GList* node_collection;
		GList* node_it;
		node_collection = module->entries;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = node_it->data;
			{
				if (node->type == G_IDL_NODE_CALLBACK) {
					ValaCallback* cb;
					char* __temp31;
					cb = vala_gidl_parser_parse_callback (self, ((GIdlNodeFunction*) node));
					if (cb == NULL) {
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						continue;
					}
					__temp31 = NULL;
					vala_symbol_set_name (VALA_SYMBOL (cb), (__temp31 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (((ValaSymbol*) cb)), module)));
					(__temp31 = (g_free (__temp31), NULL));
					vala_namespace_add_callback (ns, cb);
					vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cb));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
				} else {
					if (node->type == G_IDL_NODE_STRUCT) {
						ValaStruct* st;
						char* __temp32;
						st = vala_gidl_parser_parse_struct (self, ((GIdlNodeStruct*) node));
						if (st == NULL) {
							(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
							continue;
						}
						__temp32 = NULL;
						vala_symbol_set_name (VALA_SYMBOL (st), (__temp32 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (((ValaSymbol*) st)), module)));
						(__temp32 = (g_free (__temp32), NULL));
						vala_namespace_add_struct (ns, st);
						vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (st));
						(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
					} else {
						if (node->type == G_IDL_NODE_BOXED) {
							ValaStruct* st;
							char* __temp33;
							char* __temp34;
							st = vala_gidl_parser_parse_boxed (self, ((GIdlNodeBoxed*) node));
							__temp33 = NULL;
							vala_symbol_set_name (VALA_SYMBOL (st), (__temp33 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (((ValaSymbol*) st)), module)));
							(__temp33 = (g_free (__temp33), NULL));
							vala_namespace_add_struct (ns, st);
							__temp34 = NULL;
							vala_struct_set_type_id (st, (__temp34 = vala_data_type_get_upper_case_cname (VALA_DATA_TYPE (st), "TYPE_")));
							(__temp34 = (g_free (__temp34), NULL));
							vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (st));
							(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
						} else {
							if (node->type == G_IDL_NODE_ENUM) {
								ValaEnum* en;
								char* __temp35;
								en = vala_gidl_parser_parse_enum (self, ((GIdlNodeEnum*) node));
								__temp35 = NULL;
								vala_symbol_set_name (VALA_SYMBOL (en), (__temp35 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (((ValaSymbol*) en)), module)));
								(__temp35 = (g_free (__temp35), NULL));
								vala_namespace_add_enum (ns, en);
								vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (en));
								(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
							} else {
								if (node->type == G_IDL_NODE_OBJECT) {
									ValaClass* cl;
									char* __temp36;
									cl = vala_gidl_parser_parse_object (self, ((GIdlNodeInterface*) node));
									__temp36 = NULL;
									vala_symbol_set_name (VALA_SYMBOL (cl), (__temp36 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (((ValaSymbol*) cl)), module)));
									(__temp36 = (g_free (__temp36), NULL));
									vala_namespace_add_class (ns, cl);
									vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
									(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
								} else {
									if (node->type == G_IDL_NODE_INTERFACE) {
										ValaInterface* iface;
										char* __temp37;
										iface = vala_gidl_parser_parse_interface (self, ((GIdlNodeInterface*) node));
										__temp37 = NULL;
										vala_symbol_set_name (VALA_SYMBOL (iface), (__temp37 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (((ValaSymbol*) iface)), module)));
										(__temp37 = (g_free (__temp37), NULL));
										vala_namespace_add_interface (ns, iface);
										vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (iface));
										(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
									} else {
										if (node->type == G_IDL_NODE_CONSTANT) {
											ValaConstant* c;
											char* __temp38;
											c = vala_gidl_parser_parse_constant (self, ((GIdlNodeConstant*) node));
											__temp38 = NULL;
											vala_symbol_set_name (VALA_SYMBOL (c), (__temp38 = vala_gidl_parser_fix_const_name (self, vala_symbol_get_name (((ValaSymbol*) c)), module)));
											(__temp38 = (g_free (__temp38), NULL));
											vala_namespace_add_constant (ns, c);
											vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (c));
											(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
										} else {
											if (node->type == G_IDL_NODE_FUNCTION) {
												ValaMethod* m;
												m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) node), FALSE, FALSE);
												if (m != NULL) {
													vala_method_set_instance (m, FALSE);
													vala_namespace_add_method (ns, m);
													vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (m));
												}
												(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	if (VALA_IS_NAMESPACE (sym)) {
		gpointer __temp39;
		return VALA_NAMESPACE ((__temp39 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), (attributes = (g_strfreev (attributes), NULL)), __temp39));
	}
	__temp40 = NULL;
	return (__temp40 = ns, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), (attributes = (g_strfreev (attributes), NULL)), __temp40);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
	(attributes = (g_strfreev (attributes), NULL));
}


static ValaCallback* vala_gidl_parser_parse_callback (ValaGIdlParser* self, GIdlNodeFunction* f_node)
{
	GIdlNode* node;
	char** __temp41;
	gint attributes_length1;
	char** attributes;
	ValaTypeReference* __temp47;
	ValaCallback* __temp48;
	ValaCallback* cb;
	ValaCallback* __temp51;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f_node != NULL, NULL);
	node = ((GIdlNode*) f_node);
	__temp41 = NULL;
	attributes = (__temp41 = vala_gidl_parser_get_attributes (self, node->name), attributes_length1 = -1, __temp41);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp46;
				char* attr;
				__temp46 = NULL;
				attr = (__temp46 = attr_collection[attr_it], (__temp46 == NULL ? NULL : g_strdup (__temp46)));
				{
					char** __temp42;
					gint nv_length1;
					char** nv;
					__temp42 = NULL;
					nv = (__temp42 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp42);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* __temp43;
						gboolean __temp44;
						__temp43 = NULL;
						if ((__temp44 = g_utf8_collate ((__temp43 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp43 = (g_free (__temp43), NULL)), __temp44)) {
							gpointer __temp45;
							return VALA_CALLBACK ((__temp45 = NULL, (attr = (g_free (attr), NULL)), (nv = (g_strfreev (nv), NULL)), (attributes = (g_strfreev (attributes), NULL)), __temp45));
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	__temp47 = NULL;
	__temp48 = NULL;
	cb = (__temp48 = vala_callback_new (node->name, (__temp47 = vala_gidl_parser_parse_param (self, f_node->result)), self->priv->current_source_reference), (__temp47 == NULL ? NULL : (__temp47 = (g_object_unref (__temp47), NULL))), __temp48);
	((ValaDataType*) cb)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = f_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaTypeReference* __temp49;
				ValaFormalParameter* __temp50;
				ValaFormalParameter* p;
				param_node = ((GIdlNode*) param);
				__temp49 = NULL;
				__temp50 = NULL;
				p = (__temp50 = vala_formal_parameter_new (param_node->name, (__temp49 = vala_gidl_parser_parse_param (self, param)), NULL), (__temp49 == NULL ? NULL : (__temp49 = (g_object_unref (__temp49), NULL))), __temp50);
				vala_callback_add_parameter (cb, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	__temp51 = NULL;
	return (__temp51 = cb, (attributes = (g_strfreev (attributes), NULL)), __temp51);
	(attributes = (g_strfreev (attributes), NULL));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
}


static ValaStruct* vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node)
{
	GIdlNode* node;
	ValaStruct* st;
	char** __temp53;
	gint st_attributes_length1;
	char** st_attributes;
	ValaDataType* __temp59;
	ValaStruct* __temp58;
	ValaDataType* __temp60;
	ValaStruct* __temp61;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (st_node != NULL, NULL);
	node = ((GIdlNode*) st_node);
	if (st_node->deprecated) {
		return VALA_STRUCT (NULL);
	}
	st = vala_struct_new (node->name, self->priv->current_source_reference);
	((ValaDataType*) st)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	vala_struct_set_is_reference_type (st, TRUE);
	__temp53 = NULL;
	st_attributes = (__temp53 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, __temp53);
	if (st_attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp57;
				char* attr;
				__temp57 = NULL;
				attr = (__temp57 = attr_collection[attr_it], (__temp57 == NULL ? NULL : g_strdup (__temp57)));
				{
					char** __temp54;
					gint nv_length1;
					char** nv;
					char* __temp55;
					gboolean __temp56;
					__temp54 = NULL;
					nv = (__temp54 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp54);
					__temp55 = NULL;
					if ((__temp56 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp55 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp55 = (g_free (__temp55), NULL)), __temp56)) {
						vala_struct_set_is_reference_type (st, FALSE);
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	__temp59 = NULL;
	__temp58 = NULL;
	self->priv->current_data_type = (__temp59 = VALA_DATA_TYPE ((__temp58 = st, (__temp58 == NULL ? NULL : g_object_ref (__temp58)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp59);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = st_node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					ValaMethod* m;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE, FALSE);
					if (m != NULL) {
						vala_struct_add_method (st, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_FIELD) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_struct_add_field (st, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	__temp60 = NULL;
	self->priv->current_data_type = (__temp60 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp60);
	__temp61 = NULL;
	return (__temp61 = st, (st_attributes = (g_strfreev (st_attributes), NULL)), __temp61);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	(st_attributes = (g_strfreev (st_attributes), NULL));
}


static ValaStruct* vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node)
{
	GIdlNode* node;
	ValaStruct* st;
	char** __temp62;
	gint st_attributes_length1;
	char** st_attributes;
	ValaDataType* __temp68;
	ValaStruct* __temp67;
	ValaDataType* __temp69;
	ValaStruct* __temp70;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (boxed_node != NULL, NULL);
	node = ((GIdlNode*) boxed_node);
	st = vala_struct_new (node->name, self->priv->current_source_reference);
	((ValaDataType*) st)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	vala_struct_set_is_reference_type (st, TRUE);
	__temp62 = NULL;
	st_attributes = (__temp62 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, __temp62);
	if (st_attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp66;
				char* attr;
				__temp66 = NULL;
				attr = (__temp66 = attr_collection[attr_it], (__temp66 == NULL ? NULL : g_strdup (__temp66)));
				{
					char** __temp63;
					gint nv_length1;
					char** nv;
					char* __temp64;
					gboolean __temp65;
					__temp63 = NULL;
					nv = (__temp63 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp63);
					__temp64 = NULL;
					if ((__temp65 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp64 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp64 = (g_free (__temp64), NULL)), __temp65)) {
						vala_struct_set_is_reference_type (st, FALSE);
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	__temp68 = NULL;
	__temp67 = NULL;
	self->priv->current_data_type = (__temp68 = VALA_DATA_TYPE ((__temp67 = st, (__temp67 == NULL ? NULL : g_object_ref (__temp67)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp68);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = boxed_node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					ValaMethod* m;
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE, FALSE);
					if (m != NULL) {
						vala_struct_add_method (st, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_FIELD) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_struct_add_field (st, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	__temp69 = NULL;
	self->priv->current_data_type = (__temp69 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp69);
	__temp70 = NULL;
	return (__temp70 = st, (st_attributes = (g_strfreev (st_attributes), NULL)), __temp70);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	(st_attributes = (g_strfreev (st_attributes), NULL));
}


static ValaEnum* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node)
{
	GIdlNode* node;
	ValaEnum* en;
	char* common_prefix;
	ValaEnum* __temp75;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (en_node != NULL, NULL);
	node = ((GIdlNode*) en_node);
	en = vala_enum_new (node->name, self->priv->current_source_reference);
	((ValaDataType*) en)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	common_prefix = NULL;
	{
		GList* value_collection;
		GList* value_it;
		value_collection = en_node->values;
		for (value_it = value_collection; value_it != NULL; value_it = value_it->next) {
			GIdlNode* value;
			value = value_it->data;
			{
				if (common_prefix == NULL) {
					char* __temp72;
					const char* __temp71;
					__temp72 = NULL;
					__temp71 = NULL;
					common_prefix = (__temp72 = (__temp71 = value->name, (__temp71 == NULL ? NULL : g_strdup (__temp71))), (common_prefix = (g_free (common_prefix), NULL)), __temp72);
					while (g_utf8_strlen (common_prefix, -1) > 0 && !g_str_has_suffix (common_prefix, "_")) {
						char* __temp73;
						/* FIXME: could easily be made faster*/
						__temp73 = NULL;
						common_prefix = (__temp73 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), __temp73);
					}
				} else {
					while (!g_str_has_prefix (value->name, common_prefix)) {
						char* __temp74;
						__temp74 = NULL;
						common_prefix = (__temp74 = g_strndup (common_prefix, strlen (common_prefix) - 1), (common_prefix = (g_free (common_prefix), NULL)), __temp74);
					}
				}
			}
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	{
		GList* value2_collection;
		GList* value2_it;
		value2_collection = en_node->values;
		for (value2_it = value2_collection; value2_it != NULL; value2_it = value2_it->next) {
			GIdlNode* value2;
			value2 = value2_it->data;
			{
				ValaEnumValue* ev;
				ev = vala_enum_value_new (g_utf8_offset_to_pointer (value2->name, g_utf8_strlen (common_prefix, -1)));
				vala_enum_add_value (en, ev);
				(ev == NULL ? NULL : (ev = (g_object_unref (ev), NULL)));
			}
		}
	}
	__temp75 = NULL;
	return (__temp75 = en, (common_prefix = (g_free (common_prefix), NULL)), __temp75);
	(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
	(common_prefix = (g_free (common_prefix), NULL));
}


static ValaClass* vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node)
{
	ValaClass* cl;
	char** __temp76;
	gint attributes_length1;
	char** attributes;
	ValaDataType* __temp82;
	ValaClass* __temp81;
	GeeSet* __temp83;
	GeeHashMap* current_type_vfunc_map;
	ValaDataType* __temp86;
	GeeSet* __temp87;
	ValaClass* __temp88;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (node != NULL, NULL);
	cl = vala_class_new (node->gtype_name, self->priv->current_source_reference);
	((ValaDataType*) cl)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp76 = NULL;
	attributes = (__temp76 = vala_gidl_parser_get_attributes (self, vala_symbol_get_name (((ValaSymbol*) cl))), attributes_length1 = -1, __temp76);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp79;
				char* attr;
				__temp79 = NULL;
				attr = (__temp79 = attr_collection[attr_it], (__temp79 == NULL ? NULL : g_strdup (__temp79)));
				{
					char** __temp77;
					gint nv_length1;
					char** nv;
					__temp77 = NULL;
					nv = (__temp77 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp77);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp78;
						__temp78 = NULL;
						vala_data_type_add_cheader_filename (VALA_DATA_TYPE (cl), (__temp78 = vala_gidl_parser_eval (self, nv[1])));
						(__temp78 = (g_free (__temp78), NULL));
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	if (node->parent != NULL) {
		ValaTypeReference* parent;
		parent = vala_type_reference_new ();
		vala_gidl_parser_parse_type_string (self, parent, node->parent);
		vala_class_add_base_type (cl, parent);
		(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
	}
	{
		GList* iface_name_collection;
		GList* iface_name_it;
		iface_name_collection = node->interfaces;
		for (iface_name_it = iface_name_collection; iface_name_it != NULL; iface_name_it = iface_name_it->next) {
			const char* __temp80;
			char* iface_name;
			__temp80 = NULL;
			iface_name = (__temp80 = iface_name_it->data, (__temp80 == NULL ? NULL : g_strdup (__temp80)));
			{
				ValaTypeReference* iface;
				iface = vala_type_reference_new ();
				vala_gidl_parser_parse_type_string (self, iface, iface_name);
				vala_class_add_base_type (cl, iface);
				(iface_name = (g_free (iface_name), NULL));
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	__temp82 = NULL;
	__temp81 = NULL;
	self->priv->current_data_type = (__temp82 = VALA_DATA_TYPE ((__temp81 = cl, (__temp81 == NULL ? NULL : g_object_ref (__temp81)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp82);
	__temp83 = NULL;
	self->priv->current_type_symbol_set = (__temp83 = GEE_SET (gee_hash_set_new (g_strdup, g_free, g_str_hash, g_str_equal)), (self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL))), __temp83);
	current_type_vfunc_map = gee_hash_map_new (g_strdup, g_free, g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set (GEE_MAP (current_type_vfunc_map), member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					char* __temp84;
					gboolean __temp85;
					gboolean is_virtual;
					ValaMethod* m;
					__temp84 = NULL;
					is_virtual = (__temp85 = (__temp84 = gee_map_get (GEE_MAP (current_type_vfunc_map), member->name)) != NULL, (__temp84 = (g_free (__temp84), NULL)), __temp85);
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual, FALSE);
					if (m != NULL) {
						vala_class_add_method (cl, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_PROPERTY) {
						ValaProperty* prop;
						prop = vala_gidl_parser_parse_property (self, ((GIdlNodeProperty*) member));
						if (prop != NULL) {
							vala_class_add_property (cl, prop, FALSE);
						}
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					} else {
						if (member->type == G_IDL_NODE_SIGNAL) {
							ValaSignal* sig;
							sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
							if (sig != NULL) {
								vala_class_add_signal (cl, sig);
							}
							(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
						}
					}
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FIELD) {
					if (!gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), member->name)) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_class_add_field (cl, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	{
		GeeCollection* prop_collection;
		GeeIterator* prop_it;
		prop_collection = vala_class_get_properties (cl);
		prop_it = gee_iterable_iterator (GEE_ITERABLE (prop_collection));
		while (gee_iterator_next (prop_it)) {
			ValaProperty* prop;
			prop = gee_iterator_get (prop_it);
			{
				char* getter;
				char* setter;
				getter = g_strdup_printf ("get_%s", vala_symbol_get_name (((ValaSymbol*) prop)));
				if (vala_property_get_get_accessor (prop) != NULL && !gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), getter)) {
					vala_property_set_no_accessor_method (prop, TRUE);
				}
				setter = g_strdup_printf ("set_%s", vala_symbol_get_name (((ValaSymbol*) prop)));
				if (vala_property_get_set_accessor (prop) != NULL && !gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), setter)) {
					vala_property_set_no_accessor_method (prop, TRUE);
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				(getter = (g_free (getter), NULL));
				(setter = (g_free (setter), NULL));
			}
		}
		(prop_collection == NULL ? NULL : (prop_collection = (g_object_unref (prop_collection), NULL)));
		(prop_it == NULL ? NULL : (prop_it = (g_object_unref (prop_it), NULL)));
	}
	__temp86 = NULL;
	self->priv->current_data_type = (__temp86 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp86);
	__temp87 = NULL;
	self->priv->current_type_symbol_set = (__temp87 = NULL, (self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL))), __temp87);
	__temp88 = NULL;
	return (__temp88 = cl, (attributes = (g_strfreev (attributes), NULL)), (current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL))), __temp88);
	(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	(attributes = (g_strfreev (attributes), NULL));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL)));
}


static ValaInterface* vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node)
{
	ValaInterface* iface;
	char** __temp89;
	gint attributes_length1;
	char** attributes;
	ValaDataType* __temp95;
	ValaInterface* __temp94;
	GeeHashMap* current_type_vfunc_map;
	ValaDataType* __temp98;
	ValaInterface* __temp99;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (node != NULL, NULL);
	iface = vala_interface_new (node->gtype_name, self->priv->current_source_reference);
	((ValaDataType*) iface)->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp89 = NULL;
	attributes = (__temp89 = vala_gidl_parser_get_attributes (self, vala_symbol_get_name (((ValaSymbol*) iface))), attributes_length1 = -1, __temp89);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp92;
				char* attr;
				__temp92 = NULL;
				attr = (__temp92 = attr_collection[attr_it], (__temp92 == NULL ? NULL : g_strdup (__temp92)));
				{
					char** __temp90;
					gint nv_length1;
					char** nv;
					__temp90 = NULL;
					nv = (__temp90 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp90);
					if (g_utf8_collate (nv[0], "cheader_filename") == 0) {
						char* __temp91;
						__temp91 = NULL;
						vala_data_type_add_cheader_filename (VALA_DATA_TYPE (iface), (__temp91 = vala_gidl_parser_eval (self, nv[1])));
						(__temp91 = (g_free (__temp91), NULL));
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	{
		GList* prereq_name_collection;
		GList* prereq_name_it;
		prereq_name_collection = node->prerequisites;
		for (prereq_name_it = prereq_name_collection; prereq_name_it != NULL; prereq_name_it = prereq_name_it->next) {
			const char* __temp93;
			char* prereq_name;
			__temp93 = NULL;
			prereq_name = (__temp93 = prereq_name_it->data, (__temp93 == NULL ? NULL : g_strdup (__temp93)));
			{
				ValaTypeReference* prereq;
				prereq = vala_type_reference_new ();
				vala_gidl_parser_parse_type_string (self, prereq, prereq_name);
				vala_interface_add_prerequisite (iface, prereq);
				(prereq_name = (g_free (prereq_name), NULL));
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
			}
		}
	}
	__temp95 = NULL;
	__temp94 = NULL;
	self->priv->current_data_type = (__temp95 = VALA_DATA_TYPE ((__temp94 = iface, (__temp94 == NULL ? NULL : g_object_ref (__temp94)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp95);
	current_type_vfunc_map = gee_hash_map_new (g_strdup, g_free, g_strdup, g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set (GEE_MAP (current_type_vfunc_map), member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = member_it->data;
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					char* __temp96;
					gboolean __temp97;
					gboolean is_virtual;
					ValaMethod* m;
					__temp96 = NULL;
					is_virtual = (__temp97 = (__temp96 = gee_map_get (GEE_MAP (current_type_vfunc_map), member->name)) != NULL, (__temp96 = (g_free (__temp96), NULL)), __temp97);
					m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), is_virtual, TRUE);
					if (m != NULL) {
						vala_interface_add_method (iface, m);
					}
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (member->type == G_IDL_NODE_SIGNAL) {
						ValaSignal* sig;
						sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
						if (sig != NULL) {
							vala_interface_add_signal (iface, sig);
						}
						(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
					}
				}
			}
		}
	}
	__temp98 = NULL;
	self->priv->current_data_type = (__temp98 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), __temp98);
	__temp99 = NULL;
	return (__temp99 = iface, (attributes = (g_strfreev (attributes), NULL)), (current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL))), __temp99);
	(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
	(attributes = (g_strfreev (attributes), NULL));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL)));
}


static ValaTypeReference* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node)
{
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (type_node != NULL, NULL);
	type = vala_type_reference_new ();
	if (type_node->tag == TYPE_TAG_VOID) {
		if (type_node->is_pointer) {
			vala_type_reference_set_type_name (type, "pointer");
		} else {
			vala_type_reference_set_type_name (type, "void");
		}
	} else {
		if (type_node->tag == TYPE_TAG_BOOLEAN) {
			vala_type_reference_set_type_name (type, "bool");
		} else {
			if (type_node->tag == TYPE_TAG_INT8) {
				vala_type_reference_set_type_name (type, "char");
			} else {
				if (type_node->tag == TYPE_TAG_UINT8) {
					vala_type_reference_set_type_name (type, "uchar");
				} else {
					if (type_node->tag == TYPE_TAG_INT16) {
						vala_type_reference_set_type_name (type, "short");
					} else {
						if (type_node->tag == TYPE_TAG_UINT16) {
							vala_type_reference_set_type_name (type, "ushort");
						} else {
							if (type_node->tag == TYPE_TAG_INT32) {
								vala_type_reference_set_type_name (type, "int");
							} else {
								if (type_node->tag == TYPE_TAG_UINT32) {
									vala_type_reference_set_type_name (type, "uint");
								} else {
									if (type_node->tag == TYPE_TAG_INT64) {
										vala_type_reference_set_type_name (type, "int64");
									} else {
										if (type_node->tag == TYPE_TAG_UINT64) {
											vala_type_reference_set_type_name (type, "uint64");
										} else {
											if (type_node->tag == TYPE_TAG_INT) {
												vala_type_reference_set_type_name (type, "int");
											} else {
												if (type_node->tag == TYPE_TAG_UINT) {
													vala_type_reference_set_type_name (type, "uint");
												} else {
													if (type_node->tag == TYPE_TAG_LONG) {
														vala_type_reference_set_type_name (type, "long");
													} else {
														if (type_node->tag == TYPE_TAG_ULONG) {
															vala_type_reference_set_type_name (type, "ulong");
														} else {
															if (type_node->tag == TYPE_TAG_SSIZE) {
																vala_type_reference_set_type_name (type, "long");
															} else {
																if (type_node->tag == TYPE_TAG_SIZE) {
																	vala_type_reference_set_type_name (type, "ulong");
																} else {
																	if (type_node->tag == TYPE_TAG_FLOAT) {
																		vala_type_reference_set_type_name (type, "float");
																	} else {
																		if (type_node->tag == TYPE_TAG_DOUBLE) {
																			vala_type_reference_set_type_name (type, "double");
																		} else {
																			if (type_node->tag == TYPE_TAG_UTF8) {
																				vala_type_reference_set_type_name (type, "string");
																			} else {
																				if (type_node->tag == TYPE_TAG_FILENAME) {
																					vala_type_reference_set_type_name (type, "string");
																				} else {
																					if (type_node->tag == TYPE_TAG_ARRAY) {
																						ValaTypeReference* __temp100;
																						__temp100 = NULL;
																						type = (__temp100 = vala_gidl_parser_parse_type (self, type_node->parameter_type1), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp100);
																						vala_type_reference_set_array_rank (type, 1);
																					} else {
																						if (type_node->tag == TYPE_TAG_LIST) {
																							vala_type_reference_set_namespace_name (type, "GLib");
																							vala_type_reference_set_type_name (type, "List");
																						} else {
																							if (type_node->tag == TYPE_TAG_SLIST) {
																								vala_type_reference_set_namespace_name (type, "GLib");
																								vala_type_reference_set_type_name (type, "SList");
																							} else {
																								if (type_node->tag == TYPE_TAG_HASH) {
																									vala_type_reference_set_namespace_name (type, "GLib");
																									vala_type_reference_set_type_name (type, "HashTable");
																								} else {
																									if (type_node->tag == TYPE_TAG_ERROR) {
																										vala_type_reference_set_namespace_name (type, "GLib");
																										vala_type_reference_set_type_name (type, "Error");
																									} else {
																										if (type_node->is_interface) {
																											const char* __temp101;
																											char* n;
																											__temp101 = NULL;
																											n = (__temp101 = type_node->interface, (__temp101 == NULL ? NULL : g_strdup (__temp101)));
																											if (g_utf8_collate (n, "") == 0) {
																												gpointer __temp102;
																												return VALA_TYPE_REFERENCE ((__temp102 = NULL, (n = (g_free (n), NULL)), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp102));
																											}
																											if (g_str_has_prefix (n, "const-")) {
																												char* __temp104;
																												const char* __temp103;
																												__temp104 = NULL;
																												__temp103 = NULL;
																												n = (__temp104 = (__temp103 = g_utf8_offset_to_pointer (n, g_utf8_strlen ("const-", -1)), (__temp103 == NULL ? NULL : g_strdup (__temp103))), (n = (g_free (n), NULL)), __temp104);
																											}
																											if (type_node->is_pointer && (g_utf8_collate (n, "gchar") == 0 || g_utf8_collate (n, "char") == 0)) {
																												vala_type_reference_set_type_name (type, "string");
																											} else {
																												if (g_utf8_collate (n, "gunichar") == 0) {
																													vala_type_reference_set_type_name (type, "unichar");
																												} else {
																													if (g_utf8_collate (n, "gchar") == 0) {
																														vala_type_reference_set_type_name (type, "char");
																													} else {
																														if (g_utf8_collate (n, "guchar") == 0 || g_utf8_collate (n, "guint8") == 0) {
																															vala_type_reference_set_type_name (type, "uchar");
																															if (type_node->is_pointer) {
																																vala_type_reference_set_array_rank (type, 1);
																															}
																														} else {
																															if (g_utf8_collate (n, "gushort") == 0) {
																																vala_type_reference_set_type_name (type, "ushort");
																															} else {
																																if (g_utf8_collate (n, "gshort") == 0) {
																																	vala_type_reference_set_type_name (type, "short");
																																} else {
																																	if (g_utf8_collate (n, "gconstpointer") == 0 || g_utf8_collate (n, "void") == 0) {
																																		vala_type_reference_set_type_name (type, "pointer");
																																	} else {
																																		if (g_utf8_collate (n, "goffset") == 0) {
																																			vala_type_reference_set_type_name (type, "int64");
																																		} else {
																																			if (g_utf8_collate (n, "value_array") == 0) {
																																				vala_type_reference_set_namespace_name (type, "GLib");
																																				vala_type_reference_set_type_name (type, "ValueArray");
																																			} else {
																																				if (g_utf8_collate (n, "time_t") == 0) {
																																					vala_type_reference_set_type_name (type, "ulong");
																																				} else {
																																					if (g_utf8_collate (n, "gint") == 0 || g_utf8_collate (n, "pid_t") == 0) {
																																						vala_type_reference_set_type_name (type, "int");
																																					} else {
																																						if (g_utf8_collate (n, "FILE") == 0) {
																																							vala_type_reference_set_namespace_name (type, "GLib");
																																							vala_type_reference_set_type_name (type, "FileStream");
																																						} else {
																																							vala_gidl_parser_parse_type_string (self, type, n);
																																							if (type_node->is_pointer && vala_gidl_parser_is_value_type (self, n)) {
																																								vala_type_reference_set_is_out (type, TRUE);
																																							}
																																						}
																																					}
																																				}
																																			}
																																		}
																																	}
																																}
																															}
																														}
																													}
																												}
																											}
																											(n = (g_free (n), NULL));
																										} else {
																											fprintf (stdout, "%d\n", type_node->tag);
																										}
																									}
																								}
																							}
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static gboolean vala_gidl_parser_is_value_type (ValaGIdlParser* self, const char* type_name)
{
	char** __temp106;
	gint type_attributes_length1;
	char** type_attributes;
	gboolean __temp112;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	g_return_val_if_fail (type_name != NULL, FALSE);
	/* FIXME only works if both types are in current package, e.g. doesn't work when Gtk uses GdkRectangle*/
	__temp106 = NULL;
	type_attributes = (__temp106 = vala_gidl_parser_get_attributes (self, type_name), type_attributes_length1 = -1, __temp106);
	if (type_attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = type_attributes;
			for (attr_it = 0; (type_attributes_length1 != -1 && attr_it < type_attributes_length1) || (type_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp111;
				char* attr;
				__temp111 = NULL;
				attr = (__temp111 = attr_collection[attr_it], (__temp111 == NULL ? NULL : g_strdup (__temp111)));
				{
					char** __temp107;
					gint nv_length1;
					char** nv;
					char* __temp108;
					gboolean __temp109;
					__temp107 = NULL;
					nv = (__temp107 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp107);
					__temp108 = NULL;
					if ((__temp109 = g_utf8_collate (nv[0], "is_value_type") == 0 && g_utf8_collate ((__temp108 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp108 = (g_free (__temp108), NULL)), __temp109)) {
						gboolean __temp110;
						return (__temp110 = TRUE, (attr = (g_free (attr), NULL)), (nv = (g_strfreev (nv), NULL)), (type_attributes = (g_strfreev (type_attributes), NULL)), __temp110);
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	return (__temp112 = FALSE, (type_attributes = (g_strfreev (type_attributes), NULL)), __temp112);
	(type_attributes = (g_strfreev (type_attributes), NULL));
}


static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaTypeReference* type, const char* n)
{
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	g_return_if_fail (n != NULL);
	/* Generated GIDL misses explicit namespace specifier,
	 so try to guess namespace*/
	if (g_str_has_prefix (n, "H") || g_str_has_suffix (n, "Class") || g_utf8_collate (n, "va_list") == 0 || g_str_has_prefix (n, "LOGFONT") || g_str_has_prefix (n, "xml")) {
		/* unsupported*/
		vala_type_reference_set_type_name (type, "pointer");
	} else {
		if (g_str_has_prefix (n, "cairo")) {
			vala_type_reference_set_namespace_name (type, "Cairo");
			if (g_utf8_collate (n, "cairo_t") == 0) {
				vala_type_reference_set_type_name (type, "Context");
			} else {
				if (g_utf8_collate (n, "cairo_surface_t") == 0) {
					vala_type_reference_set_type_name (type, "Surface");
				} else {
					vala_type_reference_set_namespace_name (type, NULL);
					vala_type_reference_set_type_name (type, "pointer");
				}
			}
		} else {
			if (g_str_has_prefix (n, "Atk")) {
				vala_type_reference_set_namespace_name (type, "Atk");
				vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Atk", -1)));
				if (g_utf8_collate (vala_type_reference_get_type_name (type), "AttributeSet") == 0) {
					vala_type_reference_set_namespace_name (type, "GLib");
					vala_type_reference_set_type_name (type, "SList");
				}
			} else {
				if (g_str_has_prefix (n, "Gst")) {
					vala_type_reference_set_namespace_name (type, "Gst");
					vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Gst", -1)));
					if (g_utf8_collate (vala_type_reference_get_type_name (type), "ClockTime") == 0) {
						vala_type_reference_set_namespace_name (type, NULL);
						vala_type_reference_set_type_name (type, "uint64");
					} else {
						if (g_utf8_collate (vala_type_reference_get_type_name (type), "ClockTimeDiff") == 0) {
							vala_type_reference_set_namespace_name (type, NULL);
							vala_type_reference_set_type_name (type, "int64");
						} else {
							if (g_utf8_collate (vala_type_reference_get_type_name (type), "ClockID") == 0 || g_str_has_prefix (vala_type_reference_get_type_name (type), "Xml")) {
								vala_type_reference_set_namespace_name (type, NULL);
								vala_type_reference_set_type_name (type, "pointer");
							}
						}
					}
				} else {
					if (g_str_has_prefix (n, "Gtk")) {
						vala_type_reference_set_namespace_name (type, "Gtk");
						vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Gtk", -1)));
						if (g_utf8_collate (vala_type_reference_get_type_name (type), "TextLayout") == 0) {
							vala_type_reference_set_namespace_name (type, NULL);
							vala_type_reference_set_type_name (type, "pointer");
						}
					} else {
						if (g_str_has_prefix (n, "Gdk")) {
							vala_type_reference_set_namespace_name (type, "Gdk");
							vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Gdk", -1)));
							if (g_utf8_collate (vala_type_reference_get_type_name (type), "NativeWindow") == 0 || g_utf8_collate (vala_type_reference_get_type_name (type), "XEvent") == 0) {
								vala_type_reference_set_namespace_name (type, NULL);
								vala_type_reference_set_type_name (type, "pointer");
							}
						} else {
							if (g_str_has_prefix (n, "Pango")) {
								vala_type_reference_set_namespace_name (type, "Pango");
								vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Pango", -1)));
								if (g_utf8_collate (vala_type_reference_get_type_name (type), "Glyph") == 0) {
									vala_type_reference_set_namespace_name (type, NULL);
									vala_type_reference_set_type_name (type, "uint");
								} else {
									if (g_utf8_collate (vala_type_reference_get_type_name (type), "GlyphUnit") == 0) {
										vala_type_reference_set_namespace_name (type, NULL);
										vala_type_reference_set_type_name (type, "int");
									}
								}
							} else {
								if (g_str_has_prefix (n, "Vte")) {
									vala_type_reference_set_namespace_name (type, "Vte");
									vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Vte", -1)));
								} else {
									if (g_str_has_prefix (n, "Goo")) {
										vala_type_reference_set_namespace_name (type, "Goo");
										vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("Goo", -1)));
									} else {
										if (g_str_has_prefix (n, "G")) {
											vala_type_reference_set_namespace_name (type, "GLib");
											vala_type_reference_set_type_name (type, g_utf8_offset_to_pointer (n, g_utf8_strlen ("G", -1)));
											if (g_utf8_collate (vala_type_reference_get_type_name (type), "Strv") == 0) {
												vala_type_reference_set_namespace_name (type, NULL);
												vala_type_reference_set_type_name (type, "string");
												vala_type_reference_set_array_rank (type, 1);
											} else {
												if (g_utf8_collate (vala_type_reference_get_type_name (type), "StaticRecMutex") == 0) {
													vala_type_reference_set_namespace_name (type, NULL);
													vala_type_reference_set_type_name (type, "pointer");
												}
											}
										} else {
											char** __temp113;
											gint name_parts_length1;
											char** name_parts;
											__temp113 = NULL;
											name_parts = (__temp113 = g_strsplit (n, ".", 2), name_parts_length1 = -1, __temp113);
											if (name_parts[1] == NULL) {
												vala_type_reference_set_type_name (type, name_parts[0]);
											} else {
												vala_type_reference_set_namespace_name (type, name_parts[0]);
												vala_type_reference_set_type_name (type, name_parts[1]);
											}
											(name_parts = (g_strfreev (name_parts), NULL));
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
}


static ValaTypeReference* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param)
{
	ValaTypeReference* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (param != NULL, NULL);
	type = vala_gidl_parser_parse_type (self, param->type);
	return type;
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


/* disable for now as null_ok not yet correctly set
 type.non_null = !param.null_ok;*/
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_virtual, gboolean is_interface)
{
	GIdlNode* node;
	ValaTypeReference* return_type;
	ValaMethod* m;
	gboolean add_ellipsis;
	char** __temp127;
	gint attributes_length1;
	char** attributes;
	gboolean first;
	ValaFormalParameter* last_param;
	ValaMethod* __temp142;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (f != NULL, NULL);
	node = ((GIdlNode*) f);
	if (f->deprecated) {
		return VALA_METHOD (NULL);
	}
	return_type = NULL;
	if (f->result != NULL) {
		ValaTypeReference* __temp116;
		__temp116 = NULL;
		return_type = (__temp116 = vala_gidl_parser_parse_param (self, f->result), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), __temp116);
	}
	m = NULL;
	if (!is_interface && (f->is_constructor || g_str_has_prefix (node->name, "new"))) {
		ValaMethod* __temp117;
		__temp117 = NULL;
		m = (__temp117 = VALA_METHOD (vala_creation_method_new (node->name, self->priv->current_source_reference)), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), __temp117);
		if (g_utf8_collate (vala_symbol_get_name (((ValaSymbol*) m)), "new") == 0) {
			vala_symbol_set_name (VALA_SYMBOL (m), NULL);
		} else {
			if (g_str_has_prefix (vala_symbol_get_name (((ValaSymbol*) m)), "new_")) {
				vala_symbol_set_name (VALA_SYMBOL (m), g_utf8_offset_to_pointer (vala_symbol_get_name (((ValaSymbol*) m)), g_utf8_strlen ("new_", -1)));
			}
		}
	} else {
		ValaMethod* __temp118;
		__temp118 = NULL;
		m = (__temp118 = vala_method_new (node->name, return_type, self->priv->current_source_reference), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), __temp118);
	}
	m->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	vala_method_set_is_virtual (m, is_virtual);
	/* GIDL generator can't provide array parameter information yet*/
	vala_method_set_no_array_length (m, TRUE);
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), node->name);
	}
	if (self->priv->current_data_type != NULL) {
		char** __temp121;
		gint sig_attributes_length1;
		char* __temp119;
		char** __temp120;
		char** sig_attributes;
		__temp121 = NULL;
		__temp119 = NULL;
		__temp120 = NULL;
		sig_attributes = (__temp121 = (__temp120 = vala_gidl_parser_get_attributes (self, (__temp119 = g_strdup_printf ("%s::%s", vala_symbol_get_name (((ValaSymbol*) self->priv->current_data_type)), node->name))), (__temp119 = (g_free (__temp119), NULL)), __temp120), sig_attributes_length1 = -1, __temp121);
		if (sig_attributes != NULL) {
			{
				char** attr_collection;
				int attr_it;
				attr_collection = sig_attributes;
				for (attr_it = 0; (sig_attributes_length1 != -1 && attr_it < sig_attributes_length1) || (sig_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
					const char* __temp126;
					char* attr;
					__temp126 = NULL;
					attr = (__temp126 = attr_collection[attr_it], (__temp126 == NULL ? NULL : g_strdup (__temp126)));
					{
						char** __temp122;
						gint nv_length1;
						char** nv;
						char* __temp123;
						gboolean __temp124;
						__temp122 = NULL;
						nv = (__temp122 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp122);
						__temp123 = NULL;
						if ((__temp124 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((__temp123 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp123 = (g_free (__temp123), NULL)), __temp124)) {
							gpointer __temp125;
							return VALA_METHOD ((__temp125 = NULL, (attr = (g_free (attr), NULL)), (nv = (g_strfreev (nv), NULL)), (sig_attributes = (g_strfreev (sig_attributes), NULL)), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), __temp125));
						}
						(attr = (g_free (attr), NULL));
						(nv = (g_strfreev (nv), NULL));
					}
				}
			}
		}
		(sig_attributes = (g_strfreev (sig_attributes), NULL));
	}
	add_ellipsis = FALSE;
	__temp127 = NULL;
	attributes = (__temp127 = vala_gidl_parser_get_attributes (self, f->symbol), attributes_length1 = -1, __temp127);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp134;
				char* attr;
				__temp134 = NULL;
				attr = (__temp134 = attr_collection[attr_it], (__temp134 == NULL ? NULL : g_strdup (__temp134)));
				{
					char** __temp128;
					gint nv_length1;
					char** nv;
					__temp128 = NULL;
					nv = (__temp128 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp128);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* __temp129;
						gboolean __temp130;
						__temp129 = NULL;
						if ((__temp130 = g_utf8_collate ((__temp129 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp129 = (g_free (__temp129), NULL)), __temp130)) {
							gpointer __temp131;
							return VALA_METHOD ((__temp131 = NULL, (attr = (g_free (attr), NULL)), (nv = (g_strfreev (nv), NULL)), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), (attributes = (g_strfreev (attributes), NULL)), __temp131));
						}
					} else {
						if (g_utf8_collate (nv[0], "ellipsis") == 0) {
							char* __temp132;
							gboolean __temp133;
							__temp132 = NULL;
							if ((__temp133 = g_utf8_collate ((__temp132 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp132 = (g_free (__temp132), NULL)), __temp133)) {
								add_ellipsis = TRUE;
							}
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	vala_method_set_cname (m, f->symbol);
	first = TRUE;
	last_param = NULL;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = f->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaTypeReference* __temp135;
				ValaFormalParameter* __temp136;
				ValaFormalParameter* p;
				char* __temp137;
				gboolean __temp138;
				ValaFormalParameter* __temp140;
				ValaFormalParameter* __temp139;
				param_node = ((GIdlNode*) param);
				if (first) {
					first = FALSE;
					if (self->priv->current_data_type != NULL && param->type->is_interface && (g_utf8_collate (param_node->name, "self") == 0 || g_str_has_suffix (param->type->interface, vala_symbol_get_name (((ValaSymbol*) self->priv->current_data_type))))) {
						/* instance method*/
						if (!vala_data_type_is_reference_type (self->priv->current_data_type) && param->type->is_pointer) {
							vala_method_set_instance_by_reference (m, TRUE);
						}
						continue;
					} else {
						/* static method*/
						vala_method_set_instance (m, FALSE);
					}
				}
				__temp135 = NULL;
				__temp136 = NULL;
				p = (__temp136 = vala_formal_parameter_new (param_node->name, (__temp135 = vala_gidl_parser_parse_param (self, param)), NULL), (__temp135 == NULL ? NULL : (__temp135 = (g_object_unref (__temp135), NULL))), __temp136);
				vala_method_add_parameter (m, p);
				__temp137 = NULL;
				if ((__temp138 = last_param != NULL && g_utf8_collate (vala_symbol_get_name (((ValaSymbol*) p)), (__temp137 = g_strconcat ("n_", vala_symbol_get_name (((ValaSymbol*) last_param)), NULL))) == 0, (__temp137 = (g_free (__temp137), NULL)), __temp138)) {
					/* last_param is array, p is array length*/
					vala_type_reference_set_array_rank (vala_formal_parameter_get_type_reference (last_param), 1);
					vala_type_reference_set_is_out (vala_formal_parameter_get_type_reference (last_param), FALSE);
				}
				__temp140 = NULL;
				__temp139 = NULL;
				last_param = (__temp140 = (__temp139 = p, (__temp139 == NULL ? NULL : g_object_ref (__temp139))), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), __temp140);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	if (first) {
		/* no parameters => static method*/
		vala_method_set_instance (m, FALSE);
	}
	if (last_param != NULL && g_str_has_prefix (vala_symbol_get_name (((ValaSymbol*) last_param)), "first_")) {
		vala_formal_parameter_set_ellipsis (last_param, TRUE);
	} else {
		if (add_ellipsis) {
			ValaFormalParameter* __temp141;
			__temp141 = NULL;
			vala_method_add_parameter (m, (__temp141 = vala_formal_parameter_new_with_ellipsis (NULL)));
			(__temp141 == NULL ? NULL : (__temp141 = (g_object_unref (__temp141), NULL)));
		}
	}
	__temp142 = NULL;
	return (__temp142 = m, (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (attributes = (g_strfreev (attributes), NULL)), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), __temp142);
	(return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL)));
	(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	(attributes = (g_strfreev (attributes), NULL));
	(last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL)));
}


static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name)
{
	GString* str;
	const char* __temp143;
	char* i;
	const char* __temp146;
	char* __temp147;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	str = g_string_new ("");
	__temp143 = NULL;
	i = (__temp143 = name, (__temp143 == NULL ? NULL : g_strdup (__temp143)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* __temp145;
		const char* __temp144;
		c = g_utf8_get_char (i);
		if (c == '-') {
			g_string_append_c (str, '_');
		} else {
			g_string_append_unichar (str, c);
		}
		__temp145 = NULL;
		__temp144 = NULL;
		i = (__temp145 = (__temp144 = g_utf8_next_char (i), (__temp144 == NULL ? NULL : g_strdup (__temp144))), (i = (g_free (i), NULL)), __temp145);
	}
	__temp146 = NULL;
	__temp147 = NULL;
	return (__temp147 = (__temp146 = str->str, (__temp146 == NULL ? NULL : g_strdup (__temp146))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), __temp147);
	(str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL)));
	(i = (g_free (i), NULL));
}


static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node)
{
	GIdlNode* node;
	ValaPropertyAccessor* get_acc;
	ValaPropertyAccessor* set_acc;
	ValaTypeReference* __temp152;
	char* __temp151;
	ValaProperty* __temp153;
	ValaProperty* prop;
	ValaProperty* __temp154;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (prop_node != NULL, NULL);
	node = ((GIdlNode*) prop_node);
	if (prop_node->deprecated) {
		return VALA_PROPERTY (NULL);
	}
	if (!prop_node->readable && !prop_node->writable) {
		/* buggy GIDL definition*/
		prop_node->readable = TRUE;
		prop_node->writable = TRUE;
	}
	get_acc = NULL;
	set_acc = NULL;
	if (prop_node->readable) {
		ValaPropertyAccessor* __temp149;
		__temp149 = NULL;
		get_acc = (__temp149 = vala_property_accessor_new (TRUE, FALSE, FALSE, NULL, NULL), (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), __temp149);
	}
	if (prop_node->writable) {
		ValaPropertyAccessor* __temp150;
		__temp150 = NULL;
		set_acc = (__temp150 = vala_property_accessor_new (FALSE, FALSE, FALSE, NULL, NULL), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), __temp150);
		if (prop_node->construct_only) {
			vala_property_accessor_set_construction (set_acc, TRUE);
		} else {
			vala_property_accessor_set_writable (set_acc, TRUE);
			vala_property_accessor_set_construction (set_acc, prop_node->construct);
		}
	}
	__temp152 = NULL;
	__temp151 = NULL;
	__temp153 = NULL;
	prop = (__temp153 = vala_property_new ((__temp151 = vala_gidl_parser_fix_prop_name (self, node->name)), (__temp152 = vala_gidl_parser_parse_type (self, prop_node->type)), get_acc, set_acc, self->priv->current_source_reference), (__temp152 == NULL ? NULL : (__temp152 = (g_object_unref (__temp152), NULL))), (__temp151 = (g_free (__temp151), NULL)), __temp153);
	vala_property_set_access (prop, VALA_MEMBER_ACCESSIBILITY_PUBLIC);
	vala_property_set_interface_only (prop, TRUE);
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), vala_symbol_get_name (((ValaSymbol*) prop)));
	}
	__temp154 = NULL;
	return (__temp154 = prop, (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), __temp154);
	(get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL)));
	(set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL)));
	(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
}


static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node)
{
	GIdlNode* node;
	ValaTypeReference* type;
	ValaConstant* c;
	ValaConstant* __temp156;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (const_node != NULL, NULL);
	node = ((GIdlNode*) const_node);
	type = vala_gidl_parser_parse_type (self, const_node->type);
	if (type == NULL) {
		gpointer __temp155;
		return VALA_CONSTANT ((__temp155 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp155));
	}
	c = vala_constant_new (node->name, type, NULL, self->priv->current_source_reference);
	__temp156 = NULL;
	return (__temp156 = c, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp156);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
}


static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node)
{
	GIdlNode* node;
	ValaTypeReference* type;
	char** __temp161;
	gint attributes_length1;
	char* __temp159;
	char** __temp160;
	char** attributes;
	ValaField* field;
	ValaField* __temp167;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (field_node != NULL, NULL);
	node = ((GIdlNode*) field_node);
	type = vala_gidl_parser_parse_type (self, field_node->type);
	if (type == NULL) {
		gpointer __temp157;
		return VALA_FIELD ((__temp157 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp157));
	}
	if (!field_node->readable) {
		gpointer __temp158;
		return VALA_FIELD ((__temp158 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), __temp158));
	}
	__temp161 = NULL;
	__temp159 = NULL;
	__temp160 = NULL;
	attributes = (__temp161 = (__temp160 = vala_gidl_parser_get_attributes (self, (__temp159 = g_strdup_printf ("%s.%s", vala_symbol_get_name (((ValaSymbol*) self->priv->current_data_type)), node->name))), (__temp159 = (g_free (__temp159), NULL)), __temp160), attributes_length1 = -1, __temp161);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp166;
				char* attr;
				__temp166 = NULL;
				attr = (__temp166 = attr_collection[attr_it], (__temp166 == NULL ? NULL : g_strdup (__temp166)));
				{
					char** __temp162;
					gint nv_length1;
					char** nv;
					__temp162 = NULL;
					nv = (__temp162 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp162);
					if (g_utf8_collate (nv[0], "hidden") == 0) {
						char* __temp163;
						gboolean __temp164;
						__temp163 = NULL;
						if ((__temp164 = g_utf8_collate ((__temp163 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp163 = (g_free (__temp163), NULL)), __temp164)) {
							gpointer __temp165;
							return VALA_FIELD ((__temp165 = NULL, (attr = (g_free (attr), NULL)), (nv = (g_strfreev (nv), NULL)), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (attributes = (g_strfreev (attributes), NULL)), __temp165));
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), node->name);
	}
	field = vala_field_new (node->name, type, NULL, self->priv->current_source_reference);
	field->access = VALA_MEMBER_ACCESSIBILITY_PUBLIC;
	__temp167 = NULL;
	return (__temp167 = field, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (attributes = (g_strfreev (attributes), NULL)), __temp167);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	(attributes = (g_strfreev (attributes), NULL));
	(field == NULL ? NULL : (field = (g_object_unref (field), NULL)));
}


static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode)
{
	char* attributes;
	char** __temp169;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (codenode != NULL, NULL);
	attributes = gee_map_get (self->priv->codenode_attributes_map, codenode);
	if (attributes == NULL) {
		gpointer __temp168;
		return (__temp168 = NULL, (attributes = (g_free (attributes), NULL)), __temp168);
	}
	__temp169 = NULL;
	return (__temp169 = g_strsplit (attributes, " ", 0), (attributes = (g_free (attributes), NULL)), __temp169);
	(attributes = (g_free (attributes), NULL));
}


static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s)
{
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (s != NULL, NULL);
	return g_strndup (g_utf8_offset_to_pointer (s, 1), strlen (s) - 2);
}


static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node)
{
	GIdlNode* node;
	ValaTypeReference* __temp173;
	char* __temp172;
	ValaSignal* __temp174;
	ValaSignal* sig;
	char** __temp177;
	gint attributes_length1;
	char* __temp175;
	char** __temp176;
	char** attributes;
	gboolean first;
	ValaSignal* __temp188;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (sig_node != NULL, NULL);
	node = ((GIdlNode*) sig_node);
	if (sig_node->deprecated || sig_node->result == NULL) {
		return VALA_SIGNAL (NULL);
	}
	__temp173 = NULL;
	__temp172 = NULL;
	__temp174 = NULL;
	sig = (__temp174 = vala_signal_new ((__temp172 = vala_gidl_parser_fix_prop_name (self, node->name)), (__temp173 = vala_gidl_parser_parse_param (self, sig_node->result)), self->priv->current_source_reference), (__temp173 == NULL ? NULL : (__temp173 = (g_object_unref (__temp173), NULL))), (__temp172 = (g_free (__temp172), NULL)), __temp174);
	vala_signal_set_access (sig, VALA_MEMBER_ACCESSIBILITY_PUBLIC);
	__temp177 = NULL;
	__temp175 = NULL;
	__temp176 = NULL;
	attributes = (__temp177 = (__temp176 = vala_gidl_parser_get_attributes (self, (__temp175 = g_strdup_printf ("%s::%s", vala_symbol_get_name (((ValaSymbol*) self->priv->current_data_type)), vala_symbol_get_name (((ValaSymbol*) sig))))), (__temp175 = (g_free (__temp175), NULL)), __temp176), attributes_length1 = -1, __temp177);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* __temp185;
				char* attr;
				__temp185 = NULL;
				attr = (__temp185 = attr_collection[attr_it], (__temp185 == NULL ? NULL : g_strdup (__temp185)));
				{
					char** __temp178;
					gint nv_length1;
					char** nv;
					__temp178 = NULL;
					nv = (__temp178 = g_strsplit (attr, "=", 2), nv_length1 = -1, __temp178);
					if (g_utf8_collate (nv[0], "name") == 0) {
						char* __temp179;
						vala_signal_set_cname (sig, vala_symbol_get_name (((ValaSymbol*) sig)));
						__temp179 = NULL;
						vala_symbol_set_name (VALA_SYMBOL (sig), (__temp179 = vala_gidl_parser_eval (self, nv[1])));
						(__temp179 = (g_free (__temp179), NULL));
					} else {
						char* __temp180;
						gboolean __temp181;
						__temp180 = NULL;
						if ((__temp181 = g_utf8_collate (nv[0], "has_emitter") == 0 && g_utf8_collate ((__temp180 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp180 = (g_free (__temp180), NULL)), __temp181)) {
							vala_signal_set_has_emitter (sig, TRUE);
						} else {
							if (g_utf8_collate (nv[0], "hidden") == 0) {
								char* __temp182;
								gboolean __temp183;
								__temp182 = NULL;
								if ((__temp183 = g_utf8_collate ((__temp182 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (__temp182 = (g_free (__temp182), NULL)), __temp183)) {
									gpointer __temp184;
									return VALA_SIGNAL ((__temp184 = NULL, (attr = (g_free (attr), NULL)), (nv = (g_strfreev (nv), NULL)), (sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL))), (attributes = (g_strfreev (attributes), NULL)), __temp184));
								}
							}
						}
					}
					(attr = (g_free (attr), NULL));
					(nv = (g_strfreev (nv), NULL));
				}
			}
		}
	}
	first = TRUE;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = sig_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = param_it->data;
			{
				GIdlNode* param_node;
				ValaTypeReference* __temp186;
				ValaFormalParameter* __temp187;
				ValaFormalParameter* p;
				if (first) {
					/* ignore implicit first signal parameter (sender)*/
					first = FALSE;
					continue;
				}
				param_node = ((GIdlNode*) param);
				__temp186 = NULL;
				__temp187 = NULL;
				p = (__temp187 = vala_formal_parameter_new (param_node->name, (__temp186 = vala_gidl_parser_parse_param (self, param)), NULL), (__temp186 == NULL ? NULL : (__temp186 = (g_object_unref (__temp186), NULL))), __temp187);
				vala_signal_add_parameter (sig, p);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
	}
	__temp188 = NULL;
	return (__temp188 = sig, (attributes = (g_strfreev (attributes), NULL)), __temp188);
	(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
	(attributes = (g_strfreev (attributes), NULL));
}


static void vala_gidl_parser_class_init (ValaGIdlParserClass * klass)
{
	vala_gidl_parser_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaGIdlParserPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_gidl_parser_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gidl_parser_real_visit_source_file;
}


static void vala_gidl_parser_init (ValaGIdlParser * self)
{
	self->priv = VALA_GIDL_PARSER_GET_PRIVATE (self);
}


static void vala_gidl_parser_dispose (GObject * obj)
{
	ValaGIdlParser * self;
	ValaGIdlParserClass * klass;
	GObjectClass * parent_class;
	self = VALA_GIDL_PARSER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL)));
	(self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL)));
	(self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_object_unref (self->priv->codenode_attributes_map), NULL)));
	(self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL)));
	klass = VALA_GIDL_PARSER_CLASS (g_type_class_peek (VALA_TYPE_GIDL_PARSER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_gidl_parser_get_type (void)
{
	static GType vala_gidl_parser_type_id = 0;
	if (G_UNLIKELY (vala_gidl_parser_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlParser), 0, (GInstanceInitFunc) vala_gidl_parser_init };
		vala_gidl_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlParser", &g_define_type_info, 0);
	}
	return vala_gidl_parser_type_id;
}




