/* valareport.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valareport.h>
#include <stdio.h>
#include <vala/valasourcefile.h>




gint vala_report_warnings = 0;
gint vala_report_errors = 0;
gboolean vala_report_verbose_errors = FALSE;



/**
 * Set the error verbosity.
 */
void vala_report_set_verbose_errors (gboolean verbose) {
	vala_report_verbose_errors = verbose;
}


/**
 * Returns the total number of warnings reported.
 */
gint vala_report_get_warnings (void) {
	return vala_report_warnings;
}


/**
 * Returns the total number of errors reported.
 */
gint vala_report_get_errors (void) {
	return vala_report_errors;
}


/**
 * Pretty-print the actual line of offending code if possible.
 */
void vala_report_report_source (ValaSourceReference* source) {
	char* offending_line;
	g_return_if_fail (VALA_IS_SOURCE_REFERENCE (source));
	if (vala_source_reference_get_first_line (source) != vala_source_reference_get_last_line (source)) {
		/* FIXME Cannot report multi-line issues currently*/
		return;
	}
	offending_line = vala_source_file_get_source_line (vala_source_reference_get_file (source), vala_source_reference_get_first_line (source));
	if (offending_line != NULL) {
		gint idx;
		fprintf (stderr, "%s\n", offending_line);
		idx = 0;
		/* We loop in this manner so that we don't fall over on differing
		 * tab widths. This means we get the ^s in the right places.
		 */
		for (idx = 1; idx < vala_source_reference_get_first_column (source); (idx = idx + 1)) {
			if (g_utf8_get_char (g_utf8_offset_to_pointer (offending_line, idx - 1)) == '\t') {
				fprintf (stderr, "\t");
			} else {
				fprintf (stderr, " ");
			}
		}
		for (idx = vala_source_reference_get_first_column (source); idx <= vala_source_reference_get_last_column (source); (idx = idx + 1)) {
			if (g_utf8_get_char (g_utf8_offset_to_pointer (offending_line, idx - 1)) == '\t') {
				fprintf (stderr, "\t");
			} else {
				fprintf (stderr, "^");
			}
		}
		fprintf (stderr, "\n");
	}
	offending_line = (g_free (offending_line), NULL);
}


/**
 * Reports the specified message as warning.
 *
 * @param source  reference to source code
 * @param message warning message
 */
void vala_report_warning (ValaSourceReference* source, const char* message) {
	g_return_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source));
	g_return_if_fail (message != NULL);
	vala_report_warnings++;
	if (source == NULL) {
		fprintf (stderr, "warning: %s\n", message);
	} else {
		char* _tmp0;
		_tmp0 = NULL;
		fprintf (stderr, "%s: warning: %s\n", (_tmp0 = vala_source_reference_to_string (source)), message);
		_tmp0 = (g_free (_tmp0), NULL);
		if (vala_report_verbose_errors) {
			vala_report_report_source (source);
		}
	}
}


/**
 * Reports the specified message as error.
 *
 * @param source  reference to source code
 * @param message error message
 */
void vala_report_error (ValaSourceReference* source, const char* message) {
	g_return_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source));
	g_return_if_fail (message != NULL);
	vala_report_errors++;
	if (source == NULL) {
		fprintf (stderr, "error: %s\n", message);
	} else {
		char* _tmp0;
		_tmp0 = NULL;
		fprintf (stderr, "%s: error: %s\n", (_tmp0 = vala_source_reference_to_string (source)), message);
		_tmp0 = (g_free (_tmp0), NULL);
		if (vala_report_verbose_errors) {
			vala_report_report_source (source);
		}
	}
}




