/* valapropertyaccessor.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_PROPERTY_ACCESSOR (vala_property_accessor_get_type ())
#define VALA_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessor))
#define VALA_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))
#define VALA_IS_PROPERTY_ACCESSOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_IS_PROPERTY_ACCESSOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY_ACCESSOR))
#define VALA_PROPERTY_ACCESSOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorClass))

typedef struct _ValaPropertyAccessor ValaPropertyAccessor;
typedef struct _ValaPropertyAccessorClass ValaPropertyAccessorClass;
typedef struct _ValaPropertyAccessorPrivate ValaPropertyAccessorPrivate;

#define VALA_TYPE_BASIC_BLOCK (vala_basic_block_get_type ())
#define VALA_BASIC_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_BASIC_BLOCK, ValaBasicBlock))
#define VALA_BASIC_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_BASIC_BLOCK, ValaBasicBlockClass))
#define VALA_IS_BASIC_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_BASIC_BLOCK))
#define VALA_IS_BASIC_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_BASIC_BLOCK))
#define VALA_BASIC_BLOCK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_BASIC_BLOCK, ValaBasicBlockClass))

typedef struct _ValaBasicBlock ValaBasicBlock;
typedef struct _ValaBasicBlockClass ValaBasicBlockClass;

#define VALA_TYPE_FORMAL_PARAMETER (vala_formal_parameter_get_type ())
#define VALA_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameter))
#define VALA_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))
#define VALA_IS_FORMAL_PARAMETER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_IS_FORMAL_PARAMETER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FORMAL_PARAMETER))
#define VALA_FORMAL_PARAMETER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FORMAL_PARAMETER, ValaFormalParameterClass))

typedef struct _ValaFormalParameter ValaFormalParameter;
typedef struct _ValaFormalParameterClass ValaFormalParameterClass;

#define VALA_TYPE_BLOCK (vala_block_get_type ())
#define VALA_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_BLOCK, ValaBlock))
#define VALA_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_BLOCK, ValaBlockClass))
#define VALA_IS_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_BLOCK))
#define VALA_IS_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_BLOCK))
#define VALA_BLOCK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_BLOCK, ValaBlockClass))

typedef struct _ValaBlock ValaBlock;
typedef struct _ValaBlockClass ValaBlockClass;
#define _vala_basic_block_unref0(var) ((var == NULL) ? NULL : (var = (vala_basic_block_unref (var), NULL)))
#define _vala_code_node_unref0(var) ((var == NULL) ? NULL : (var = (vala_code_node_unref (var), NULL)))
#define _g_free0(var) (var = (g_free (var), NULL))

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_PROPERTY (vala_property_get_type ())
#define VALA_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_PROPERTY, ValaProperty))
#define VALA_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_PROPERTY, ValaPropertyClass))
#define VALA_IS_PROPERTY(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_PROPERTY))
#define VALA_IS_PROPERTY_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_PROPERTY))
#define VALA_PROPERTY_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_PROPERTY, ValaPropertyClass))

typedef struct _ValaProperty ValaProperty;
typedef struct _ValaPropertyClass ValaPropertyClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;

#define VALA_TYPE_INTERFACE (vala_interface_get_type ())
#define VALA_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTERFACE, ValaInterface))
#define VALA_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTERFACE, ValaInterfaceClass))
#define VALA_IS_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTERFACE))
#define VALA_IS_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTERFACE))
#define VALA_INTERFACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTERFACE, ValaInterfaceClass))

typedef struct _ValaInterface ValaInterface;
typedef struct _ValaInterfaceClass ValaInterfaceClass;

#define VALA_TYPE_MEMBER_ACCESS (vala_member_access_get_type ())
#define VALA_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccess))
#define VALA_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))
#define VALA_IS_MEMBER_ACCESS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER_ACCESS))
#define VALA_IS_MEMBER_ACCESS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER_ACCESS))
#define VALA_MEMBER_ACCESS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER_ACCESS, ValaMemberAccessClass))

typedef struct _ValaMemberAccess ValaMemberAccess;
typedef struct _ValaMemberAccessClass ValaMemberAccessClass;

#define VALA_TYPE_STATEMENT (vala_statement_get_type ())
#define VALA_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STATEMENT, ValaStatement))
#define VALA_IS_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STATEMENT))
#define VALA_STATEMENT_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_STATEMENT, ValaStatementIface))

typedef struct _ValaStatement ValaStatement;
typedef struct _ValaStatementIface ValaStatementIface;

#define VALA_TYPE_RETURN_STATEMENT (vala_return_statement_get_type ())
#define VALA_RETURN_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_RETURN_STATEMENT, ValaReturnStatement))
#define VALA_RETURN_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_RETURN_STATEMENT, ValaReturnStatementClass))
#define VALA_IS_RETURN_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_RETURN_STATEMENT))
#define VALA_IS_RETURN_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_RETURN_STATEMENT))
#define VALA_RETURN_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_RETURN_STATEMENT, ValaReturnStatementClass))

typedef struct _ValaReturnStatement ValaReturnStatement;
typedef struct _ValaReturnStatementClass ValaReturnStatementClass;

#define VALA_TYPE_ASSIGNMENT_OPERATOR (vala_assignment_operator_get_type ())

#define VALA_TYPE_ASSIGNMENT (vala_assignment_get_type ())
#define VALA_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ASSIGNMENT, ValaAssignment))
#define VALA_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ASSIGNMENT, ValaAssignmentClass))
#define VALA_IS_ASSIGNMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ASSIGNMENT))
#define VALA_IS_ASSIGNMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ASSIGNMENT))
#define VALA_ASSIGNMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ASSIGNMENT, ValaAssignmentClass))

typedef struct _ValaAssignment ValaAssignment;
typedef struct _ValaAssignmentClass ValaAssignmentClass;

#define VALA_TYPE_EXPRESSION_STATEMENT (vala_expression_statement_get_type ())
#define VALA_EXPRESSION_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION_STATEMENT, ValaExpressionStatement))
#define VALA_EXPRESSION_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION_STATEMENT, ValaExpressionStatementClass))
#define VALA_IS_EXPRESSION_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION_STATEMENT))
#define VALA_IS_EXPRESSION_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION_STATEMENT))
#define VALA_EXPRESSION_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION_STATEMENT, ValaExpressionStatementClass))

typedef struct _ValaExpressionStatement ValaExpressionStatement;
typedef struct _ValaExpressionStatementClass ValaExpressionStatementClass;

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;
#define _gee_collection_object_unref0(var) ((var == NULL) ? NULL : (var = (gee_collection_object_unref (var), NULL)))

#define VALA_TYPE_REFERENCE_TYPE (vala_reference_type_get_type ())
#define VALA_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceType))
#define VALA_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))
#define VALA_IS_REFERENCE_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REFERENCE_TYPE))
#define VALA_IS_REFERENCE_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REFERENCE_TYPE))
#define VALA_REFERENCE_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REFERENCE_TYPE, ValaReferenceTypeClass))

typedef struct _ValaReferenceType ValaReferenceType;
typedef struct _ValaReferenceTypeClass ValaReferenceTypeClass;

#define VALA_TYPE_ERROR_TYPE (vala_error_type_get_type ())
#define VALA_ERROR_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_TYPE, ValaErrorType))
#define VALA_ERROR_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_TYPE, ValaErrorTypeClass))
#define VALA_IS_ERROR_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_TYPE))
#define VALA_IS_ERROR_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_TYPE))
#define VALA_ERROR_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_TYPE, ValaErrorTypeClass))

typedef struct _ValaErrorType ValaErrorType;
typedef struct _ValaErrorTypeClass ValaErrorTypeClass;

struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
	gboolean (*is_instance_member) (ValaSymbol* self);
	gboolean (*is_class_member) (ValaSymbol* self);
};

struct _ValaPropertyAccessor {
	ValaSymbol parent_instance;
	ValaPropertyAccessorPrivate * priv;
};

struct _ValaPropertyAccessorClass {
	ValaSymbolClass parent_class;
};

struct _ValaPropertyAccessorPrivate {
	gboolean _readable;
	gboolean _writable;
	gboolean _construction;
	ValaBasicBlock* _entry_block;
	ValaBasicBlock* _exit_block;
	gboolean _automatic_body;
	ValaFormalParameter* _value_parameter;
	ValaDataType* _value_type;
	char* _cname;
	ValaBlock* _body;
};

struct _ValaStatementIface {
	GTypeInterface parent_iface;
};

typedef enum  {
	VALA_ASSIGNMENT_OPERATOR_NONE,
	VALA_ASSIGNMENT_OPERATOR_SIMPLE,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_OR,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_AND,
	VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR,
	VALA_ASSIGNMENT_OPERATOR_ADD,
	VALA_ASSIGNMENT_OPERATOR_SUB,
	VALA_ASSIGNMENT_OPERATOR_MUL,
	VALA_ASSIGNMENT_OPERATOR_DIV,
	VALA_ASSIGNMENT_OPERATOR_PERCENT,
	VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT,
	VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT
} ValaAssignmentOperator;


static gpointer vala_property_accessor_parent_class = NULL;

gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_property_accessor_get_type (void);
gpointer vala_basic_block_ref (gpointer instance);
void vala_basic_block_unref (gpointer instance);
GParamSpec* vala_param_spec_basic_block (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_basic_block (GValue* value, gpointer v_object);
gpointer vala_value_get_basic_block (const GValue* value);
GType vala_basic_block_get_type (void);
GType vala_formal_parameter_get_type (void);
GType vala_block_get_type (void);
#define VALA_PROPERTY_ACCESSOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_PROPERTY_ACCESSOR, ValaPropertyAccessorPrivate))
enum  {
	VALA_PROPERTY_ACCESSOR_DUMMY_PROPERTY
};
GType vala_member_get_type (void);
GType vala_property_get_type (void);
ValaProperty* vala_property_accessor_get_prop (ValaPropertyAccessor* self);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
GType vala_typesymbol_get_type (void);
gboolean vala_property_accessor_get_readable (ValaPropertyAccessor* self);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
const char* vala_symbol_get_name (ValaSymbol* self);
char* vala_property_accessor_get_cname (ValaPropertyAccessor* self);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
void vala_property_accessor_set_readable (ValaPropertyAccessor* self, gboolean value);
void vala_property_accessor_set_writable (ValaPropertyAccessor* self, gboolean value);
void vala_property_accessor_set_construction (ValaPropertyAccessor* self, gboolean value);
void vala_property_accessor_set_value_type (ValaPropertyAccessor* self, ValaDataType* value);
void vala_property_accessor_set_body (ValaPropertyAccessor* self, ValaBlock* value);
ValaPropertyAccessor* vala_property_accessor_new (gboolean readable, gboolean writable, gboolean construction, ValaDataType* value_type, ValaBlock* body, ValaSourceReference* source_reference);
ValaPropertyAccessor* vala_property_accessor_construct (GType object_type, gboolean readable, gboolean writable, gboolean construction, ValaDataType* value_type, ValaBlock* body, ValaSourceReference* source_reference);
void vala_code_visitor_visit_property_accessor (ValaCodeVisitor* self, ValaPropertyAccessor* acc);
static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaDataType* vala_property_accessor_get_value_type (ValaPropertyAccessor* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaBlock* vala_property_accessor_get_body (ValaPropertyAccessor* self);
static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
const char* vala_attribute_get_name (ValaAttribute* self);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
void vala_property_accessor_process_attributes (ValaPropertyAccessor* self);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
ValaSymbol* vala_semantic_analyzer_get_current_symbol (ValaSemanticAnalyzer* self);
void vala_semantic_analyzer_set_current_symbol (ValaSemanticAnalyzer* self, ValaSymbol* value);
gboolean vala_symbol_get_external_package (ValaSymbol* self);
gboolean vala_property_get_interface_only (ValaProperty* self);
gboolean vala_property_get_is_abstract (ValaProperty* self);
GType vala_object_type_symbol_get_type (void);
GType vala_interface_get_type (void);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
void vala_property_accessor_set_automatic_body (ValaPropertyAccessor* self, gboolean value);
ValaBlock* vala_block_new (ValaSourceReference* source_reference);
ValaBlock* vala_block_construct (GType object_type, ValaSourceReference* source_reference);
ValaMemberAccess* vala_member_access_new_simple (const char* member_name, ValaSourceReference* source_reference);
ValaMemberAccess* vala_member_access_construct_simple (GType object_type, const char* member_name, ValaSourceReference* source_reference);
GType vala_member_access_get_type (void);
GType vala_statement_get_type (void);
void vala_block_add_statement (ValaBlock* self, ValaStatement* stmt);
ValaReturnStatement* vala_return_statement_new (ValaExpression* return_expression, ValaSourceReference* source_reference);
ValaReturnStatement* vala_return_statement_construct (GType object_type, ValaExpression* return_expression, ValaSourceReference* source_reference);
GType vala_return_statement_get_type (void);
GType vala_assignment_operator_get_type (void);
ValaAssignment* vala_assignment_new (ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
ValaAssignment* vala_assignment_construct (GType object_type, ValaExpression* left, ValaExpression* right, ValaAssignmentOperator operator, ValaSourceReference* source_reference);
GType vala_assignment_get_type (void);
ValaExpressionStatement* vala_expression_statement_new (ValaExpression* expression, ValaSourceReference* source_reference);
ValaExpressionStatement* vala_expression_statement_construct (GType object_type, ValaExpression* expression, ValaSourceReference* source_reference);
GType vala_expression_statement_get_type (void);
gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self);
gboolean vala_property_accessor_get_construction (ValaPropertyAccessor* self);
ValaFormalParameter* vala_formal_parameter_new (const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
ValaFormalParameter* vala_formal_parameter_construct (GType object_type, const char* name, ValaDataType* parameter_type, ValaSourceReference* source_reference);
void vala_property_accessor_set_value_parameter (ValaPropertyAccessor* self, ValaFormalParameter* value);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym);
ValaFormalParameter* vala_property_accessor_get_value_parameter (ValaPropertyAccessor* self);
GeeList* vala_code_node_get_error_types (ValaCodeNode* self);
GType vala_reference_type_get_type (void);
GType vala_error_type_get_type (void);
gboolean vala_error_type_get_dynamic_error (ValaErrorType* self);
void vala_report_warning (ValaSourceReference* source, const char* message);
char* vala_code_node_to_string (ValaCodeNode* self);
static gboolean vala_property_accessor_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_property_accessor_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
void vala_code_node_set_parent_node (ValaCodeNode* self, ValaCodeNode* value);
void vala_symbol_set_owner (ValaSymbol* self, ValaScope* value);
ValaBasicBlock* vala_property_accessor_get_entry_block (ValaPropertyAccessor* self);
void vala_property_accessor_set_entry_block (ValaPropertyAccessor* self, ValaBasicBlock* value);
ValaBasicBlock* vala_property_accessor_get_exit_block (ValaPropertyAccessor* self);
void vala_property_accessor_set_exit_block (ValaPropertyAccessor* self, ValaBasicBlock* value);
gboolean vala_property_accessor_get_automatic_body (ValaPropertyAccessor* self);
static void vala_property_accessor_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



static gpointer _vala_code_node_ref0 (gpointer self) {
	return self ? vala_code_node_ref (self) : NULL;
}


char* vala_property_accessor_get_cname (ValaPropertyAccessor* self) {
	char* result;
	ValaTypeSymbol* t;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->_cname != NULL) {
		result = g_strdup (self->priv->_cname);
		return result;
	}
	t = _vala_code_node_ref0 (VALA_TYPESYMBOL (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_property_accessor_get_prop (self))));
	if (self->priv->_readable) {
		char* _tmp0_;
		char* _tmp1_;
		result = (_tmp1_ = g_strdup_printf ("%sget_%s", _tmp0_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) t), vala_symbol_get_name ((ValaSymbol*) vala_property_accessor_get_prop (self))), _g_free0 (_tmp0_), _tmp1_);
		_vala_code_node_unref0 (t);
		return result;
	} else {
		char* _tmp2_;
		char* _tmp3_;
		result = (_tmp3_ = g_strdup_printf ("%sset_%s", _tmp2_ = vala_symbol_get_lower_case_cprefix ((ValaSymbol*) t), vala_symbol_get_name ((ValaSymbol*) vala_property_accessor_get_prop (self))), _g_free0 (_tmp2_), _tmp3_);
		_vala_code_node_unref0 (t);
		return result;
	}
	_vala_code_node_unref0 (t);
}


ValaPropertyAccessor* vala_property_accessor_construct (GType object_type, gboolean readable, gboolean writable, gboolean construction, ValaDataType* value_type, ValaBlock* body, ValaSourceReference* source_reference) {
	ValaPropertyAccessor* self;
	self = (ValaPropertyAccessor*) vala_symbol_construct (object_type, NULL, source_reference);
	vala_property_accessor_set_readable (self, readable);
	vala_property_accessor_set_writable (self, writable);
	vala_property_accessor_set_construction (self, construction);
	vala_property_accessor_set_value_type (self, value_type);
	vala_property_accessor_set_body (self, body);
	return self;
}


ValaPropertyAccessor* vala_property_accessor_new (gboolean readable, gboolean writable, gboolean construction, ValaDataType* value_type, ValaBlock* body, ValaSourceReference* source_reference) {
	return vala_property_accessor_construct (VALA_TYPE_PROPERTY_ACCESSOR, readable, writable, construction, value_type, body, source_reference);
}


static void vala_property_accessor_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = (ValaPropertyAccessor*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_property_accessor (visitor, self);
}


static void vala_property_accessor_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaPropertyAccessor * self;
	self = (ValaPropertyAccessor*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_property_accessor_get_value_type (self), visitor);
	if (vala_property_accessor_get_body (self) != NULL) {
		vala_code_node_accept ((ValaCodeNode*) vala_property_accessor_get_body (self), visitor);
	}
}


void vala_property_accessor_process_attributes (ValaPropertyAccessor* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = _vala_code_node_ref0 ((ValaAttribute*) a_it->data);
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					if (vala_attribute_has_argument (a, "cname")) {
						char* _tmp0_;
						self->priv->_cname = (_tmp0_ = vala_attribute_get_string (a, "cname"), _g_free0 (self->priv->_cname), _tmp0_);
					}
				}
				_vala_code_node_unref0 (a);
			}
		}
	}
}


static gboolean vala_property_accessor_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaPropertyAccessor * self;
	gboolean result;
	ValaSymbol* old_symbol;
	self = (ValaPropertyAccessor*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		result = !vala_code_node_get_error ((ValaCodeNode*) self);
		return result;
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_property_accessor_process_attributes (self);
	if (!vala_code_node_check ((ValaCodeNode*) vala_property_accessor_get_value_type (self), analyzer)) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		result = FALSE;
		return result;
	}
	old_symbol = _vala_code_node_ref0 (vala_semantic_analyzer_get_current_symbol (analyzer));
	vala_semantic_analyzer_set_current_symbol (analyzer, (ValaSymbol*) self);
	if (!vala_symbol_get_external_package ((ValaSymbol*) vala_property_accessor_get_prop (self))) {
		gboolean _tmp0_;
		gboolean _tmp1_;
		_tmp0_ = FALSE;
		_tmp1_ = FALSE;
		if (vala_property_accessor_get_body (self) == NULL) {
			_tmp1_ = !vala_property_get_interface_only (vala_property_accessor_get_prop (self));
		} else {
			_tmp1_ = FALSE;
		}
		if (_tmp1_) {
			_tmp0_ = !vala_property_get_is_abstract (vala_property_accessor_get_prop (self));
		} else {
			_tmp0_ = FALSE;
		}
		if (_tmp0_) {
			ValaBlock* _tmp2_;
			char* _tmp3_;
			ValaMemberAccess* _tmp4_;
			ValaMemberAccess* ma;
			if (VALA_IS_INTERFACE (vala_symbol_get_parent_symbol ((ValaSymbol*) vala_property_accessor_get_prop (self)))) {
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Automatic properties can't be used in interfaces");
				result = FALSE;
				_vala_code_node_unref0 (old_symbol);
				return result;
			}
			vala_property_accessor_set_automatic_body (self, TRUE);
			vala_property_accessor_set_body (self, _tmp2_ = vala_block_new (vala_code_node_get_source_reference ((ValaCodeNode*) self)));
			_vala_code_node_unref0 (_tmp2_);
			ma = (_tmp4_ = vala_member_access_new_simple (_tmp3_ = g_strdup_printf ("_%s", vala_symbol_get_name ((ValaSymbol*) vala_property_accessor_get_prop (self))), vala_code_node_get_source_reference ((ValaCodeNode*) self)), _g_free0 (_tmp3_), _tmp4_);
			if (self->priv->_readable) {
				ValaReturnStatement* _tmp5_;
				vala_block_add_statement (vala_property_accessor_get_body (self), (ValaStatement*) (_tmp5_ = vala_return_statement_new ((ValaExpression*) ma, vala_code_node_get_source_reference ((ValaCodeNode*) self))));
				_vala_code_node_unref0 (_tmp5_);
			} else {
				ValaMemberAccess* _tmp6_;
				ValaAssignment* _tmp7_;
				ValaAssignment* assignment;
				ValaExpressionStatement* _tmp8_;
				assignment = (_tmp7_ = vala_assignment_new ((ValaExpression*) ma, (ValaExpression*) (_tmp6_ = vala_member_access_new_simple ("value", vala_code_node_get_source_reference ((ValaCodeNode*) self))), VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference ((ValaCodeNode*) self)), _vala_code_node_unref0 (_tmp6_), _tmp7_);
				vala_block_add_statement (vala_property_accessor_get_body (self), (ValaStatement*) (_tmp8_ = vala_expression_statement_new ((ValaExpression*) assignment, NULL)));
				_vala_code_node_unref0 (_tmp8_);
				_vala_code_node_unref0 (assignment);
			}
			_vala_code_node_unref0 (ma);
		}
	}
	if (vala_property_accessor_get_body (self) != NULL) {
		gboolean _tmp9_;
		_tmp9_ = FALSE;
		if (self->priv->_writable) {
			_tmp9_ = TRUE;
		} else {
			_tmp9_ = self->priv->_construction;
		}
		if (_tmp9_) {
			ValaFormalParameter* _tmp10_;
			vala_property_accessor_set_value_parameter (self, _tmp10_ = vala_formal_parameter_new ("value", vala_property_accessor_get_value_type (self), vala_code_node_get_source_reference ((ValaCodeNode*) self)));
			_vala_code_node_unref0 (_tmp10_);
			vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) vala_property_accessor_get_body (self)), vala_symbol_get_name ((ValaSymbol*) self->priv->_value_parameter), (ValaSymbol*) self->priv->_value_parameter);
		}
		vala_code_node_check ((ValaCodeNode*) vala_property_accessor_get_body (self), analyzer);
		{
			GeeList* _tmp11_;
			GeeIterator* _tmp12_;
			GeeIterator* _body_error_type_it;
			_body_error_type_it = (_tmp12_ = gee_iterable_iterator ((GeeIterable*) (_tmp11_ = vala_code_node_get_error_types ((ValaCodeNode*) vala_property_accessor_get_body (self)))), _gee_collection_object_unref0 (_tmp11_), _tmp12_);
			while (TRUE) {
				ValaDataType* body_error_type;
				if (!gee_iterator_next (_body_error_type_it)) {
					break;
				}
				body_error_type = (ValaDataType*) gee_iterator_get (_body_error_type_it);
				if (!vala_error_type_get_dynamic_error (VALA_ERROR_TYPE (body_error_type))) {
					char* _tmp14_;
					char* _tmp13_;
					vala_report_warning (vala_code_node_get_source_reference ((ValaCodeNode*) body_error_type), _tmp14_ = g_strdup_printf ("unhandled error `%s'", _tmp13_ = vala_code_node_to_string ((ValaCodeNode*) body_error_type)));
					_g_free0 (_tmp14_);
					_g_free0 (_tmp13_);
				}
				_vala_code_node_unref0 (body_error_type);
			}
			_gee_collection_object_unref0 (_body_error_type_it);
		}
	}
	vala_semantic_analyzer_set_current_symbol (analyzer, old_symbol);
	result = !vala_code_node_get_error ((ValaCodeNode*) self);
	_vala_code_node_unref0 (old_symbol);
	return result;
}


static void vala_property_accessor_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaPropertyAccessor * self;
	self = (ValaPropertyAccessor*) base;
	g_return_if_fail (old_type != NULL);
	g_return_if_fail (new_type != NULL);
	if (vala_property_accessor_get_value_type (self) == old_type) {
		vala_property_accessor_set_value_type (self, new_type);
	}
}


ValaProperty* vala_property_accessor_get_prop (ValaPropertyAccessor* self) {
	ValaProperty* result;
	ValaSymbol* _tmp0_;
	g_return_val_if_fail (self != NULL, NULL);
	result = (_tmp0_ = vala_symbol_get_parent_symbol ((ValaSymbol*) self), VALA_IS_PROPERTY (_tmp0_) ? ((ValaProperty*) _tmp0_) : NULL);
	return result;
}


ValaDataType* vala_property_accessor_get_value_type (ValaPropertyAccessor* self) {
	ValaDataType* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_value_type;
	return result;
}


void vala_property_accessor_set_value_type (ValaPropertyAccessor* self, ValaDataType* value) {
	ValaDataType* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_value_type = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_value_type), _tmp0_);
	if (value != NULL) {
		vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_value_type, (ValaCodeNode*) self);
	}
}


gboolean vala_property_accessor_get_readable (ValaPropertyAccessor* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_readable;
	return result;
}


void vala_property_accessor_set_readable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_readable = value;
}


gboolean vala_property_accessor_get_writable (ValaPropertyAccessor* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_writable;
	return result;
}


void vala_property_accessor_set_writable (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_writable = value;
}


gboolean vala_property_accessor_get_construction (ValaPropertyAccessor* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_construction;
	return result;
}


void vala_property_accessor_set_construction (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_construction = value;
}


ValaBlock* vala_property_accessor_get_body (ValaPropertyAccessor* self) {
	ValaBlock* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_body;
	return result;
}


void vala_property_accessor_set_body (ValaPropertyAccessor* self, ValaBlock* value) {
	ValaBlock* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_body = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_body), _tmp0_);
	if (self->priv->_body != NULL) {
		vala_symbol_set_owner ((ValaSymbol*) self->priv->_body, vala_symbol_get_scope ((ValaSymbol*) self));
	}
}


ValaBasicBlock* vala_property_accessor_get_entry_block (ValaPropertyAccessor* self) {
	ValaBasicBlock* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_entry_block;
	return result;
}


static gpointer _vala_basic_block_ref0 (gpointer self) {
	return self ? vala_basic_block_ref (self) : NULL;
}


void vala_property_accessor_set_entry_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_entry_block = (_tmp0_ = _vala_basic_block_ref0 (value), _vala_basic_block_unref0 (self->priv->_entry_block), _tmp0_);
}


ValaBasicBlock* vala_property_accessor_get_exit_block (ValaPropertyAccessor* self) {
	ValaBasicBlock* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_exit_block;
	return result;
}


void vala_property_accessor_set_exit_block (ValaPropertyAccessor* self, ValaBasicBlock* value) {
	ValaBasicBlock* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_exit_block = (_tmp0_ = _vala_basic_block_ref0 (value), _vala_basic_block_unref0 (self->priv->_exit_block), _tmp0_);
}


gboolean vala_property_accessor_get_automatic_body (ValaPropertyAccessor* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_automatic_body;
	return result;
}


void vala_property_accessor_set_automatic_body (ValaPropertyAccessor* self, gboolean value) {
	g_return_if_fail (self != NULL);
	self->priv->_automatic_body = value;
}


ValaFormalParameter* vala_property_accessor_get_value_parameter (ValaPropertyAccessor* self) {
	ValaFormalParameter* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_value_parameter;
	return result;
}


void vala_property_accessor_set_value_parameter (ValaPropertyAccessor* self, ValaFormalParameter* value) {
	ValaFormalParameter* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_value_parameter = (_tmp0_ = _vala_code_node_ref0 (value), _vala_code_node_unref0 (self->priv->_value_parameter), _tmp0_);
}


static void vala_property_accessor_class_init (ValaPropertyAccessorClass * klass) {
	vala_property_accessor_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_property_accessor_finalize;
	g_type_class_add_private (klass, sizeof (ValaPropertyAccessorPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_property_accessor_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_property_accessor_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->check = vala_property_accessor_real_check;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_property_accessor_real_replace_type;
}


static void vala_property_accessor_instance_init (ValaPropertyAccessor * self) {
	self->priv = VALA_PROPERTY_ACCESSOR_GET_PRIVATE (self);
}


static void vala_property_accessor_finalize (ValaCodeNode* obj) {
	ValaPropertyAccessor * self;
	self = VALA_PROPERTY_ACCESSOR (obj);
	_vala_basic_block_unref0 (self->priv->_entry_block);
	_vala_basic_block_unref0 (self->priv->_exit_block);
	_vala_code_node_unref0 (self->priv->_value_parameter);
	_vala_code_node_unref0 (self->priv->_value_type);
	_g_free0 (self->priv->_cname);
	_vala_code_node_unref0 (self->priv->_body);
	VALA_CODE_NODE_CLASS (vala_property_accessor_parent_class)->finalize (obj);
}


GType vala_property_accessor_get_type (void) {
	static GType vala_property_accessor_type_id = 0;
	if (vala_property_accessor_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaPropertyAccessorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_property_accessor_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaPropertyAccessor), 0, (GInstanceInitFunc) vala_property_accessor_instance_init, NULL };
		vala_property_accessor_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaPropertyAccessor", &g_define_type_info, 0);
	}
	return vala_property_accessor_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




