/*
 * ===========================
 * VDK Visual Development Kit
 * xdb subsystem
 * Version 0.1
 * Revision 0.0
 * April 2000
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _vdkxdbtable_h
#define _vdkxdbtable_h
#include <vdk/vdkstring.h>
#include <vdk/vdkprops.h>
#include <vdk/dlist.h>
#include <xdb/dbf.h>
#include <xdb/ndx.h>

// see into xdb/xdb/dbf.h
typedef xbSchema VDKXRecordTemplate ;

class VDKXDatabase;
class VDKXTableIndex;


typedef VDKList<VDKXTableIndex> IndexList;
typedef VDKListIterator<VDKXTableIndex> IndexListIterator;


/*!
  \class VDKXTable
  \brief Provides a xbDbf wrapper

  Since it is a derived class of xbDbf, most behaviours are inherited.
  Refer to xdb reference for further informations.
*/
class VDKXTable: public xbDbf
{
 protected:
  //! table name
  VDKString name;
  //! indexes list
  IndexList iList;
  void SetOrder(xbShort ndx);
  bool open;
  VDKXTableIndex* activeIndex;
 public:
  /*
    table properties
  */
  //! table size in records
  VDKReadOnlyValueProp<VDKXTable,xbLong> Size;

  /*! 
    sets/gets active index ordinal position.
    Setting Order to -1 no active index will be used.
  */
  VDKReadWriteValueProp<VDKXTable,xbShort> Order;
  /*! 
    sets/gets active index 
    Setting Order to NULL no active index will be used.
    \param name index file name
  */
  void SetOrder(char* name);
  /*!
    If false all record marked for deletion
    will be ignored on common table operations.
  */
  VDKReadWriteValueProp<VDKXTable,bool> ShowDeleted;
  //---------------------------------------------
  /*
    Internal use only (mainly for builder)
    User don't care.
  */
  VDKReadWriteValueProp<VDKXTable,bool> Active;
  //---------------------------------------------
  /*! 
    Constructor
    \param owner owner xdb
    \param name table name can be either with or without .dbf extension
  */

  VDKXTable(VDKXDatabase* owner, char* name);
  //! Destructor
  virtual ~VDKXTable();
  //! Returns table name
  VDKString Name() { return name; }
  /*!
    Creates a table
    \param tpl a VDKXRecordTemplate
    \param overwrite if true and table exist will be truncated and
    overwritten.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_FILE_EXISTS file exist and overwrite flag = false
    \arg XB_OPEN_ERROR couldn't open file
    \arg XB_NO_MEMORY mem allocation error
    \arg XB_WRITE_ERROR couldn't write on device
   */
  xbShort Create(VDKXRecordTemplate* tpl, bool overwrite = false);
  /*! 
    Opens table.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_OPEN_ERROR couldn't open file
    \arg XB_NO_MEMORY mem allocation error
    \arg XB_WRITE_ERROR couldn't write on device
    \arg XB_NOT_XBASE not an xdb file type
  */
  xbShort Open(void);
  /*! 
    Close table and all associated indexes.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_NOT_OPEN file wasn't opened
  */
  xbShort Close(void);
  /*!
    Removes an index from table
    \param name index name.
    Returns true on success
   */
  bool RemoveIndex(char* name);
  /*!
    Returns an index given a name
    \param name index name
    Returns (VDKXTableIndex*) NULL on failure
  */
  VDKXTableIndex* Index(char* name);
  /*!
    Returns an index given an ordinal position
    \param order index ordinal position
    Returns (VDKXTableIndex*) NULL on failure
  */
  VDKXTableIndex* Index(xbShort order);
  /*!
    Moves database pointer to first logical record.
    If there is no active index moves on first phisical record
    If ShowDeleted property is fale (default)moves to
    first logical/phisical undeleted record.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_INVALID_RECORD invalid record number
    \arg XB_LOCK_FAILED couldn't lock file
    \arg XB_NOT_OPEN file not open
    \arg XB_SEEK_ERROR seek routine error
    \arg XB_WRITE_ERROR couldn't write
   */
  xbShort First();
  /*!
    Moves database pointer to last logical record
    If there is no active index moves on last phisical record
    If ShowDeleted property is fale (default)moves to
    last logical/phisical undeleted record.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_INVALID_RECORD invalid record number
    \arg XB_LOCK_FAILED couldn't lock file
    \arg XB_NOT_OPEN file not open
    \arg XB_SEEK_ERROR seek routine error
    \arg XB_WRITE_ERROR couldn't write
   */
  xbShort Last();
  /*!
    Moves database pointer to next logical record
    If there is no active index moves on next phisical record
    If ShowDeleted property is fale (default)moves to
    next logical/phisical undeleted record.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_INVALID_RECORD invalid record number
    \arg XB_LOCK_FAILED couldn't lock file
    \arg XB_NOT_OPEN file not open
    \arg XB_SEEK_ERROR seek routine error
    \arg XB_WRITE_ERROR couldn't write
   */
  xbShort Next();
  /*!
    Moves database pointer to previous logical record
    If there is no active index moves on previous phisical record
    If ShowDeleted property is fale (default)moves to
    previous logical/phisical undeleted record.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_INVALID_RECORD invalid record number
    \arg XB_LOCK_FAILED couldn't lock file
    \arg XB_NOT_OPEN file not open
    \arg XB_SEEK_ERROR seek routine error
    \arg XB_WRITE_ERROR couldn't write
   */
  xbShort Prev();
  //! /internal
  IndexList* NdxList() { return &iList; }
  //! return if table is open or not
  bool IsOpen() { return open; }
};

/*!
  \class VDKXTableIndex
  \brief Provides a table index
*/
class VDKXTableIndex: public xbNdx
{
 protected:
  VDKString name,key;
  VDKXTable* table;
  bool open;
 public:
  /*!
    Constructor
    \param owner VDKXtable that owns index
    \param s index file name
  */
  VDKXTableIndex(VDKXTable* owner, char* s);
  //! Destructor
  virtual ~VDKXTableIndex();
  /*!
    Creates an index
    \param k index key (can be an expression as well)
    \param unique if true allows duplicated keys
    \param overwrite if true and table exist will be truncated and
    overwritten.
    Returns:
    \arg XB_NO_ERROR no errors
    \arg XB_OPEN_ERROR couldn't open file
    \arg XB_NO_MEMORY mem allocation error
    \arg XB_READ_ERROR couldn't read on device
    \arg XB_INVALID_KEY_EXPRESSION bad index key expression
    \arg XB_SEEK_ERROR seek error encountered
    \arg XB_NOT_OPEN database wasn't opened
  */
  xbShort Create(char* key,
		 bool unique = true , 
		 bool overwrite = false
		 );
  //! Open index
  xbShort Open(void);
  /*!
    Close index.
    \arg Programming note:
    Isn't necessary to explicitely close an index, closing the table
    will close all associated indexes as well.
  */
  xbShort Close(void);
  //! return index name
  VDKString Name() { return name; }
  //! return index key
  VDKString Key() { return key; }
  //! return if index is open or not
  bool IsOpen() { return open; }
};

#endif

