/*
 * channel.c:	Get and forward AC3 stream from DVB card to the
 *		S/P-DIF interface of sound cards supported by ALSA.
 *		(NO decoding!)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * Copyright (C) 2002-2004 Werner Fink, <werner@suse.de>
 * Copyright (C) 2003 Sven Goethel, <sven@jausoft.com>
 */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif
#include <stdio.h>
#include "types.h"
#include "channel.h"
#include "ac3.h"
#include "dts.h"
#include "lpcm.h"
#include "mp2.h"

// #define DEBUG_CHL
#ifdef  DEBUG_CHL
# define debug_chl(args...) fprintf(stderr, args)
#else
# define debug_chl(args...)
#endif

// --- cInStream : Get AC3 stream for redirecting it to  S/P-DIF of a sound card--------

const uint_32 cInStream::PS1magic = 0x000001bd;
uint_8  * cInStream::tsdata;
cBounce * cInStream::bounce;

cInStream::cInStream(int Pid, spdif *dev, ctrl_t &up, cBounce *bPtr)
#if defined(VDRVERSNUM) && VDRVERSNUM >= 10300
:cReceiver(0, -1, 1, Pid), cThread("bso(instream): Forwarding bitstream"),
 ctr(), wait(), spdifDev(dev), setup(up)
#else
:cReceiver(0, -1, 1, Pid), cThread(), ctr(), wait(), spdifDev(dev), setup(up)
#endif
{
    flags = 0;
    ctr.Lock();
    stream = NULL;
    ctr.Unlock();
    Apid = Pid;
    ResetScan(true);
    bounce = bPtr;
    bounce->bank(0);
    tsdata = setup.buf + TRANSFER_START;
    audioType = audioTypes[IEC_NONE];
}

cInStream::~cInStream(void)
{
    if (test_and_set_flag(DETACHED))
	Detach();
    ActivateInStream(false);
    bounce->flush();
    ResetScan(true);
    clear_flag(FAILED);
}

void cInStream::ActivateInStream(bool onoff)
{
    if (test_setup(CLEAR))
	onoff = false;

    debug_chl("%s %d (%s)\n", __FUNCTION__, __LINE__, onoff ? "on" : "off");

    if (onoff) {
	if (*spdifDev)
	    goto out;		// An other has opened the S/P-DIF
	if (test_setup(RESET)) {
	    clear_flag(FAILED);
	    clear_setup(RESET);
	}
	if (test_flag(FAILED) || test_flag(ACTIVE))
	    goto out;
	set_flag(ACTIVE);
	set_setup(LIVE);
	bounce->flush();
	Start();				// Warning: Start() sleeps 10ms after work
    } else {
	int n = 50;				// 500 ms

	clear_flag(ACTIVE);			// Set only above
	do {
	    bounce->flush();
	    bounce->signal();
	    wait.msec(10);			// More than 2ms
	    if (!Active())
		break;
	} while (test_flag(RUNNING) && (n-- > 0));
	Cancel(1);

	if (test_flag(RUNNING) || Active()) {
	    esyslog("INSTREAM: Forwarding bitstream thread was broken");
	    wait.msec(10);			// More than 2ms
	    ctr.Lock();
	    iec60958* curr = stream;
	    ctr.Unlock();
	    if (curr) {				// thread broken
		LOCK_THREAD;
		spdifDev->Close();
		clear_flag(BOUNDARY);
		curr->Reset();
		bounce->leaveio();
	    }
	    clear_flag(RUNNING);		// Should not happen
	}

	ResetScan(true);
	clear_setup(LIVE);
	ctr.Lock();
	stream = NULL;
	ctr.Unlock();
    }
out:
    return;
}

void cInStream::Action(void)
{
    set_flag(RUNNING);
    ctr.Lock();
    iec60958 *curr = stream;
    ctr.Unlock();

    realtime("INSTREAM: ");
    if (!curr) {
	esyslog("INSTREAM: no stream for spdif interface");
	set_flag(FAILED);
	goto out;
    } else {
	LOCK_THREAD;
	if (!spdifDev->Open(curr)) {
	    esyslog("INSTREAM: can't open spdif interface");
	    set_flag(FAILED);
	    goto out;
	}
    }
    debug_chl("%s %d\n", __FUNCTION__, __LINE__);

#if defined(VDRVERSNUM) && VDRVERSNUM < 10300
    dsyslog("INSTREAM: Forwarding bitstream thread started (pid=%d)", getpid());
#endif
    bounce->takeio();
    while (test_flag(ACTIVE)) {
	ssize_t len;

	if (test_setup(MUTE)) {
	    if (!test_flag(WASMUTED)) {
		LOCK_THREAD;
		spdifDev->Clear();
		clear_flag(BOUNDARY);
		bounce->flush();
		curr->Clear();
	    }
	    set_flag(WASMUTED);
	}

	if (!spdifDev->Synchronize(bounce))	// Wait on bounce->signal()
	    break;

	if (!test_flag(ACTIVE)) {
	    debug_chl("%s %d\n", __FUNCTION__, __LINE__);
	    break;
	}

	if (test_setup(MUTE)) {
	    if (!test_flag(WASMUTED)) {
		LOCK_THREAD;
		spdifDev->Clear();
		clear_flag(BOUNDARY);
		bounce->flush();
		curr->Clear();
	    }
	    set_flag(WASMUTED);
	    continue;
	}

	if ((len = bounce->fetch(tsdata, spdifDev->Available(TRANSFER_MEM))))
	    spdifDev->Forward(tsdata, len, bounce);
    }
    Lock();
    spdifDev->Close();
    clear_flag(BOUNDARY);
    curr->Reset();
    Unlock();
#if defined(VDRVERSNUM) && VDRVERSNUM < 10300
    dsyslog("INSTREAM: Forwarding bitstream thread ended (pid=%d)", getpid());
#endif
out:
    ctr.Lock();
    stream = NULL;
    ctr.Unlock();
    bounce->flush();
    bounce->leaveio();
    clear_flag(RUNNING);
}

void cInStream::Receive(uchar *b, int cnt)
{
    uint_8 off = 4;
    uint_8 start = 0;
    static uint_16 skip = 0;

    if (test_flag(DETACHED))
	goto out;

    if (!test_setup(ACTIVE)) {
	if (test_flag(ACTIVE))
	    Activate(false);
	goto out;
    }

    if (skip) {
	debug("cReplayOutSPDif::Play(skip=%d)\n", skip);
	skip--;
	goto out;
    }

    if (test_setup(MUTE)) {
	clear_flag(PAYSTART);
	goto out;
    }

    if (!b || *b != 0x47) {
	esyslog("INSTREAM: have seen broken TS packet");
	goto out;
    }

    if (cnt != TS_SIZE) {
	esyslog("INSTREAM: have seen broken TS packet");
	goto out;
    }

    start = (b[1] & PAY_START);

    // Start engine only if PES frame starts
    if (!test_flag(PAYSTART)) {
	if (!start)
	    goto out;
	set_flag(PAYSTART);
    }

    if (Apid != (((uint_16)(b[1]&0x1F)<<8)|b[2]))
	goto out;

#if 0
    if (b[1] & TS_ERROR)
	TSerr++;			// Skip?

    if ((b[3] ^ TScount) & CONT_CNT_MASK) {
	if (/* TScount != -1 && */ ((b[3] ^ (TScount + 1)) & CONT_CNT_MASK))
	    TScnterr++;			// Skip?
	TScount = (b[3] & CONT_CNT_MASK);
    }
#endif
    if (b[3] & ADAPT_FIELD) {
	off += (b[4] + 1);
	if (off > 187)
	    goto out;
    }

    if (!ScanTSforAudio(&b[off], TS_SIZE-off, (start != 0)))
	skip = 20;
out:
    return;
}

inline void cInStream::ResetScan(bool err)
{
    bfound = paklen = ptsoff = subfnd = suboff = 0;
    bytes = 9;
    memset(pts, 0, sizeof(pts));
    syncword = 0xffffffff;
    submag = 0xff;
    if (err)
	clear_flag(BOUNDARY);
}

inline bool cInStream::ScanTSforAudio(const uint_8 *buf, const int cnt, const bool wakeup)
{
    const uint_8 *const tail = buf + cnt;
    bool ret = true, search;
    ctr.Lock();
    iec60958 *curr = stream;
    ctr.Unlock();

resync:
    if (bfound < bytes) {
	//
	// Scan for payload length and PTS, the switch is used here
	// to be able to start at any offset (controlled by bfound)
	//
	switch (bfound) {
	case 0 ... 3:
	    //
	    // Scan for start sequence of the Private Stream 1 or Mpeg2 Audio
	    //
	    search = true;
	    while (search) {
		if (buf >= tail)
		    goto out;
		syncword = (syncword << 8) | *buf++;

		switch (syncword) {
		case PS1magic:				// Audio of PES Private Stream 1
		    if (curr == &mp2)
			goto out;
		    search = false;
		    set_flag(PS1AUDIO);
		    break;
		case 0x000001c0 ... 0x000001df:		// Audio of PES Mpeg Audio frame
		    if (!test_setup(MP2ENABLE))
			goto out;
		    if (curr && curr != &mp2)
			goto out;
		    search = false;
		    clear_flag(PS1AUDIO);
		    break;
		default:
		    break;
		}
	    }
	    bfound = 4;
	case 4:
	    if (buf >= tail)
		goto out;
	    paklen  = (uint_16)(*buf) << 8;
	    buf++;
	    bfound++;
	case 5:
	    if (buf >= tail)
		goto out;
	    paklen |= *buf;
	    paklen += 6;				// Full packet length
	    buf++;
	    bfound++;
	case 6:
	    if (buf >= tail)
		goto out;
	    if ((*buf & 0xC0) != 0x80) {		// First mpeg2 flag byte
		ResetScan();				// Reset for next scan
		esyslog("INSTREAM: broken TS stream");
		goto resync;				// Check for rest
	    }
	    if (*buf & 0x04)				// Is the sub stream aligned?
		set_flag(BOUNDARY);
	    buf++;
	    bfound++;
	case 7:
	    if (buf >= tail)
		goto out;
	    memset(pts, 0, sizeof(pts));
	    if (*buf & 0xC0) {				// Second mpeg2 flag byte (PTS/DTS only)
		bytes = 14;
		set_flag(BOUNDARY);
	    } else {
		bytes = 9;
	    }
	    buf++;
	    bfound++;
	case 8:
	    if (buf >= tail)
		goto out;
	    ptsoff = *buf;				// Third mpeg2 byte (PTS offset)
	    buf++;
	    bfound++;
	    if (bytes <= 9)				// No PTS data: out here
		break;
	case 9 ... 13:					// Save PTS
	    do {
		if (buf >= tail)
		    goto out;
		pts[bfound - 9] = *buf;
		buf++;
		bfound++;
		ptsoff--;
	    } while (bfound < 14);
            set_flag(SET_PTS);
	default:
	    break;
	}
    }

    //
    // Start only at a logical begin of a sub stream
    //
    if (!test_flag(BOUNDARY)) {
	ResetScan();
	goto out;
    }

    //
    // Skip PTS
    //
    while (ptsoff) {
	register int skip;
	if ((skip = tail - buf) <= 0)
	    goto out;
	if (skip > ptsoff)
	    skip = ptsoff;
	buf    += skip;
	bfound += skip;
	ptsoff -= skip;
    }

    if (!curr) {
	if (test_flag(PS1AUDIO)) {			// Scan for DVB/DVD data
	    //
	    // BOUNDARY is set and stream is NULL at first time therefore we
	    // are able to scan for magics in payload due the data are aligned
	    // (start at first payload byte).  For Radio stations using DTS ...
	    //
	    switch (subfnd) {
	    case 0:
		if (buf >= tail)
		    goto out;
		if ((submag = *buf) == 0x0b) {		// DVB AC3 data stream
		    curr = &ac3;
		    curr->Reset();
		    curr->isDVD = false;
		    curr->track = 0x0b;
		    clear_setup(AUDIO);
		    clear_setup(MP2);
		    audioType = audioTypes[IEC_AC3];
		    break;				// found, skip rest
		}
		if ((submag = *buf) == 0x7f) {		// DVB DTS data stream (??)
		    curr = &dts;
		    curr->Reset();
		    curr->isDVD = false;
		    curr->track = 0x7f;
		    clear_setup(AUDIO);
		    clear_setup(MP2);
		    audioType = audioTypes[IEC_DTS];
		    break;				// found, skip rest
		}
		buf++; subfnd++; bfound++;
	    case 1:
		if (buf >= tail)
		    goto out;
		buf++; subfnd++; bfound++;
	    case 2:
		if (buf >= tail)
		    goto out;
		suboff = (uint_16)(*buf) << 8;
		buf++; subfnd++; bfound++;
	    case 3:
		if (buf >= tail)
		    goto out;
		suboff |= *buf;
		buf++; subfnd++; bfound++;
		if (submag >= 0x80 && submag <= 0x87) {	// DVD AC3 data stream may used over DVB (??)
		    curr = &ac3;
		    curr->Reset();
		    curr->isDVD = true;
		    curr->track = (submag & 7) + 1;
		    clear_setup(AUDIO);
		    clear_setup(MP2);
		    audioType = audioTypes[IEC_AC3];
		    break;				// found, skip rest
		}
		if (submag >= 0x88 && submag <= 0x8f) {	// DVD DTS data stream may used over DVB (??)
		    curr = &dts;
		    curr->Reset();
		    curr->isDVD = true;
		    curr->track = (submag & 7) + 1;
		    clear_setup(AUDIO);
		    clear_setup(MP2);
		    audioType = audioTypes[IEC_DTS];
		    break;				// found, skip rest
		}
	    case 4: case 5:				// Never heard about linear PCM over DVB
	    default:
		ResetScan();
		goto out;
		break;
	    }
	} else {
	    //
	    // At BOUNDARY scanning for Mpeg2 Audio data start is not needed
	    //
	    curr = &mp2;
	    curr->Reset();
	    curr->isDVD = false;
	    curr->track = (((uint_8)(syncword)) & 7) + 1;
	    set_setup(AUDIO);
	    set_setup(MP2);
	    audioType = audioTypes[IEC_MP2];
	}
	ctr.Lock();
	stream = curr;
	ctr.Unlock();
    } else if (test_flag(PS1AUDIO) && curr && curr->isDVD) {
	// Skip sub stream head for DVD case
	suboff = 4;
	switch (*buf) {
	case 0x80 ... 0x87:
	    if (curr == &ac3 && curr->track == (*buf & 7) + 1)
		break;
	case 0x88 ... 0x8f:
	    if (curr == &dts && curr->track == (*buf & 7) + 1)
		break;
	default:
	    ResetScan();
	    goto out;
	    break;
	}
    }

    //
    // We have to had a stream behind this point
    //
    if (!curr) {
	ResetScan();
	goto out;
    }

    //
    // Start or continue after mute the forwarding thread
    // only on PES boundary and if we got a Present Time
    // Stamp (PTS). With this we should get as fast as
    // possible into A/V sync.
    //
    if (!test_flag(RUNNING)) {
	if (!wakeup || !test_flag(SET_PTS)) {
	    ResetScan();
	    goto out;
	}
	ActivateInStream(true);
	clear_flag(WASMUTED);
    } else if (test_flag(WASMUTED)) {
	if (!wakeup || !test_flag(SET_PTS)) {
	    ResetScan();
	    goto out;
	}
	clear_flag(WASMUTED);
    }

    //
    // Skip substream offset, if any
    //
    while (suboff) {
	register int skip;
	if ((skip = tail - buf) <= 0)
	    goto out;
	if (skip > suboff)
	    skip = suboff;
	buf    += skip;
	bfound += skip;
	ptsoff -= skip;
    }

    //
    // PTS given, remember value than
    //
    if (test_and_clear_flag(SET_PTS) && (bounce->getused() == 0))
	curr->pts.mark(pts);

    //
    // Send data to payload scanner
    //
    while (bfound < paklen) {
	register int transmit, pay;
	bool start;

	if ((transmit = tail - buf) <= 0)
	    goto out;
	if (transmit > (pay = paklen - bfound))	// The rest is handled below
	    transmit = pay;
	bfound += transmit;

	//
	// Check if we have one or more data frames
	// within the submitted payload
	//
	start = curr->Count(buf, buf+transmit);

	// Submit data
	ret  = bounce->store(buf, transmit, start);
	buf += transmit;
    }

    //
    // Bytes found are equal to packet length if we have found the PES packet,
    // for the rest restart scanning.
    //
    if (paklen && (bfound == paklen)) {
	ResetScan(false);				// Reset for next scan
	if (buf >= tail)
	    goto out;
	goto resync;				// Check the rest for next frame
    }
out:
    return ret;
}

// --- cChannelOutSPDif : Live AC3 stream over S/P-DIF of a sound card with ALSA ---------
cBounce * cChannelOutSPDif::bounce;
const char * cChannelOutSPDif::audioType = audioTypes[IEC_NONE];
uint_16 cChannelOutSPDif::Apid = 0x1FFF;

cChannelOutSPDif::cChannelOutSPDif(spdif &dev, ctrl_t &up, cBounce * bPtr, const char *script)
#if defined(VDRVERSNUM) && VDRVERSNUM >= 10300
:cStatus(), cThread("bso(channelout): Switching bitstream"),
 Channel(NULL), wait(), SPDIFmute(script), spdifDev(&dev), setup(up)
#else
:cStatus(), cThread(), Channel(NULL), wait(), SPDIFmute(script), spdifDev(&dev), setup(up)
#endif
{
    in = NULL;
    replaying = false;
    bounce = bPtr;
    Player = NULL;
    Activate(true);
}

cChannelOutSPDif::~cChannelOutSPDif(void)
{
    Activate(false);
    if (in)
	AttachReceiver(false);
}

void cChannelOutSPDif::Activate(bool onoff)
{
    if (test_setup(CLEAR))
	onoff = false;

    if (onoff) {
	if (test_flag(ACTIVE))
	    goto out;
	set_flag(ACTIVE);
	Start();				// Warning: Start() sleeps 10ms after work
    } else {
	int n = 50;				// 500 ms

	clear_flag(ACTIVE);			// Set only above
	do {
	    ctrl.Lock();
	    watch.Broadcast();
	    ctrl.Unlock();
	    pthread_yield();
	    wait.msec(10);			// More than 2ms
	    if (!Active())
		break;
	} while (test_flag(RUNNING) && (n-- > 0));
	Cancel(1);

	if (test_flag(RUNNING) || Active()) {
	    esyslog("CHANNELOUT: Switching bitstream thread broken");
	    ctrl.Unlock();
	    clear_flag(RUNNING);		// Should not happen
	}
    }
out:
    return;
}

void cChannelOutSPDif::Action(void)
{
    set_flag(RUNNING);
#if defined(VDRVERSNUM) && VDRVERSNUM < 10300
    dsyslog("CHANNELOUT: Switching bitstream thread started (pid=%d)\n", getpid());
#endif

    ctrl.Lock();
    while (test_flag(ACTIVE)) {
	uint_16 apid;
	const char* type;

	watch.TimedWait(ctrl, 200);			// 200ms

	if (test_setup(CLEAR))
	    break;

	if (!test_flag(ACTIVE))
	    break;

	if (!in || !Channel)
	    continue;

	if (!GetCurrentAudioTrack(apid, type))		// Get audio pid and type
	    continue;

	if (strcmp(type, audioType) != 0)		// The green button switch
	    continue;					// changes languages only

	if (Apid == apid)
	    continue;

	AudioSwitch(apid, type);
    }
    ctrl.Unlock();

#if defined(VDRVERSNUM) && VDRVERSNUM < 10300
    dsyslog("CHANNELOUT: Switching bitstream thread ended (pid=%d)\n", getpid());
#endif
    clear_flag(RUNNING);
}

void cChannelOutSPDif::AudioSwitch(uint_16 apid, const char* type)
{
    sw.Lock();
    if (!test_setup(ACTIVE) && test_flag(ACTIVE))
	Activate(false);

    if (in)
	AttachReceiver(false);

    if (replaying) {
	if (Player) {
	    int numtracks = Player->NumAudioTracks();
	    int num = (int)apid - 1;

	    if (0 <= num && num < numtracks)
		Player->SetAudioTrack(num);
	}
    } else {
	Apid = apid;
	audioType = type;

	if (!test_setup(MP2ENABLE) && (strcmp(audioType, audioTypes[IEC_AC3]) != 0))
	    goto out;

	if (Apid && Apid < 0x1FFF)
	    AttachReceiver(true);
	else
	    test_setup(RESET);
    }
out:
    sw.Unlock();
    IfNeededMuteSPDIF();	// On close the S/P-DIF is not muted anymore
    return;
}

void cChannelOutSPDif::AttachReceiver(bool onoff)
{
    cDevice *PrimaryDevice = cDevice::PrimaryDevice();

    if (test_setup(CLEAR))
	onoff = false;

    if (in) {
	if (PrimaryDevice)
	    PrimaryDevice->Detach(in);
	delete in;
	in = NULL;
	bounce->flush();
	Apid = 0x1FFF;
	audioType = audioTypes[IEC_NONE];
    }

    if (!onoff)
	goto out;

    if (!PrimaryDevice)
	goto out;

    in = new cInStream(Apid, spdifDev, setup, bounce);
    if (!in) {
	esyslog("ERROR: out of memory");
	Apid = 0x1FFF;
	audioType = audioTypes[IEC_NONE];
	goto out;
    }
    PrimaryDevice->AttachReceiver(in);
out:
    IfNeededMuteSPDIF();	// On close the S/P-DIF is not muted anymore
    return;
}

void cChannelOutSPDif::ChannelSwitch(const cDevice *Device, int channelNumber)
{
    cDevice *PrimaryDevice = cDevice::PrimaryDevice();
    uint_16 apid;
    const char * type;

    if (!PrimaryDevice)
	goto out;

    // LiveView is only on the primary device possible, isn't it?
    if (!Device->IsPrimaryDevice())
	goto out;

    // Sanity check not be attached in transfer mode.
    // In this case the replay part should take the data
    if (PrimaryDevice != cDevice::ActualDevice())
	goto out;

    // Next sanity check not be attached in transfer mode
    // from the primary device to the primary device.
    // As metioned above the replay part should take the data
    if (PrimaryDevice == cTransferControl::ReceiverDevice())
	goto out;

    // If we're noticed only for channel switch we're done.
    // Also during a replay we're done.
    if (!channelNumber || replaying) {
	//
	// Problem: Channel switches for recording on the same
	// transponder will cause a small audio pause.
	//
	if (in)
	    AttachReceiver(false);
	Channel = NULL;			// Reset Channel
	goto out;
    }

    // Now get the current live channel but _not_ the given
    // channel number (maybe this one is for recording).
    Channel = Channels.GetByNumber(cDevice::CurrentChannel());

    // Oops, no channel is no audio
    if (!Channel)
	goto out;

    Apid = 0x1FFF;			// Reset Apid
    audioType = audioTypes[IEC_NONE];	// ... and its type
    if (!GetCurrentAudioTrack(apid, type, Channel))
	goto out;

    AudioSwitch(apid, type);		// Set Apid and its type
out:
    return;
}

void cChannelOutSPDif::Replaying(const cControl *Control, const char *Name)
{
    replaying = (Name != NULL);
    Player = ((cControlPerm *)Control)->Player();

    if (replaying) {
	if (in)
	    AttachReceiver(false);
	Channel = NULL;			// Reset Channel
    }
    //else
    //	Do nothing here. Attaching a receiver will be done by ChannelSwitch.
    //	Anything else will cause trouble in transfer and replay mode.
    //
    IfNeededMuteSPDIF();	// On close the S/P-DIF is not muted anymore
}

void cChannelOutSPDif::IfNeededMuteSPDIF(void)
{
    if (SPDIFmute) {    // Normal TV mode
	char *cmd = NULL;
	asprintf(&cmd, "%s %s 2> /dev/null", SPDIFmute, test_setup(MUTE) ? "mute" : "unmute");
	if (cmd) {
	    if (test_setup(MP2ENABLE))
		setenv("loop", "off", 1);
	    else
		setenv("loop", "on", 1);
	    system(cmd);
	    free(cmd);
	    cmd = NULL;
	}
    }
}

bool cChannelOutSPDif::GetCurrentAudioTrack(uint_16 &apid, const char* &type)
{
    bool ret = false;

    if (replaying)
	goto out;

    if (!Channel)
	goto out;

    ret = GetCurrentAudioTrack(apid, type, Channel);
out:
    return ret;
}

bool cChannelOutSPDif::GetCurrentAudioTrack(uint_16 &apid, const char* &type, cChannel *channel)
{
    cDevice *PrimaryDevice = cDevice::PrimaryDevice();
    uint_16 mpid = 0x1FFF;
    uint_16 dpid = 0x1FFF;
    int CurrentAudioTrack = 0;
    bool ret = false;

    type = audioTypes[IEC_NONE];
    apid = 0x1FFF;
    if (!channel || !PrimaryDevice)
	goto out;

    if (PrimaryDevice->GetAudioTracks(&CurrentAudioTrack) == NULL)
	goto out;

    switch (CurrentAudioTrack) {
    case 0:
	dpid = channel->Dpid1();
	mpid = channel->Apid1();
	break;
    case 1:
	dpid = channel->Dpid2();
	mpid = channel->Apid2();
	break;
    default:
	break;
    }

    if (dpid == 0 || dpid >= 0x1FFF) {
	apid = mpid;
	type = audioTypes[IEC_MP2];
    } else {
	if (strcmp(audioType, audioTypes[IEC_MP2]) == 0) {
	    apid = mpid;
	    type = audioTypes[IEC_MP2];
	} else {
	    apid = dpid;
	    type = audioTypes[IEC_AC3];
	}
    }

    ret = (apid && apid < 0x1FFF);
out:
    return ret;
}

bool cChannelOutSPDif::AudioTrack(int num, const char* type, int &apid)
{
    if (replaying) {
	if (Player) {
	    int numtracks = Player->NumAudioTracks();
	    if (0 <= num && num < numtracks && (strcmp(type, audioTypes[IEC_AC3]) != 0))
		apid = num+1;
	    else
		apid = 0x1FFF;
	}
    } else {
	apid = 0x1FFF;

	if (!Channel)
	   goto out;

	switch (num) {
	case 0:
	    apid = (strcmp(type, audioTypes[IEC_AC3]) == 0) ? Channel->Dpid1() : Channel->Apid1();
	    break;
	case 1:
	    apid = (strcmp(type, audioTypes[IEC_AC3]) == 0) ? Channel->Dpid2() : Channel->Apid2();
	    break;
	default:
	    break;
	}
    }
out:
    return (apid && apid < 0x1FFF);
}

uint_16 cChannelOutSPDif::AudioPid(void)
{
    if (replaying) {
	if (Player) {
	    int apid;
	    Player->GetAudioTracks(&apid);
	    return apid+1;
	}
    } else if (in)
	return in->AudioPid();
	
    return Apid;
}

const char* cChannelOutSPDif::AudioType(void)
{
    if (replaying)
	return audioTypes[IEC_NONE];
    if (in)
	return in->AudioType();
    return audioType;
}
