/*
 * mp2.c:	Use the S/P-DIF interface for redirecting MP2 stream
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * Copyright (C) 2003,2004 Werner Fink, <werner@suse.de>
 * Copyright (C) 2003 Sven Goethel, <sven@jausoft.com>
 */

#include <vdr/thread.h>
#include "types.h"
#include "mp2.h"
#include "mp2_decoder.h"

//#define DEBUG_MP2
#ifdef  DEBUG_MP2
# define debug_mp2(args...) fprintf(stderr, args)
#else
# define debug_mp2(args...)
#endif

// --- cMP2 : Scanning MP2 stream for counting and decoding into PCM frames ------------

cMP2 mp2(48000);

const uint_32 cMP2::samplerate_table[3] = {44100, 48000, 32000};
const uint_16 cMP2::magic = 0xffe0;

cMP2::cMP2(unsigned int rate)
: iec60958(rate, SPDIF_BURST_SIZE, 0)
{
}

cMP2::~cMP2(void)
{
    if (Decoder) Release();
}

const bool cMP2::Initialize(void)
{
    return (Decoder = new MP2Decoder) != NULL;
}

const void cMP2::Release(void)
{
    if (Decoder) delete Decoder;
    Decoder = NULL;
}

// This function requires two arguments:
//   first is the start of the data segment
//   second is the tail of the data segment
const frame_t & cMP2::Frame(const uint_8 *&out, const uint_8 *const tail)
{
    play.burst = (uint_32 *)0;
    play.size  = 0;

    if (!Decoder)
	goto done;

resync:
    //
    // First try to empty the decoders buffer and if this fails we
    // refill this buffer due pushData() checks for MP2 frame start
    //
    if (!Decoder->decode()) {
	register int avail, used;
	if ((avail = tail - out) <= 0)
	    goto done;		// Nothing to refill, we are done

	if ((used = Decoder->pushData(out, avail)) > 0) {
	    out += used;
	    goto resync;	// Refilled, try to decode
	} else
	    goto done;		// Error, avoid loop (should not happen)
    }

    if (Decoder->pcm_netto_size > 0) { 
	memcpy(&current[0], Decoder->lpcmFrame().Data, Decoder->pcm_netto_size);

	swab(payload, payload, Decoder->pcm_netto_size);

	play.burst = (uint_32 *)(&current[0]);
	play.size  = Decoder->pcm_netto_size;
	play.pay   = Decoder->pcm_netto_size;

	Switch();

	last.burst = (uint_32 *)(&remember[0]);
	last.size  = play.size;
	last.pay   = play.pay;
    }

done:
    return play;
}

const bool cMP2::Count(const uint_8 *buf, const uint_8 *const tail)
{
    bool ret = false;

resync:
    if (c.bfound < 3) {
	switch(c.bfound) {
	case 0 ... 1:
	    while((c.syncword & magic) != magic) {
		if (buf >= tail)
		    goto out;
		c.syncword = (c.syncword << 8) | *buf++;
	    }
	    {
		bool ext   = ((c.syncword & 0x0010) >> 4) == 0;
		bool lsf   = ((c.syncword & 0x0008) >> 3) == 0;
		char layer = 4 - ((c.syncword & 0x0006) >> 1);

		// Sanity check for mpeg audio
		if ((layer == 4) || (!lsf && ext)) {	// Combination
		    c.bfound = 0;
		    c.syncword = 0x001f;
		    goto resync;			// ... not valid
		}

		c.fsize = 16 * ((layer == 1) ? 12 : (((layer == 3) && lsf) ? 18 : 36));
		debug_mp2("count_mp2_frame: %d %d\n", layer, c.fsize);
	    }
	    c.bfound = 2;
	case 2:
	    if (buf >= tail)
		goto out;

	    // Sanity checks for mpeg audio
	    if ((((*buf) & 0xf0) >> 4) == 15) {		// Bit rate index
		c.bfound = 0;
		c.syncword = 0x001f;
		goto resync;				// ... not valid
	    }
	    if ((((*buf) & 0x0c) >> 2) == 3) {		// Sample rate index
		c.bfound = 0;
		c.syncword = 0x001f;
		goto resync;				// ... not valid
	    }

	    // Not used
	    // uint_32 rate = samplerate_table[(ub & 0x0c) >> 2];

	    buf++;
	    c.bfound++;
	default:
	    break;
	}
    }

    while (c.bfound < c.fsize) {
	register int skip, pay;
	if ((skip = tail - buf) <= 0)
	    goto out;
	if (skip > (pay = c.fsize - c.bfound))
	    skip = pay;
	c.bfound += skip;
	buf	 += skip;
    }

    if (c.fsize && (c.bfound == c.fsize)) {
	c.bfound = 0;
	c.syncword = 0x001f;
	ret = true;
	debug_mp2("count_mp2_frame: frame crossed\n");
	if (buf >= tail)
	    goto out;
	goto resync;
    }
out:
    return ret;
}
