/*
 * MP3/MPlayer plugin to VDR (C++)
 *
 * (C) 2001-2004 Stefan Huelswitt <s.huelswitt@gmx.de>
 *
 * This code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 */

#include <ctype.h>
#include <dirent.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <vdr/tools.h>
#include <vdr/config.h>

#include "common.h"
#include "data.h"

// ----------------------------------------------------------------

const char *mountscript = "mount.sh";

char *Quote(const char *str)
{
  char *nstr=MALLOC(char,strlen(str)*2);
  char *p=nstr;
  while(*str) {
    switch(*str) {
      case '$':  // dollar
      case '\\': // backslash
      case '\"': // double quote
      case '`':  // back tick
                 *p++='\\'; break;
      }
    *p++=*str++;
    }
  *p=0;
  return nstr;
}

char *AddPath(const char *dir, const char *filename)
{
  char *name=0;
  asprintf(&name,"%s/%s",dir,filename);
  return name;
}

bool CheckVDRVersion(int Version, int Major, int Minor, const char *text)
{
  static char vv[] = VDRVERSION;
  int version, major, minor;
  if(sscanf(vv,"%d.%d.%d",&version,&major,&minor)==3) {
    if(version<Version ||
       (version==Version && major<Major) ||
       (version==Version && major==Major && minor<Minor)) {
      if(text) {
        esyslog("ERROR: %s plugin needs at least VDR version %d.%d.%d",text,Version,Major,Minor);
        fprintf(stderr,"%s plugin needs at least VDR version %d.%d.%d\n",text,Version,Major,Minor);
        }
      return false;
      }
    }
  else esyslog("ERROR: cannot parse VDR version string '%s'",vv);
  return true;
}

// -- cScanDir --------------------------------------------------------------

bool cScanDir::ScanDir(cFileSource *src, const char *subdir, eScanType type, const char *spec, const char *excl, bool recursiv)
{
  bool result=true;
  char *cmd=0, *dir=0, *s=0, *e=0, tc;

  switch(type) {
    default:
    case stFile: tc='f'; break;
    case stDir:  tc='d'; break;
    }
  if(subdir) asprintf(&dir,"%s/%s",src->BaseDir(),subdir); else asprintf(&dir,"%s",src->BaseDir());
  if(spec) asprintf(&s,"-iname \"%s\"",QuoteString(spec));
  if(excl) asprintf(&e,"-not -iname \"%s\"",QuoteString(excl));
  asprintf(&cmd, "find \"%s\" -follow -type %c %s %s %s 2>/dev/null | sort -df",
             QuoteString(dir), tc, s?s:"", e?e:"", recursiv?"":"-maxdepth 1");

  FILE *p=popen(cmd,"r");
  if(p) {
    int len=strlen(dir);
    char *s;
    while((s=readline(p))!=0) {
      char *ss=strstr(s,dir);
      if(ss) { s=ss+len; if(*s=='/') s++; }
      if(*s) DoItem(src,subdir,s);
      }
    pclose(p);
    }
  else result=false;

  free(cmd); free(dir); free(s); free(e);
  return result;
}

char *cScanDir::QuoteString(const char *str)
{
  static char *nstr=0;
  free(nstr); nstr=Quote(str);
  return nstr;
}

// -- cFileObj --------------------------------------------------------------

cFileObj::cFileObj(cFileSource *Source, const char *Subdir, const char *Name, const eObjType Type)
{
  path=fpath=0;
  source=Source;
  subdir=Subdir ? strdup(Subdir):0;
  name=Name ? strdup(Name):0;
  type=Type;
  Set();
}

cFileObj::cFileObj(const cFileObj *obj)
{
  path=fpath=0;
  source=obj->source;
  subdir=obj->subdir ? strdup(obj->subdir):0;
  name=obj->name ? strdup(obj->name):0;
  type=obj->type;
  Set();
}

cFileObj::~cFileObj()
{
  free(name);
  free(subdir);
  free(path);
  free(fpath);
}

void cFileObj::SetName(const char *Name)
{
  free(name);
  name=Name ? strdup(Name):0;
  Set();
}

void cFileObj::SplitAndSet(const char *Path)
{
  free(subdir); subdir=0;
  char buff[PATH_MAX+1];
  char *s;
  if(Path[0]=='/') s=realpath(Path,buff);
  else {
    char *p;
    MakeFullName(&p,Path);
    s=realpath(p,buff);
    free(p);
    }
  if(!s) {
    if(errno!=ENOENT && errno!=ENOTDIR)
      esyslog("ERROR: realpath: %s: %s",Path,strerror(errno));
    strn0cpy(buff,Path,sizeof(buff));
    }
  if(buff[0]=='/') {
    const int l=strlen(source->BaseDir());
    if(!strncasecmp(buff,source->BaseDir(),l) && buff[l]=='/')
      strcpy(buff,buff+l+1);
    else
      strcpy(buff,buff+1);
    }
  s=rindex(buff,'/');
  if(s) {
    const int l=s-buff+1;
    subdir=MALLOC(char,l);
    if(subdir) strn0cpy(subdir,buff,l);
    SetName(s+1);
    }
  else
    SetName(buff);
}

void cFileObj::Set(void)
{
  free(path); path=0;
  asprintf(&path,subdir ? "%2$s/%1$s":"%s",name,subdir);
  free(fpath); fpath=0;
  MakeFullName(&fpath,name);
}

void cFileObj::MakeFullName(char **fp, const char *Name)
{
  asprintf(fp,subdir ? "%1$s/%3$s/%2$s":"%s/%s",source->BaseDir(),Name,subdir);
}

bool cFileObj::Exists(void)
{
  if(type==otFile) {
    struct stat64 ds;
    if(!stat64(fpath,&ds) &&
       S_ISREG(ds.st_mode) &&
       !access(fpath,R_OK)) return true;
    }
  return false;
}

bool cFileObj::TestName(const char *newName)
{
  bool r=false;
  if(type==otFile) {
    char *fname;
    MakeFullName(&fname,newName);
    if(access(fname,F_OK)==0) r=true;
    free(fname);
    }
  return r;
}

bool cFileObj::Rename(const char *newName)
{
  bool r=false;
  if(type==otFile) {
    char *fname;
    MakeFullName(&fname,newName);
    if(access(fname,F_OK) && (!rename(fpath,fname))) {
      SetName(newName);
      r=true;
      }
    free(fname);
    }
  return r;
}

bool cFileObj::Create(const char *newName)
{
  bool r=false;
  if(type==otFile) {
    char *fname;
    MakeFullName(&fname,newName);
    FILE *newf;
    if(access(fname,F_OK) && (newf=fopen(fname,"w"))) {
      fclose(newf);
      SetName(newName);
      r=true;
      }
    free(fname);
    }
  return r;
}

bool cFileObj::Delete(void)
{
  if(type==otFile && !unlink(fpath)) return true;
  return false;
}

// -- cDirList --------------------------------------------------------------

bool cDirList::Load(cFileSource *src, const char *subdir)
{
  bool res=false;
  Clear();
  if(subdir) Add(new cFileObj(src,subdir,"..",otParent));
  otype=otDir;
  if(ScanDir(src,subdir,stDir,0,0,false)) {
    otype=otFile;
    if(ScanDir(src,subdir,stFile,src->Include(),".*",false)) res=true;
    }
  return res;
}

void cDirList::DoItem(cFileSource *src, const char *subdir, const char *name)
{
  Add(new cFileObj(src,subdir,name,otype));
}

// -- cFileSource --------------------------------------------------------------

cFileSource::cFileSource(void)
{
  browsedir=browseparent=0;
  basedir=description=include=0; useCount=0;
  needsmount=false;
}

cFileSource::cFileSource(const char *Basedir, const char *Description, const bool NeedsMount, const char *Include)
{
  browsedir=browseparent=0;
  basedir=description=include=0; useCount=0;
  Set(Basedir,Description,NeedsMount,Include);
}

cFileSource::~cFileSource()
{
  ClearRemember();
  free(basedir);
  free(description);
  free(include);
}

void cFileSource::Set(const char *Basedir, const char *Description, const bool NeedsMount, const char *Include)
{
  free(basedir); basedir=strdup(Basedir);
  free(description); description=strdup(Description);
  free(include); include=Include ? strdup(Include) : 0;
  needsmount=NeedsMount;
}

void cFileSource::SetRemember(const char *dir, const char *parent)
{
  ClearRemember();
  if(dir) browsedir=strdup(dir);
  if(parent) browseparent=strdup(parent);
}

void cFileSource::ClearRemember(void)
{
  free(browsedir); browsedir=0;
  free(browseparent); browseparent=0;
}

bool cFileSource::GetRemember(char * &dir, char * &parent)
{
  dir=parent=0;
  if(browsedir) {
    if(browseparent) parent=strdup(browseparent);
    dir=strdup(browsedir);
    return true;
    }
  return false;
}

bool cFileSource::Parse(char *s)
{
  char base[256], des[256], incl[256];
  int needsmount, n;
  if((n=sscanf(s,"%255[^;];%255[^;];%d;%255[^;]",base,des,&needsmount,incl))>=3) {
    char *base2=skipspace(stripspace(base));
    int l=strlen(base2);
    while(l>0 && base2[l-1]=='/') {
      esyslog("WARNING: removing trailing '/' from base %s",base2);
      base2[l-1]=0;
      l--;
      }
    Set(base2,skipspace(stripspace(des)),needsmount!=0,n>3?skipspace(stripspace(incl)):0);

    // do some checking of the basedir and issue a warning if apropriate
    if(access(base2,R_OK)) { esyslog("WARNING: source base %s not found/permission denied",base2); }
    else {
      struct stat64 ds;
      if(lstat64(base2,&ds)) { esyslog("WARNING: can't stat source base %s",base2); }
      else {
        if(S_ISLNK(ds.st_mode)) { esyslog("WARNING: source base %s is a symbolic link",base2); }
        else if(!S_ISDIR(ds.st_mode)) { esyslog("WARNING: source base %s is not a directory",base2); }
        }
      }
    return true;
    }
  return false;
}

bool cFileSource::Action(eAction act)
{
  static char *str[] = { "mount","unmount","eject","status" };
  
  char *cmd=0;
  asprintf(&cmd,"%s %s %s",mountscript,str[act],basedir);
  bool res=(system(cmd)==0);
  free(cmd);
  return res;
}

bool cFileSource::Mount(void)
{
  bool res=false;
  if(needsmount && (res=Action(acMount))) ClearRemember();
  return res;
}

bool cFileSource::Unmount(void)
{
  bool res=false;
  if(needsmount) {
    if(!useCount && (res=Action(acUnmount))) ClearRemember();
    }
  return res;
}

bool cFileSource::Eject(void)
{
  bool res=false;
  if(needsmount) {
    if(!useCount && (res=Action(acEject))) ClearRemember();
    }
  return res;
}

bool cFileSource::Status(void)
{
  if(needsmount) return Action(acStatus);
  return true;
}

// -- cFileSources --------------------------------------------------------------

bool cFileSources::Load(const char *filename, bool dummy)
{
  if(cConfig<cFileSource>::Load(filename,true)) {
    SetSource(First());
    return true;
    }
  return false;
}

cFileSource *cFileSources::FindSource(const char *filename)
{
  cFileSource *src=First();
  while(src) {
    if(startswith(filename,src->BaseDir())) return src;
    src=Next(src);
    }
  return 0;
}
