/*
 * MP3/MPlayer plugin to VDR (C++)
 *
 * (C) 2001-2003 Stefan Huelswitt <huels@iname.com>
 *
 * This code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 */

#include <vdr/player.h>
#include <vdr/status.h>
#include <vdr/font.h>
#ifdef HAVE_BEAUTYPATCH
#include <vdr/fontsym.h>
#endif
#ifndef VDRVERSNUM
#include <vdr/config.h>
#endif

#include "common.h"
#include "menu.h"
#include "menu-mplayer.h"
#include "player-mplayer.h"
#include "i18n.h"

cFileSources MPlaySources;

// --- cMenuSetupMPlayer --------------------------------------------------------

cMenuSetupMPlayer::cMenuSetupMPlayer(void)
{
  data=MPlayerSetup;
  SetSection(tr("MPlayer"));
  Add(new cMenuEditBoolItem(tr("Setup.MPlayer$Control mode"), &data.SlaveMode, tr("Traditional"), tr("Slave")));
}

void cMenuSetupMPlayer::Store(void)
{
  MPlayerSetup=data;
  SetupStore("ControlMode",MPlayerSetup.SlaveMode);
}

// --- cMPlayerControl ---------------------------------------------------------

class cMPlayerControl : public cControl {
private:
  static char * fileName;
  cMPlayerPlayer *player;
  bool visible, modeOnly, haveBeauty;
  time_t timeoutShow;
  int lastCurrent, lastTotal, osdPos;
  //
  bool jumpactive, jumphide, jumpmode;
  int jumpval;
  //
  void Stop(void);
  void ShowTimed(int Seconds=0);
  void DisplayAtBottom(const char *s);
  void ShowProgress(void);
  void ShowMode(void);
  void Jump(void);
  void JumpProcess(eKeys Key);
  void JumpDisplay(void);
public:
  cMPlayerControl(void);
  virtual ~cMPlayerControl();
  virtual eOSState ProcessKey(eKeys Key);
  virtual void Show(void) { ShowTimed(); }
  virtual void Hide(void);
  static void SetFilename(const char *FileName);
  };

char *cMPlayerControl::fileName=0;

cMPlayerControl::cMPlayerControl(void)
:cControl(player=new cMPlayerPlayer(fileName))
{
  visible=modeOnly=jumpactive=haveBeauty=false; osdPos=0;
  cStatus::MsgReplaying(this,fileName);
#ifdef HAVE_BEAUTYPATCH
#if VDRVERSNUM>=10300
  const cFont *sym=cFont::GetFont(fontSym);
  const cFont *osd=cFont::GetFont(fontOsd);
  const cFont::tCharData *symD=sym->CharData(32);
  const cFont::tCharData *osdD=osd->CharData(32);
#else
  cFont *sym=new cFont(fontSym);
  cFont *osd=new cFont(fontOsd);
  const cFont::tCharData *symD=sym->CharData(32);
  const cFont::tCharData *osdD=osd->CharData(32);
  delete sym;
  delete osd;
#endif
  if(symD != osdD) haveBeauty=true;
  d(printf("mplayer: beauty patch %sdetected\n",haveBeauty?"":"NOT "))
#endif
}

cMPlayerControl::~cMPlayerControl()
{
  Stop();
  cStatus::MsgReplaying(this, NULL);
}

void cMPlayerControl::SetFilename(const char *FileName)
{
  free(fileName);
  fileName=FileName ? strdup(FileName):0;
}

void cMPlayerControl::Stop(void)
{
  delete player; player=0;
}

void cMPlayerControl::ShowTimed(int Seconds)
{
  if(modeOnly) Hide();
  if(!visible) {
    ShowProgress();
    timeoutShow = Seconds>0 ? time(0)+Seconds : 0;
    }
}

void cMPlayerControl::Hide(void)
{
  if(visible) {
    Interface->Close();
    needsFastResponse=visible=modeOnly=false;
    }
}

void cMPlayerControl::DisplayAtBottom(const char *s)
{
  const int p=modeOnly ? 0 : 2;
  if(s) {
    const int d=max(Width()-cOsd::WidthInCells(s),0) / 2;
    if(modeOnly) Interface->Fill(0, p, Interface->Width(), 1, clrTransparent);
    Interface->Write(d, p, s);
    }
  else
    Interface->Fill(12, p, Width() - 22, 1, clrBackground);
}

void cMPlayerControl::ShowProgress(void)
{
  int Current, Total;

  if(GetIndex(Current,Total) && Total>0) {
    bool flush=false;
    if(!visible) {
      Interface->Open(Setup.OSDwidth,osdPos-3);
      Interface->Clear();
      if(osdPos<0) Interface->Fill(0,3,Interface->Width(),-osdPos,clrTransparent);
      needsFastResponse=visible=true;
      if(fileName) {
        int n=strlen(fileName);
        if(n>Width()) {
          n=n-Width()+4; if(n<0) n=0;
          char str[72];
          snprintf(str,sizeof(str),"... %s",fileName+n);
          Interface->Write(0,0,str);
          }
        else Interface->Write(0,0,fileName);
        }
      lastCurrent=lastTotal=-1;
      flush=true;
      }

    if(Current!=lastCurrent) {
       cProgressBar ProgressBar(Width() * cOsd::CellWidth(), cOsd::LineHeight(), Current, Total);
       Interface->SetBitmap(0, cOsd::LineHeight(), ProgressBar);
       Interface->Write(0,2,IndexToHMSF(Current));
       flush=true;
       lastCurrent=Current; lastTotal=Total;
       }
    if(flush) Interface->Flush();
    ShowMode();
    }
}

#ifdef HAVE_BEAUTYPATCH
int forwSym[] = { FSYM_FORW,FSYM_FORW1,FSYM_FORW2,FSYM_FORW3 };
int backSym[] = { FSYM_BACK,FSYM_BACK1,FSYM_BACK2,FSYM_BACK3 };
#endif

void cMPlayerControl::ShowMode(void)
{
  if(Setup.ShowReplayMode && !jumpactive) {
    bool Play, Forward;
    int Speed;
    if(GetReplayMode(Play, Forward, Speed)) {
       bool NormalPlay = (Play && Speed == -1);

       if(!visible) {
         if(NormalPlay) return;
         Interface->Open(0,osdPos-1);
         visible=modeOnly=true;
         }

       if(modeOnly && !timeoutShow && NormalPlay) timeoutShow=time(0)+SELECTHIDE_TIMEOUT;

       char buf[16];
       eDvbFont OldFont;
#ifdef HAVE_BEAUTYPATCH
       if(haveBeauty) {
         int i=0;
         if(!(Width()&1)) buf[i++]=' ';
         buf[i]=FSYM_EMPTY; if(Speed>=0 && !Forward) buf[i]=backSym[Speed];
         i++;
         buf[i++]=Play?(Speed==-1?FSYM_PLAY:FSYM_EMPTY):FSYM_PAUSE;
         buf[i]=FSYM_EMPTY; if(Speed>=0 && Forward) buf[i]=forwSym[Speed];
         i++;
         if(!(Width()&1)) buf[i++]=' ';
         buf[i]=0;
         OldFont = Interface->SetFont(fontSym);
         }
       else {
#endif
         const char *Mode;
         if (Speed == -1) Mode = Play    ? "  >  " : " ||  ";
         else if (Play)   Mode = Forward ? " X>> " : " <<X ";
         else             Mode = Forward ? " X|> " : " <|X ";
         strn0cpy(buf, Mode, sizeof(buf));
         char *p = strchr(buf, 'X');
         if(p) *p = Speed > 0 ? '1' + Speed - 1 : ' ';
         OldFont = Interface->SetFont(fontFix);
#ifdef HAVE_BEAUTYPATCH
         }
#endif
       DisplayAtBottom(buf);
       Interface->SetFont(OldFont);
       }
    }
}

void cMPlayerControl::JumpDisplay(void)
{
  char buf[64];
  const char *j=tr("Jump: "), u=jumpmode?'%':'m';
  if(!jumpval) sprintf(buf,"%s- %c",  j,u);
  else         sprintf(buf,"%s%d- %c",j,jumpval,u);
  DisplayAtBottom(buf);
}

void cMPlayerControl::JumpProcess(eKeys Key)
{
  const int n=Key-k0;
  switch (Key) {
    case k0 ... k9:
      {
      const int max=jumpmode?100:lastTotal;
      if(jumpval*10+n <= max) jumpval=jumpval*10+n;
      JumpDisplay();
      }
      break;
    case kBlue:
      jumpmode=!jumpmode; jumpval=0;
      DisplayAtBottom(0); JumpDisplay();
      break;
    case kPlay:
    case kUp:
      player->Goto(jumpval*(jumpmode?1:60),jumpmode,false);
      jumpactive=false;
      break;
    case kFastRew:
    case kFastFwd:
    case kLeft:
    case kRight:
      if(!jumpmode) {
        player->SkipSeconds(jumpval*60 * ((Key==kLeft || Key==kFastRew) ? -1:1));
        jumpactive=false;
        }
      break;
    default:
      jumpactive=false;
      break;
    }

  if(!jumpactive) {
    if(jumphide) Hide();
    else DisplayAtBottom(0);
    }
}

void cMPlayerControl::Jump(void)
{
  jumpval=0; jumphide=jumpmode=false;
  if(!visible) {
    ShowTimed(); if(!visible) return;
    jumphide=true;
    }
  JumpDisplay();
  jumpactive=true;
}

eOSState cMPlayerControl::ProcessKey(eKeys Key)
{
  if(!player->Active()) { Hide(); Stop(); return osEnd; }

  if(!player->SlaveMode()) {
    if(Key==kBlue) { Hide(); Stop(); return osEnd; }
    }
  else {
    if(visible) {
      if(timeoutShow && time(0)>timeoutShow) {
        Hide(); ShowMode();
        timeoutShow = 0;
        }
      else {
        if(modeOnly) ShowMode();
        else ShowProgress();
        }
      }

    if(jumpactive && Key != kNone) {
      JumpProcess(Key);
      return osContinue;
      }

    bool DoShowMode = true;
    switch (Key) {
      case kPlay:
      case kUp:      player->Play(); break;

      case kPause:
      case kDown:    player->Pause(); break;

      case kFastRew|k_Repeat:
      case kFastRew:
      case kLeft|k_Repeat:
      case kLeft:    player->SkipSeconds(-10); break;

      case kFastFwd|k_Repeat:
      case kFastFwd:
      case kRight|k_Repeat:
      case kRight:   player->SkipSeconds(10); break;

      case kRed:     Jump(); break;

      case kGreen|k_Repeat:                      // temporary use
      case kGreen:   player->SkipSeconds(-60); break;
      case kYellow|k_Repeat:
      case kYellow:  player->SkipSeconds(60); break;
  //    case kGreen|k_Repeat:                      // reserved for future use
  //    case kGreen:   player->SkipPrev(); break;
  //    case kYellow|k_Repeat:
  //    case kYellow:  player->SkipNext(); break;

      case kStop:
      case kBlue:
      case kBack:    Hide(); Stop(); return osEnd;

      default:
        DoShowMode = false;
        switch(Key) {
          case kOk: if(visible && !modeOnly) { Hide(); DoShowMode=true; }
                    else ShowTimed();
                    break;
          case k3:  player->Osd(); break;
          case k1:  player->AudioDelay(-1); break;
          case k2:  player->AudioDelay(1); break;
          case k9:  if(visible && !modeOnly) {
                      Hide();
                      osdPos++; if(osdPos>0) osdPos=0;
                      ShowTimed();
                      }
                    break;
          case k6:  if(visible && !modeOnly) {
                      Hide();
                      osdPos--; if(osdPos<-6) osdPos=-6;
                      ShowTimed();
                      }
                    break;
          default:  break;
          }
        break;
      }

    if(DoShowMode) ShowMode();
    }
  return osContinue;
}

// --- cMenuMPlayBrowse ---------------------------------------------------------

cMenuMPlayBrowse::cMenuMPlayBrowse(void)
:cMenuBrowse(MPlaySources.GetSource(),false,false,tr("MPlayer browser"))
{
  sourcing=false;
  SetButtons();
}

void cMenuMPlayBrowse::SetButtons(void)
{
  SetHelp(tr("Play"), currentdir?tr("Parent"):0, tr("Source"), 0);
  Display();
}

eOSState cMenuMPlayBrowse::Source(bool second)
{
  if(HasSubMenu()) return osContinue;

  if(!second) {
    sourcing=true;
    return AddSubMenu(new cMenuSource(&MPlaySources,tr("MPlayer source")));
    }
  sourcing=false;
  cFileSource *src=cMenuSource::GetSelected();
  if(src) {
    MPlaySources.SetSource(src);
    SetSource(src);
    NewDir(0);
    }
  return osContinue;
}

eOSState cMenuMPlayBrowse::ProcessKey(eKeys Key)
{
  eOSState state=cMenuBrowse::ProcessKey(Key);

  if(!HasSubMenu() && state==osContinue) { // eval the return value from submenus
    if(sourcing) return Source(true);
    }

  if(state==osBack && lastselect) {
    char *name=lastselect->Path();
    char *full=source->BuildName(name);
    cMPlayerControl::SetFilename(full);
    free(full); free(name);
    cControl::Launch(new cMPlayerControl);
    return osEnd;
    }

  if(state==osUnknown && Key==kYellow) return Source(false);
  return state;
}
