#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <GL/gl.h>

#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <gtkgl/gtkglarea.h>

#include "msglist.h"
#include "guiutils.h"

#include "v3dmp.h"
#include "v3dmodel.h"
#include "v3dtex.h"

#include "view.h"
#include "viewcb.h"
#include "editor.h"
#include "editorselect.h"
#include "editorcb.h"
#include "vmacfg.h"
#include "vmacfglist.h"
#include "vma.h"

#include "config.h"
#include "messages.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


#include "images/icon_menu_map_20x20.xpm"
#include "images/icon_restore_20x20.xpm"
#include "images/icon_maximize_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_grid_20x20.xpm"
#include "images/icon_viewable_dim_20x20.xpm"
#include "images/icon_move_rate_20x20.xpm"
#include "images/icon_render_20x20.xpm"
#include "images/icon_cull_20x20.xpm"
#include "images/icon_translations_20x20.xpm"
#include "images/icon_undo_20x20.xpm"
#include "images/icon_redo_20x20.xpm"
#include "images/icon_goto_20x20.xpm"


gdouble ViewSanitizeDegrees(gdouble d);
gdouble ViewSanitizeRadians(gdouble r);

guint ViewPositionDecimals(void);
guint ViewAngleDecimals(void);
gbool ViewRenderState(void);
gbool ViewCullState(void);
gbool ViewCullDirection(void);
gbool ViewTranslationsState(void);
gbool ViewEnableAlphaChannel(void);
gdouble ViewCameraClipNear(void);
gdouble ViewCameraClipFar(void);
gdouble ViewCameraFOV(void);

GtkWidget *ViewMakePixmapLabel(guint8 **data);
GtkWidget *ViewMakeInanimateButtonPixmap(
	guint8 **data,
	gint (*func_cb)(GtkWidget *, gpointer, gpointer),
	gpointer client_data
);

void View2DBuildCursors(vma_view2d_struct *v);
vma_view2d_struct *View2DCreate(
	void *core_ptr, void *editor_ptr,
	gint type,
	GtkWidget *restored_parent, GtkWidget *maximized_parent,
	void *client_data,
	void (*event_cb)(void *, void *)
);
void View2DSetPalette(vma_view2d_struct *v, vma_view_palette_struct *palette);
gdouble View2DGetVToWCoeff(vma_view2d_struct *v);
gdouble View2DGetWToVCoeff(vma_view2d_struct *v);
gint View2DGLEnableContext(vma_view2d_struct *v);
void View2DUpdateMenus(vma_view2d_struct *v);
void View2DResetDragMods(vma_view2d_struct *v);
void View2DReset(vma_view2d_struct *v);
void View2DDestroy(vma_view2d_struct *v);

void View3DBuildCursors(vma_view3d_struct *v);
vma_view3d_struct *View3DCreate(
	void *core_ptr, void *editor, gint type,
	GtkWidget *restored_parent, GtkWidget *maximized_parent,
	void *client_data,
	void (*event_cb)(void *, void *)
);
void View3DSetPalette(vma_view3d_struct *v, vma_view_palette_struct *palette);
gint View3DGLEnableContext(vma_view3d_struct *v);
void View3DUpdateMenus(vma_view3d_struct *v);
void View3DResetDragMods(vma_view3d_struct *v);
void View3DReset(vma_view3d_struct *v);
void View3DDestroy(vma_view3d_struct *v);


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define RADTODEG(r)     ((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180)


static gint gl_attributes_list[] = {
	GDK_GL_RGBA,
	GDK_GL_DOUBLEBUFFER,
	GDK_GL_DEPTH_SIZE,	1,
	GDK_GL_NONE
};


/*
 *	Sanitizes degree value d into [0, 360).
 */
gdouble ViewSanitizeDegrees(gdouble d)
{
	while(d >= 360.0)
		d -= 360.0;
	while(d < 0.0)
		d += 360.0;
	return(d);
}

/*
 *	Sanitizes radian value r into [0, 2 PI).
 */
gdouble ViewSanitizeRadians(gdouble r)
{
	while(r >= (2.0 * PI))
		r -= (2.0 * PI);
	while(r < 0.0)
		r += (2.0 * PI);
	return(r);
}


/*
 *	Returns number of decimals to represent position on a view from
 *	the global configuration options list.
 */
guint ViewPositionDecimals(void)
{
	return((guint)VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_VIEW_DECIMALS_POSITION
	));
}

/*
 *      Returns number of decimals to represent angles on a view from
 *      the global configuration options list.
 */
guint ViewAngleDecimals(void)
{
	return((guint)VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_VIEW_DECIMALS_ANGLE
	));
}

/*
 *      Returns render state from global configuration options list.
 */
gbool ViewRenderState(void)
{
	return(VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_VIEW_RENDER
	) ? TRUE : FALSE);
}

/*
 *	Returns cull state from global configuration options list.
 */
gbool ViewCullState(void)
{
	return(VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_VIEW_CULL_FACES
	) ? TRUE : FALSE);
}

/*
 *      Returns cull direction from global configuration options list,
 *	where TRUE is counter clockwise and FALSE is clockwise.
 */
gbool ViewCullDirection(void)
{
	return(VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_CULL_DIRECTION
	) ? TRUE : FALSE);
}

/*
 *      Returns handle translations and rotations state from global
 *	configuration options list,
 */
gbool ViewTranslationsState(void)
{
	return(VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_VIEW_TRANSLATIONS_STATE
	) ? TRUE : FALSE);
} 

/*
 *	Returns enable alpha channel state from global configuration
 *	options list.
 */
gbool ViewEnableAlphaChannel(void)
{
	return(VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_VIEW_ENABLE_ALPHA_CHANNEL
	) ? TRUE : FALSE);
}

/*
 *      Returns camera clip near value from global
 *      configuration options list,
 */
gdouble ViewCameraClipNear(void)
{
	return(VMACFGItemListGetValueD(
	    option, VMA_CFG_PARM_VIEW_CAM_CLIP_NEAR
	));
}

/*
 *      Returns camera flip far value from global
 *      configuration options list,
 */
gdouble ViewCameraClipFar(void)
{
	return(VMACFGItemListGetValueD(
	    option, VMA_CFG_PARM_VIEW_CAM_CLIP_FAR
	));
}

/*
 *	Returns the field of view (FOV) in radians from global
 *	configuration options list.
 */
gdouble ViewCameraFOV(void)
{
	return(VMACFGItemListGetValueD(
	    option, VMA_CFG_PARM_VIEW_CAM_FOV
	));
}



/*
 *	Creates a fixed widget containing the specified pixmap data.
 */
GtkWidget *ViewMakePixmapLabel(guint8 **data)
{
	gint width, height;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkStyle *style = gtk_widget_get_default_style();
	GtkWidget *w;


	if(data == NULL)
	    return(NULL);

	pixmap = gdk_pixmap_create_from_xpm_d(
	    (GdkWindow *)GDK_ROOT_PARENT(), &mask,
	    (style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,
	    (gchar **)data
	);
	if(pixmap == NULL)
	    return(NULL);

	gdk_window_get_size(pixmap, &width, &height);

	w = gtk_pixmap_new(pixmap, mask);
	gtk_widget_set_usize(w, width, height);
	gtk_widget_show(w);

	gdk_pixmap_unref(pixmap);
	if(mask != NULL)
	    gdk_pixmap_unref(mask);

	return(w);
}

/*
 *	Creates an event box with a fixed widget in there containing
 *	the specified pixmap data.
 *
 *	The event box will only be watching for a button press event.
 */
GtkWidget *ViewMakeInanimateButtonPixmap(
	guint8 **data,
	gint (*func_cb)(GtkWidget *, gpointer, gpointer),
	gpointer client_data
)
{
	GtkWidget *w, *event_box, *parent;
	GtkWidget *pixmapwid;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkStyle *style;
	gint width, height;


	/* Create event box (but do not show it) */
	w = gtk_event_box_new();
	if((w != NULL) && (func_cb != NULL))
	{
	    gtk_widget_add_events(
		w,
		GDK_BUTTON_PRESS_MASK
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w),
		"button_press_event",
		GTK_SIGNAL_FUNC(func_cb),
		client_data
	    );
	}
	event_box = w;
	parent = w;


	if((data != NULL) && (parent != NULL))
	{
	    style = gtk_widget_get_style(parent);

	    /* Now on to the xpm stuff */
	    pixmap = gdk_pixmap_create_from_xpm_d(
		(GdkWindow *)GDK_ROOT_PARENT(), &mask,
		&style->bg[GTK_STATE_NORMAL],
		(gchar **)data
	    );

	    pixmapwid = gtk_pixmap_new(pixmap, mask);

	    gdk_window_get_size((GdkWindow *)pixmap, &width, &height);

	    gtk_container_add(GTK_CONTAINER(parent), pixmapwid);

	    /* Match event box size to pixmap size */
	    gtk_widget_set_usize(event_box, width, height);

	    /* Unref pixmap and mask pair, they are no longer needed */
	    gdk_pixmap_unref(pixmap);
	    pixmap = NULL;
	    if(mask != NULL)
	    {
		gdk_pixmap_unref(mask);
		mask = NULL;
	    }

	    gtk_widget_show(pixmapwid);
	}

	return(event_box);
}

/*
 *	Creates the cursors for the 2D view.
 */
void View2DBuildCursors(vma_view2d_struct *v)
{
	/* Remember that bytes are swapped */
	static guint8 select_cd[] = {
0x00, 0x00,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x3e, 0x7c,
0x3e, 0x7c,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x00, 0x00
	};
	static guint8 select_mask_cd[] = {
0x80, 0x01,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xfe, 0x7f,
0x7f, 0xfe,
0x7f, 0xfe,
0xfe, 0x7f,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0x80, 0x01
};
	gint select_cd_w = 16, select_cd_h = 16;

	static guint8 translate_cd[] = {
0x00, 0x00,
0x80, 0x01,
0xc0, 0x03,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x84, 0x21,
0xfe, 0x7f,
0xfe, 0x7f,
0x84, 0x21,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0xc0, 0x03,
0x80, 0x01,
0x00, 0x00
	};
	static guint8 translate_mask_cd[] = {
0x80, 0x01,
0xc0, 0x03,
0xe0, 0x07,
0xf0, 0x0f,
0xc8, 0x13,
0xcc, 0x33,
0xfe, 0x7f,
0xff, 0xff, 
0xff, 0xff, 
0xfe, 0x7f,
0xcc, 0x33,
0xc8, 0x13,
0xf0, 0x0f,
0xe0, 0x07,
0xc0, 0x03,
0x80, 0x01
	};
	int translate_cd_w = 16, translate_cd_h = 16;

	static guint8 zoom_cd[] = {
0x00, 0x00,
0xfe, 0x7f,
0x06, 0x60,
0x0a, 0x50,
0x12, 0x48,
0xe2, 0x47,
0x22, 0x44,
0x22, 0x44,
0x22, 0x44,
0x22, 0x44,
0xe2, 0x47,
0x12, 0x48,
0x0a, 0x50,
0x06, 0x60,
0xfe, 0x7f,
0x00, 0x00
	};
	static guint8 zoom_mask_cd[] = {
0xff, 0xff,
0xff, 0xff,
0xff, 0xff,
0x3f, 0xfc,
0xff, 0xff,
0xff, 0xff,
0xf7, 0xef,
0x77, 0xee,
0x77, 0xee,
0xf7, 0xef,
0xff, 0xff,
0xff, 0xff,
0x3f, 0xfc,
0xff, 0xff,
0xff, 0xff,
0xff, 0xff
	};
	int zoom_cd_w = 16, zoom_cd_h = 16;


	guint8 *cd, *mask_cd;
	int cd_w, cd_h;
	double cd_x, cd_y;
	GdkCursor *cursor;
	GdkPixmap *source, *mask;
	GdkColor fg = {
	    0,
	    (u_int16_t)(-1),
	    (u_int16_t)(-1),
	    (u_int16_t)(-1)
	};
	GdkColor bg = {
	    0, 0, 0, 0
	};



#define DO_CREATE_CURSOR	\
{ \
 source = gdk_bitmap_create_from_data( \
  NULL, cd, cd_w, cd_h \
 ); \
 mask = gdk_bitmap_create_from_data( \
  NULL, mask_cd, cd_w, cd_h \
 ); \
 cursor = gdk_cursor_new_from_pixmap( \
  source, mask, &fg, &bg, \
  (gint)(cd_x * cd_w), (gint)(cd_y * cd_h) \
 ); \
 gdk_pixmap_unref(source); \
 gdk_pixmap_unref(mask); \
}

	cd = select_cd;
	mask_cd = select_mask_cd;
	cd_w = select_cd_w;
	cd_h = select_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	v->select_cur = cursor;

	cd = translate_cd;
	mask_cd = translate_mask_cd;
	cd_w = translate_cd_w;
	cd_h = translate_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	v->translate_cur = cursor;

	cd = zoom_cd;
	mask_cd = zoom_mask_cd;
	cd_w = zoom_cd_w;
	cd_h = zoom_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	v->zoom_cur = cursor;

#undef DO_CREATE_CURSOR

	return;
}


/*
 *      Allocates and sets up a 2D view.
 *
 *      The parent is assumed to be any valid container widget.
 */
vma_view2d_struct *View2DCreate(
	void *core_ptr, void *editor_ptr,
	gint type,
	GtkWidget *restored_parent, GtkWidget *maximized_parent,
	void *client_data,
	void (*event_cb)(void *, void *)
)
{
	const gchar *msglist[] = VMA_MSGLIST_EDITOR_TOOLTIPS;
	GtkAdjustment *adj;
/*	gint bw = 20, bh = 20; */
	gint accel_key;
	gpointer accel_group;
	guint accel_mods;
	guint8 **icon;
	const gchar *label = NULL;
	gpointer mclient_data;
	void (*func_cb)(GtkWidget *w, gpointer);
	GtkWidget *w, *parent2, *parent3, *parent4, *menu, *fw;
	const gchar *coord_names_xy[] = { "X:", "Y:" };
	const gchar *coord_names_yz[] = { "Y:", "Z:" };
	const gchar *coord_names_xz[] = { "X:", "Z:" };
	const gchar **coord_names_ptr;
	vma_view2d_struct *v = (vma_view2d_struct *)g_malloc0(
	    sizeof(vma_view2d_struct)
	);

	if(v == NULL)
	    return(NULL);

	if(restored_parent == NULL)
	{
	    g_free(v);
	    return(NULL);
	}


	/* Initialize implicit values */
	v->type = type;
	v->initialized = TRUE;

	v->editor_ptr = editor_ptr;
	v->bgimage_filename = NULL;
	v->client_data = client_data;
	v->event_cb = event_cb;


	/* Create cursors */
	View2DBuildCursors(v);


	/* Begin creating widgets */
	v->restored_parent = restored_parent;
	v->maximized_parent = maximized_parent;
	  
	w = gtk_vbox_new(FALSE, 0);
	v->toplevel = w;
	gtk_container_add(GTK_CONTAINER(restored_parent), w);
	gtk_widget_show(w);
	parent2 = w;
		
		  
	/* First row of widget table parents */
	w = gtk_table_new(7, 1, FALSE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Map menu button (an event box with a fixed widget) */
	w = GUIButtonPixmap((guint8 **)icon_menu_map_20x20_xpm);
	v->menu_map_btn = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    0, 1,       /* Left and right */  
	    0, 1,       /* Top and bottom */  
	    0, 
	    0, 
	    2, 2
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",
	    GTK_SIGNAL_FUNC(View2DButtonMenuMapCB),
	    (gpointer)v
	);
	gtk_widget_show(w);

	/* Select coordinate labels */
	switch(type)
	{
	  case VMA_VIEW2D_TYPE_YZ:   
	    coord_names_ptr = coord_names_yz;
	    break;
 
	  case VMA_VIEW2D_TYPE_XZ:
	    coord_names_ptr = coord_names_xz;
	    break;

	  default:
	    coord_names_ptr = coord_names_xy;
	    break;
	}
	/* I coordinate label and text entry */
	w = gtk_label_new(coord_names_ptr[0]);
	v->label_i = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    1, 2,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_i = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(View2DTextEnterCB),
	    (gpointer)v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    2, 3,	/* Left and right */
	    0, 1,	/* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);

	/* J coordinate label and text entry */
	w = gtk_label_new(coord_names_ptr[1]);
	v->label_j = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    3, 4,	/* Left and right */
	    0, 1,	/* Top and bottom */  
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_j = w;
	gtk_signal_connect(
	    GTK_OBJECT(w),
	    "activate",
	    GTK_SIGNAL_FUNC(View2DTextEnterCB),
	    v         
	);
	gtk_table_attach(GTK_TABLE(parent3), w,       
	    4, 5,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);

	/* Maximize button */
	w = GUIButtonPixmap((guint8 **)icon_maximize_20x20_xpm);
	v->maximize_btn = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(View2DMaximizeCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    5, 6,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist,  VMA_MSGNAME_EDITOR_VIEW_MAXIMIZE
	    )
	);
	gtk_widget_show(w);	/* Show this button, so current is restore */

	/* Restore button */
	w = GUIButtonPixmap((guint8 **)icon_restore_20x20_xpm);
	v->restore_btn = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(View2DRestoreCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    5, 6,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,         
	    0,  
	    2, 2
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_RESTORE
	    )
	);
 	/* Do not show restored button */

	/* Enabled button */
	w = gtk_button_new();
	gtk_table_attach(GTK_TABLE(parent3), w,
	    6, 7,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(View2DEnabledCB),
	    (gpointer)v
	);
	gtk_widget_show(w);
	parent4 = w;

	/* Enabled drawing area inside of enabled button */
	v->enabled_da = w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, 20, 20);
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_EXPOSURE_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(View2DEnabledDACB),
	    (gpointer)v
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_ENABLE
	    )
	);
	gtk_widget_show(w);

	/* Glarea widget */
	w = gtk_gl_area_new(gl_attributes_list);
	if(w == NULL)
	{
	    g_free(v);
	    fprintf(stderr, "Cannot create glarea widget.\n");
	    return(NULL);  
	}
	v->view = w;
	v->view_realized = FALSE;

	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK |
	    GDK_KEY_RELEASE_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
/*
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
 */
/*
	gtk_signal_connect(
	    GTK_OBJECT(w), "destroy",
	    GTK_SIGNAL_FUNC(View2DViewEventCB),
	    v
	);
 */
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);


	/* Row three grid size and viewable dimension */
	w = gtk_table_new(4, 1, FALSE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Grid size in meters */
	w = ViewMakePixmapLabel((guint8 **)icon_grid_20x20_xpm);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    0, 1,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_widget_show(w);

	adj = (GtkAdjustment *)gtk_adjustment_new(
	    VMA_DEF_GRID_SPACING,
	    0.0,		/* Min */
	    10000000.0,		/* Big max limit */
	    1.0,                /* Step inc */
	    5.0,                /* Page inc */
	    5.0                 /* Page size */
	);
	w = gtk_spin_button_new(
	    adj,
	    1.0,        /* Climb rate (0.0 to 1.0) */
	    2           /* Digits */   
	);
	v->grid_spacing_adj = adj;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    1, 2,
	    0, 1,
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(View2DSpinChangeCB),
	    v
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_GRID_SPACING
	    )
	);
	gtk_widget_show(w);

	/* Viewable dimension (zoom) in meters */
	w = ViewMakePixmapLabel((guint8 **)icon_viewable_dim_20x20_xpm);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    2, 3,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_widget_show(w);

	adj = (GtkAdjustment *)gtk_adjustment_new(
	    VMA_DEF_VIEW2D_VIEWABLE_DIM,
	    0.0001,		/* Min */
	    10000000.0,		/* Big max limit */
	    0.1,		/* Step inc */
	    1.0,		/* Page inc */
	    1.0			/* Page size */
	);
	w = gtk_spin_button_new(
	    adj,
	    1.0,        /* Climb rate (0.0 to 1.0) */
	    3           /* Digits */
	);
	v->viewable_dim_adj = adj;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    3, 4,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(View2DSpinChangeCB),
	    v
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_VIEWABLE_DIM
	    )
	);
	gtk_widget_show(w);



	/* Right click menu */
	v->menu = menu = GUIMenuCreate();
	accel_group = NULL;
	mclient_data = v;

#define DO_ADD_MENU_ITEM_LABEL  \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_ITEM_CHECK	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_CHECK, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_SEP \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}
	icon = NULL;
	label = "Enabled";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DEnabledCB;
	DO_ADD_MENU_ITEM_CHECK
	v->enabled_micheck = (GtkWidget *)fw;

	DO_ADD_MENU_SEP

	icon = (guint8 **)icon_undo_20x20_xpm;
	label = "Undo";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DUndoCB;
	DO_ADD_MENU_ITEM_LABEL
	v->undo_mi = w;
	v->undo_milabel = (GtkWidget *)fw;

	icon = (guint8 **)icon_redo_20x20_xpm; 
	label = "Redo";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DRedoCB;
	DO_ADD_MENU_ITEM_LABEL
	v->redo_mi = w;
	v->redo_milabel = (GtkWidget *)fw;

	DO_ADD_MENU_SEP

	icon = (guint8 **)icon_goto_20x20_xpm;
	label = "Jump To Vertex";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DJumpToVertexCB;
	DO_ADD_MENU_ITEM_LABEL
	v->jump_to_vertex_mi = w;

	icon = NULL;
	label = "Set To Cursor";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DSetToCursorCB;
	DO_ADD_MENU_ITEM_LABEL
	v->set_to_cursor_mi = w;

	DO_ADD_MENU_SEP

	icon = NULL;
	label = "Set Background Image...";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DSetBGImageCB;
	DO_ADD_MENU_ITEM_LABEL
	v->set_bgimg_mi = w;

	icon = NULL;
	label = "Clear Background Image";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DClearBGImageCB;
	DO_ADD_MENU_ITEM_LABEL
	v->clear_bgimg_mi = w;

	DO_ADD_MENU_SEP

	icon = NULL;     
	label = "Maximize/Restore"; 
	accel_key = 0;
	accel_mods = 0;
	func_cb = View2DMaximizeRestoreMCB;
	DO_ADD_MENU_ITEM_LABEL
	v->maximize_restore_mi = w;

#undef DO_ADD_MENU_SEP
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_ITEM_LABEL

	/* Need to attach basic "hide" signal to menu so we know when
	 * it is unmapped.
	 */
	gtk_signal_connect(
	    GTK_OBJECT(menu), "hide",
	    GTK_SIGNAL_FUNC(View2DMenuHideCB),
	    (gpointer)v
	);


	/* Reset values */
	View2DReset(v);


	return(v);
}

/*
 *	Sets the palette for the 2d view.
 */
void View2DSetPalette(
	vma_view2d_struct *v, vma_view_palette_struct *palette
)
{
	if(v != NULL)
	{
	    if(palette == NULL)
		memset(&v->palette, 0x00, sizeof(vma_view_palette_struct));
	    else
		memcpy(&v->palette, palette, sizeof(vma_view_palette_struct));
	}
	return;
}

/*
 *	Returns the view coordinates to window coordinates conversion
 *	coefficient value. Can return 0.0 on error.
 */
gdouble View2DGetVToWCoeff(vma_view2d_struct *v)
{
	gint w_max;      /* Longer window size in window units */
	GtkWidget *w;

	if((v == NULL) ? 1 : !v->initialized)
	    return(0.0);

	w = v->view; 
	if(w == NULL)
	    return(0.0);

	w_max = MAX(w->allocation.width, w->allocation.height); 
	if(v->viewable_dim > 0.0)
	    return((gdouble)w_max / v->viewable_dim);
	else
	    return(0.0);
}

/*      
 *      Returns the window coordinates to view coordinates conversion
 *      coefficient value. Can return 0.0 on error.
 */
gdouble View2DGetWToVCoeff(vma_view2d_struct *v)
{
	int w_max;	/* Longer window size in window units */
	GtkWidget *w;

	if((v == NULL) ? 1 : !v->initialized)
	    return(0.0);

	w = v->view;
	if(w == NULL)
	    return(0.0);

	w_max = MAX(w->allocation.width, w->allocation.height);
	if(w_max > 0)
	    return(v->viewable_dim / (gdouble)w_max);
	else
	    return(0.0);
}


/*
 *	Makes the glarea view widget on the view into context.
 *	Returns 0 on success or -1 on error.
 */
gint View2DGLEnableContext(vma_view2d_struct *v)
{
	GtkWidget *w;
	gint status;


	if(v == NULL)
	    return(-1);

	if(!v->view_realized)
	    return(-1);

	w = v->view;
	if(w != NULL)
	{
	    status = gtk_gl_area_make_current(GTK_GL_AREA(w));
	    if(status)
		return(0);
	    else
		return(-1);
	}

	return(-1);
}

/*
 *	Updates some memu items to correspond to the other members
 *	of the given 2d view structure.
 */
void View2DUpdateMenus(vma_view2d_struct *v)
{
	static gbool reenterant = FALSE;
	gbool sensitivity, state;
	gint model_num = -1, pn = -1;
	GtkWidget *w;
	v3d_model_struct *model_ptr = NULL;
	gpointer p = NULL;
	ma_editor_struct *editor;


	if(v == NULL)
	    return;

	if(!v->view_realized)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Get pointer to editor structure (can be NULL) */
	editor = v->editor_ptr;

	/* Get currently selected model if any */
	if(editor != NULL)
	{
	    model_num = EditorSelectedModelIndex(editor);
	    model_ptr = V3DModelListGetPtr(
	        editor->model, editor->total_models, model_num
	    );
	}

	/* Get currently single selected primitive if any */
	if((editor != NULL) && (model_ptr != NULL))
	{
	    pn = ((editor->total_selected_primitives > 0) ?
		editor->selected_primitive[0] : -1
	    );
	    p = V3DMPListGetPtr(
		model_ptr->primitive, model_ptr->total_primitives,
		pn
	    );
	}


#define SET_WIDGET_SENSITIVITY  \
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitivity); \
}

#define SET_CHECK_MENU_ITEM_STATE	\
{ \
 if(w != NULL) \
   GTK_CHECK_MENU_ITEM(w)->active = state; \
}

	/* Begin updating menu items */

	/* Enabled state */
	w = v->enabled_micheck;
	state = ((v->flags & VMA_VIEW_FLAG_ENABLED) ? TRUE : FALSE);
	SET_CHECK_MENU_ITEM_STATE

	/* Undo */
	w = v->undo_milabel;
	if((w != NULL) && (editor != NULL))
	{
	    gint total = editor->total_undos;
	    gchar text[256];

	    *text = '\0';
	    strcat(text, "Undo");
	    if(total > 0)
	    {
		vma_undo_common_struct *u = editor->undo[total - 1];
		gchar numstr[80];

		if(u != NULL)
		{
		    if(u->name != NULL)
		    {
			strcat(text, " ");
			strncat(text, u->name, 80);
			text[256 - 1] = '\0';
		    }
		}
		sprintf(numstr, "(%i)", total);
		strcat(text, numstr);
	    }
	    gtk_label_set_text(GTK_LABEL(w), text);
	}
	w = v->undo_mi;
	sensitivity = FALSE;
	if((w != NULL) && (editor != NULL))
	{
	    if(editor->total_undos > 0)
		sensitivity = TRUE;
	}
	SET_WIDGET_SENSITIVITY

	/* Redo */
	w = v->redo_milabel;
	if((w != NULL) && (editor != NULL))  
	{
	    gint total = editor->total_redos;
	    gchar text[256];
	    
	    *text = '\0';
	    strcat(text, "Redo");
	    if(total > 0)
	    {
		vma_undo_common_struct *u = editor->redo[total - 1];
		gchar numstr[80];

		if(u != NULL)
		{
		    if(u->name != NULL)
		    {
			strcat(text, " ");
			strncat(text, u->name, 80);
			text[256 - 1] = '\0';
		    }
		}
		sprintf(numstr, "(%i)", total);
		strcat(text, numstr);
	    }
	    gtk_label_set_text(GTK_LABEL(w), text);
	}
	w = v->redo_mi;
	sensitivity = FALSE;
	if((w != NULL) && (editor != NULL))
	{
	    if(editor->total_redos > 0)
		sensitivity = TRUE;
	}
	SET_WIDGET_SENSITIVITY

	/* Jump to vertex */
	w = v->jump_to_vertex_mi;
	sensitivity = FALSE;
	if(editor != NULL)
	{
	    /* Attempt to get a pointer to the vertex of the primitive.
	     * If successful it means the primitive has a vertex.
	     */
	    gint vtx_num;
	    mp_vertex_struct *vertex;

	    vtx_num = EditorGetSelected(
		editor->selected_value, editor->total_selected_values,
		0
	    );
	    vertex = V3DMPGetVertex(p, vtx_num);
	    if(vertex != NULL)
		sensitivity = TRUE;
	}
	SET_WIDGET_SENSITIVITY
	/* Set to cursor (same criteria as jump to vertex) */
	w = v->set_to_cursor_mi;
	SET_WIDGET_SENSITIVITY

	/* Clear background image */
	w = v->clear_bgimg_mi;
	sensitivity = ((v->bg_image == NULL) ? FALSE : TRUE);
	SET_WIDGET_SENSITIVITY


#undef SET_WIDGET_SENSITIVITY
#undef SET_CHECK_MENU_ITEM_STATE

	reenterant = FALSE;

	return;
}


/*
 *	Reset all drag modifiers on the given view and resets the
 *	cursor. This is needed in some cases where the drag is not
 *	reset (ie after selecting multiple primitives when the pointer
 *	is busy with the confirmation dialog with the user).
 */
void View2DResetDragMods(vma_view2d_struct *v)
{
	GtkWidget *w;

	if(v == NULL)
	    return;

	w = v->view;
	if(w == NULL)
	    return;

	v->flags &= ~VMA_VIEW_FLAG_BUTTON1;
	v->flags &= ~VMA_VIEW_FLAG_BUTTON2;
	v->flags &= ~VMA_VIEW_FLAG_BUTTON3;
	v->drag_state = VMA_VIEW2D_DRAG_NONE;

	if(!GTK_WIDGET_NO_WINDOW(w))
	    gdk_window_set_cursor(w->window, NULL);
}

/*
 *	Resets values on the 2d view.
 *
 *	Does not reset the palette color pointers.
 */
void View2DReset(vma_view2d_struct *v)
{
	GtkWidget *w;

	if(v == NULL)
	    return;

	if(!v->initialized)
	    return;

	/* Begin resetting values */

	v->flags |= VMA_VIEW_FLAG_ENABLED;
	w = v->enabled_micheck;
	if(w != NULL)
	    GTK_CHECK_MENU_ITEM(w)->active = TRUE;

	if(VMACFGItemListGetValueI(option, VMA_CFG_PARM_POINTER_EMULATE_2BUTTON))
	    v->flags |= VMA_VIEW_FLAG_POINTER_EMULATE_2BUTTON;
	else
	    v->flags &= ~VMA_VIEW_FLAG_POINTER_EMULATE_2BUTTON;

	v->position_decimals = ViewPositionDecimals();
	v->angle_decimals = ViewAngleDecimals();

	v->viewable_dim = VMA_DEF_VIEW2D_VIEWABLE_DIM;

	v->v_ti = 0.0;
	v->v_tj = 0.0;

 	v->w_cur_i = 0;
	v->w_cur_j = 0;
	v->v_cur_i = 0.0;
	v->v_cur_j = 0.0;

	v->grid_spacing = VMA_DEF_GRID_SPACING;

	v->drag_state = VMA_VIEW2D_DRAG_NONE;
	v->w_last_i = 0;
	v->w_last_j = 0;

	v->w_sel_rect_i0 = 0;
	v->w_sel_rect_j0 = 0;
	v->w_sel_rect_i1 = 0;
	v->w_sel_rect_j1 = 0;

	/* Unload background image */
	ViewBGImageUnload(v->bg_image);
	v->bg_image = NULL;

	g_free(v->bgimage_filename);
	v->bgimage_filename = NULL;

	v->bgimage_offset_i = 0.0;
	v->bgimage_offset_j = 0.0;
	v->bgimage_scale_i = 0.0;
	v->bgimage_scale_j = 0.0;


	View2DTranslateCB(v);
	View2DZoomCB(v);
	View2DGridCB(v);
	View2DUpdateMenus(v);
}

/*
 *	Destroys a 2d view, deallocating any sub resources and the
 *	structure itself.
 */
void View2DDestroy(vma_view2d_struct *v)
{
	GtkWidget **w;
	GdkCursor **cur;


	if(v == NULL)
	    return;

	if(v->initialized)
	{
#define DO_DESTROY_CURSOR       \
{ \
 if((*cur) != NULL) \
 { \
  GdkCursor *tc = *cur; \
  (*cur) = NULL; \
  gdk_cursor_destroy(tc); \
 } \
}

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
	    /* Unload background image */
	    ViewBGImageUnload(v->bg_image);
	    v->bg_image = NULL;

	    g_free(v->bgimage_filename);
	    v->bgimage_filename = NULL;


	    /* Begin destroying widgets */

	    w = &v->view;
	    DO_DESTROY_WIDGET

	    w = &v->menu;
	    v->enabled_micheck = NULL;
	    v->undo_mi = NULL;
	    v->undo_milabel = NULL;
	    v->redo_mi = NULL;
	    v->redo_milabel = NULL;
	    v->jump_to_vertex_mi = NULL;
	    v->set_to_cursor_mi = NULL;
	    v->set_bgimg_mi = NULL;
	    v->clear_bgimg_mi = NULL;
	    v->maximize_restore_mi = NULL;
	    DO_DESTROY_WIDGET

	    w = &v->enabled_da;
	    DO_DESTROY_WIDGET

	    w = &v->toplevel;
	    DO_DESTROY_WIDGET


	    cur = &v->translate_cur;
	    DO_DESTROY_CURSOR
	    cur = &v->zoom_cur;
	    DO_DESTROY_CURSOR
	    cur = &v->select_cur;
	    DO_DESTROY_CURSOR

#undef DO_DESTROY_CURSOR
#undef DO_DESTROY_WIDGET
	}

	/* Deallocate structure itself */
	free(v);
}




/*
 *      Creates the cursors for the 3d view.
 */
void View3DBuildCursors(vma_view3d_struct *v)
{
	/* Remember that bytes are swapped */
	static guint8 move_strafe_cd[] = {
0x00, 0x00,
0x80, 0x01,
0xc0, 0x03,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x84, 0x21,
0xfe, 0x7f,
0xfe, 0x7f,
0x84, 0x21,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0xc0, 0x03,
0x80, 0x01,
0x00, 0x00 
	}; 
	static guint8 move_strafe_mask_cd[] = {
0x80, 0x01,
0xc0, 0x03,
0xe0, 0x07,   
0xf0, 0x0f,
0xc8, 0x13,
0xcc, 0x33,
0xfe, 0x7f,
0xff, 0xff,
0xff, 0xff,
0xfe, 0x7f,
0xcc, 0x33,
0xc8, 0x13,
0xf0, 0x0f,
0xe0, 0x07,   
0xc0, 0x03,
0x80, 0x01 
	}; 
	gint move_strafe_cd_w = 16, move_strafe_cd_h = 16;

	static guint8 move_rotate_cd[] = {
0x00, 0x00, 
0x80, 0x01,
0xc0, 0x03,
0xe0, 0x07,
0x80, 0x01,
0x88, 0x11,
0x8c, 0x31,
0xbe, 0x7d,
0xfe, 0x7f,
0xec, 0x37,
0xc8, 0x13,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x80, 0x01,
0x00, 0x00
	};
	static guint8 move_rotate_mask_cd[] = {
0x80, 0x01,
0xc0, 0x03,
0xe0, 0x07,
0xf0, 0x0f,
0xe8, 0x17,
0xdc, 0x3b,
0xfe, 0x7f,
0xff, 0xff,
0xff, 0xff,
0xfe, 0x7f,
0xfc, 0x3f,
0xc8, 0x13,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03,
0x80, 0x01
	};
	gint move_rotate_cd_w = 16, move_rotate_cd_h = 16;

	static guint8 move_height_cd[] = {
0x00, 0x00,
0x60, 0x06,
0xb0, 0x0d,
0xd0, 0x0b,
0x00, 0x00,
0xd0, 0x0b,
0xd0, 0x0b,
0x00, 0x00,
0xa0, 0x05,
0xa0, 0x05,
0x40, 0x02,
0x00, 0x00,
0x00, 0x00,
0x00, 0x00,
0x80, 0x01,
0x00, 0x00
	};
	static guint8 move_height_mask_cd[] = {
0xe0, 0x07,
0xf0, 0x0f,
0xf8, 0x1f,
0xf8, 0x1f,
0xf8, 0x1f,
0xf8, 0x1f,
0xf8, 0x1f,
0xf8, 0x1f,
0xf0, 0x0f,
0xf0, 0x0f,
0xe0, 0x07,
0xe0, 0x07,
0x40, 0x02,
0xc0, 0x03,
0xc0, 0x03,
0xc0, 0x03
	};
	int move_height_cd_w = 16, move_height_cd_h = 16;

	static guint8 rotate_cd[] = {
0x00, 0x00,
0x00, 0x00,
0xc0, 0x03,
0xf0, 0x0f,
0x30, 0x1c,
0x00, 0x18,
0x00, 0x30,
0x0c, 0xfc,
0x1e, 0x78,
0x3f, 0x30,
0x0c, 0x00,
0x18, 0x00,
0x38, 0x0c,
0xf0, 0x0f,
0xc0, 0x03,
0x00, 0x00
	};
	static guint8 rotate_mask_cd[] = {
0x00, 0x00,
0xc0, 0x03,
0xf0, 0x0f,
0xf8, 0x1f,
0xf8, 0x3f,
0x30, 0x3c,
0x0c, 0xfc,
0x1e, 0xfe,
0x3f, 0xfc,
0x7f, 0x78,
0x3f, 0x30,
0x3c, 0x0c,
0xfc, 0x1f,
0xf8, 0x1f,
0xf0, 0x0f,
0xc0, 0x03
	};
	gint rotate_cd_w = 16, rotate_cd_h = 16;


	guint8 *cd, *mask_cd;
	gint cd_w, cd_h;
	gdouble cd_x, cd_y;
	GdkCursor *cursor;
	GdkPixmap *source, *mask;
	GdkColor fg = {
	    0,
	    (u_int16_t)(-1),
	    (u_int16_t)(-1),
	    (u_int16_t)(-1)
	};
	GdkColor bg = {
	    0, 0, 0, 0
	};

#define DO_CREATE_CURSOR	\
{ \
 source = gdk_bitmap_create_from_data( \
  NULL, cd, cd_w, cd_h \
 ); \
 mask = gdk_bitmap_create_from_data( \
  NULL, mask_cd, cd_w, cd_h \
 ); \
 cursor = gdk_cursor_new_from_pixmap( \
  source, mask, &fg, &bg, \
  (gint)(cd_x * cd_w), (gint)(cd_y * cd_h) \
 ); \
 gdk_pixmap_unref(source); \
 gdk_pixmap_unref(mask); \
}

	cd = move_strafe_cd;
	mask_cd = move_strafe_mask_cd; 
	cd_w = move_strafe_cd_w;
	cd_h = move_strafe_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR  
	v->move_strafe_cur = cursor;

	cd = move_rotate_cd;
	mask_cd = move_rotate_mask_cd;
	cd_w = move_rotate_cd_w;
	cd_h = move_rotate_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	v->move_rotate_cur = cursor;

	cd = move_height_cd;
	mask_cd = move_height_mask_cd;
	cd_w = move_height_cd_w;
	cd_h = move_height_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	v->move_height_cur = cursor;

	cd = rotate_cd;
	mask_cd = rotate_mask_cd;
	cd_w = rotate_cd_w;
	cd_h = rotate_cd_h;
	cd_x = 0.5;
	cd_y = 0.5;
	DO_CREATE_CURSOR
	v->rotate_cur = cursor;

#undef DO_CREATE_CURSOR

	return;
}

/*
 *	Allocates and sets up a 3D view.
 *
 *	The parent is assumed to be any valid container widget.
 */
vma_view3d_struct *View3DCreate(
	void *core_ptr, void *editor_ptr,
	gint type,
	GtkWidget *restored_parent, GtkWidget *maximized_parent,
	void *client_data,
	void (*event_cb)(void *, void *)
)
{
	const gchar *msglist[] = VMA_MSGLIST_EDITOR_TOOLTIPS;
	gint accel_key;
	gpointer accel_group;
	guint accel_mods;
	guint8 **icon;
	const gchar *label = NULL;
	gpointer mclient_data;
	void (*func_cb)(GtkWidget *w, void *);
	GtkWidget *w, *parent2, *parent3, *parent4, *menu, *fw;
	GtkAdjustment *adj;
	vma_view3d_struct *v = (vma_view3d_struct *)g_malloc0(
	    sizeof(vma_view3d_struct)
	);
   

	if(v == NULL)
	    return(NULL);

	if(restored_parent == NULL)
	{
	    g_free(v);
	    return(NULL);
	}


	/* Initialize implicit values */
	v->type = type;
	/* Type must be VMA_VIEW3D_TYPE_STANDARD */
	if(v->type != VMA_VIEW3D_TYPE_STANDARD)
	    v->type = VMA_VIEW3D_TYPE_STANDARD;
	v->initialized = TRUE;
	v->flags = VMA_VIEW_FLAG_SHOW_GRID;

	v->editor_ptr = editor_ptr;

	v->client_data = client_data;
	v->event_cb = event_cb;

	memset(&v->palette, 0x00, sizeof(vma_view_palette_struct));


	/* Build cursors */
	View3DBuildCursors(v);


	/* Begin creating widgets */
	v->restored_parent = restored_parent;
	v->maximized_parent = maximized_parent;

	w = gtk_vbox_new(FALSE, 0);
	v->toplevel = w;
	gtk_container_add(GTK_CONTAINER(restored_parent), w);
	gtk_widget_show(w);
	parent2 = w;


	/* First row of widget table parents */
	w = gtk_table_new(7, 1, FALSE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Map menu button (an event box with a fixed widget) */
	w = GUIButtonPixmap((guint8 **)icon_menu_map_20x20_xpm);
	v->menu_map_btn = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    0, 1,	/* Left and right */
	    0, 1,	/* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "pressed",
	    GTK_SIGNAL_FUNC(View3DButtonMenuMapCB),
	    (gpointer)v
	);
	gtk_widget_show(w);


	/* Tempory first row filler label */
	w = gtk_label_new("3D View");
	gtk_table_attach(GTK_TABLE(parent3), w,
	    1, 2,
	    0, 1,
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);


	/* Render actual toggle */
	w = GUIToggleButtonPixmap((guint8 **)icon_render_20x20_xpm);
	v->render_toggle = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(View3DRenderActualToggleCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    2, 3,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,  
	    2, 2
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_RENDER_ACTUAL
	    )
	);
	gtk_widget_show(w);

	/* Cull toggle */
	w = GUIToggleButtonPixmap((guint8 **)icon_cull_20x20_xpm);
	v->cull_toggle = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(View3DCullToggleCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,  
	    3, 4,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_CULL_FACES
	    )
	);
	gtk_widget_show(w);

	/* Translations toggle */
	w = GUIToggleButtonPixmap((guint8 **)icon_translations_20x20_xpm);
	v->translations_toggle = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(View3DTranslationsToggleCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    4, 5,       /* Left and right */ 
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_TRANSLATIONS
	    )
	);
	gtk_widget_show(w);

	/* Maximize button */
	w = GUIButtonPixmap((guint8 **)icon_maximize_20x20_xpm);
	v->maximize_btn = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(View3DMaximizeCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    5, 6,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_MAXIMIZE
	    )
	);
	gtk_widget_show(w);     /* Show this button, so current is restore */

	/* Restore button */
	w = GUIButtonPixmap((guint8 **)icon_restore_20x20_xpm);
	v->restore_btn = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(View3DRestoreCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    5, 6,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_RESTORE
	    )
	);
	/* Do not show restore button */

	/* Enabled button */
	w = gtk_button_new();
	gtk_table_attach(GTK_TABLE(parent3), w,
	    6, 7,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,   
	    0,
	    2, 2
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(View3DEnabledCB),
	    (gpointer)v
	);
	gtk_widget_show(w);
	parent4 = w;

	/* Enabled drawing area inside of enabled button */
	v->enabled_da = w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, 20, 20);
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_EXPOSURE_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(View3DEnabledDACB),
	    (gpointer)v
	);   
	gtk_container_add(GTK_CONTAINER(parent4), w);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_ENABLE
	    )
	);
	gtk_widget_show(w);


	/* Coordinate position labels and text entries */
	w = gtk_table_new(6, 1, FALSE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* X coordinate position */
	w = gtk_label_new("X:");
	v->label_x = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    0, 1,       /* Left and right */
	    0, 1,       /* Top and bottom */  
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_x = w;
	gtk_signal_connect(
	    GTK_OBJECT(w),
	    "activate",
	    GTK_SIGNAL_FUNC(View3DTextEnterCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    1, 2,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);

	/* Y coordinate position */
	w = gtk_label_new("Y:");
	v->label_y = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    2, 3,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_y = w;
	gtk_signal_connect(
	    GTK_OBJECT(w),
	    "activate",
	    GTK_SIGNAL_FUNC(View3DTextEnterCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    3, 4,       /* Left and right */  
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);

	/* Z coordinate position */
	w = gtk_label_new("Z:");
	v->label_z = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    4, 5,       /* Left and right */
	    0, 1,       /* Top and bottom */  
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_z = w;
	gtk_signal_connect(
	    GTK_OBJECT(w),
	    "activate",
	    GTK_SIGNAL_FUNC(View3DTextEnterCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    5, 6,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);  
	gtk_widget_show(w);



	/* Third row, attitude widgets table parent */
	w = gtk_table_new(6, 1, FALSE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Heading direction position */
	w = gtk_label_new("H:");
	v->label_h = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    0, 1,       /* Left and right */
	    0, 1,       /* Top and bottom */  
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_h = w; 
	gtk_signal_connect(
	    GTK_OBJECT(w),
	    "activate",
	    GTK_SIGNAL_FUNC(View3DTextEnterCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    1, 2,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);

	/* Pitch direction position */
	w = gtk_label_new("P:");
	v->label_p = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    2, 3,       /* Left and right */
	    0, 1,       /* Top and bottom */  
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_p = w;
	gtk_signal_connect(
	    GTK_OBJECT(w),
	    "activate",
	    GTK_SIGNAL_FUNC(View3DTextEnterCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    3, 4,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);

	/* Bank direction position */
	w = gtk_label_new("B:");
	v->label_b = w;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    4, 5,       /* Left and right */
	    0, 1,       /* Top and bottom */  
	    0,
	    0,
	    2, 2
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_widget_show(w);

	w = gtk_entry_new_with_max_length(VMA_VALUE_TEXT_MAX);
	v->text_b = w;
	gtk_signal_connect(
	    GTK_OBJECT(w),
	    "activate",
	    GTK_SIGNAL_FUNC(View3DTextEnterCB),
	    v
	);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    5, 6,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_widget_show(w);


	/* Glarea widget */
	w = gtk_gl_area_new(gl_attributes_list);
	if(w == NULL)
	{
	    g_free(v);
	    fprintf(stderr, "Cannot create glarea widget.\n");
	    return(NULL);  
	}
	v->view = w;
	v->view_realized = FALSE;

	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK |
	    GDK_KEY_RELEASE_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v   
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
/*
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
 */
/*
	gtk_signal_connect(
	    GTK_OBJECT(w), "destroy",
	    GTK_SIGNAL_FUNC(View3DViewEventCB),
	    v
	);
 */
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);


	/* Row five grid size and move rate */
	w = gtk_table_new(4, 1, FALSE);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Grid size in meters */
	w = ViewMakePixmapLabel((guint8 **)icon_grid_20x20_xpm);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    0, 1,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_widget_show(w);

	adj = (GtkAdjustment *)gtk_adjustment_new(
	    VMA_DEF_GRID_SPACING,
	    0.0,		/* Min */
	    100000.0,		/* Big max limit */
	    1.0,		/* Step inc */
	    5.0,		/* Page inc */
	    5.0			/* Page size */
	);
	w = gtk_spin_button_new(
	    adj,
	    1.0,	/* Climb rate (0.0 to 1.0) */
	    2		/* Digits */
	);
	v->grid_spacing_adj = adj;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    1, 2,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(View3DSpinChangeCB),
	    v
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_GRID_SPACING
	    )
	);
	gtk_widget_show(w);

	/* Move rate in meters */
	w = ViewMakePixmapLabel((guint8 **)icon_move_rate_20x20_xpm);
	gtk_table_attach(GTK_TABLE(parent3), w,
	    2, 3,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    0,
	    0,
	    2, 2
	);
	gtk_widget_show(w);

	adj = (GtkAdjustment *)gtk_adjustment_new(
	    VMA_DEF_VIEW3D_MOVE_RATE,
	    0.0,		/* Min */
	    100000.0,           /* Big max limit */
	    0.01,		/* Step inc */
	    0.1,		/* Page inc */ 
	    0.5			/* Page size */
	);
	w = gtk_spin_button_new(
	    adj,
	    1.0,        /* Climb rate (0.0 to 1.0) */
	    4           /* Digits */   
	);
	v->move_rate_adj = adj;
	gtk_table_attach(GTK_TABLE(parent3), w,
	    3, 4,       /* Left and right */
	    0, 1,       /* Top and bottom */
	    GTK_FILL | GTK_SHRINK | GTK_EXPAND,
	    GTK_SHRINK,
	    2, 2
	);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(View3DSpinChangeCB),
	    v
	);
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_EDITOR_VIEW_CAM_MOVE_RATE
	    )
	);
	gtk_widget_show(w);



	/* Right click menu */
	v->menu = menu = GUIMenuCreate();
	accel_group = NULL;
	mclient_data = v;

#define DO_ADD_MENU_ITEM_LABEL	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_ITEM_CHECK	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_CHECK, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_SEP	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

	icon = NULL;
	label = "Enabled";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DEnabledCB;
	DO_ADD_MENU_ITEM_CHECK
	v->enabled_micheck = (GtkWidget *)fw;

	DO_ADD_MENU_SEP  

	icon = (guint8 **)icon_undo_20x20_xpm;
	label = "Undo";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DUndoCB;
	DO_ADD_MENU_ITEM_LABEL
	v->undo_mi = w;
	v->undo_milabel = (GtkWidget *)fw;

	icon = (guint8 **)icon_redo_20x20_xpm;
	label = "Redo";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DRedoCB;
	DO_ADD_MENU_ITEM_LABEL
	v->redo_mi = w;
	v->redo_milabel = (GtkWidget *)fw;

	DO_ADD_MENU_SEP

	icon = NULL;
	label = "Render Actual";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DRenderActualToggleCB;
	DO_ADD_MENU_ITEM_CHECK
	v->render_micheck = w;

	icon = NULL;  
	label = "Cull";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DCullToggleCB;
	DO_ADD_MENU_ITEM_CHECK
	v->cull_micheck = w;

	icon = NULL;   
	label = "Rotations & Translations";
	accel_key = 0; 
	accel_mods = 0;
	func_cb = View3DTranslationsToggleCB;
	DO_ADD_MENU_ITEM_CHECK
	v->translations_micheck = w;

	icon = NULL;
	label = "Alpha Channel";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DEnableAlphaChannelToggleCB;
	DO_ADD_MENU_ITEM_CHECK
	v->enable_alpha_channel_micheck = w;

	DO_ADD_MENU_SEP

	icon = (guint8 **)icon_properties_20x20_xpm;
	label = "Camera Properties...";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DCameraPropertiesMapCB;
	DO_ADD_MENU_ITEM_LABEL

	DO_ADD_MENU_SEP

	icon = NULL;
	label = "Set Background Image...";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DSetBGImageCB;
	DO_ADD_MENU_ITEM_LABEL
	v->set_bgimg_mi = w;

	icon = NULL;
	label = "Clear Background Image";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DClearBGImageCB;
	DO_ADD_MENU_ITEM_LABEL
	v->clear_bgimg_mi = w;

	DO_ADD_MENU_SEP

	icon = NULL;
	label = "Maximize/Restore";
	accel_key = 0;
	accel_mods = 0;
	func_cb = View3DMaximizeRestoreMCB;
	DO_ADD_MENU_ITEM_LABEL
	v->maximize_restore_mi = w;

#undef DO_ADD_MENU_SEP
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_ITEM_LABEL

	/* Need to attach basic "hide" signal to menu so we know when
	 * it is unmapped.
	 */
	gtk_signal_connect(
	    GTK_OBJECT(menu), "hide",
	    GTK_SIGNAL_FUNC(View3DMenuHideCB),
	    (gpointer)v
	);


	/* Reset values */
	View3DReset(v);


	return(v);
}

/*
 *      Sets the color palette for the 3d view. If the palette is NULL
 *	then the current view palette will be reset.
 */
void View3DSetPalette(
	vma_view3d_struct *v, vma_view_palette_struct *palette
)
{
	if(v != NULL)
	{
	    if(palette == NULL)
		memset(&v->palette, 0x00, sizeof(vma_view_palette_struct));
	    else
		memcpy(&v->palette, palette, sizeof(vma_view_palette_struct));
	}
	return;
}

/*
 *      Makes the glarea view widget on the view into context.
 *      Returns 0 on success or -1 on error.
 */
gint View3DGLEnableContext(vma_view3d_struct *v)
{
	GtkWidget *w;
	gint status;

 
	if(v == NULL)
	    return(-1);
 
	if(!v->view_realized)
	    return(-1);

	w = v->view;
	if(w != NULL)
	{
	    status = gtk_gl_area_make_current(GTK_GL_AREA(w));
	    if(status)
		return(0);
	    else
		return(-1);
	}
  
	return(-1);    
}

/*
 *      Updates some memu items to correspond to the other members
 *      of the given 3d view structure.
 */
void View3DUpdateMenus(vma_view3d_struct *v)
{
	static gbool reenterant = FALSE;
	gbool sensitivity, state;
	GtkWidget *w;
	ma_editor_struct *editor;


	if(v == NULL)
	    return;

	if(!v->initialized)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Get pointer to editor structure (can be NULL) */
	editor = v->editor_ptr;

#define SET_WIDGET_SENSITIVITY  \
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitivity); \
}

#define SET_CHECK_MENU_ITEM_STATE	\
{ \
 if(w != NULL) \
   GTK_CHECK_MENU_ITEM(w)->active = state; \
}

	/* Begin updating menu */

	/* Enabled state */
	w = v->enabled_micheck;
	state = ((v->flags & VMA_VIEW_FLAG_ENABLED) ? TRUE : FALSE);
	SET_CHECK_MENU_ITEM_STATE

	/* Undo */
	w = v->undo_milabel;
	if((w != NULL) && (editor != NULL))  
	{
	    gint total = editor->total_undos;
	    gchar text[256];

	    (*text) = '\0';
	    strcat(text, "Undo");
	    if(total > 0)
	    {
		vma_undo_common_struct *u = editor->undo[total - 1];
		gchar numstr[80];
	 
		if(u != NULL)
		{
		    if(u->name != NULL)
		    {
			strcat(text, " ");
			strncat(text, u->name, 80);
			text[256 - 1] = '\0';
		    }
		}
		sprintf(numstr, "(%i)", total);
		strcat(text, numstr);
	    }
	    gtk_label_set_text(GTK_LABEL(w), text);
	}
	w = v->undo_mi;  
	sensitivity = FALSE;
	if((w != NULL) && (editor != NULL))
	{
	    if(editor->total_undos > 0)
		sensitivity = TRUE;
	}
	SET_WIDGET_SENSITIVITY

	/* Redo */
	w = v->redo_milabel;
	if((w != NULL) && (editor != NULL))  
	{
	    gint total = editor->total_redos;
	    gchar text[256];

	    *text = '\0';
	    strcat(text, "Redo");
	    if(total > 0)
	    {
		vma_undo_common_struct *u = editor->redo[total - 1];
		gchar numstr[80];

		if(u != NULL)
		{
		    if(u->name != NULL)
		    {
			strcat(text, " ");
			strncat(text, u->name, 80);
			text[256 - 1] = '\0';
		    }
		}
		sprintf(numstr, "(%i)", total);
		strcat(text, numstr);
	    }
	    gtk_label_set_text(GTK_LABEL(w), text);
	}
	w = v->redo_mi;  
	sensitivity = FALSE;
	if((w != NULL) && (editor != NULL))
	{
	    if(editor->total_redos > 0)
		sensitivity = TRUE;
	}
	SET_WIDGET_SENSITIVITY


	/* Render actual toggle */
	w = v->render_toggle;
	if(w != NULL)
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w),
		v->render_state
	    );
	/* Render actual check menu item */
	w = v->render_micheck;
	if(w != NULL)
	    gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(w),
		v->render_state
	    );

	/* Cull toggle */
	w = v->cull_toggle;
	if(w != NULL)
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w),
		v->cull_state
	    );
	/* Cull check menu item */
	w = v->cull_micheck;
	if(w != NULL)
	    gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(w),
		v->cull_state
	    );

	/* Translations toggle */
	w = v->translations_toggle; 
	if(w != NULL)
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w),
		v->translations_state
	    );
	/* Translations check menu item */
	w = v->translations_micheck; 
	if(w != NULL)
	    gtk_check_menu_item_set_active( 
		GTK_CHECK_MENU_ITEM(w),
		v->translations_state
	    );

	/* Enable alpha channel check menu item */
	w = v->enable_alpha_channel_micheck;
	if(w != NULL)
	    gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(w),
		v->enable_alpha_channel
	    );

	/* Clear background image */
	w = v->clear_bgimg_mi;
	sensitivity = ((v->bg_image == NULL) ? FALSE : TRUE);
	SET_WIDGET_SENSITIVITY


#undef SET_WIDGET_SENSITIVITY
#undef SET_CHECK_MENU_ITEM_STATE

	reenterant = FALSE;

	return;
} 

/*
 *      Reset all drag modifiers on the given view and resets the
 *      cursor. This is needed in some cases where the drag is not
 *      reset (ie after selecting multiple primitives when the pointer
 *      is busy with the confirmation dialog with the user).
 */
void View3DResetDragMods(vma_view3d_struct *v)
{
	GtkWidget *w;

	if(v == NULL)
	    return;

	w = v->view;
	if(w == NULL)
	    return;

	v->flags &= ~VMA_VIEW_FLAG_BUTTON1;
	v->flags &= ~VMA_VIEW_FLAG_BUTTON2;
	v->flags &= ~VMA_VIEW_FLAG_BUTTON3;
	v->drag_state = VMA_VIEW3D_DRAG_NONE;

	if(!GTK_WIDGET_NO_WINDOW(w))
	    gdk_window_set_cursor(w->window, NULL);
}

/*
 *	Resets values on the 3d view.
 */
void View3DReset(vma_view3d_struct *v)
{
	GtkWidget *w;


	if(v == NULL)
	    return;

	if(!v->initialized)
	    return;

	/* Begin resetting values */

	v->flags |= VMA_VIEW_FLAG_ENABLED;
	w = v->enabled_micheck;
	if(w != NULL)
	    GTK_CHECK_MENU_ITEM(w)->active =  TRUE;

	if(VMACFGItemListGetValueI(option, VMA_CFG_PARM_POINTER_EMULATE_2BUTTON))
	    v->flags |= VMA_VIEW_FLAG_POINTER_EMULATE_2BUTTON;
	else
	    v->flags &= ~VMA_VIEW_FLAG_POINTER_EMULATE_2BUTTON;

	v->position_decimals = ViewPositionDecimals();
	v->angle_decimals = ViewAngleDecimals();
	v->render_state = ViewRenderState();
	v->cull_state =  ViewCullState();
	v->cull_direction = ViewCullDirection();
	v->translations_state = ViewTranslationsState();
	v->enable_alpha_channel = ViewEnableAlphaChannel();

	v->cam_clip_near = ViewCameraClipNear();
	v->cam_clip_far = ViewCameraClipFar();
	v->cam_fov = ViewCameraFOV();

	v->cam_x = 5.0;
	v->cam_y = 5.0;
	v->cam_z = 5.0;

	v->cam_h = 1.25 * PI;
	v->cam_p = 0.25 * PI;
	v->cam_b = 0.0 * PI;

	v->grid_spacing = VMA_DEF_GRID_SPACING;
	v->move_rate = VMA_DEF_VIEW3D_MOVE_RATE;

	v->drag_state = VMA_VIEW3D_DRAG_NONE;
	v->w_last_i = 0;
	v->w_last_j = 0;

	/* Unload background image */
	ViewBGImageUnload(v->bg_image);
	v->bg_image = NULL;

	View3DMoveStrafeCB(v);
	View3DTurnCB(v);
	View3DGridCB(v);
	View3DMoveRateCB(v);
	View3DUpdateMenus(v);

	return;
}

/*
 *	Destroys a 3d view, deallocating any sub resources and the
 *	structure itself..
 */
void View3DDestroy(vma_view3d_struct *v)
{
	GtkWidget **w;
	GdkCursor **cur;


	if(v == NULL)
	    return;

	if(v->initialized)
	{
#define DO_DESTROY_CURSOR       \
{ \
 if((*cur) != NULL) \
 { \
  GdkCursor *tc = *cur; \
  (*cur) = NULL; \
  gdk_cursor_destroy(tc); \
 } \
}

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
	    /* Unload background image */
	    ViewBGImageUnload(v->bg_image);
	    v->bg_image = NULL;


	    /* Begin destroying widgets */

	    w = &v->view;
	    DO_DESTROY_WIDGET

	    w = &v->menu;
	    v->enabled_micheck = NULL;
	    v->undo_mi = NULL;
	    v->undo_milabel = NULL;
	    v->redo_mi = NULL;
	    v->redo_milabel = NULL;
	    v->render_micheck = NULL;
	    v->cull_micheck = NULL;
	    v->translations_micheck = NULL;
	    v->enable_alpha_channel_micheck = NULL;
	    v->set_bgimg_mi = NULL;
	    v->clear_bgimg_mi = NULL;
	    v->maximize_restore_mi = NULL;
	    DO_DESTROY_WIDGET

	    w = &v->enabled_da;
	    DO_DESTROY_WIDGET

	    w = &v->toplevel;
	    DO_DESTROY_WIDGET


	    cur = &v->move_strafe_cur;
	    DO_DESTROY_CURSOR
	    cur = &v->move_rotate_cur;
	    DO_DESTROY_CURSOR
	    cur = &v->move_height_cur;
	    DO_DESTROY_CURSOR
	    cur = &v->rotate_cur;
	    DO_DESTROY_CURSOR

#undef DO_DESTROY_WIDGET
#undef DO_DESTROY_CURSOR
	}

	/* Deallocate structure itself */
	g_free(v);
}
