/*
 * VFenc.c  - char encoding routines
 *
 *  Programmmed by Werner Lemberg
 *  E-Mail:  a7621gac@awiuni11.bitnet
 *
 *  Edition History
 *  19 Feb 1996  First version.
 */


/* This file is part of VFlib
 *
 * Copyright (C) 1993 Hirotsugu KAKUGAWA.   All rights reserved.
 *
 * VFlib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
 * to anyone for the consequences of using it or for whether it serves any
 * particular purpose or works at all, unless he says so in writing.  Refer
 * to the GNU General Public License for full details.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * VFlib, but only under the conditions described in the GNU
 * General Public License.  A copy of this license is supposed to have been
 * given to you along with VFlib so you can know your rights and
 * responsibilities.  It should be in a file named COPYING.  Among other
 * things, the copyright notice and this notice must be preserved on all
 * copies. 
 */

#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <ctype.h>
#include  "config.h"
#include  "defs.h"
#include  "_VF.h"
#include  "VF.h"
#include  "VFenc.h"


Import FontTable FTable[];  /* defined in VFlib.c */

int Mapping[MAX_FONTS];


Private int init_mappings();
Private int deinit_mappings();
Private int map_enc();
Private int convert_enc();


/*
 * VFE_SearchEncoding
 *
 *    Converts a string which names an encoding (e.g. as used for the "en" 
 *    entry in the vfoncap database) into a number representing this
 *    encoding. If the entry is not known or an error has occurred, -1 is
 *    returned.
 *
 *    To simplify the interface, comparison of strings will ignore lowercase
 *    and uppercase. Both encoding strings and encoding numbers are defined
 *    in the VFenc.h header file.
 *
 * ARGUMENTS:
 *   entry: encoding string.
 * RETURN VALUE (integer):
 *   -1 :   entry illegal or not known.
 *   (non-negative integer): encoding number.
 */

int
VFE_SearchEncoding(entry)
  char *entry;
{
  int  i;
  char *p;
  char buf[160];

  if (entry == NULL || *entry == '\0')
    return -1;  /* ERR: ILL PARAM */

  strncpy(buf, entry, sizeof(buf) - 1);
  p = buf;
  while (*p)
    *p++ = toupper(*p);

  for (i = 0; EncTable[i].EncString != NULL; i++)
    if (strcmp(EncTable[i].EncString, buf) == 0)
      return EncTable[i].EncValue;

  return -1;  /* ERR: entry not found */
}


/*
 * VFE_SearchCharSet
 *
 *    Converts a string which names a character set (e.g. as used for the
 *    "cs" entry in the vfontcap database) into a number representing this
 *    character set. If the entry is not known or an error has occurred, -1
 *    is returned.
 *
 *    To simplify the interface, comparison of strings will ignore lowercase
 *    and uppercase. Both character set strings and character set numbers are
 *    defined in the VFenc.h header file.
 *
 * ARGUMENTS:
 *   entry: character set string.
 * RETURN VALUE (integer):
 *   -1 :   entry illegal or not known.
 *   (non-negative integer): character set number.
 */

int
VFE_SearchCharSet(entry)
  char *entry;
{
  int  i;
  char *p;
  char buf[160];

  if (entry == NULL || *entry == '\0')
    return -1;  /* ERR: ILL PARAM */

  strncpy(buf, entry, sizeof(buf) - 1);
  p = buf;
  while (*p)
    *p++ = toupper(*p);

  for (i = 0; ChSetTable[i].CharSetString != NULL; i++)
    if (strcmp(ChSetTable[i].CharSetString, buf) == 0)
      return ChSetTable[i].CharSetValue;

  return -1;  /* ERR: entry not found */
}


/*
 * VFE_GetEncoding
 *
 *   Gets the encoding of an opened font.
 *
 * ARGUMENTS:
 *   fid :  font ID
 * RETURN VALUE (integer):
 *   -1 : illegal fid.
 *   (non-negative integer): encoding number.
 */

int
VFE_GetEncoding(fid)
  int  fid;
{
  if (FTable[fid].Fobj == (FontObj*) NULL)
    return -1;  /* ERR: ill fid */
  if (FTable[fid].Fobj->GetEnc == NULL)
    return -1;  /* ERR: ill module */

  return (FTable[fid].Fobj->GetEnc)(FTable[fid].Fobj);
}


/*
 * VFE_GetCharSet
 *
 *   Gets the character set of an opened font.
 *
 * ARGUMENTS:
 *   fid :  font ID
 * RETURN VALUE (integer):
 *   -1 : illegal fid.
 *   (non-negative integer): character set number.
 */

int
VFE_GetCharSet(fid)
  int  fid;
{
  if (FTable[fid].Fobj == (FontObj*) NULL)
    return -1;  /* ERR: ill fid */
  if (FTable[fid].Fobj->GetCharSet == NULL)
    return -1;  /* ERR: ill module */

  return (FTable[fid].Fobj->GetCharSet)(FTable[fid].Fobj);
}


/*
 * VFE_Init
 *
 *   Initializes VFlib and mapping tables.
 *
 * ARGUMENTS:
 *   vfcap :  A path name for vfontcap file.  If it is null pointer, default
 *            file is used.
 * RETURN VALUE (integer):
 *   0 :  if succeeded.
 *  -1 :  if failed to initialize.  (maybe, vfontcap is not found, or lack of 
 *         memory) 
 */

int
VFE_Init(vfcap)
  char  *vfcap;
{
  if (! init_mappings())
    return -1;

  return VF_Init(vfcap);
}


/*
 * VFE_Deinit
 *
 *   Deinitializes VFlib and mapping tables.
 *
 * ARGUMENTS:
 *   none
 * RETURN VALUE (integer):
 *   0 : always succeeds.
 */

int
VFE_Deinit()
{
  deinit_mappings();
  return VF_Deinit();
}


/*
 * VFE_OpenFont
 *
 *    Opens a font with a certain encoding.
 *
 * ARGUMENTS:
 *   fontname: font name entry.
 *   encoding: encoding number.
 * RETURN VALUE (integer):
 *   -1 :   error.
 *   (non-negative integer): font ID.
 */

int
VFE_OpenFont(fontname, encoding)
  char *fontname;
  int  encoding;
{
  int fid;

  if ((fid = VF_OpenFont(fontname)) == -1)
    return -1;

  if (! map_enc(encoding, VFE_GetEncoding(fid))){
    VF_CloseFont(fid);
    return -1;  /* ERR: mapping not possible */
  }

  return fid;
}


/*
 * VFE_CloseFont
 *
 *   Closes a font (specified by font ID). This function is identical to
 *   VF_CloseFont() and is only provided for orthogonality.
 *
 * ARGUMENTS:
 *   fid :  A font ID to close.
 * RETURN VALUE (integer):
 *   0 : (always succeeds)
 */

int
VFE_CloseFont(fid)
  int  fid;
{
  return VF_CloseFont(fid);
}


/*
 * VFE_CloseAllFonts
 *
 *   Closes all opened fonts. This function is identical to
 *   VF_CloseAllFonts() and is only provided for orthogonality.
 *
 * ARGUMENTS:
 *   none
 * RETURN VALUE (integer):
 *   0 : (always succeeds)
 */

int
VFE_CloseAllFonts()
{
  return VF_CloseAllFonts();
}


/*
 * VFE_GetBitmap
 *
 *   Gets a bitmap image for a character (in a certain encoding).
 *
 * ARGUMENTS:
 *   CJKcode : character code of wanted glyph.
 *   fid     : font ID.
 *   w       : width of bitmap.
 *   h       : height of bitmap.
 *   bw      : bytes per line of bitmap.
 *   bo      : bit offset of the generated bitmap image from left side.
 *   bm_buf  : a memory area to hold the generated bitmap.
 * RETURN VALUE (integer):
 *    0 : if successful
 *   -1 : if error
 */

int
VFE_GetBitmap(CJKcode, fid, w, h, bw, bo, bm_buf)
  int   CJKcode;
  int   fid;
  int   w;
  int   h;
  int   bw;
  int   bo;
  char  *bm_buf;
{
  int ccode;

  if ((ccode = convert_enc(CJKcode, fid)) == -1)
    return -1;  /* ERR: illegal character */

  return VF_GetBitmap(ccode, fid, w, h, bw, bo, bm_buf);
}


/*
 *  VFE_GetOutline
 *
 *   Gets outline data of a character (in a certain encoding).
 *
 * ARGUMENTS:
 *   CJKcode  : character code of wanted glyph.
 *   fid      : font ID
 * RETURN VALUE (pointer to long):
 *   NULL : if failed to get outline.
 *   otherwise, pointer to outline data.  Use VF_DrawOutline() to generate
 *      bitmap image from this outline data, and VF_FreeOutline() to
 *      release memory allocated for outline data.
 */

long*
VFE_GetOutline(CJKcode, fid)
  int   CJKcode;
  int   fid;
{
  int ccode;

  if ((ccode = convert_enc(CJKcode, fid)) == -1)
    return NULL;  /* ERR: illegal character */

  return VF_GetOutline(ccode, fid);
}


/*
 *  VFE_GetOutline2
 *
 *   Gets outline data of a character (in a certain encoding) in font
 *   dependent coordinates.
 *
 * ARGUMENTS:
 *   CJKcode  : character code of wanted glyph.
 *   fid      : font ID
 * RETURN VALUE (pointer to long):
 *   NULL : if failed to get outline.
 *   otherwise, pointer to outline data.  Use VFE_DrawOutline() to generate
 *      bitmap image from this outline data, and VFE_FreeOutline() to
 *      release memory allocated for outline data.
 */

long*
VFE_GetOutline2(CJKcode, fid)
  int   CJKcode;
  int   fid;
{
  int ccode;

  if ((ccode = convert_enc(CJKcode, fid)) == -1)
    return NULL;  /* ERR: illegal character */

  return VF_GetOutline2(ccode, fid);
}


/*
 * VFE_DrawOutline
 *
 *   Gets bitmap image from outline data. This function is identical to
 *   VF_DrawOutline() and is only provided for orthogonality.
 *
 * ARGUMENTS:
 *   vfdata  : outline data.
 *   fid     : font ID.
 *   w       : width of bitmap.
 *   h       : height of bitmap.
 *   bw      : bytes per line of bitmap.
 *   bo      : bit offset of the generated bitmap image from left side.
 *   bm_buf  : a memory area to hold the generated bitmap.
 * RETURN VALUE (integer):
 *    0 : if successful
 *   -1 : if error
 */

int
VFE_DrawOutline(vfdata, fid, w, h, bw, bo, bm_buf)
  long  *vfdata;
  int   fid;
  int   w;
  int   h;
  int   bw;
  int   bo;
  char  *bm_buf;
{
  return VF_DrawOutline(vfdata, fid, w, h, bw, bo, bm_buf);
}


/*
 * VFE_FreeOutline
 *
 *   Releases outline data allocated by VFE_GetOutline(). This function is
 *   identical to VF_FreeOutline() and is only provided for orthogonality.
 *
 * ARGUMENTS:
 *   vfdata : outline data allocated by VFE_GetOutline().
 *   fid    : font ID.
 * RETURN VALUE (integer):
 *    0 : if successful
 *   -1 : if error
 */ 
 
int
VFE_FreeOutline(vfdata, fid)
  long  *vfdata;
  int   fid;
{
  return VF_FreeOutline(vfdata, fid);
}


/*
 * Initialize mapping tables.
 *
 * Returns 1 on success, 0 on error.
 */

Private int 
init_mappings()
{
  return 1;
}


/*
 * Deinitialize mapping tables.
 *
 * Always returns 1.
 */

Private int
deinit_mappings()
{
  return 1;
}


/*
 * Search mapping table.
 *
 * Returns 1 on success, 0 if mapping is not possible.
 */

Private int
map_enc(in_enc, out_enc)
  int in_enc;
  int out_enc;
{
  return 1;
}


/*
 * Convert encoding.
 *
 * Returns converted character or -1 if CJKcode is not valid or not mappable.
 */

Private int
convert_enc(CJKcode, fid)
  int CJKcode;
  int fid;
{
  return CJKcode;
}
