
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL

#include "PDFContentView.h"
#include "ExtendedImageView.h"
#include "ViewPDF.h"

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSNotification.h>
#include <Foundation/NSValue.h>
#include <AppKit/NSClipView.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSScroller.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSImageView.h>


/*
 * Non-Public methods.
 */
@interface PDFContentView(Private)
- (void) _createScrollView;
- (void) _createImageView;
- (void) _createMessageView;
- (void) _centerImageInScrollView;
- (void) _updateScrollers;
- (void) _notifyDragScroll: (id)notification;
@end

/*
 * This view displays an image within a ScrollView. Additionaly,
 * some controls are provided to manipulate the display properties
 * (eg. the zoom factor).
 */
@implementation PDFContentView

/*
 * Designated initializer.
 */
- (id) initWithFrame: (NSRect)frame
{
   if (self = [super initWithFrame: frame])
   {
      [self _createScrollView];
      [self _createImageView];
      [self _createMessageView];

      [self setAutoresizesSubviews: YES];
      [self setAutoresizingMask: (NSViewWidthSizable |
                                  NSViewHeightSizable)];

      messageDisplayed = NO;

      usePDFImageRep = 
         [[NSUserDefaults standardUserDefaults] boolForKey: PREFS_USE_PDF_IMAGEREP];

      imageRep  = nil;

      pageNum = 1;
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc PDFContentView, retain count is %d", [self retainCount]);

   [scrollView release];
   [imageView release];
   [messageView release];
   [imageRep release];
   
   [super dealloc];
}


/*
 * Returns the NSImageView that is embedded in this
 * view.
 */
- (id) imageView
{
   return imageView;
}


/*
 * When resized, center the image in the scrollview.
 */
- (void) resizeWithOldSuperviewSize: (NSSize)oldBoundsSize 
{
   [super resizeWithOldSuperviewSize: oldBoundsSize];
   //[self centerImageInScrollView];
   //[self updateScrollers];
}



/*
 * Set the document to be displayed. Note that the
 * document is _not_ retained by this method.
 */
- (void) setDocument: (PDFDocument*)_pdfDoc
{
   NSAssert(_pdfDoc, @"pdfDoc is nil");
   pdfDoc = _pdfDoc;

   NSSize   imageSize;
   NSImage* image;

   imageSize = NSMakeSize([pdfDoc pageWidth: 1],
                          [pdfDoc pageHeight: 1]);
      
   imageRep = [[PDFImageRep alloc] initWithDocument: pdfDoc];
   [imageRep setSize: imageSize];

   image = [[[NSImage alloc] init] autorelease];
   [image addRepresentation: imageRep];

   [image setCacheMode: NSImageCacheAlways];
   [image setBackgroundColor: [NSColor whiteColor]];

   NSAssert([self imageView], @"No ImageView");
   [[self imageView] setImage: image];
   // image is retained by the imageview
}


/*
 * Set the page to be displayed. The content of this
 * page is scaled by the specified factor when drawn
 * to the screen. A factor of 1.0 is equal to 100
 * percent. Note that updateView has to be called
 * to let the changes take effect.
 */
- (void) setPage: (int)_pageNum scaledBy: (float)factor
{
   NSSize scaledSize;
   NSSize unscaledSize;
   double hDPI, vDPI;

   NSAssert(pdfDoc, @"No PDFDocument is associated with the content view");
   
   unscaledSize = NSMakeSize([pdfDoc pageWidth: pageNum],
                             [pdfDoc pageHeight: pageNum]);
   
   scaledSize = NSMakeSize(unscaledSize.width * factor,
                           unscaledSize.height * factor);
   
   hDPI = (scaledSize.width / unscaledSize.width) * PDFBaseResolution;
   vDPI = (scaledSize.height / unscaledSize.height) * PDFBaseResolution;
   
   pageNum = _pageNum;
   [imageRep setPageNum: pageNum];
   [imageRep setResolution: (hDPI < vDPI ? hDPI : vDPI)];
   [[[self imageView] image] recache];
   [[[self imageView] image] setSize: [imageRep size]];

   [self removeMessage];
}


/*
 * Set the size of the internal image view. This can be used
 * to scale the displayed page becaue  the internal image view
 * is configured to scale it's image when resized.
 */
- (void) setPageSize: (NSSize)size
{
   NSRect viewFrame;

   viewFrame = NSMakeRect(0, 0, size.width, size.height);
   //?? [imageView setFrame: viewFrame];
}


/*
 * Repaint the the view, update the scrollers ...
 */
- (void) updateView
{
   NSLog(@"[PDFContentView updateView]");

   [self sizeToFitPage];
   [imageView setNeedsDisplay: YES];
   [self setNeedsDisplay: YES];
}


/*
 * Get the embedded image scroll view.
 */
- (ExtendedScrollView*) scrollView
{
   return scrollView;
}


/*
 * Display a message in the internal ImageView.
 * This can be used to display informations for
 * the user (eg. when work is progress). The
 * message will disapper when removeMessage
 * or setImage is called.
 */
- (void) displayMessage: (NSString*)message
{
   NSRect mFrame;
   NSRect iFrame;

   [messageView setStringValue: message];
   [messageView sizeToFit];
   
   iFrame = [imageView frame];
   mFrame = [messageView frame];

   mFrame = NSMakeRect(NSMidX(iFrame) - (NSWidth(mFrame) / 2),
                       NSMidY(iFrame) - (NSHeight(mFrame) / 2),
                       NSWidth(mFrame),
                       NSHeight(mFrame));

   [messageView setFrame: mFrame];
                             
   [imageView addSubview: messageView];
   messageDisplayed = YES;
}


/*
 * Remove a displayed message from the ImageView.
 * Use this only if displayMessage has been
 * called before.
 */
- (void) removeMessage
{
   if (messageDisplayed)
   {
      [messageView removeFromSuperview];
      messageDisplayed = NO;
   }
}


/*
 * Resize the internal ImageView so that is shows the
 * complete current page.
 */
- (void) sizeToFitPage
{
   NSSize imageSize;
   NSRect viewFrame;

   imageSize = [[imageView image] size];
   viewFrame = NSMakeRect(0, 0, imageSize.width, imageSize.height);

   [imageView setFrame: viewFrame];
}


/*
 * Center the image in it's scrollview when necessary.
 */
- (void) _centerImageInScrollView
{
   NSRect scrollViewFrame;
   NSSize imageSize;
   NSRect imageFrame;
   BOOL   needsResize = NO;

   //NSLog(@"center image");
   scrollViewFrame = [scrollView documentVisibleRect];
   imageFrame      = [imageView frame];

   imageSize = [imageView frame].size;

   if (scrollViewFrame.size.width > imageSize.width)
   {
      imageFrame.size.width = scrollViewFrame.size.width;
      needsResize = YES;
   }
   else if (scrollViewFrame.size.width < imageSize.width)
   {
      imageFrame.size.width = imageSize.width;
      needsResize = YES;
   }

   if (scrollViewFrame.size.height > imageSize.height)
   {
      imageFrame.size.height = scrollViewFrame.size.height;
      needsResize = YES;
   }
   else if (scrollViewFrame.size.height < imageSize.height)
   {
      imageFrame.size.height = imageSize.height;
      needsResize = YES;
   }
      
   
   if (needsResize)
   {
      /*
      NSLog(@"   x:%f y:%f w:%f h:%f",
            imageFrame.origin.x,
            imageFrame.origin.y,
            imageFrame.size.width,
            imageFrame.size.height);
      */
   
      [imageView setFrame: imageFrame];
   }
}


/*
 * Determine whether Scrollers should be visible or
 * not. This depends on the images size.
 */
- (void) _updateScrollers
{
   NSSize scrollViewSize;
   NSSize imageViewSize;

   //scrollViewSize = [scrollView documentVisibleRect].size;
   scrollViewSize = [scrollView contentSize];
   imageViewSize  = [imageView frame].size;

   if (![scrollView displayPageControls])
   {
      [scrollView setHasHorizontalScroller: scrollViewSize.width < imageViewSize.width];
   }
   else
   {
      [scrollView setHasHorizontalScroller: YES];
   }

   [scrollView setHasVerticalScroller: scrollViewSize.height < imageViewSize.height];
}


/*
 * Create the ScrollView that holds the image.
 */
- (void) _createScrollView
{
   NSRect scrollViewFrame;

   scrollViewFrame = [self frame];
   scrollView = [[ExtendedScrollView alloc] initWithFrame: scrollViewFrame];

   [scrollView setHasHorizontalScroller: YES];
   [scrollView setHasVerticalScroller: YES];
   [scrollView setAutoresizesSubviews: YES];
   [scrollView setAutoresizingMask: (NSViewWidthSizable |
                                     NSViewHeightSizable)];
   [scrollView setDrawsBackground: NO];

   [self addSubview: scrollView];
}


- (void) _createImageView
{
   NSRect   frame;

   frame = NSMakeRect(0, 0,
                      [scrollView contentSize].width,
                      [scrollView contentSize].height);
   imageView = [[ExtendedImageView alloc] initWithFrame: frame];
   //imageView = [[NSImageView alloc] initWithFrame: frame];
   [imageView setImageAlignment: NSImageAlignCenter];
   //[imageView setImageScaling: NSScaleNone];
   [imageView setImageScaling: NSScaleProportionally];

   [scrollView setDocumentView: imageView];

   // get informed about scrolling through mouse dragging
   [[NSNotificationCenter defaultCenter] addObserver: self
                                         selector: @selector(_notifyDragScroll:)
                                         name: N_ScrollingRequested
                                         object: nil];
}


- (void) _createMessageView
{
   NSRect   frame;

   frame = NSMakeRect(0, 0, 0, 0);
   messageView = [[NSTextField alloc] initWithFrame: frame];
   [messageView setFont: [NSFont boldSystemFontOfSize: 16.0]];
   [messageView setBordered: NO];
   [messageView setBezeled: NO];
   [messageView setSelectable: NO];
   [messageView setDrawsBackground: NO];
   [messageView setEditable: NO];
}


- (void) _notifyDragScroll: (id)notification
{
   NSPoint  newOrigin;
   NSSize   scrollAmount;
   NSSize   contentSize;
   NSRect   vRect;

   scrollAmount =
      [[[notification userInfo] objectForKey: UserInfoKeyScrollAmount] sizeValue];

   vRect       = [scrollView documentVisibleRect];
   contentSize = [scrollView contentSize];

   newOrigin = NSMakePoint(vRect.origin.x + scrollAmount.width,
                           vRect.origin.y + scrollAmount.height);

   // check bounds
   /*
   if (newOrigin.x < 0)
   {
      newOrigin.x = 0;
   }
   if (newOrigin. x > (contentSize.height - vRect.size.height))
   {
      newOrigin.x = contentSize.height - vRect.size.height;
   }
   */

   [[scrollView contentView] scrollToPoint:
       [[scrollView contentView] constrainScrollPoint:newOrigin]];
}

@end
