/* TIFF parts: Copyright (c) 1988, 1990 by Sam Leffler.
 * All rights reserved.
 *
 * This file is provided for unrestricted use provided that this
 * legend is included on all tape media and as a part of the
 * software program in whole or part.  Users may copy, modify or
 * distribute this file at will.
 * -----------------------------
 * Modifications for VIPS:  Kirk Martinez 1994
 * 22/11/94 JC
 *	- more general
 *	- memory leaks fixed
 * 20/3/95 JC
 *	- TIFF error handler added
 *	- read errors detected correctly
 *
 * Modified to handle LAB in tiff format.
 * It convert LAB-tiff format to IM_TYPE_LABQ in vips format.
 *  Copyright July-1995 Ahmed Abbood.
 *
 *
 * 19/9/95 JC
 *	- now calls TIFFClose ... stupid
 * 25/1/96 JC
 *	- typo on MINISBLACK ...
 * 7/4/97 JC
 *	- completely redone for TIFF 6
 *	- now full baseline TIFF 6 reader, and does CIELAB as well
 * 11/4/97 JC
 *	- added partial read for tiled images
 * 23/4/97 JC
 *	- extra subsample parameter
 *	- im_istiffpyramid() added
 * 5/12/97 JC
 *	- if loading YCbCr, convert to IM_CODING_LABQ
 * 1/5/98 JC
 *	- now reads 16-bit greyscale and RGB
 * 26/10/98 JC
 *	- now used "rb" mode on systems that need binary open
 * 12/11/98 JC
 *	- no sub-sampling if sub == 1
 * 26/2/99 JC
 *	- ooops, else missing for subsample stuff above
 * 2/10/99 JC
 *	- tiled 16-bit greyscale read was broken
 *	- added mutex for TIFF*() calls
 * 11/5/00 JC
 *	- removed TIFFmalloc/TIFFfree usage
 * 23/4/01 JC
 *	- HAVE_TIFF turns on TIFF goodness
 * 24/5/01 JC
 *	- im_tiff2vips_header() added
 * 11/7/01 JC
 *	- subsample now in input filename
 *	- ... and it's a page number (from 0) instead
 * 21/8/02 JC
 *	- now reads CMYK
 *	- hmm, dpi -> ppm conversion was wrong!
 * 10/9/02 JC
 *	- oops, handle TIFF errors better
 * 2/12/02 JC
 *	- reads 8-bit RGBA
 * 12/12/02 JC
 *	- reads 16-bit LAB
 * 13/2/03 JC
 *	- pixels/cm res read was wrong
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* Turn on debugging output.
#define DEBUG
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#ifndef HAVE_TIFF

#include <vips/vips.h>

int
im_tiff2vips( const char *tiffile, IMAGE *im )
{
	im_errormsg( "im_tiff2vips: TIFF support disabled" );
	return( -1 );
}

int
im_istiffpyramid( const char *name )
{
	return( 0 );
}

int
im_tiff2vips_header( const char *tiffile, IMAGE *im )
{
	im_errormsg( "im_tiff2vips_header: TIFF support disabled" );
	return( -1 );
}

#else /*HAVE_TIFF*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>
#include <vips/thread.h>

#include <tiffio.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Lower and upper bounds for tile cache size. Choose an exact number based on
 * tile size.
 */
#define IM_MAX_TILE_CACHE (250)
#define IM_MIN_TILE_CACHE (5)

/* Scanline-type process function.
 */
typedef PEL *(*scanline_process_fn)( PEL *q, PEL *p, int n, void *user );

/* How to read a TIFF file ... process function, plus some data.
 */
typedef struct {
	scanline_process_fn sfn;	/* Process function */
	void *a;			/* Optional arg to process */
} ReadParams;

/* A tile in our cache.
 */
typedef struct {
	REGION *tile;			/* REGION with private mem for data */
	int time;			/* Time of last use for flush */
} CachedTile;

/* Stuff we track during a partial read.
 */
typedef struct {
	/* Parameters.
	 */
	TIFF *in;
	IMAGE *out;
	scanline_process_fn sfn;	/* Process for this image type */
	void *client;

	/* Geometry.
	 */
	int twidth, theight;		/* Tile size */
	int csize;			/* Cache size */

	/* Only need one of these, since we mutex around TIFF*().
	 */
	im_lock_t tlock;		/* Lock for TIFF*() calls */
	tdata_t *buf;			/* TIFF read buffer */
} ReadInfo;

/* Per-thread stuff - give each one a separate cache for simplicity.
 */
typedef struct {
	/* Parent.
	 */
	ReadInfo *rinf;

	/* Cache. 
	 */
	CachedTile tcache[IM_MAX_TILE_CACHE];
	int time;			/* Increment and mark tiles on use */
} SeqInfo;

/* Reading a YCbCr image ... parameters we use for conversion.
 */
typedef struct {
	/* Input and output.
	 */
	TIFF *tif;			/* From here */
	IMAGE *im;			/* To here */

	/* RGB <-> YCbCr conversion.
	 */
	float LumaRed, LumaGreen, LumaBlue;

	/* RGB -> LAB conversion.
	 */
	void *table;
} YCbCrParams;

/* Handle TIFF errors here. 
 */
static void 
vhandle( char *module, char *fmt, va_list ap )
{
	im_errormsg( "TIFF error in \"%s\": ", module );
	im_verrormsg( fmt, ap );
}

/* Test for field exists.
 */
static int
tfexists( TIFF *tif, ttag_t tag )
{
	uint32 a, b;

	if( TIFFGetField( tif, tag, &a, &b ) ) 
		return( 1 );
	else 
		return( 0 );
}

/* Test a uint16 field. Field must be defined and equal to the value.
 */
static int
tfequals( TIFF *tif, ttag_t tag, uint16 val )
{
	uint16 fld;

	if( !TIFFGetFieldDefaulted( tif, tag, &fld ) ) {
		im_errormsg( "im_tiff2vips: required field %d missing", tag );
		return( 0 );
	}
	if( fld != val ) {
		im_errormsg( "im_tiff2vips: required field %d=%d, not %d",
			tag, fld, val );
		return( 0 );
	}

	/* All ok.
	 */
	return( 1 );
}

/* Get a uint32 field.
 */
static int
tfget32( TIFF *tif, ttag_t tag, int *out )
{
	uint32 fld;

	if( !TIFFGetFieldDefaulted( tif, tag, &fld ) ) {
		im_errormsg( "im_tiff2vips: required field %d missing", tag );
		return( 0 );
	}

	/* All ok.
	 */
	*out = fld;
	return( 1 );
}

/* Get a uint16 field.
 */
static int
tfget16( TIFF *tif, ttag_t tag, int *out )
{
	uint16 fld;

	if( !TIFFGetFieldDefaulted( tif, tag, &fld ) ) {
		im_errormsg( "im_tiff2vips: required field %d missing", tag );
		return( 0 );
	}

	/* All ok.
	 */
	*out = fld;
	return( 1 );
}

/* Free a cache entry.
 */
static void
cache_free( SeqInfo *seq, int i )
{
	if( seq->tcache[i].tile ) {
		(void) im_region_free( seq->tcache[i].tile );
		seq->tcache[i].tile = NULL;
	}
}

/* Free a ReadInfo.
 */
static int
rinfo_free( ReadInfo *rinf )
{
	if( rinf->in ) {
		TIFFClose( rinf->in );
		rinf->in = NULL;
	}

	if( rinf->buf ) {
		im_free( rinf->buf );
		rinf->buf = NULL;
	}

	im_lock_destroy( &rinf->tlock );

	return( 0 );
}

/* Free a SeqInfo.
 */
static int
seq_free( SeqInfo *seq )
{
	int i;

	for( i = 0; i < seq->rinf->csize; i++ )
		cache_free( seq, i );
	
	return( 0 );
}

/* Make a ReadInfo.
 */
static ReadInfo *
rinfo_build( IMAGE *out, TIFF **ptif, ReadParams *params )
{
	TIFF *tif = *ptif;
	ReadInfo *rinf;
	int tpl;

	if( !(rinf = IM_NEW( out, ReadInfo )) )
		return( NULL );
	rinf->in = tif;
	rinf->out = out;
	rinf->sfn = params->sfn;
	rinf->client = params->a;
	rinf->buf = NULL;
	im_lock_init( &rinf->tlock );

	/* We have now taken responsibility for TIFFClose(), so zap ptif to
	 * stop our callers freeing it for us.
	 */
	*ptif = NULL;

	/* We need a close callback to free TIFF resources.
	 */
	if( im_add_close_callback( out, 
		(im_callback_fn) rinfo_free, rinf, NULL ) ) {
		rinfo_free( rinf );
		return( NULL );
	}

	/* Get tiling geometry.
	 */
	if( !tfget32( tif, TIFFTAG_TILEWIDTH, &rinf->twidth ) ||
		!tfget32( tif, TIFFTAG_TILELENGTH, &rinf->theight ) )
		return( NULL );

	/* Find upper bound for cache size ... aim for enough tiles to cover a
	 * line of a 20,000 pixel wide image, plus 50%.
	 */
	tpl = (3 * 20000 / rinf->twidth) / 2;
	rinf->csize = IM_CLIP( IM_MIN_TILE_CACHE, tpl, IM_MAX_TILE_CACHE );
#ifdef DEBUG
	printf( "im_tiff2vips: caching %d %dx%d tiles\n", 
		rinf->csize, rinf->twidth, rinf->theight );
#endif /*DEBUG*/

	if( !(rinf->buf = im_malloc( NULL, TIFFTileSize( rinf->in ) )) )
		return( NULL );

	return( rinf );
}

/* Make a SeqInfo.
 */
static void *
seq_build( IMAGE *out, ReadInfo *rinf )
{
	int i;
	SeqInfo *seq = IM_NEW( rinf->out, SeqInfo );

	if( !seq )
		return( NULL );
	seq->rinf = rinf;

	for( i = 0; i < rinf->csize; i++ )
		seq->tcache[i].tile = NULL;

	seq->time = 0;

	return( (void *) seq );
}

/* Touch a cache entry.
 */
static void
cache_touch( SeqInfo *seq, int i )
{
	assert( i >=0 && i < seq->rinf->csize );

	seq->tcache[i].time = seq->time++;
}

/* Find an empty cache slot and fill it, or find the LRU and reuse that.
 * Return <0 for error.
 */
static int
cache_find( SeqInfo *seq )
{
	ReadInfo *rinf = seq->rinf;
	int i;
	int min, minpos;

	/* Look for empty.
	 */
	for( i = 0; i < rinf->csize; i++ )
		if( !seq->tcache[i].tile ) {
			if( !(seq->tcache[i].tile = 
				im_region_create( rinf->out )) )
				return( -1 );
			return( i );
		}

	/* Find LRU.
	 */
	min = seq->tcache[0].time;
	minpos = 0;
	for( i = 1; i < rinf->csize; i++ )
		if( seq->tcache[i].time < min ) {
			min = seq->tcache[i].time;
			minpos = i;
		}

	return( minpos );
}

/* Run over a TIFF tile with the scanline process function, filling a
 * region.
 */
static void
vips_tile( ReadInfo *rinf, tdata_t *td, REGION *out )
{
	Rect *r = &out->valid;
	PEL *p = (PEL *) td;
	int y;

	for( y = r->top; y < IM_RECT_BOTTOM( r ); y++ ) {
		PEL *q = (PEL *) IM_REGION_ADDR( out, r->left, y );

		p = rinf->sfn( q, p, r->width, rinf->client );
	}
}

/* Find a tile ... if we don't have it in cache, get it. Return the cache
 * index, or <0 for error.
 */
static int
cache_fill( SeqInfo *seq, int ix, int iy )
{
	ReadInfo *rinf = seq->rinf;
	int i;
	REGION *reg;
	Rect area;

	/* Do we have this tile? Should really use a hashtable I guess.
	 */
	for( i = 0; i < rinf->csize; i++ ) {
		reg = seq->tcache[i].tile;

		if( reg && im_rect_includespoint( &reg->valid, ix, iy ) ) {
			/* In cache .. note use.
			 */
			cache_touch( seq, i );

			return( i );
		}
	}

	/* Not there ... pick a slot to use.
	 */
	if( (i = cache_find( seq )) < 0 )
		return( -1 );
	cache_touch( seq, i );
	reg = seq->tcache[i].tile;
	area.left = ix; area.top = iy;
	area.width = rinf->twidth; area.height = rinf->theight;
	if( im_region_local( reg, &area ) )
		return( -1 );

	/* Read that tile.
	 */
	im_lock( &rinf->tlock );
	if( TIFFReadTile( rinf->in, rinf->buf, ix, iy, 0, 0 ) < 0 )
		return( -1 );
	vips_tile( rinf, rinf->buf, reg );
	im_unlock( &rinf->tlock );

	return( i );
}

/* Copy useful parts of from to to.
 */
static void
copy_region( REGION *from, REGION *to )
{
	Rect ovlap;
	int y;

	/* Find common pixels.
	 */
	im_rect_intersectrect( &from->valid, &to->valid, &ovlap );
	assert( !im_rect_isempty( &ovlap ) );

	/* Loop down common area, copying.
	 */
	for( y = ovlap.top; y < IM_RECT_BOTTOM( &ovlap ); y++ ) {
		PEL *p = (PEL *) IM_REGION_ADDR( from, ovlap.left, y );
		PEL *q = (PEL *) IM_REGION_ADDR( to, ovlap.left, y );

		memcpy( q, p, IM_IMAGE_SIZEOF_PEL( from->im ) * ovlap.width );
	}
}

/* Loop over the output region, filling with data from cache.
 */
static int
fill_region( REGION *out, SeqInfo *seq )
{
	ReadInfo *rinf = seq->rinf;
	int x, y;
	Rect *r = &out->valid;

	/* Find top left of tiles we need.
	 */
	int xs = (r->left / rinf->twidth) * rinf->twidth;
	int ys = (r->top / rinf->theight) * rinf->theight;

	for( y = ys; y < IM_RECT_BOTTOM( r ); y += rinf->theight )
		for( x = xs; x < IM_RECT_RIGHT( r ); x += rinf->twidth ) {
			int i;

			if( (i = cache_fill( seq, x, y )) < 0 )
				return( -1 );
			copy_region( seq->tcache[i].tile, out );
		}

	return( 0 );
}

/* Tile-type TIFF reader core - pass in a per-tile transform. Generate into
 * the im and do it all partially.
 */
static int
read_tilewise( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	ReadInfo *rinf;

	/* Make sure we can write PIO-style.
	 */
	if( !(rinf = rinfo_build( im, ptif, params )) || im_poutcheck( im ) )
		return( -1 );

	/* Process and save as VIPS.
	 */
	if( im_demand_hint( im, IM_SMALLTILE, NULL ) )
		return( -1 );
	if( im_generate( im, seq_build, fill_region, seq_free, rinf, NULL ) )
		return( -1 );

	return( 0 );
}

/* Scanline-type TIFF reader core - pass in a per-scanline transform.
 */
static int
read_scanlinewise( TIFF *tif, IMAGE *im, ReadParams *params )
{
	PEL *vbuf;
	tdata_t tbuf;
	int y;

	/* Make sure we can write WIO-style.
	 */
	if( im_outcheck( im ) || im_setupout( im ) )
		return( -1 );

	/* Make VIPS output buffer.
	 */
	if( !(vbuf = IM_ARRAY( im, IM_IMAGE_SIZEOF_LINE( im ), PEL )) )
		return( -1 );

	/* Make TIFF line input buffer.
	 */
	if( !(tbuf = im_malloc( NULL, TIFFScanlineSize( tif ) )) ) 
		return( -1 );

	for( y = 0; y < im->Ysize; y++ ) {
		/* Read TIFF scanline.
		 */
		if( TIFFReadScanline( tif, tbuf, y, 0 ) < 0 ) {
			im_errormsg( "im_tiff2vips: read error" );
			im_free( tbuf );
			return( -1 );
		}

		/* Process and save as VIPS.
		 */
		(void) params->sfn( vbuf, tbuf, im->Xsize, params->a );
		if( im_writeline( y, im, vbuf ) ) {
			im_free( tbuf );
			return( -1 );
		}
	}

	im_free( tbuf );

	return( 0 );
}

/* Read scan-linewise or tile-wise, depending on TIFF organisation.
 */
static int
read_tiff( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;

	if( TIFFIsTiled( tif ) ) {
		if( read_tilewise( ptif, im, params ) )
			return( -1 );
	}
	else {
		/* Read the whole thing into the VIPS image scanline-wise.
		 */
		if( read_scanlinewise( tif, im, params ) ) 
			return( -1 );
	}

	return( 0 );
}

/* Read resolution from a TIFF image.
 */
static int
parse_resolution( TIFF *tif, IMAGE *im )
{
	float x, y;
	int ru;

	if( !TIFFGetFieldDefaulted( tif, TIFFTAG_XRESOLUTION, &x ) ||
		!TIFFGetFieldDefaulted( tif, TIFFTAG_YRESOLUTION, &y ) ||
		!tfget16( tif, TIFFTAG_RESOLUTIONUNIT, &ru ) ) {
		im_warning( "im_tiff2vips: no resolution information "
			"for TIFF image \"%s\"", TIFFFileName( tif ) );
		im->Xres = 1.0;
		im->Yres = 1.0;
		return( 0 );
	}

	switch( ru ) {
	case RESUNIT_NONE:
		break;

	case RESUNIT_INCH:
		/* In pixels-per-inch ... convert to mm.
		 */
		x /= 10.0 * 2.54;
		y /= 10.0 * 2.54;
		break;

	case RESUNIT_CENTIMETER:
		/* In pixels-per-centimetre ... convert to mm.
		 */
		x /= 10.0;
		y /= 10.0;
		break;

	default:
		im_errormsg( "im_tiff2vips: unknown resolution unit" );
		return( -1 );
	}

	/* Set VIPS.
	 */
	im->Xres = x;
	im->Yres = y;

	return( 0 );
}

/* Per-scanline process function for IM_CODING_LABQ.
 */
static PEL *
labpack_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;

	for( x = 0; x < n; x++ ) {
		q[0] = p[0];
		q[1] = p[1];
		q[2] = p[2];
		q[3] = 0;

		q += 4;
		p += 3;
	}

	return( p );
}

/* Read an 8-bit LAB image.
 */
static int
parse_labpack( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 3 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 4; 
	im->Bbits = 8; 
	im->BandFmt = IM_BANDFMT_UCHAR; 
	im->Coding = IM_CODING_LABQ; 
	im->Type = IM_TYPE_LAB; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = labpack_line;
	params->a = NULL;

	return( 0 );
}

/* Per-scanline process function for IM_CODING_LABQ.
 */
static PEL *
labs_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;
	unsigned short *p1 = (unsigned short *) p;
	short *q1 = (short *) q;

	for( x = 0; x < n; x++ ) {
		q1[0] = p1[0] >> 1;
		q1[1] = p1[1] - 32768;
		q1[2] = p1[2] - 32768;

		q1 += 3;
		p1 += 3;
	}

	return( p );
}

/* Read a 16-bit LAB image.
 */
static int
parse_labs( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 3 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 16 ) )
		return( -1 );

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 3; 
	im->Bbits = 16; 
	im->BandFmt = IM_BANDFMT_SHORT; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_LABS; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = labs_line;
	params->a = NULL;

	return( 0 );
}

/* Per-scanline process function for IM_CODING_LABQ.
 */
static PEL *
ycbcr_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;

	for( x = 0; x < n; x++ ) {
		q[0] = p[0];
		q[1] = p[1];
		q[2] = p[2];
		q[3] = 0;

		q += 4;
		p += 3;
	}

	return( p );
}

/* Read a YCbCr image.
 */
static int
parse_ycbcr( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 3 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 4; 
	im->Bbits = 8; 
	im->BandFmt = IM_BANDFMT_UCHAR; 
	im->Coding = IM_CODING_LABQ; 
	im->Type = IM_TYPE_LAB; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = ycbcr_line;
	params->a = NULL;

	return( 0 );
}

/* Per-scanline process function for 1 bit images.
 */
static PEL *
onebit_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract PHOTOMETRIC_INTERPRETATION.
	 */
	int pm = *((int *) flg);
	int x, i, z;
	PEL bits;

	int black = (pm == PHOTOMETRIC_MINISBLACK) ? 0 : 255;
	int white = black ^ -1;

	/* (sigh) how many times have I written this?
	 */
	for( x = 0, i = 0; i < (n >> 3); i++ ) {
		bits = (PEL) p[i];

		for( z = 0; z < 8; z++, x++ ) {
			q[x] = (bits & 128) ? white : black;
			bits <<= 1;
		}
	}

	/* Do last byte in line.
	 */
	bits = p[i];
	for( z = 0; z < (n & 7); z++ ) {
		q[x + z] = (bits & 128) ? white : black;
		bits <<= 1;
	}

	/* Fits exactly? Don't need to skip padding.
	 */
	if( (n & 7) == 0 )
		return( p + i );
	else
		return( p + i + 1 );
}

/* Read a 1-bit TIFF image. Pass in pixel values to use for black and white.
 */
static int
parse_onebit( TIFF **ptif, IMAGE *im, ReadParams *params, int pm )
{
	TIFF *tif = *ptif;
	int *ppm;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 1 ) )
		return( -1 );

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 1; 
	im->Bbits = 8; 
	im->BandFmt = IM_BANDFMT_UCHAR; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_B_W; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	/* Note pm for later.
	 */
	if( !(ppm = IM_ARRAY( im, 1, int )) )
		return( -1 );
	ppm[0] = pm;

	params->sfn = onebit_line;
	params->a = ppm;

	return( 0 );
}

/* Per-scanline process function for 8-bit greyscale images.
 */
static PEL *
greyscale8_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract swap mask.
	 */
	PEL mask = *((PEL *) flg);
	int x;

	/* Read bytes, swapping sense if necessary.
	 */
	for( x = 0; x < n; x++ ) 
		q[x] = p[x] ^ mask;

	return( p + x );
}

/* Read a 8-bit grey-scale TIFF image. 
 */
static int
parse_greyscale8( TIFF **ptif, IMAGE *im, ReadParams *params, int pm )
{
	TIFF *tif = *ptif;
	PEL *mask;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );

	/* Eor each pel with this later.
	 */
	if( !(mask = IM_ARRAY( im, 1, PEL )) )
		return( -1 );
	mask[0] = (pm == PHOTOMETRIC_MINISBLACK) ? 0 : 255;

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 1; 
	im->Bbits = 8; 
	im->BandFmt = IM_BANDFMT_UCHAR; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_B_W; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = greyscale8_line;
	params->a = mask;

	return( 0 );
}

/* Per-scanline process function for 16-bit greyscale images.
 */
static PEL *
greyscale16_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract swap mask.
	 */
	unsigned short mask = *((unsigned short *) flg);
	unsigned short *p1 = (unsigned short *) p;
	unsigned short *q1 = (unsigned short *) q;
	int x;

	/* Read bytes, swapping sense if necessary.
	 */
	for( x = 0; x < n; x++ ) 
		q1[x] = p1[x] ^ mask;

	return( (PEL *) (p1 + x) );
}

/* Read a 16-bit grey-scale TIFF image. 
 */
static int
parse_greyscale16( TIFF **ptif, IMAGE *im, ReadParams *params, int pm )
{
	TIFF *tif = *ptif;
	unsigned short *mask;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 16 ) )
		return( -1 );

	/* Eor each pel with this later.
	 */
	if( !(mask = IM_ARRAY( im, 1, unsigned short )) )
		return( -1 );
	mask[0] = (pm == PHOTOMETRIC_MINISBLACK) ? 0 : 65535;

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 1; 
	im->Bbits = 16; 
	im->BandFmt = IM_BANDFMT_USHORT; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_B_W; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = greyscale16_line;
	params->a = mask;

	return( 0 );
}

/* Per-scanline process function for palette images.
 */
static PEL *
palette_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract maps.
	 */
	PEL *red = ((PEL **) flg)[0];
	PEL *green = ((PEL **) flg)[1];
	PEL *blue = ((PEL **) flg)[2];
	int x;

	/* Read bytes, generating colour.
	 */
	for( x = 0; x < n; x++ ) {
		int i = *p++;

		q[0] = red[i];
		q[1] = green[i];
		q[2] = blue[i];

		q += 3;
	}

	return( p );
}

/* Read a palette-ised TIFF image. Again, we only allow 8-bits for now.
 */
static int
parse_palette( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;
	uint16 *tred, *tgreen, *tblue;
	PEL *red, *green, *blue;
	PEL **maps;
	int i;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );
	
	/* Allocate mem for VIPS colour maps.
	 */
	if( !(red = IM_ARRAY( im, 256, PEL )) ||
		!(green = IM_ARRAY( im, 256, PEL )) ||
		!(blue = IM_ARRAY( im, 256, PEL )) ||
		!(maps = IM_ARRAY( im, 3, PEL * )) )
		return( -1 );

	/* Get maps, convert to 8-bit data.
	 */
	if( !TIFFGetField( tif, TIFFTAG_COLORMAP, &tred, &tgreen, &tblue ) ) {
		im_errormsg( "im_tiff2vips: bad colormap" );
		return( -1 );
	}
	for( i = 0; i < 256; i++ ) {
		red[i] = tred[i] >> 8;
		green[i] = tgreen[i] >> 8;
		blue[i] = tblue[i] >> 8;
	}
	maps[0] = red; maps[1] = green; maps[2] = blue;

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 3; 
	im->Bbits = 8; 
	im->BandFmt = IM_BANDFMT_UCHAR; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_sRGB; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = palette_line;
	params->a = maps;

	return( 0 );
}

/* Per-scanline process function for 8-bit RGB/RGBA.
 */
static PEL *
rgb8_line( PEL *q, PEL *p, int n, IMAGE *im )
{
	int x, b;

	for( x = 0; x < n; x++ ) {
		for( b = 0; b < im->Bands; b++ )
			q[b] = p[b];

		q += im->Bands;
		p += im->Bands;
	}

	return( p );
}

/* Read an 8-bit RGB/RGBA image.
 */
static int
parse_rgb8( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;
	int bands;

	/* Check other TIFF fields to make sure we can read this. Can have 4
	 * bands for RGBA.
	 */
	if( !tfequals( tif, TIFFTAG_BITSPERSAMPLE, 8 ) ||
		!tfget16( tif, TIFFTAG_SAMPLESPERPIXEL, &bands ) )
		return( -1 );
	if( bands != 3 && bands != 4 ) {
		im_errormsg( "im_tiff2vips: 3 or 4 bands RGB TIFF only" );
		return( -1 );
	}

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = bands; 
	im->Bbits = 8; 
	im->BandFmt = IM_BANDFMT_UCHAR; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_sRGB; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = (scanline_process_fn) rgb8_line;
	params->a = im;

	return( 0 );
}

/* Per-scanline process function for RGB16.
 */
static PEL *
rgb16_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;
	unsigned short *p1 = (unsigned short *) p;
	unsigned short *q1 = (unsigned short *) q;

	for( x = 0; x < n; x++ ) {
		q1[0] = p1[0];
		q1[1] = p1[1];
		q1[2] = p1[2];

		q1 += 3;
		p1 += 3;
	}

	return( (PEL *) p1 );
}

/* Read a 16-bit RGB image.
 */
static int
parse_rgb16( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 3 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 16 ) )
		return( -1 );

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 3; 
	im->Bbits = 16; 
	im->BandFmt = IM_BANDFMT_USHORT; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_sRGB; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = rgb16_line;
	params->a = NULL;

	return( 0 );
}

/* Per-scanline process function for CMYK8.
 */
static PEL *
cmyk8_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;

	for( x = 0; x < n; x++ ) {
		q[0] = p[0];
		q[1] = p[1];
		q[2] = p[2];
		q[3] = p[3];

		q += 4;
		p += 4;
	}

	return( p );
}

/* Read a CMYK image.
 */
static int
parse_cmyk( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;

	/* Check other TIFF fields to make sure we can read this.
	 */
	if( !tfequals( tif, TIFFTAG_SAMPLESPERPIXEL, 4 ) ||
		!tfequals( tif, TIFFTAG_BITSPERSAMPLE, 8 ) ||
		!tfequals( tif, TIFFTAG_INKSET, INKSET_CMYK ) )
		return( -1 );

	/* Set VIPS header.
	 */
	if( !tfget32( tif, TIFFTAG_IMAGEWIDTH, &im->Xsize ) ||
		!tfget32( tif, TIFFTAG_IMAGELENGTH, &im->Ysize ) )
		return( -1 );
	im->Bands = 4; 
	im->Bbits = 8; 
	im->BandFmt = IM_BANDFMT_UCHAR; 
	im->Coding = IM_CODING_NONE; 
	im->Type = IM_TYPE_CMYK; 
	if( parse_resolution( tif, im ) )
		return( -1 );

	params->sfn = cmyk8_line;
	params->a = NULL;

	return( 0 );
}

/* Look at PhotometricInterpretation and BitsPerPixel, and try to figure out 
 * which of the image classes this is.
 */
static int
parse_header( TIFF **ptif, IMAGE *im, ReadParams *params )
{
	TIFF *tif = *ptif;
	int pm, bps;

	/* Ban separate planes.
	 */
	if( tfexists( tif, TIFFTAG_PLANARCONFIG ) && 
		!tfequals( tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG ) ) 
		return( -1 );

	/* Try to find out which type of TIFF image it is.
	 */
	if( !tfget16( tif, TIFFTAG_PHOTOMETRIC, &pm ) ||
		!tfget16( tif, TIFFTAG_BITSPERSAMPLE, &bps ) ) 
		return( -1 );

	/* Can set this now.
	 */
	im->Hist = im_strdup( NULL, "TIFF image imported by im_tiff2vips()\n" );

	switch( pm ) {
	case PHOTOMETRIC_CIELAB:
		switch( bps ) {
		case 8:
			if( parse_labpack( ptif, im, params ) )
				return( -1 );
			break;

		case 16:
			if( parse_labs( ptif, im, params ) )
				return( -1 );
			break;

		default:
			im_errormsg( "im_tiff2vips: unsupported depth %d "
				"for LAB image", bps );
			return( -1 );
		}

		break;

	case PHOTOMETRIC_YCBCR:
		/* Easy decision!
		 */
		if( parse_ycbcr( ptif, im, params ) )
			return( -1 );

		break;

	case PHOTOMETRIC_MINISWHITE:
	case PHOTOMETRIC_MINISBLACK:
		switch( bps ) {
		case 1:
			if( parse_onebit( ptif, im, params, pm ) )
				return( -1 );

			break;

		case 8:
			if( parse_greyscale8( ptif, im, params, pm ) )
				return( -1 );

			break;

		case 16:
			if( parse_greyscale16( ptif, im, params, pm ) )
				return( -1 );

			break;

		default:
			im_errormsg( "im_tiff2vips: unsupported depth %d "
				"for greyscale image", bps );
			return( -1 );
		}

		break;

	case PHOTOMETRIC_PALETTE:
		/* Full colour pallette.
		 */
		if( parse_palette( ptif, im, params ) )
			return( -1 );

		break;

	case PHOTOMETRIC_RGB:
		/* Plain RGB.
		 */
		switch( bps ) {
		case 8:
			if( parse_rgb8( ptif, im, params ) )
				return( -1 );
			break;

		case 16:
			if( parse_rgb16( ptif, im, params ) )
				return( -1 );
			break;

		default:
			im_errormsg( "im_tiff2vips: unsupported depth %d "
				"for RGB image", bps );
			return( -1 );
		}

		break;

	case PHOTOMETRIC_SEPARATED:
		if( parse_cmyk( ptif, im, params ) )
			return( -1 );

		break;

	default:
		im_errormsg( "im_tiff2vips: unknown photometric "
			"interpretation %d", pm );
		return( -1 );
	}

	return( 0 );
}

/* Open TIFF for input.
 */
static TIFF *
tiff_openin( char *name )
{
	TIFF *tif;

#ifdef BINARY_OPEN
	if( !(tif = TIFFOpen( name, "rb" )) ) {
#else /*BINARY_OPEN*/
	if( !(tif = TIFFOpen( name, "r" )) ) {
#endif /*BINARY_OPEN*/
		im_errormsg( "im_vips2tiff: unable to open \"%s\" for input",
			name );
		return( NULL );
	}

	return( tif );
}

/* Pull out the nth directory.
 */
static TIFF *
get_directory( const char *filename, int page )
{
	TIFF *tif;
	int i;

	if( !(tif = tiff_openin( (char *) filename )) )
		return( NULL );

	for( i = 0; i < page; i++ ) 
		if( !TIFFReadDirectory( tif ) ) {
			/* Run out of directories.
			 */
			TIFFClose( tif );
			return( NULL );
		}

	return( tif );
}

int
im_istiffpyramid( const char *name )
{
	TIFF *tif;

	if( (tif = get_directory( name, 2 )) ) {
		/* We can see page 2 ... assume it is.
		 */
		TIFFClose( tif );
		return( 1 );
	}

	return( 0 );
}

/* Just parse the header.
 */
int
im_tiff2vips_header( const char *tiffile, IMAGE *im )
{
	ReadParams params;
	TIFF *tif;
	int res;

	TIFFSetErrorHandler( (TIFFErrorHandler) vhandle );

	if( !(tif = tiff_openin( (char *) tiffile )) ) 
		return( -1 );

	res = parse_header( &tif, im, &params );

	if( tif )
		TIFFClose( tif );

	return( res );
}

int
im_tiff2vips( const char *filename, IMAGE *im )
{
	ReadParams params;
	TIFF *tif;
	char name[FILENAME_MAX];
	char mode[FILENAME_MAX];
	int res;
	int page;

#ifdef DEBUG
	printf( "im_tiff2vips: libtiff version is \"%s\"\n", TIFFGetVersion() );
#endif /*DEBUG*/

	/* Override the default TIFF error handler.
	 */
	TIFFSetErrorHandler( (TIFFErrorHandler) vhandle );
	
	/* Parse the page number out of the filename.
	 */
	im__filename_split( filename, name, mode );
	if( strcmp( mode, "" ) != 0 )
		page = atoi( mode );
	else
		page = 0;

	/* Sanity!
	 */
	if( page < 0 || page > 1000 ) {
		im_errormsg( "im_tiff2vips: bad page number %d", page );
		return( -1 );
	}

	/* Try to find that page.
	 */
	if( !(tif = get_directory( name, page )) ) {
		im_errormsg( "im_tiff2vips: TIFF file does not "
			"contain page %d", page );
		return( -1 );
	}

	res = parse_header( &tif, im, &params ) ||
		read_tiff( &tif, im, &params );

	/* If the TIFF pointer hasn't been 'stolen' by the partial reader (see
	 * rinfo_free() above), then it's up to us to free it.
	 */
	if( tif )
		TIFFClose( tif );

	return( res );
}

#endif /*HAVE_TIFF*/
