/* @(#) im_gradient: Extracts features from an image using the 
 * @(#) gradient operator.  Iput mask of any size.  Two mask are used:
 * @(#) a given mask and a rotated by 90 degrees
 * @(#)  Based on A Jain book Fundamentals of digital image processing
 * @(#) page 349: out = |g1| + |g2|
 * @(#)  The program can be easily modified in order to produce magitude and
 * @(#) phase. The input mask should have been set by a call to im_read_imask().
 * @(#)  Usage:
 * @(#) int im_gradient(in, out, m)
 * @(#) IMAGE *in, *out;
 * @(#) INTMASK *m;
 * @(#)
 * @(#)  Returns 0 on sucess and -1 
 * @(#)
 * Author: N. Dessipris (Copyright 1991)
 * Written on: 07/05/1991
 * Modifiied on: 29/4/93 K.Martinez  for Sys5
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define PIM_RINT 1
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/*

	old code ... now implemented in im_compass.c in terms of im_conv()

 */

int im_gradient_old(in, out, m)
IMAGE *in, *out;
INTMASK *m;
{
	int *pm;	/* pointer to mask */
	int ms;
	int *rot_offs; /* offsets for rotating the input mask 90 degrees */
	int x, y, i;	/* counters*/
	int xstart, xend, ystart, yend;
	PEL **pnt, **pnts, **cpnt, **cpnts;
	int **lut_orig, **lut, **cplut;
	int lutcnt; /* needed for freeing pointers malloc by im_create_... */
	PEL *input;
	int *line1, *cpline1;	/* output for the entered mask */
	int *line2, *cpline2;	/* output for the rotated entered mask */
	int *line, *cpline;	/* combined output */
	int os;			/* size of a line */
	int sum1, sum2, rounding;

/* ensure that input is square
 */
	if ( m->xsize != m->ysize )
		{
		im_errormsg("im_gradient: Unable to accept non square mask");
		return(-1);
		}
/* check IMAGEs parameters
 */
	if(im_iocheck(in, out) == -1)
		{ im_errormsg("im_gradient: im_iocheck failed"); return(-1); }
	if((in->Coding!=IM_CODING_NONE)||(in->Bbits!=8)||(in->BandFmt!=IM_BANDFMT_UCHAR))
		{
		im_errormsg("im_gradient: input not unsigned char uncoded");
		return(-1);
		}
/*  Prepare output IMAGE 
 */
	if ( im_cp_desc(out, in) == -1)	/* copy image descriptors */
		{ im_errormsg("im_gradient: im_cp_desc failed"); return(-1); }
	out->Bbits = IM_BBITS_INT; out->BandFmt = IM_BANDFMT_INT;
/* Set the output file or buffer
 */ 
	if ( im_setupout(out) == -1)
		{ im_errormsg("im_gradient: im_setupout failed"); return(-1); }
/* calculate luts
 */
	ms = m->xsize * m->ysize;
	lut_orig = (int**)calloc((unsigned)ms, sizeof(int**) );
	lut = (int**)calloc((unsigned)ms, sizeof(int**) );
	cplut = (int**)calloc((unsigned)ms, sizeof(int**) );
	if ( ( lut_orig == NULL )||( lut == NULL )||( cplut == NULL ) )
		{ im_errormsg("im_gradient: calloc failed(1)"); return(-1); }
	ms = m->xsize * m->ysize;
	pm = m->coeff;
	if ( im__create_int_luts( pm, ms, lut_orig, lut, &lutcnt ) == -1 )
		{im_errormsg("im_gradient: lut creation failed(1)");return(-1);}

/* allocate line buffers */
	os = out->Xsize * out->Bands;
	line = (int *)calloc( (unsigned) 3 * os, sizeof(int) );
	if ( line == NULL )
		{ im_errormsg("im_gradient: calloc failed(2)"); return(-1); }
	line1 = line + os;
	line2 = line1 + os;

/* Do the processing */
	xstart = m->xsize/2 * out->Bands;
        xend = ( out->Xsize - (m->xsize - m->xsize/2) ) * out->Bands;
        ystart = m->ysize/2;
        yend = out->Ysize - (m->ysize - m->ysize/2);
        rounding = m->scale/2;
	
/* set input pointers */
	pnts = (PEL**)calloc((unsigned)ms, sizeof(PEL**) );
	cpnts = (PEL**)calloc((unsigned)ms, sizeof(PEL**) );
	if ( (pnts == NULL) || (cpnts == NULL) )
		{ im_errormsg("im_gradient: calloc failed(3)"); return(-1); }
	pnt = pnts; 
	cpnt = cpnts;
	input = (PEL*)in->data;
	i = 0;
	for ( y=0; y<m->ysize; y++ )
		for ( x=0; x<m->xsize; x++ )
			pnt[i++] = (input+(x + y*in->Xsize)* in->Bands);

/* ZERO top part */
	memset((char*)line,0, os * sizeof(int));
	for (y=0; y<ystart; y++)
		if (im_writeline( y, out, (PEL*)line ) == -1)
			{
			free( (char*)line );
			for (i=0; i<lutcnt; i++)
                                free( (char*)lut_orig[i] );
                        free( (char*)lut_orig ); free( (char*)lut );
			free( (char*)pnts ); free( (char*)cpnts );
			im_errormsg("im_gradient: im_writeline failed (1)");
			return(-1);
			}

	/* find the rotation coefficients */
	if ( (rot_offs = im_offsets90( m->xsize )) == NULL )
		{im_errormsg("im_gradient: im_offsets90 failed");return(-1);}

	/* Create copies of luts corresponding to rotated input mask */
	for ( i=0; i<ms; i++ )
		cplut[i] = lut[ rot_offs[i] ];

#ifdef PIM_RINT
	fprintf(stderr, "input and rotated mask\n");
	pm = m->coeff;
	i=0;
	for (y=0; y<m->ysize; y++)
		{
		for (x=0; x<m->xsize; x++)
			{
			fprintf(stderr, "%4d\t", *(pm+i));
			i++;
			}
		fprintf(stderr, "\n");
		}
	fprintf(stderr, "\n");
	i=0;
	for (y=0; y<m->ysize; y++)
		{
		for (x=0; x<m->xsize; x++)
			fprintf(stderr, "%4d\t", *(pm+rot_offs[i++]));
		fprintf(stderr, "\n");
		}
#endif

	for ( y = ystart; y < yend; y++ )
		{
		/* process mask only */
		cpline1 = line1;
		cpline2 = line2;
		for ( x = 0; x < xstart ; x++ )
			{
			*cpline1++ = (int)0;
			*cpline2++ = (int)0;
			}

		for ( i=0; i<ms; i++ )
			cpnt[i] = pnt[i];

		for ( x = xstart; x < xend ; x++ )
			{
			sum1 = 0;
			sum2 = 0;
			for ( i=0; i<ms; i++ )
				{
				sum1 += *(lut[i] + (*cpnt[i]) );
				sum2 += *(cplut[i] + (*cpnt[i]) );
				cpnt[i]++;
				}
			*cpline1++ = ( (sum1+rounding)/m->scale ) + m->offset;
			*cpline2++ = ( (sum2+rounding)/m->scale ) + m->offset;
			}

		for ( x = xend; x < os; x++ )
			{
			*cpline1++ = (int)0;
			*cpline2++ = (int)0;
			}

		/* process line and write it */
		cpline = line; cpline1 = line1; cpline2 = line2;
		for ( x = 0; x < os; x++ )
			*cpline++ = abs(*cpline1++) + abs(*cpline2++);
		if (im_writeline( y, out, (PEL*)line ) == -1)
			{
			free( (char*)line );
			for (i=0; i<lutcnt; i++)
                                free( (char*)lut_orig[i] );
                        free( (char*)lut_orig );
			free( (char*)lut ); free( (char*)cplut );
			free( (char*)pnts ); free( (char*)cpnts );
			free( (char*)rot_offs );
			im_errormsg("im_gradient: im_writeline failed (2)");
			return(-1);
			}
		
		/* advance input pointers one input line */
		for ( i=0; i<ms; i++ )
			pnt[i] += os;
		}	/* end of the y loop */

/* ZERO bottom part */
	memset((char*)line,0, os * sizeof(int));
	for (y=yend; y<out->Ysize; y++)
		if (im_writeline( y, out, (PEL*)line ) == -1)
			{
			free( (char*)line );
			for (i=0; i<lutcnt; i++)
                                free( (char*)lut_orig[i] );
                        free( (char*)lut_orig );
			free( (char*)lut ); free( (char*)cplut );
			free( (char*)pnts ); free( (char*)cpnts );
			free( (char*)rot_offs );
			im_errormsg("im_gradient: im_writeline failed (3)");
			return(-1);
			}

	free( (char*)line );
	for (i=0; i<lutcnt; i++)
		free( (char*)lut_orig[i] );
	free( (char*)lut_orig );
	free( (char*)lut ); free( (char*)cplut );
	free( (char*)pnts ); free( (char*)cpnts );
	free( (char*)rot_offs );

	return(0);
}
