/* @(#) Find the maximum at every point of a set of images. All images must
 * @(#) be the same size and type. Useful for im_lindetect(), im_compass()
 * @(#) etc.
 * @(#)
 * @(#) int im_maxvalue( imarray, imout, no )
 * @(#) IMAGE *imarray[], *imout;
 * @(#) int no;
 * @(#)
 * @(#) All functions return 0 on success and -1 on error
 * @(#)
 *
 * 11/3/01 JC
 *	- from im_gbandjoin()
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Struct we carry stuff around in.
 */
typedef struct Max {
	IMAGE **in;		/* Array of input images, NULL-terminated */
	IMAGE *out;
	int n;			/* Number of input images */
} Max;

/* Make a Max struct.
 */
static Max *
max_new( IMAGE **in, IMAGE *out, int n )
{
	int i;
	Max *mx;

	if( !(mx = IM_NEW( out, Max )) )
		return( NULL );

	mx->n = n;
	mx->out = out;
	if( !(mx->in = IM_ARRAY( out, n + 1, IMAGE * )) ) 
		return( NULL );
	for( i = 0; i < n; i++ ) 
		mx->in[i] = in[i];
	mx->in[n] = NULL;

	return( mx );
}

/* Our sequence value.
 */
typedef struct {
	Max *mx;

	REGION **ir;		/* Input regions */
	PEL **pts;		/* Per-input region data pointer */
} MaxSequence;

/* Free a sequence value.
 */
static int
stop_max( MaxSequence *seq, IMAGE **in, Max *mx )
{
	int i;

	for( i = 0; i < mx->n; i++ ) 
		if( seq->ir[i] ) {
			im_region_free( seq->ir[i] );
			seq->ir[i] = NULL;
		}

	return( 0 );
}

/* Convolution start function.
 */
static void *
start_max( IMAGE *out, IMAGE **in, Max *mx )
{
	MaxSequence *seq;
	int i;

	if( !(seq = IM_NEW( out, MaxSequence )) )
		return( NULL );

	/* Init!
	 */
	seq->mx = mx;
	seq->ir = NULL;
	seq->pts = NULL;

	/* Attach regions and arrays.
	 */
	seq->ir = IM_ARRAY( out, mx->n + 1, REGION * );
	seq->pts = IM_ARRAY( out, mx->n + 1, PEL * );
	if( !seq->ir || !seq->pts ) {
		stop_max( seq, in, mx );
		return( NULL );
	}

	for( i = 0; i < mx->n; i++ )
		if( !(seq->ir[i] = im_region_create( in[i] )) ) {
			stop_max( seq, in, mx );
			return( NULL );
		}
	seq->ir[i] = NULL;

	return( (void *) seq );
}

#define FIND_IM_MAX( TYPE ) { \
	for( x = 0; x < sz; x++ ) { \
		TYPE top = ((TYPE *) seq->pts[0])[x]; \
 		\
		for( i = 1; i < mx->n; i++ ) { \
			TYPE v = ((TYPE *) seq->pts[i])[x]; \
 			\
			if( v > top ) \
				top = v; \
		} \
 		\
		((TYPE *) q)[x] = top; \
	} \
}

static int
find_max( REGION *or, MaxSequence *seq, IMAGE **in, Max *mx )
{
	Rect *r = &or->valid;
        int le = r->left;
        int to = r->top;
        int bo = IM_RECT_BOTTOM(r);
	int sz = IM_REGION_N_ELEMENTS( or );

	int x, y, i;

	/* Prepare each input area.
	 */
	for( i = 0; i < mx->n; i++ )
		if( im_prepare( seq->ir[i], r ) )
			return( -1 );

	/* Loop over output!
	 */
	for( y = to; y < bo; y++ ) {
		PEL *q = (PEL *) IM_REGION_ADDR( or, le, y );

		for( i = 0; i < mx->n; i++ )
			seq->pts[i] = (PEL *) IM_REGION_ADDR( seq->ir[i], le, y );

		switch( mx->out->BandFmt ) {
		case IM_BANDFMT_UCHAR: 	FIND_IM_MAX( unsigned char ); break;
		case IM_BANDFMT_CHAR:   FIND_IM_MAX( signed char ); break;
		case IM_BANDFMT_USHORT: FIND_IM_MAX( unsigned short ); break;
		case IM_BANDFMT_SHORT:  FIND_IM_MAX( signed short ); break;
		case IM_BANDFMT_UINT:   FIND_IM_MAX( unsigned int ); break;
		case IM_BANDFMT_INT:    FIND_IM_MAX( signed int ); break;
		case IM_BANDFMT_FLOAT:  FIND_IM_MAX( float ); break;
		case IM_BANDFMT_DOUBLE: FIND_IM_MAX( double ); break;

		default:
			assert( 0 );
		}
	}

	return( 0 );
}

/* pair-wise max of a vector of image descriptors.
 */
int
im_maxvalue( IMAGE **in, IMAGE *out, int n )
{
	int i;
	Max *mx;

	if( n < 1 ) {
		im_errormsg( "im_maxvalue: zero input images!" );
		return( -1 );
	}
	if( im_poutcheck( out ) )
		return( -1 );
	for( i = 0; i < n; i++ ) {
		if( im_pincheck( in[i] ) )
			return( -1 );

		if( in[i]->Coding != IM_CODING_NONE || im_iscomplex( in[i] ) ) {
			im_errormsg( "im_maxvalue: uncoded non-complex only" );
			return( -1 );
		}

		if( in[0]->BandFmt != in[i]->BandFmt ) {
			im_errormsg( "im_maxvalue: input images differ "
				"in format" );
			return( -1 );
		}
		if( in[0]->Xsize != in[i]->Xsize ||
			in[0]->Ysize != in[i]->Ysize ) {
			im_errormsg( "im_maxvalue: input images differ "
				"in size" );
			return( -1 );
		}
		if( in[0]->Bands != in[i]->Bands ) {
			im_errormsg( "im_maxvalue: input images differ "
				"in number of bands" );
			return( -1 );
		}
	}

	if( !(mx = max_new( in, out, n )) ||
		im_cp_desc( out, in[0] ) ||
		im_demand_hint_array( out, IM_THINSTRIP, mx->in )  ||
		im_generate( out, start_max, find_max, stop_max, mx->in, mx ) )
		return( -1 );
	
	return( 0 );
}
