/* @(#) Rank filter.
 * @(#)
 * @(#) int 
 * @(#) im_rank( in, out, xsize, ysize, n )
 * @(#) IMAGE *in, *out;
 * @(#) int xsize, ysize, n;
 * @(#)
 * @(#) Also: im_rank_raw(). As above, but does not add a black border.
 * @(#)
 * @(#) Returns either 0 (success) or -1 (fail)
 * @(#) 
 *
 * Author: JC
 * Written on: 19/8/96
 * Modified on: 
 * JC 20/8/96
 *	- now uses insert-sort rather than bubble-sort
 *	- now works for any non-complex type
 * JC 22/6/01 
 *	- oops, sanity check on n wrong
 * JC 27/10/03 
 *	- use memmove(), grr
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Global state: save our parameters here.
 */
typedef struct {
	IMAGE *in, *out;	/* Images we run */
	int xsize, ysize;	/* Window size */
	int n;			/* Element select */
} RankInfo;

/* Sequence value: just the array we sort in.
 */
typedef struct {
	REGION *ir;
	PEL *sort;
} SeqInfo;

/* Free a sequence value.
 */
static int
stop_rank( SeqInfo *seq, IMAGE *in, RankInfo *rnk )
{
	/* Free attached objects.
	 */
	if( seq->ir ) {
		im_region_free( seq->ir );
		seq->ir = NULL;
	}

	return( 0 );
}

/* Rank start function.
 */
static void *
start_rank( IMAGE *out, IMAGE *in, RankInfo *rnk )
{
	SeqInfo *seq = IM_NEW( out, SeqInfo );

	if( !seq ) 
		return( NULL );

	/* Init!
	 */
	seq->ir = NULL;
	seq->sort = NULL;

	/* Attach region and arrays.
	 */
	seq->ir = im_region_create( in );
	seq->sort = IM_ARRAY( out, 
		IM_IMAGE_SIZEOF_ELEMENT( in ) * rnk->xsize * rnk->ysize, PEL );
	if( !seq->ir || !seq->sort ) {
		stop_rank( seq, in, rnk );
		return( NULL );
	}

	return( (void *) seq );
}

/* Inner loop for sorting rank of TYPE.
 */
#define rank_loop( TYPE ) \
{ \
	TYPE *q = (TYPE *) IM_REGION_ADDR( or, le, y ); \
	TYPE *p = (TYPE *) IM_REGION_ADDR( ir, le, y ); \
	TYPE *sort = (TYPE *) seq->sort;\
	\
	/* Loop across. \
	 */ \
	for( x = 0; x < sz; x++ ) { \
		TYPE *d = &p[x];\
		\
		/* Insert-sort pels in window into array.\
		 */\
		for( k = 0, j = 0; j < rnk->ysize; j++ ) {\
			TYPE *e = d;\
			\
			for( i = 0; i < rnk->xsize; i++, k++ ) {\
				TYPE v = *e;\
				\
				/* Search for element >v.\
				 */\
				for( l = 0; l < k; l++ )\
					if( sort[l] > v )\
						break;\
				\
				/* Move remaining elements down.\
				 */\
				memmove( &sort[l+1], &sort[l], \
					sizeof( TYPE ) * (k - l) );\
				\
				/* Insert this element.\
				 */\
				sort[l] = v;\
				\
				e += ps; \
			}\
			\
			d += ls;\
		}\
		\
		/* Output nth!\
		 */\
		q[x] = sort[n];\
	} \
}

/* Rank of a REGION.
 */
static int
gen_rank( REGION *or, SeqInfo *seq, IMAGE *in, RankInfo *rnk )
{
	REGION *ir = seq->ir;

	Rect *r = &or->valid;
	Rect s;
	int le = r->left;
	int to = r->top;
	int bo = IM_RECT_BOTTOM(r);
	int sz = IM_REGION_N_ELEMENTS( or );

	int ls;
	int ps = IM_IMAGE_SIZEOF_PEL( in ) / IM_IMAGE_SIZEOF_ELEMENT( in );

	int x, y;
	int i, j;
	int k;
	int l;

	int n = rnk->n - 1;

	/* Prepare the section of the input image we need. A little larger
	 * than the section of the output image we are producing.
	 */
	s = *r;
	s.width += rnk->xsize - 1;
	s.height += rnk->ysize - 1;
	if( im_prepare( ir, &s ) )
		return( -1 );
	ls = IM_REGION_LSKIP( ir ) / IM_IMAGE_SIZEOF_ELEMENT( in );

	for( y = to; y < bo; y++ ) { 
		switch( in->BandFmt ) {
		case IM_BANDFMT_UCHAR:	rank_loop( unsigned char ); break;
		case IM_BANDFMT_USHORT:	rank_loop( unsigned short ); break;
		case IM_BANDFMT_UINT:	rank_loop( unsigned int ); break;
		case IM_BANDFMT_CHAR:	rank_loop( signed char ); break;
		case IM_BANDFMT_SHORT:	rank_loop( signed short ); break;
		case IM_BANDFMT_INT:	rank_loop( signed int ); break;
		case IM_BANDFMT_FLOAT:	rank_loop( float ); break;
		case IM_BANDFMT_DOUBLE:	rank_loop( double ); break;

		default:
			im_errormsg( "im_rank: unsupported image type" );
			return( -1 );
			/*NOTREACHED*/
		}
	}

	return( 0 );
}

/* Loop for find max of window.
 */
#define max_loop( TYPE ) \
{ \
	TYPE *q = (TYPE *) IM_REGION_ADDR( or, le, y ); \
	TYPE *p = (TYPE *) IM_REGION_ADDR( ir, le, y ); \
	\
	/* Loop across. \
	 */ \
	for( x = 0; x < sz; x++ ) { \
		TYPE *d = &p[x];\
		TYPE max;\
		\
		/* Search window for max.\
		 */\
		max = *d;\
		for( j = 0; j < rnk->ysize; j++ ) {\
			TYPE *e = d;\
			\
			for( i = 0; i < rnk->xsize; i++ ) {\
				if( *e > max )\
					max = *e;\
				\
				e += ps;\
			}\
			\
			d += ls;\
		}\
		\
		/* Output max.\
		 */\
		q[x] = max;\
	} \
}

/* Rank of a REGION - special case: choose max.
 */
static int
gen_rank_max( REGION *or, SeqInfo *seq, IMAGE *in, RankInfo *rnk )
{
	REGION *ir = seq->ir;

	Rect *r = &or->valid;
	Rect s;
	int le = r->left;
	int to = r->top;
	int bo = IM_RECT_BOTTOM(r);
	int sz = IM_REGION_N_ELEMENTS( or );

	int ls;
	int ps = IM_IMAGE_SIZEOF_PEL( in ) / IM_IMAGE_SIZEOF_ELEMENT( in );

	int x, y;
	int i, j;

	/* Prepare the section of the input image we need. A little larger
	 * than the section of the output image we are producing.
	 */
	s = *r;
	s.width += rnk->xsize - 1;
	s.height += rnk->ysize - 1;
	if( im_prepare( ir, &s ) )
		return( -1 );
	ls = IM_REGION_LSKIP( ir ) / IM_IMAGE_SIZEOF_ELEMENT( in );

	for( y = to; y < bo; y++ ) { 
		switch( in->BandFmt ) {
		case IM_BANDFMT_UCHAR:	max_loop( unsigned char ); break;
		case IM_BANDFMT_USHORT:	max_loop( unsigned short ); break;
		case IM_BANDFMT_UINT:	max_loop( unsigned int ); break;
		case IM_BANDFMT_CHAR:	max_loop( signed char ); break;
		case IM_BANDFMT_SHORT:	max_loop( signed short ); break;
		case IM_BANDFMT_INT:	max_loop( signed int ); break;
		case IM_BANDFMT_FLOAT:	max_loop( float ); break;
		case IM_BANDFMT_DOUBLE:	max_loop( double ); break;

		default:
			im_errormsg( "im_rank: unsupported image type" );
			return( -1 );
			/*NOTREACHED*/
		}
	}

	return( 0 );
}

/* Loop for find min of window.
 */
#define min_loop( TYPE ) \
{ \
	TYPE *q = (TYPE *) IM_REGION_ADDR( or, le, y ); \
	TYPE *p = (TYPE *) IM_REGION_ADDR( ir, le, y ); \
	\
	/* Loop across. \
	 */ \
	for( x = 0; x < sz; x++ ) { \
		TYPE *d = &p[x];\
		TYPE min;\
		\
		/* Search window for min.\
		 */\
		min = *d;\
		for( j = 0; j < rnk->ysize; j++ ) {\
			TYPE *e = d;\
			\
			for( i = 0; i < rnk->xsize; i++ ) {\
				if( *e < min )\
					min = *e;\
				\
				e += ps;\
			}\
			\
			d += ls;\
		}\
		\
		/* Output min.\
		 */\
		q[x] = min;\
	} \
}

/* Rank of a REGION - special case: choose min.
 */
static int
gen_rank_min( REGION *or, SeqInfo *seq, IMAGE *in, RankInfo *rnk )
{
	REGION *ir = seq->ir;

	Rect *r = &or->valid;
	Rect s;
	int le = r->left;
	int to = r->top;
	int bo = IM_RECT_BOTTOM(r);
	int sz = IM_REGION_N_ELEMENTS( or );

	int ls;
	int ps = IM_IMAGE_SIZEOF_PEL( in ) / IM_IMAGE_SIZEOF_ELEMENT( in );

	int x, y;
	int i, j;

	/* Prepare the section of the input image we need. A little larger
	 * than the section of the output image we are producing.
	 */
	s = *r;
	s.width += rnk->xsize - 1;
	s.height += rnk->ysize - 1;
	if( im_prepare( ir, &s ) )
		return( -1 );
	ls = IM_REGION_LSKIP( ir ) / IM_IMAGE_SIZEOF_ELEMENT( in );

	for( y = to; y < bo; y++ ) { 
		switch( in->BandFmt ) {
		case IM_BANDFMT_UCHAR:	min_loop( unsigned char ); break;
		case IM_BANDFMT_USHORT:	min_loop( unsigned short ); break;
		case IM_BANDFMT_UINT:	min_loop( unsigned int ); break;
		case IM_BANDFMT_CHAR:	min_loop( signed char ); break;
		case IM_BANDFMT_SHORT:	min_loop( signed short ); break;
		case IM_BANDFMT_INT:	min_loop( signed int ); break;
		case IM_BANDFMT_FLOAT:	min_loop( float ); break;
		case IM_BANDFMT_DOUBLE:	min_loop( double ); break;

		default:
			im_errormsg( "im_rank: unsupported image type" );
			return( -1 );
			/*NOTREACHED*/
		}
	}

	return( 0 );
}

/* Rank filter.
 */
int
im_rank_raw( IMAGE *in, IMAGE *out, int xsize, int ysize, int n )
{
	RankInfo *rnk;

	/* Check parameters.
	 */
	if( !in || in->Coding != IM_CODING_NONE || im_iscomplex( in ) ) {
		im_errormsg( "im_rank: input non-complex uncoded only" );
		return( -1 );
	}
	if( xsize > 1000 || ysize > 1000 || xsize <= 0 || ysize <= 0 || 
		n < 0 || n > xsize * ysize - 1 ) {
		im_errormsg( "im_rank: bad parameters" );
		return( -1 );
	}
	if( im_piocheck( in, out ) )
		return( -1 );
	
	/* Save parameters.
	 */
	if( !(rnk = IM_NEW( out, RankInfo )) )
		return( -1 );
	rnk->in = in;
	rnk->out = out;
	rnk->xsize = xsize;
	rnk->ysize = ysize;
	rnk->n = n;

	/* Prepare output. Consider a 7x7 window and a 7x7 image --- the output
	 * would be 1x1.
	 */
	if( im_cp_desc( out, in ) )
		return( -1 );
	out->Xsize -= xsize - 1;
	out->Ysize -= ysize - 1;
	if( out->Xsize <= 0 || out->Ysize <= 0 ) {
		im_errormsg( "im_rank: image too small for window" );
		return( -1 );
	}

	/* Set demand hints. FATSTRIP is good for us, as THINSTRIP will cause
	 * too many recalculations on overlaps.
	 */
	if( im_demand_hint( out, IM_FATSTRIP, in, NULL ) )
		return( -1 );

	/* Generate! 
	 */
	if( n == xsize * ysize - 1 ) {
		/* Special case: find max.
		 */
		if( im_generate( out, start_rank, gen_rank_max, stop_rank, 
			in, rnk ) )
			return( -1 );
	}
	else if( n == 0 ) {
		/* Special case: find min.
		 */
		if( im_generate( out, start_rank, gen_rank_min, stop_rank, 
			in, rnk ) )
			return( -1 );
	}
	else {
		/* General case: need sorter.
		 */
		if( im_generate( out, start_rank, gen_rank, stop_rank, 
			in, rnk ) )
			return( -1 );
	}

	return( 0 );
}

/* The above, with a border to make out the same size as in.
 */
int 
im_rank( IMAGE *in, IMAGE *out, int xsize, int ysize, int n )
{
	IMAGE *t1 = im_open_local( out, "im_rank:1", "p" );

	if( !t1 || im_rank_raw( in, t1, xsize, ysize, n ) || im_embed( t1, out,
		0, xsize/2, ysize/2, in->Xsize, in->Ysize ) )
		return( -1 );

	return( 0 );
}
