/* @(#) error handling
 * @(#) 
 * @(#)  Usage:
 * @(#)  void im_errormsg(variable_list)
 * @(#)  (variable_list) is (format, arg1, arg2, ...)
 * @(#) format, arg1, arg2, etc are the same as in fprintf
 * @(#) 
 * Copyright: N. Dessipris 
 * Written on: 18/03/1991
 * Updated on: 9/7/92 KM
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include <vips/vips.h>

#ifdef HAVE_WINDOWS_H
#include <windows.h>
#include <lmerr.h>
#endif /*HAVE_WINDOWS_H*/

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Make global array to keep the error message buffer.
 */
#define IM_MAX_ERROR (10240)
static char im_errorstring_buffer[IM_MAX_ERROR] = "";

#define IM_DIAGNOSTICS "IM_DIAGNOSTICS"
#define IM_WARNING "IM_WARNING"

const char *
im_errorstring( void )
{
	return( im_errorstring_buffer );
}

void 
im_verrormsg( const char *fmt, va_list ap )
{	
	int l = strlen( im_errorstring() );
	char *p = (char *) im_errorstring() + l;

	(void) im_vsnprintf( p, IM_MAX_ERROR - l - 3, fmt, ap );
	strcat( p, "\n" );
}

void 
im_errormsg( const char *fmt, ... )
{	
	va_list ap;

	va_start( ap, fmt );
	im_verrormsg( fmt, ap );
	va_end( ap );
}

void
im_errormsg_system( int err, const char *fmt, ... )
{
	va_list ap;

	va_start( ap, fmt );
	im_verrormsg( fmt, ap );
	va_end( ap );

#ifdef HAVE_WINDOWS_H
{
	char *buf;

	if( FormatMessageA(
		FORMAT_MESSAGE_ALLOCATE_BUFFER |
		FORMAT_MESSAGE_IGNORE_INSERTS |
		FORMAT_MESSAGE_FROM_SYSTEM,
		NULL,
		err,
		MAKELANGID( LANG_NEUTRAL, SUBLANG_DEFAULT ), 
		(LPSTR) &buf, 0, NULL ) ) {
		im_errormsg( "windows error: %s", buf );
		LocalFree( buf );
	}
}
#else /*HAVE_WINDOWS_H*/
	im_errormsg( "unix error: %s", strerror( err ) );
#endif /*HAVE_WINDOWS_H*/
}

void 
im_clear_error_string( void )
{	
	im_errorstring_buffer[0] = '\0';
}

void 
im_diagnostics( const char *fmt, ... )
{	
	va_list ap;

	if( !getenv( IM_DIAGNOSTICS ) ) {
		(void) fprintf( stderr, "vips diagnostic: " );
		va_start( ap, fmt );
		(void) vfprintf( stderr, fmt, ap );
		va_end( ap );
		(void) fprintf( stderr, "\n" );
	}
}

void 
im_warning( const char *fmt, ... )
{	
	va_list ap;

	if( !getenv( IM_WARNING ) ) {
		(void) fprintf( stderr, "vips warning: " );
		va_start( ap, fmt );
		(void) vfprintf( stderr, fmt, ap );
		va_end( ap );
		(void) fprintf( stderr, "\n" );
	}
}
