/* @(#)  display VIPS IMAGE descriptor for debugging
 * @(#)  a valid IMAGE descriptor is needed
 * @(#) HANDLESIMAGE
 * @(#)  Static strings must be modified if header files are changed
 * @(#) void im_printdesc(image)
 * @(#) IMAGE *image;
 *
 * Copyright: Nicos Dessipris
 * Written on: 15/01/1990
 * Modified on : 3/6/92 Kirk Martinez
 * 15/4/93 JC
 *	- partial regions dumped too
 *	- resolution bug fixed
 * 11/5/93 JC
 *	- formatting changed for ctags
 * 1/7/93 JC
 *	- adapted for partial v2
 * 15/11/94 JC
 *	- new Coding types added
 *	- new Type values added
 *	- new Compression type added
 * 2/3/98 JC
 *	- IM_ANY added
 * 5/11/98 JC
 *	- prints byte-order too
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <string.h>

#include <vips/vips.h>
#include <vips/list.h>
#include <vips/util.h>
#include <vips/region.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

static const char *im_Type[] = {
	"IM_TYPE_MULTIBAND", "IM_TYPE_B_W", 
	"LUMINACE", "XRAY", "IR", "YUV", 
	"RED_ONLY", "GREEN_ONLY", "BLUE_ONLY", 
	"POWER_SPECTRUM", "IM_TYPE_HISTOGRAM", "LUT", "IM_TYPE_XYZ",
	"IM_TYPE_LAB", "CMC", "IM_TYPE_CMYK", "IM_TYPE_LABQ", "IM_TYPE_RGB", 
	"IM_TYPE_UCS", "IM_TYPE_LCH", "IM_TYPE_LABS",
	"<unknown>", "IM_TYPE_sRGB", "IM_TYPE_YXY", "IM_TYPE_FOURIER", NULL
};

static const char *im_BandFmt[] = {
	"IM_BANDFMT_UCHAR", "IM_BANDFMT_CHAR", "IM_BANDFMT_USHORT", 
	"IM_BANDFMT_SHORT", "IM_BANDFMT_UINT", "IM_BANDFMT_INT", 
	"IM_BANDFMT_FLOAT", "IM_BANDFMT_COMPLEX", "IM_BANDFMT_DOUBLE", 
	"IM_BANDFMT_DPCOMPLEX", NULL
};

static const char *im_Coding[] = {
	"IM_CODING_NONE", "COLQUANT8", "IM_CODING_LABQ", 
	"IM_CODING_LABQ_COMPRESSED", NULL
};

static const char *im_Compression[] = {
	"NO_COMPRESSION", "TCSF_COMPRESSION", "JPEG_COMPRESSION", NULL
};

static const char *im_dtype[] = {
	"IM_NONE", "IM_SETBUF", "IM_SETBUF_FOREIGN", "IM_OPENIN", "IM_MMAPIN", 
	"IM_MMAPINRW", "IM_OPENOUT", "IM_PARTIAL", NULL
};

static const char *im_dhint[] = {
	"IM_SMALLTILE", "IM_FATSTRIP", "IM_THINSTRIP", "IM_ANY", NULL
};

/* Prettyprint various header fields.
 */
const char *
im_Type2char( int n )
{
        if( n < 0 )
                return( "<bad Type>" );
        else if( n < IM_NUMBER( im_Type ) )
                return( im_Type[n] );
        else
                return( "<unknown Type>" );
}

const char *
im_Compression2char( int n )
{
        if( n < 0 )
                return( "<bad Compression>" );
        else if( n < IM_NUMBER( im_Type ) )
                return( im_Compression[n] );
        else
                return( "<unknown Compression>" );
}

const char *
im_BandFmt2char( int n )
{
        if( n < 0 )
                return( "<bad BandFmt>" );
        else if( n < IM_NUMBER( im_BandFmt ) )
                return( im_BandFmt[n] );
        else
                return( "<unknown BandFmt>" );
}

const char *
im_Coding2char( int n )
{
        if( n < 0 )
                return( "<bad Coding>" );
        else if( n < IM_NUMBER( im_Coding ) )
                return( im_Coding[n] );
        else
                return( "<unknown Coding>" );
}

const char *
im_dhint2char( im_demand_type n )
{
        if( (int) n < 0 )
                return( "<bad dhint>" );
        else if( (int) n < IM_NUMBER( im_dhint ) )
                return( im_dhint[(int) n] );
        else
                return( "<unknown dhint>" );
}

const char *
im_dtype2char( im_desc_type n )
{
        if( (int) n < 0 )
                return( "<bad dtype>" );
        else if( (int) n < IM_NUMBER( im_dtype ) )
                return( im_dtype[(int) n] );
        else
                return( "<unknown dtype>" );
}

int
im_char2Type( const char *str )
{
	int i;

	for( i = 0 ; i < IM_NUMBER( im_Type ); i++ )
		if( strcmp( str, im_Type[i] ) == 0 )
			return( i );

	im_errormsg( "bad type '%s'", str );
	return( -1 );
}

int
im_char2BandFmt( const char *str )
{
	int i;

	for( i = 0 ; i < IM_NUMBER( im_BandFmt ); i++ )
		if( strcmp( str, im_BandFmt[i] ) == 0 )
			return( i );
	
	im_errormsg( "bad band format '%s'", str );
	return( -1 );
}

int
im_char2Coding( const char *str )
{
	int i;

	for( i = 0 ; i < IM_NUMBER( im_Coding ); i++ )
		if( strcmp( str, im_Coding[i] ) == 0 )
			return( i );

	im_errormsg( "bad coding '%s'", str );
	return( -1 );
}

int
im_char2Compression( const char *str )
{
	int i;

	for( i = 0 ; i < IM_NUMBER( im_Compression ); i++ )
		if( strcmp( str, im_Compression[i] ) == 0 )
			return( i );

	im_errormsg( "bad Compression '%s'", str );
	return( -1 );
}

static void *
print_region( REGION *reg, void *a, void *b )
{	
	printf( "Region defined for area at %dx%d size %dx%d\n",
		reg->valid.left, reg->valid.top,
		reg->valid.width, reg->valid.height );
	if( reg->seq )
		printf( "sequence in progress on region\n" );
	if( reg->buf )
		printf( "local memory allocated\n" );

	return( NULL );
}

void 
im_printdesc( IMAGE *image )
{	
	if( !image ) {
		printf( "NULL descriptor\n" );
		return;
	}

	if( im_isMSBfirst( image ) )
		printf( "SPARC (MSB first) " );
	else
		printf( "Intel (LSB first) " );
	printf( "byte order image, on a " );
	if( im_amiMSBfirst() )
		printf( "SPARC (MSB first) " );
	else
		printf( "Intel (LSB first) " );
	printf( "byte order machine\n" );
 
	printf( "Xsize: %d\nYsize: %d\nBands: %d\nBbits: %d\n",
		image->Xsize, image->Ysize, 
		image->Bands, image->Bbits );

	printf( "BandFmt: %d --> %s\nCoding: %d --> %s\nType: %d --> %s\n",
		image->BandFmt, im_BandFmt2char( image->BandFmt ),
		image->Coding, im_Coding2char( image->Coding ),
		image->Type, im_Type2char( image->Type ) );

	printf( "Xres: %g\nYres: %g\n", 
		image->Xres, image->Yres );

	printf( "Xoffset: %d\nYoffset: %d\n", 
		image->Xoffset, image->Yoffset ); 

	if( image->Hist ) 
		printf( "Hist:\n%s\n", image->Hist );

	if( image->filename ) 
		printf( "filename: %s\n", image->filename );
	if( image->generate )
		printf( "user generate function attached\n" );
	if( image->closefns )
		printf( "user close callbacks attached\n" );
	if( image->evalfns )
		printf( "user eval callbacks attached\n" );
	if( image->evalendfns )
		printf( "user evalend callbacks attached\n" );
	if( image->regions ) {
		printf( "%d regions present\n", im_list_len( image->regions ) );
		im_list_map( image->regions, 
			(im_list_map_fn) print_region, NULL, NULL );
	}
	if( image->kill )
		printf( "kill flag set\n" );
	if( image->closing )
		printf( "closing flag set\n" );
	if( image->close_pending )
		printf( "close_pending flag set\n" );
	printf( "dhint: %s\n", im_dhint2char( image->dhint ) );
	printf( "dtype: %s\n", im_dtype2char( image->dtype ) );
}
