/* @(#)  Reads one line of description and the history of the filename
 * @(#)  This is done by replacing the ending .v of the filename with .desc
 * @(#) and trying to read the new file.  If the file ending in .desc
 * @(#) does exist it is read and put into the Hist pointer of the
 * @(#) image descriptor
 * @(#)  If the .desc file does not exist or if the input file is not
 * @(#) ending with .v the Hist pointer in initialised to "filename\n" 
 * @(#) and history is kept from the current processing stage.
 * @(#)
 * @(#) int im_readhist(image)
 * @(#) IMAGE *image;
 * @(#)
 * @(#)  Returns either 0 (success) or -1 (fail)
 * Copyright: Nicos Dessipris
 * Written on: 15/01/1990
 * Modified on : 
 * 28/10/92 JC
 *	- no more wild freeing!
 *	- behaves itself, thank you
 * 13/1/94 JC
 *	- array-bounds write found and fixed 
 * 26/10/98 JC
 *	- binary open for stupid systems
 * 24/9/01 JC
 *	- slight clean up
 * 6/8/02 JC
 *	- another cleanup
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif /*HAVE_SYS_FILE_H*/
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /*HAVE_UNISTD_H*/
#ifdef HAVE_IO_H
#include <io.h>
#define strcasecmp(a,b) _stricmp(a,b)
#endif /*HAVE_IO_H*/

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Strip off any of a set of old suffixes (eg. [".v", ".jpg"]), add a single 
 * new suffix (eg. ".tif"). 
 */
void
im__change_suffix( const char *name, char *out, int mx,
        const char *new, const char **olds, int nolds )
{
        char *p;
        int i;
	int len;

        /* Copy start string.
         */
        im_strncpy( out, name, mx );

        /* Drop all matching suffixes.
         */
        while( (p = strrchr( out, '.' )) ) {
                /* Found suffix - test against list of alternatives. Ignore
                 * case.
                 */
                for( i = 0; i < nolds; i++ )
                        if( strcasecmp( p, olds[i] ) == 0 ) {
                                *p = '\0';
                                break;
                        }

                /* Found match? If not, break from loop.
                 */
                if( *p )
                        break;
        }

        /* Add new suffix.
         */
	len = strlen( out );
	im_strncpy( out + len, new, mx - len );
}

int 
im_readhist( IMAGE *image )
{	
	/* Junk any old history.
	 */
	if( image->Hist ) {
		im_free( image->Hist );
		image->Hist = NULL;
	}

	/* Check whether the filename ends with .v 
	 */
	if( im__ispostfix( image->filename, ".v" ) ) {
		int fd;
		char buf[1024];
		const char *exts[] = { ".v" };

		/* Yes: try to read the '.desc' file.
		 */
		im__change_suffix( image->filename, 
			buf, 1024, ".desc", exts, 1 );

		/* Have to read binary, otherwise we can't predict the length.
		 */
#ifdef BINARY_OPEN
		if( (fd = open( buf, O_RDONLY | O_BINARY )) != -1 ) {
#else /*BINARY_OPEN*/
		if( (fd = open( buf, O_RDONLY )) != -1 ) {
#endif /*BINARY_OPEN*/
			char *buffer;
			struct stat st;

			if( fstat( fd, &st ) == -1 ) {
				im_errormsg( "im_readhist: unable to fstat" );
				close( fd );
				return( -1 );
			}
			if( !(buffer = im_malloc( NULL, st.st_size + 1 )) ) {
				close( fd );
				return( -1 );
			}

			if( read( fd, buffer, st.st_size ) != st.st_size ) {
				im_errormsg( "im_readhist: unable to read" );
				close( fd );
				im_free( buffer ); 
				return( -1 );
			}
			close( fd );

			/* Make into a string and attach to image.
			 */
			buffer[st.st_size] = '\0';
			image->Hist = buffer;
		}
	}

	/* Make a fallback Hist.
	 */
	if( !image->Hist ) {
		char buf[1024];

		im_snprintf( buf, 1024, "%s\n", image->filename );
		image->Hist = im_strdup( NULL, buf );
	}

	return( 0 );
}
