/* @(#) memory.c: mem handling stuff
 *
 * 2/11/99 JC
 *	- from im_open.c and callback.c
 *	- malloc tracking stuff added
 * 11/3/01 JC
 * 	- im_strncpy() added
 * 20/4/01 JC
 * 	- im_(v)snprintf() added
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <assert.h>

#include <vips/vips.h>
#include <vips/region.h>
#include <vips/util.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Define for simple malloc tracking ... better to use dmalloc if you can.
#define DEBUGM
 */

/* Temp buffer for snprintf() layer on old systems.
 */
#define MAX_BUF (32768)

/* Track total alloc/total free here for debugging.
 */
#ifdef DEBUGM
static unsigned int total_mem_alloc = 0;
static unsigned int total_allocs = 0;
#endif /*DEBUGM*/

/* VIPS free function. Try to put all vips free() through this.
 */
int
im_free( void *s )
{
#ifdef DEBUGM
	int size;

	s = (void *) ((char*)s - 16);
	size = *((unsigned int*)s);
	total_mem_alloc -= size;
	total_allocs -= 1;
	printf( "im_free: %d, %d allocs, total %.3gM\n", 
		size,
		total_allocs,
		total_mem_alloc/(1024.0*1024.0) );

	if( total_allocs < 0 ) {
		printf( "woah! too many im_free()s!\n" );
		abort();
	}
	
	/* Probably trying to free something not allocated by im_malloc().
	 */
	if( size > 10000000 ) {
		printf( "woah! big!\n" );
		abort();
	}
#endif /*DEBUGM*/

	free( s );

	return( 0 );
}

/* Malloc local to a descriptor. Try to put all vips malloc through this.
 */
void *
im_malloc( IMAGE *im, int size )
{
        void *buf;

	if( size <= 0 ) {
#ifdef DEBUG
		error_exit( "im_malloc: size <= 0" );
#endif /*DEBUG*/
		im_warning( "im_malloc: size <= 0, expect a crash soon!" );
	}

#ifdef DEBUGM
	/* If debugging mallocs, need an extra sizeof(uint) bytes to track 
	 * size of this block. Ask for an extra 16 to make sure we don't break
	 * alignment rules.
	 */
	size += 16;
#endif /*DEBUGM*/

        if( !(buf = malloc( size )) ) {
#ifdef DEBUG
		/* Make sure we catch this ... 
		 */
		error_exit( "im_malloc: out of memory" );
#endif /*DEBUG*/
		im_warning( "malloc failed, very low on memory" );
		im_errormsg( "im_malloc: out of memory --- "
			"size == %dMB", (int) (size / (1024.0*1024.0)) );
                return( NULL );
	}

#ifdef DEBUGM
	/* Record number alloced.
	 */
	*((unsigned int*)buf) = size;
	buf = (void *) ((char*)buf + 16);
	total_mem_alloc += size;
	total_allocs += 1;
	printf( "im_malloc: %d, %d allocs, total %.3gM\n", 
		size, 
		total_allocs,
		total_mem_alloc/(1024.0*1024.0) );
#endif /*DEBUGM*/
 
        if( im && im_add_close_callback( im, 
		(im_callback_fn) im_free, buf, NULL ) ) {
                im_free( buf );
                return( NULL );
        }
 
        return( buf );
}

/* strdup local to a descriptor.
 */
char *
im_strdup( IMAGE *im, const char *str )
{
	int l = strlen( str );
	char *buf;

	if( !(buf = (char *) im_malloc( im, l + 1 )) )
		return( NULL );
	strcpy( buf, str );

	return( buf );
}

/* Like strncpy(), but always NULL-terminate, and don't pad with NULLs.
 */
char *
im_strncpy( char *dest, const char *src, int n )
{
        int i;

        assert( n > 0 );

        for( i = 0; i < n - 1; i++ )
                if( !(dest[i] = src[i]) )
                        break;
        dest[i] = '\0';

        return( dest );
}

/* Wrapper over (v)snprintf() ... missing on old systems.
 */
int
im_vsnprintf( char *str, size_t size, const char *format, va_list ap )
{
#ifdef HAVE_VSNPRINTF
	return( vsnprintf( str, size, format, ap ) );
#else /*HAVE_VSNPRINTF*/
	/* Bleurg!
	 */
	int n;
	static char buf[MAX_BUF];

	if( size > MAX_BUF )
		error_exit( "panic: buffer overflow "
			"(request to write %d bytes to buffer of %d bytes)",
			size, MAX_BUF );
	n = vsprintf( buf, format, ap );
	if( n > MAX_BUF )
		error_exit( "panic: buffer overflow "
			"(%d bytes written to buffer of %d bytes)",
			n, MAX_BUF );

	im_strncpy( str, buf, size );

	return( n );
#endif /*HAVE_VSNPRINTF*/
}

int
im_snprintf( char *str, size_t size, const char *format, ... )
{
	va_list ap;
	int n;

	va_start( ap, format );
	n = im_vsnprintf( str, size, format, ap );
	va_end( ap );

	return( n );
}

/* Find the rightmost occurrence of needle in haystack.
 */
char *
im_strrstr( const char *haystack, const char *needle )
{
	int haystack_len = strlen( haystack );
	int needle_len = strlen( needle );
	int i;

	for( i = haystack_len - needle_len; i >= 0; i-- )
		if( strncmp( needle, haystack + i, needle_len ) == 0 )
			return( (char *) haystack + i );
	
	return( NULL );
}
