/* Make and destroy partial image regions.
 * 
 * J.Cupitt, 8/4/93.
 * 1/7/93 JC
 *	- adapted for partial v2
 *	- ANSIfied
 * 15/8/94 JC
 *	- start & stop can now be NULL for no-op
 * 12/5/94 JC
 *      - threads v2.0 added
 * 22/2/95 JC
 *	- im_region_region() args changed
 * 22/6/95 JC
 *	- im_region_local() did not always reset the data pointer
 * 18/11/98 JC
 *	- init a, b, c also now, to help rtc avoid spurious checks
 * 29/6/01 JC
 *	- im_region_free() now frees immediately
 * 6/8/02 JC
 *	- new mmap() window regions
 * 5/11/02 JC
 *	- fix for mmap a local region
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG_ENVIRONMENT 1
#define DEBUG
#define DEBUG_CREATE
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /*HAVE_UNISTD_H*/
#include <errno.h>
#include <string.h>
#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif
#include <assert.h>

#include <vips/vips.h>
#include <vips/util.h>
#include <vips/list.h>
#include <vips/rect.h>
#include <vips/region.h>
#include <vips/thread.h>

#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif /*HAVE_WINDOWS_H*/

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Add this many lines above and below the mmap() window.
 */
int im__mmap_window_margin = IM__MMAP_WINDOW_MARGIN;

/* Sanity checking ... write to this during read tests to make sure we don't
 * get optimised out.
 */
int im__read_test;

#ifdef DEBUG
static int total_mmap_usage = 0;
static int max_mmap_usage = 0;
#endif /*DEBUG*/

/* Call a start function if no sequence is running on this REGION.
 */
int
im__call_start( REGION *reg )
{
	IMAGE *im = reg->im;

        /* Have we a sequence running on this region? Start one if not.
         */
        if( !reg->seq && im->start ) {
                im_lock( &im->sslock );
                reg->seq = im->start( im, im->client1, im->client2 );
                im_unlock( &im->sslock );
 
                if( !reg->seq ) {
                        im_errormsg( "user start function failed for image %s",
                                im->filename );
                        return( -1 );
                }
        }

        return( 0 );
}

/* Call a stop function if a sequence is running in this REGION. No error
 * return, really.
 */
void
im__call_stop( REGION *reg )
{
	IMAGE *im = reg->im;
	int res;

        /* Stop any running sequence.
         */
        if( reg->seq && im->stop ) {
                im_lock( &im->sslock );
                res = im->stop( reg->seq, im->client1, im->client2 );
                im_unlock( &im->sslock );

		if( res )
                        error_exit( "panic: user stop callback failed "
				"for image %s", im->filename );
 
                reg->seq = NULL;
        }
}

/* Create a region. Set no attachments. Either im_prepare() or im_generate()
 * are responsible for getting regions ready for user functions to read
 * from/write to.
 */
REGION *
im_region_create( IMAGE *im )
{	
	REGION *reg;
#ifdef DEBUG_ENVIRONMENT
	char *str;
#endif /*DEBUG_ENVIRONMENT*/

	if( !(reg = IM_NEW( NULL, REGION )) )
		return( NULL );

	reg->im = im;
	reg->valid.left = 0;
	reg->valid.top = 0;
	reg->valid.width = 0;
	reg->valid.height = 0;
	reg->type = IM_REGION_NONE;
	reg->data = NULL;
	reg->bpl = 0;
	reg->seq = NULL;
	reg->buf = NULL;
	reg->bsize = 0;
	reg->baseaddr = NULL;
	reg->length = 0;

	if( im_list_add( (List **) &im->regions, reg ) ) {
		im_region_free( reg );
		return( NULL );
	}

#ifdef DEBUG_ENVIRONMENT
	printf( "iofuncs/region.c: compiled with DEBUG_ENVIRONMENT enabled\n" );
	if( (str = getenv( "IM_MMAP_WINDOW_MARGIN" )) ) {
		im__mmap_window_margin = atoi( str );
		printf( "im_region_create: setting window margin to %d "
			"from environment\n", im__mmap_window_margin );
	}
#endif /*DEBUG_ENVIRONMENT*/

	return( reg );
}

static int
im_region_unmap_window( REGION *reg )
{
	/* unmap the old window
	 */
	if( reg->baseaddr ) {
		if( im__munmap( reg->baseaddr, reg->length ) )
			return( -1 );

#ifdef DEBUG
		total_mmap_usage -= reg->length;
		assert( total_mmap_usage >= 0 );
#endif /*DEBUG*/

		reg->type = IM_REGION_NONE;
		reg->data = NULL;
		reg->length = 0;
		reg->baseaddr = NULL;
	}

	return( 0 );
}

#ifdef DEBUG
static void
trace_mmap_usage( void )
{
	static int last_total = 0;
	int total = total_mmap_usage / (1024 * 1024);
	int max = max_mmap_usage / (1024 * 1024);

	if( total != last_total ) {
		printf( "im_region_mmap_window: current mmap usage of ~%dMB "
			"(high water mark %dMB)\n", total, max );
		last_total = total;
	}
}
#endif /*DEBUG*/

/* Free an image region. Call the region-specific free function, then
 * unlink the region from the IMAGE and free the region struct. We may be
 * called from im_list_fix(), so return dummy result 'reg'. No error returns,
 * really. 
 */
void *
im_region_free( REGION *reg )
{	
	IMAGE *im = reg->im;

        /* Stop this sequence.
         */
        im__call_stop( reg );

	/* Free any attached memory.
	 */
	if( reg->buf ) {
		im_free( reg->buf );
		reg->buf = NULL;
		reg->bsize = 0;
	}

#ifdef DEBUG
	if( reg->baseaddr ) {
		(void) im_region_unmap_window( reg );
		trace_mmap_usage();
	}
#else
	(void) im_region_unmap_window( reg );
#endif /*DEBUG*/

	/* Detach from image. 
	 */
	im_list_remove( (List **) &im->regions, reg );
	reg->im = NULL;

	/* Was this the last region on an image with close_pending? If yes,
	 * close the image too.
	 */
	if( !im->regions && im->close_pending ) {
#ifdef DEBUG_IO
		printf( "im_region_free: closing pending image \"%s\"\n",
			im->filename );
#endif /*DEBUG_IO*/
		/* Time to close the image.
		 */
		im->close_pending = 0;
		im_close( im );
	}

	im_free( reg );

	return( reg );
}

/* Move reg to position r and make sure it has enough local memory to be able
 * to write to all of r. If there is too little space, free the old local 
 * buffer and allocate some more. r is clipped against the size of the image.
 * Note that we never shrink the size of the local pool, even in
 * im_region_image() and im_region_region() below. This is to avoid possible
 * allocate/immediate free cycles in some user functions. Local memory will
 * eventually be freed when the region is removed.
 */
int
im_region_local( REGION *reg, Rect *r )
{
	int sz;
	Rect image;
	Rect clipped;

	/* Clip against image.
	 */
	image.top = 0;
	image.left = 0;
	image.width = reg->im->Xsize;
	image.height = reg->im->Ysize;
	im_rect_intersectrect( r, &image, &clipped );

	/* Test for empty.
	 */
	if( im_rect_isempty( &clipped ) ) {
		im_errormsg( "im_region_local: valid clipped to nothing" );
		return( -1 );
	}

	/* Init new stuff.
	 */
	reg->valid = clipped;
	reg->bpl = IM_REGION_SIZEOF_LINE( reg );

	/* Need new memory?
	 */
	sz = reg->bpl * reg->valid.height;
	if( sz > reg->bsize ) {
		/* Free old memory.
		 */
		if( reg->buf ) {
			im_free( reg->buf );
			reg->buf = NULL;
			reg->bsize = 0;
		}

		/* Allocate new memory.
		 */
		if( !(reg->buf = im_malloc( NULL, sz )) ) 
			return( -1 );
		reg->bsize = sz;
	}

	/* Reset us to local.
	 */
	reg->type = IM_REGION_LOCAL;
	reg->data = reg->buf;

	return( 0 );
}

/* Attach a region to a small section of the image on which it is defined.
 * The IMAGE we are attached to should be im_mmapin(), im_mmapinrw() or 
 * im_setbuf(). The Rect is clipped against the image size.
 */
int
im_region_image( REGION *reg, Rect *r )
{
	Rect image;
	Rect clipped;

	/* Sanity check.
	 */
	if( !reg->im->data ) {
		im_errormsg( "im_region_image: inappropriate image type" );
		return( -1 );
	}

	/* Clip against image.
	 */
	image.top = 0;
	image.left = 0;
	image.width = reg->im->Xsize;
	image.height = reg->im->Ysize;
	im_rect_intersectrect( r, &image, &clipped );

	/* Test for empty.
	 */
	if( im_rect_isempty( &clipped ) ) {
		im_errormsg( "im_region_image: valid clipped to nothing" );
		return( -1 );
	}

	/* Init new stuff.
	 */
	reg->valid = clipped;
	reg->bpl = IM_IMAGE_SIZEOF_LINE( reg->im );
	reg->data = reg->im->data + 
		clipped.top * IM_IMAGE_SIZEOF_LINE( reg->im ) +
		clipped.left * IM_IMAGE_SIZEOF_PEL( reg->im );;
	reg->type = IM_REGION_OTHER_IMAGE;

	return( 0 );
}

static int
im_getpagesize()
{
	int pagesize;

#ifdef HAVE_WINDOWS_H
	SYSTEM_INFO si;

	GetSystemInfo( &si );

	pagesize = si.dwAllocationGranularity;
#else /*HAVE_WINDOWS_H*/
	pagesize = getpagesize();
#endif /*HAVE_WINDOWS_H*/

	return( pagesize );
}

/* Map a window into a file.
 */
static int
im_region_set_window( REGION *reg, int window_top, int window_height )
{
	static int pagesize = -1;

	void *baseaddr;
	off_t start, end, pagestart;
	size_t length, pagelength;

	/* Calculate start and length for our window.
	 */
	start = 64 + (off_t) IM_IMAGE_SIZEOF_LINE( reg->im ) * window_top;
	length = (size_t) IM_IMAGE_SIZEOF_LINE( reg->im ) * window_height;

	/* We can only mmap multiples of the page size ... round down and up.
	 */
	if( pagesize == -1 )
		pagesize = im_getpagesize();

	pagestart = start - start % pagesize;
	end = start + length;
	pagelength = end - pagestart;

	if( !(baseaddr = im__mmap( reg->im->fd, 0, pagelength, pagestart )) )
		return( -1 ); 

	reg->baseaddr = baseaddr;
	reg->length = pagelength;
	reg->data = (char *) baseaddr + (start - pagestart);
	reg->valid.left = 0;
	reg->valid.top = window_top;
	reg->valid.width = reg->im->Xsize;
	reg->valid.height = window_height;
	reg->bpl = IM_IMAGE_SIZEOF_LINE( reg->im );

	/* Sanity check ... make sure the data pointer is readable.
	 */
	im__read_test = reg->data[0] & reg->baseaddr[0];

#ifdef DEBUG
	total_mmap_usage += reg->length;
	if( total_mmap_usage > max_mmap_usage )
		max_mmap_usage = total_mmap_usage;
#endif /*DEBUG*/

	return( 0 );
}

/* reg is defined on an OPENIN image ... do the rolling mmap() window stuff.
 * The Rect is clipped against the image size.
 */
int
im_region_mmap_window( REGION *reg, Rect *r )
{
	Rect image;
	Rect clipped;
	Rect window;

	/* Sanity check.
	 */
	if( reg->im->dtype != IM_OPENIN ) {
		im_errormsg( "im_region_mmap_window: bad image type" );
		return( -1 );
	}

#ifdef DEBUG_CREATE
	if( reg->type == IM_REGION_NONE ) {
		int height = r->height + 2 * im__mmap_window_margin;
		float size = ((float) height * 
			IM_IMAGE_SIZEOF_LINE( reg->im )) / (1024 * 1024);

		printf( "im_region_mmap_window: new window 0x%x on file "
			"\"%s\" of ~%.2fMB\n",
			(unsigned int) reg, reg->im->filename, size );
		printf( "im_region_mmap_window: \"%s\" now has %d regions\n",
			reg->im->filename, im_list_len( reg->im->regions ) );
	}
#endif /*DEBUG_CREATE*/

	/* Clip against image.
	 */
	image.top = 0;
	image.left = 0;
	image.width = reg->im->Xsize;
	image.height = reg->im->Ysize;
	im_rect_intersectrect( r, &image, &clipped );
	if( im_rect_isempty( &clipped ) ) {
		im_errormsg( "im_region_mmap_window: clipped to nothing" );
		return( -1 );
	}

	/* Does the area fit inside the current valid area? Move the window if
	 * it does not.
	 */
	if( reg->type != IM_REGION_MMAP_WINDOW ||
		!im_rect_includesrect( &reg->valid, &clipped ) ) {
		if( im_region_unmap_window( reg ) )
			return( -1 );

		/* Calculate the new window size ... the full width of the
		 * image, the height of the requested area, plus some margins.
		 */
		window.left = 0;
		window.top = r->top - im__mmap_window_margin;
		window.width = reg->im->Xsize;
		window.height = r->height + 2 * im__mmap_window_margin;
		im_rect_intersectrect( &window, &image, &clipped );

#ifdef DEBUG
		printf( "im_region_mmap_window: moving window 0x%x on file "
			"\"%s\"\n"
			"    from top=%d, height=%d to top=%d, height=%d\n",
			(unsigned int) reg, reg->im->filename,
			reg->valid.top, reg->valid.height, 
			clipped.top, clipped.height );
#endif /*DEBUG*/

		/* Map new window.
		 */
		if( im_region_set_window( reg, clipped.top, clipped.height ) ) 
			return( -1 );

		reg->type = IM_REGION_MMAP_WINDOW;
	}

#ifdef DEBUG
	trace_mmap_usage();
#endif /*DEBUG*/

	return( 0 );
}

/* Make IM_REGION_ADDR() stuff to reg go to dest instead. 
 *
 * r is the part of the reg image which you want to be able to write to (this
 * effectively becomes the valid field), (x,y) is the top LH corner of the
 * corresponding area in dest.
 *
 * Performs all clippings necessary to ensure that &reg->valid is indeed
 * valid.
 *
 * If the region we attach to is modified, we are left with dangling pointers!
 * If the region we attach to is on another image, the two images must have 
 * the same BandFmt and number of Bands.
 */
int
im_region_region( REGION *reg, REGION *dest, Rect *r, int x, int y )
{
	Rect image;
	Rect wanted;
	Rect clipped;
	Rect clipped2;
	Rect final;

	/* Sanity check.
	 */
	if( !dest->data || dest->im->BandFmt != reg->im->BandFmt ||
		dest->im->Bands != reg->im->Bands ) {
		im_errormsg( "im_region_region: inappropriate region type" );
		return( -1 );
	}

	/* Clip r against size of reg.
	 */
	image.top = 0;
	image.left = 0;
	image.width = reg->im->Xsize;
	image.height = reg->im->Ysize;
	im_rect_intersectrect( r, &image, &clipped );

	/* Translate to dest's coordinate space and clip against the available
	 * pixels.
	 */
	wanted.left = x + (clipped.left - r->left);
	wanted.top = y + (clipped.top - r->top);
	wanted.width = clipped.width;
	wanted.height = clipped.height;
	im_rect_intersectrect( &wanted, &dest->valid, &clipped2 );

	/* Translate back to reg's coordinate space and set as valid.
	 */
	final.left = r->left + (clipped2.left - wanted.left);
	final.top = r->top + (clipped2.top - wanted.top);
	final.width = clipped2.width;
	final.height = clipped2.height;

	/* Test for empty.
	 */
	if( im_rect_isempty( &final ) ) {
		im_errormsg( "im_region_region: valid clipped to nothing" );
		return( -1 );
	}

	/* Init new stuff.
	 */
	reg->valid = final;
	reg->bpl = dest->bpl;
	reg->data = IM_REGION_ADDR( dest, clipped2.left, clipped2.top );
	reg->type = IM_REGION_OTHER_REGION;

	return( 0 );
}

/* Do two regions point to the same piece of image? ie. 
 * 	IM_REGION_ADDR( reg1, x, y ) == IM_REGION_ADDR( reg2, x, y ) &&
 * 	*IM_REGION_ADDR( reg1, x, y ) == *IM_REGION_ADDR( reg2, x, y ) for all x, y, reg1, reg2.
 */
int
im_region_equalsregion( REGION *reg1, REGION *reg2 )
{
	return( reg1->im == reg2->im &&
		im_rect_equalsrect( &reg1->valid, &reg2->valid ) &&
		reg1->data == reg2->data );
}

/* Set the position of a region. This only affects reg->valid, ie. the way
 * pixels are addressed, not reg->data, the pixels which are addressed. Clip
 * against the size of the image. Do not allow negative positions, or
 * positions outside the image.
 */
int
im_region_position( REGION *reg, int x, int y )
{
	Rect req, image, clipped;

	/* Clip!
	 */
	image.top = 0;
	image.left = 0;
	image.width = reg->im->Xsize;
	image.height = reg->im->Ysize;
	req.top = y;
	req.left = x;
	req.width = reg->valid.width;
	req.height = reg->valid.height;
	im_rect_intersectrect( &image, &req, &clipped );
	if( x < 0 || y < 0 || im_rect_isempty( &clipped ) ) {
		im_errormsg( "im_region_position: bad position" );
		return( -1 );
	}

	reg->valid = clipped;
	return( 0 );
}
