// Object part of VMask class

/*

    Copyright (C) 1991-2001 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include <stdlib.h>
#include <math.h>

#include <vips/vips.h>
#include <vips/vipscpp.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Functions for VMask - refcounting layer over VPMask.
 */

VMask::~VMask()
{
	ref->nrefs--;
	if( !ref->nrefs )
		delete ref;
}

VMask &VMask::operator=( const VMask &a )
{ 
	// Loosing ref to LHS
	ref->nrefs--;

	if( ref->nrefs > 0 )
		// Need fresh refblock
		ref = new refblock;
	else 
		// Recycle old refblock
		delete ref->pmask;

	// LHS now points to RHS
	ref = a.ref; 
	ref->nrefs++; 
	
	return( *this ); 
}

// Make sure this is a private copy of pmask --- dup if nrefs != 1
void VMask::make_private()
{
	if( ref->nrefs > 1 ) {
		// Make fresh refblock
		refblock *ref2 = new refblock;

		// And copy the mask
		ref2->pmask = ref->pmask->dup();
		ref->nrefs--;
		ref = ref2;
	}
}

// Embed INTMASK in VIMask
void VIMask::embed( void *i )
{
	if( ref->pmask )
		verror( "embed: VIMask not empty" );
	ref->pmask = new VPIMask( i );
}

// Type conversions: implicit INTMASK to DOUBLEMASK 
VIMask::operator VDMask()
{
	VDMask out( xsize(), ysize() );

	((DOUBLEMASK *) out.mask())->scale = scale();
	((DOUBLEMASK *) out.mask())->offset = offset();

	for( int i = 0; i < size(); i++ )
		out[i] = (*this)[i];

	return( out );
}


// Forward ref of VImage class
class VImage;

// Type conversions: implicit DOUBLEMASK to INTMASK
VDMask::operator VIMask()
{
	VIMask out( xsize(), ysize() );

	((INTMASK *) out.mask())->scale = int( scale() );
	((INTMASK *) out.mask())->offset = int( offset() );

	for( int i = 0; i < size(); i++ )
		out[i] = (int) rint( (*this)[i] );

	return( out );
}

// Type conversions: implicit DOUBLEMASK to VImage
VDMask::operator VImage()
{
	VImage out;

	if( im_mask2vips( (DOUBLEMASK *) mask(), (IMAGE *) out.image() ) )
		verror();

	return( out );
}

// ... and INTMASK to VImage
VIMask::operator VImage() { return( VImage( VDMask( *this ) ) ); }

// Embed DOUBLEMASK in VDMask
void VDMask::embed( void *i )
{
	if( ref->pmask )
		verror( "embed: VDMask not empty" );
	ref->pmask = new VPDMask( i );
}

/* Functions for P*Mask - layer over im_*_*mask() functions.
 */

// Create empty imask
VPIMask::VPIMask( int xsize, int ysize )
{
	if( !(data = im_create_imask( "VPIMask::VPIMask", xsize, ysize )) )
		verror();
	type = VPMask::INT;
}

// Init from data
VPIMask::VPIMask( int xsize, int ysize, int scale, int offset, va_list ap )
{
	if( !(data = im_create_imask( "VPIMask::VPIMask", xsize, ysize )) )
		verror();
	type = VPMask::INT;

	((INTMASK *) data)->scale = scale;
	((INTMASK *) data)->offset = offset;
	for( int i = 0; i < xsize * ysize; i++ )
		((INTMASK *) data)->coeff[i] = va_arg( ap, int );
}

// Create from filename
VPIMask::VPIMask( const char *name )
{
	if( !(data = im_read_imask( (char *) name )) )
		verror();
	type = VPMask::INT;
}

// Create from existing INTMASK
VPIMask::VPIMask( void *imask )
{
	data = imask;
	type = VPMask::INT;
}

// Create empty
VPIMask::VPIMask()
{
	data = 0;
	type = VPMask::UNASSIGNED;
}

VPIMask::~VPIMask()
{
	if( data ) {
		im_free_imask( (INTMASK *) data );
		data = 0;
		type = VPMask::UNASSIGNED;
	}
}

// Duplicate -- we are a VPIMask, return a new VPIMask which is a copy of us.
// Return as a VPMask tho'.
VPMask *VPIMask::dup() const
{
	VPIMask *out = new VPIMask();

	INTMASK *msk;
	if( !(msk = im_dup_imask( (INTMASK *) data, "VPIMask::dup" )) ) {
		delete out;
		verror();
	}
	out->embed( msk );

	return( out );
}

// Insert INTMASK pointer
void VPIMask::embed( void *msk )
{
	if( type != VPMask::UNASSIGNED )
		verror( "VPIMask::embed: VPIMask not empty" );

	data = msk;
	type = VPMask::INT;
}

int VPIMask::xsize() const
{
	if( !data ) 
		verror( "xsize: mask not set" );

	return( ((INTMASK *) data)->xsize );
}

int VPIMask::ysize() const
{
	if( !data ) 
		verror( "ysize: mask not set" );

	return( ((INTMASK *) data)->ysize );
}

int VPIMask::scale() const
{
	if( !data ) 
		verror( "scale: mask not set" );

	return( ((INTMASK *) data)->scale );
}

int VPIMask::offset() const
{
	if( !data ) 
		verror( "offset: mask not set" );

	return( ((INTMASK *) data)->offset );
}

const char *VPIMask::filename() const
{
	if( !data ) 
		verror( "filename: mask not set" );

	return( ((INTMASK *) data)->filename );
}

void VPIMask::print( std::ostream &file ) const
{
	if( !data )
		verror( "internal error #7447234" );
	
	INTMASK *dmsk = (INTMASK *) data;
	int i, j;
	int *p = dmsk->coeff;

	file << this->xsize() << "\t" << this->ysize() << "\t";
	file << this->scale() << "\t" << this->offset() << "\n";

	for( i = 0; i < this->ysize(); i++ ) {
		for( j = 0; j < this->xsize(); j++ )
			file << *p++ << "\t";

		file << "\n";
	}
}

// Extract start of int array
int *VPIMask::array() const 
{ 
	return( ((INTMASK *)data)->coeff ); 
}

// Create empty dmask
VPDMask::VPDMask( int xsize, int ysize )
{
	if( !(data = im_create_dmask( "VPDMask::VPDMask", xsize, ysize )) )
		verror();
	type = VPMask::DOUBLE;
}

// Create from args
VPDMask::VPDMask( int xsize, int ysize, 
	double scale, double offset, va_list ap )
{
	if( !(data = im_create_dmask( "VPDMask::VPDMask", xsize, ysize )) )
		verror();
	type = VPMask::DOUBLE;

	((DOUBLEMASK *) data)->scale = scale;
	((DOUBLEMASK *) data)->offset = offset;
	for( int i = 0; i < xsize * ysize; i++ )
		((DOUBLEMASK *) data)->coeff[i] = va_arg( ap, double );
}

// Create from filename
VPDMask::VPDMask( const char *name )
{
	if( !(data = im_read_dmask( (char *) name )) )
		verror();
	type = VPMask::DOUBLE;
}

// Create empty
VPDMask::VPDMask()
{
	data = 0;
	type = VPMask::UNASSIGNED;
}

// Create from existing DOUBLEMASK
VPDMask::VPDMask( void *dmask )
{
	data = dmask;
	type = VPMask::DOUBLE;
}

VPDMask::~VPDMask()
{
	if( data ) {
		im_free_dmask( (DOUBLEMASK *) data );
		data = 0;
		type = VPMask::UNASSIGNED;
	}
}

// Duplicate -- we are a VPIMask, return a new VPIMask which is a copy of us.
// Return as a VPMask tho'.
VPMask *VPDMask::dup() const
{
	VPDMask *out = new VPDMask();

	DOUBLEMASK *msk;
	if( !(msk = im_dup_dmask( (DOUBLEMASK *) data, "VPDMask::dup" )) ) {
		delete out;
		verror();
	}
	out->embed( msk );

	return( out );
}

// Insert DOUBLEMASK pointer
void VPDMask::embed( void *msk )
{
	if( type != VPMask::UNASSIGNED )
		verror( "VPDMask::embed: VPDMask not empty" );

	data = msk;
	type = VPMask::DOUBLE;
}

int VPDMask::xsize() const
{
	if( !data ) 
		verror( "xsize: mask not set" );

	return( ((DOUBLEMASK *) data)->xsize );
}

int VPDMask::ysize() const
{
	if( !data ) 
		verror( "ysize: mask not set" );

	return( ((DOUBLEMASK *) data)->ysize );
}

double VPDMask::scale() const
{
	if( !data ) 
		verror( "scale: mask not set" );

	return( ((DOUBLEMASK *) data)->scale );
}

double VPDMask::offset() const
{
	if( !data ) 
		verror( "offset: mask not set" );

	return( ((DOUBLEMASK *) data)->offset );
}

const char *VPDMask::filename() const
{
	if( !data ) 
		verror( "filename: mask not set" );

	return( ((DOUBLEMASK *) data)->filename );
}

void VPDMask::print( std::ostream &file ) const
{
	if( !data )
		verror( "internal error #7447234" );
	
	DOUBLEMASK *dmsk = (DOUBLEMASK *) data;
	int i, j;
	double *p = dmsk->coeff;

	file << this->xsize() << "\t" << this->ysize() << "\t";
	file << this->scale() << "\t" << this->offset() << "\n";

	for( i = 0; i < this->ysize(); i++ ) {
		for( j = 0; j < this->xsize(); j++ )
			file << *p++ << "\t";

		file << "\n";
	}
}

// Extract data pointer
double *VPDMask::array() const 
{ 
	return( ((DOUBLEMASK *)data)->coeff ); 
}

// Build functions
VIMask VIMask::gauss( double sig, double minamp )
{
	VIMask out;
	INTMASK *msk;

	if( !(msk = im_gauss_imask( "VIMask::gauss", sig, minamp )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::gauss( double sig, double minamp )
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_gauss_dmask( "VDMask::gauss", sig, minamp )) )
		verror();
	out.embed( msk );

	return( out );
}

VIMask VIMask::log( double sig, double minamp )
{
	VIMask out;
	INTMASK *msk;

	if( !(msk = im_log_imask( "VIMask::log", sig, minamp )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::log( double sig, double minamp )
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_log_dmask( "VDMask::log", sig, minamp )) )
		verror();
	out.embed( msk );

	return( out );
}

// Manipulation functions
VIMask VIMask::rotate45()
{
	VIMask out;
	INTMASK *msk;

	if( !(msk = im_rotate_imask45( (INTMASK *) mask(), 
		"VIMask::rotate45" )) )
		verror();
	out.embed( msk );

	return( out );
}

VIMask VIMask::rotate90()
{
	VIMask out;
	INTMASK *msk;

	if( !(msk = im_rotate_imask90( (INTMASK *) mask(), 
		"VIMask::rotate90" )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::rotate45()
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_rotate_dmask45( (DOUBLEMASK *) mask(), 
		"VDMask::rotate45" )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::rotate90()
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_rotate_dmask90( (DOUBLEMASK *) mask(), 
		"VDMask::rotate90" )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::trn()
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_mattrn( (DOUBLEMASK *) mask(), "VDMask::trn" )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::inv()
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_matinv( (DOUBLEMASK *) mask(), "VDMask::inv" )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::mul( VDMask m )
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_matmul( (DOUBLEMASK *) mask(), (DOUBLEMASK *) m.mask(), 
		"VDMask::mul" )) )
		verror();
	out.embed( msk );

	return( out );
}

VDMask VDMask::cat( VDMask m )
{
	VDMask out;
	DOUBLEMASK *msk;

	if( !(msk = im_matcat( (DOUBLEMASK *) mask(), (DOUBLEMASK *) m.mask(), 
		"VDMask::cat" )) )
		verror();
	out.embed( msk );

	return( out );
}

VIMask VDMask::scalei()
{
	VDMask out;
	INTMASK *msk;

	if( !(msk = im_scale_dmask( (DOUBLEMASK *) mask(), "VDMask::scalei" )) )
		verror();
	out.embed( msk );

	return( out );
}

// Arithmetic on a VIMask ... just cast and use VDMask
VDMask VIMask::trn() { return( ((VDMask)*this).trn() ); }
VDMask VIMask::inv() { return( ((VDMask)*this).inv() ); }
VDMask VIMask::cat( VDMask a ) { return( ((VDMask)*this).cat( a ) ); }
VDMask VIMask::mul( VDMask a ) { return( ((VDMask)*this).mul( a ) ); }

// Overload [] to get linear array subscript.
// Our caller may write to the result, so make sure we have a private
// copy.
// Involves function call, slow anyway, so do range checking
int &VIMask::operator[]( int x )
{ 
	if( ref->nrefs != 1 )
		make_private();

	if( x > size() )
		verror( "VIMask::operator[]: subscript out of range" );

	return( ((VPIMask *)ref->pmask)->array()[x] ); 
}

double &VDMask::operator[]( int x )
{ 
	if( ref->nrefs != 1 )
		make_private();

	if( x > size() )
		verror( "VDMask::operator[]: subscript out of range" );

	return( ((VPDMask *)ref->pmask)->array()[x] ); 
}
