/* VIPS universal main program.
 *
 * J. Cupitt, 8/4/93.
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <locale.h>

#include <vips/vips.h>
#include <vips/util.h>
#include <vips/dispatch.h>

#ifdef HAVE_WINDOWS_H
#define strcasecmp(a,b) _stricmp(a,b)
#endif

/* Name we were run as.
 */
static char *our_name;

/* Print a package's name.
 */
static void *
print_package_name( im_package *pack )
{
	printf( "\t%s (%d functions)\n", pack->name, pack->nfuncs );
	return( NULL );
}

/* List all functions in a package, with their description.
 */
static void *
list_functions( im_package *pack )
{
	int i;

	printf( "functions available in package \"%s\":\n", pack->name );
	for( i = 0; i < pack->nfuncs; i++ ) 
		printf( "%-18s - %s\n", 
			pack->table[i]->name, pack->table[i]->desc );
	
	return( NULL );
}

/* Is s1 a prefix of s2?
 */
static int
isprefix( char *s1, char *s2 )
{
	while( *s1 && *s1 == *s2 ) {
		s1++;
		s2++;
	}

	return( *s1 == '\0' );
}

/* Is s1 a postfix of s2?
 */
static int
ispostfix( char *s1, char *s2 )
{
	int l1 = strlen( s1 );
	int l2 = strlen( s2 );

	if( l2 < l1 )
		return( 0 );
	
	return( strcasecmp( s1, s2 + l2 - l1 ) == 0 );
}

/* Print "ln -s" lines for this package.
 */
static void *
print_links( im_package *pack )
{
	int i;

	for( i = 0; i < pack->nfuncs; i++ ) 
#ifdef HAVE_WINDOWS_H
		printf( "rm -f %s.exe; ln -s vips.exe %s.exe\n", 
			pack->table[i]->name, pack->table[i]->name );
#else /*HAVE_WINDOWS_H*/
		printf( "rm -f %s; ln -s vips %s\n", 
			pack->table[i]->name, pack->table[i]->name );
#endif /*HAVE_WINDOWS_H*/

	return( NULL );
}

/* Does a function have any printing output?
 */
static int
has_print( im_function *fn )
{
	int i;

	for( i = 0; i < fn->argc; i++ )
		if( fn->argv[i].print )
			return( -1 );

	return( 0 );
}

/* Print a usage string from an im_function descriptor.
 */
static void
usage( im_function *fn )
{
	int i;
	im_package *pack = im_package_of_function( fn->name );

	fprintf( stderr, "%s: args: ", our_name ); 

	/* Print args requiring command-line input.
	 */
	for( i = 0; i < fn->argc; i++ )
		if( fn->argv[i].desc->flags & IM_TYPE_ARG )
			fprintf( stderr, "%s ", fn->argv[i].name );

	/* Print types of command line args.
	 */
	fprintf( stderr, "\nwhere:\n" );
	for( i = 0; i < fn->argc; i++ )
		if( fn->argv[i].desc->flags & IM_TYPE_ARG )
			fprintf( stderr, "\t%s is of type \"%s\"\n", 
				fn->argv[i].name, fn->argv[i].desc->type );

	/* Print output print args.
	 */
	if( has_print( fn ) ) {
		fprintf( stderr, "prints:\n" );
		for( i = 0; i < fn->argc; i++ )
			if( fn->argv[i].print ) 
				fprintf( stderr, "\t%s of type \"%s\"\n", 
					fn->argv[i].name, 
					fn->argv[i].desc->type );
	}

	/* Print description of this function, and package it comes from.
	 */
	fprintf( stderr, "%s", fn->desc );
	if( pack )
		fprintf( stderr, ", from package \"%s\"", pack->name );
	fprintf( stderr, "\n" );

	/* Print any flags this function has.
	 */
	fprintf( stderr, "flags: " );
	if( fn->flags & IM_FN_PIO )
		fprintf( stderr, "(PIO function) " );
	else
		fprintf( stderr, "(WIO function) " );
	if( fn->flags & IM_FN_TRANSFORM )
		fprintf( stderr, "(coordinate transformer) " );
	else
		fprintf( stderr, "(no coordinate transformation) " );
	if( fn->flags & IM_FN_PTOP )
		fprintf( stderr, "(point-to-point operation) " );
	else
		fprintf( stderr, "(area operation) " );
	fprintf( stderr, "\n" );
}

/* Help system.
 */
static void
print_help( char *cat )
{
	im_function *fn;
	im_package *pack;

	if( !cat ) {
		/* Plain "-help". 
		 */
		printf( 
"\"-help flags\" for a list of flags available\n"
"\"-help <package name>\" for a list of commands in package\n"
"where <package name> is one of:\n" );
		im_map_packages( 
			(im_list_map_fn) print_package_name, NULL );
		printf( 
"\"-help all\" for a list of all commands in all packages\n" 
"\"-help <command>\" for an explanation of a command\n" ); 
	}
	else if( strcmp( cat, "flags" ) == 0 )
		printf(
"flags supported are:\n"
"-help\t\t\tsimple help system\n"
"-plugin <file name>\tadd commands in plugin to those available\n"
"-links\t\t\tprint link lines for all functions\n"
"-ip <package name>\tprint sample ip \".def\" file for package\n"
"-c++ <package name>\tprint C++ bindings for package\n" 
"-c++.h <package name>\tprint just C++ member decls\n"
"-c++.cc <package name>\tprint just C++ function bodies\n" );
	else if( strcmp( cat, "all" ) == 0 )
		im_map_packages( (im_list_map_fn) list_functions, NULL );
	else {
		/* Search for package or function of this name.
		 */
		if( (fn = im_find_function( cat )) )
			/* Found a matching function. Print usage.
			 */
			usage( fn );
		else if( (pack = im_find_package( cat )) ) 
			(void) list_functions( pack );
		else if( !isprefix( "im_", cat ) ) {
			char buf[ 256 ];

			im_snprintf( buf, 256, "im_%s", cat );
			if( (fn = im_find_function( buf )) )
				usage( fn );
		}
		else
			error_exit( "%s: -help: no such function or "
				"package \"%s\"", our_name, cat );
	}
}

/* Print a sample .def file for package.
 */
static void
print_defs( char *pname )
{
	im_package *pack = im_find_package( pname );
	int i, j, k;

	/* VIPS types.
	 */
	static char *vtypes[] = {
		IM_TYPE_DOUBLE,
		IM_TYPE_INT,  
		IM_TYPE_COMPLEX,
		IM_TYPE_STRING,
		IM_TYPE_IMAGE         
	};

	/* Corresponding ip type predicates.
	 */
	static char *itypes[] = {
		"is_number",
		"is_number",
		"is_complex",
		"is_string",
		"is_imreg"
	};

	/* Corresponding ip names.
	 */
	static char *inames[] = {
		"number",
		"number",
		"complex",
		"string",
		"image"
	};

	if( !pack )
		error_exit( "%s: -ip: no such package \"%s\"", 
			our_name, pname );

	for( i = 0; i < pack->nfuncs; i++ ) {
		im_function *fn = pack->table[i];
		char name[ 4096 ];

		/* Make name for function: chop off "im_" prefix.
		 */
		if( isprefix( "im_", fn->name ) )
			strcpy( name, fn->name + 3 );
		/* Nope --- try capitalising the initial letter.
		 */
		else if( islower( fn->name[0] ) ) {
			strcpy( name, fn->name );
			name[0] = toupper( name[0] );
		}
		/* Nope --- add an ip_ prefix.
		 */
		else {
			strcpy( name, "ip_" );
			strcat( name, fn->name );
		}
		printf( "/* %s", name );

		/* Print names for each input arg. Ignore DISPLAY arguments.
		 */
		for( j = 0; j < fn->argc; j++ ) {
			im_type_desc *vt = fn->argv[j].desc;

			if( !(vt->flags & IM_TYPE_OUTPUT) &&
				strcmp( vt->type, IM_TYPE_DISPLAY ) != 0 )
				printf( " %s", fn->argv[j].name );
		}
		
		/* Print description.
		 */
		printf( ": %s\n */\n\n", fn->desc );

		/* Print definition!
		 */
		printf( "%s", name );
		for( j = 0; j < fn->argc; j++ ) {
			im_type_desc *vt = fn->argv[j].desc;

			if( !(vt->flags & IM_TYPE_OUTPUT) &&
				strcmp( vt->type, IM_TYPE_DISPLAY ) != 0 )
				printf( " %s", fn->argv[j].name );
		}
		printf( "\n" );

		/* Print a typecheck for each arg.
		 */
		for( j = 0; j < fn->argc; j++ ) {
			im_type_desc *ty = fn->argv[j].desc;

			/* Skip output args.
			 */
			if( ty->flags & IM_TYPE_OUTPUT )
				continue;
			
			/* Skip DISPLAY args.
			 */
			if( strcmp( ty->type, IM_TYPE_DISPLAY ) == 0 )
				continue;

			/* Find type.
			 */
			for( k = 0; k < IM_NUMBER( vtypes ); k++ )
				if( strcmp( ty->type, vtypes[k] ) == 0 ) {
					printf( 				
"      =\terror \"argument %d is not %s\", not %s %s;\n",
						j + 1, inames[k], itypes[k],
						fn->argv[j].name );
					break;
				}

			/* Help!
			 */
			if( k == IM_NUMBER( vtypes ) )
				printf( 
"      =\terror \"unknown type %s\", true;\n", ty->type );
		}

		/* Call the function itself.
		 */
		printf( "      =\t%s", fn->name );
		for( j = 0; j < fn->argc; j++ ) {
			im_type_desc *ty = fn->argv[j].desc;

			/* Skip output args.
			 */
			if( ty->flags & IM_TYPE_OUTPUT )
				continue;
			
			/* Skip DISPLAY args.
			 */
			if( strcmp( ty->type, IM_TYPE_DISPLAY ) == 0 )
				continue;

			/* If its an IMAGE, then forceim it.
			 */
			if( strcmp( ty->type, IM_TYPE_IMAGE ) == 0 )
				printf( " (forceim %s)", 
					fn->argv[j].name );
			else
				printf( " %s", 
					fn->argv[j].name );
		}
		printf( ";\n\n" );
	}
}

/* Convert VIPS type name to C++ type name. NULL for type unsupported by C++
 * layer.
 */
static char *
vips2cpp( im_type_desc *ty )
{
	int k;

	/* VIPS types.
	 */
	static char *vtypes[] = {
		IM_TYPE_DOUBLE,
		IM_TYPE_INT,  
		IM_TYPE_COMPLEX,
		IM_TYPE_STRING,
		IM_TYPE_IMAGE,
		IM_TYPE_IMASK,
		IM_TYPE_DMASK,
		IM_TYPE_DISPLAY
	};

	/* Corresponding C++ types.
	 */
	static char *ctypes[] = {
		"double",
		"int",
		"std::complex<double>",
		"char*",
		"VImage",
		"VIMask",
		"VDMask",
		"VDisplay"
	};

	for( k = 0; k < IM_NUMBER( vtypes ); k++ )
		if( strcmp( ty->type, vtypes[k] ) == 0 ) 
			return( ctypes[k] );

	return( NULL );
}

/* Test a function definition for C++ suitability.
 */
static int
is_cppable( im_function *fn )
{
	int j;

	/* Check we know all the types.
	 */
	for( j = 0; j < fn->argc; j++ ) {
		im_type_desc *ty = fn->argv[j].desc;

		if( !vips2cpp( ty ) )
			return( 0 );
	}

	/* Must be at least one image argument (input or output) ... since we 
	 * get inserted in the VImage class. Other funcs get wrapped by hand.
	 */
	for( j = 0; j < fn->argc; j++ ) 
		if( strcmp( fn->argv[j].desc->type, IM_TYPE_IMAGE ) == 0 ) 
			break;
	if( j == fn->argc )
		return( 0 );

	return( -1 );
}

/* Search for the first output arg, and the first IMAGE input arg.
 */
static void
find_ioargs( im_function *fn, int *ia, int *oa )
{
	int j;

	/* Look for first output arg - this will be the result of the
	 * function.
	 */
	*oa = -1;
	for( j = 0; j < fn->argc; j++ ) {
		im_type_desc *ty = fn->argv[j].desc;

		if( ty->flags & IM_TYPE_OUTPUT ) {
			*oa = j;
			break;
		}
	}

	/* Look for first input IMAGE arg. This will become the implicit
	 * "this" arg.
	 */
	*ia = -1;
	for( j = 0; j < fn->argc; j++ ) {
		im_type_desc *ty = fn->argv[j].desc;

		if( !(ty->flags & IM_TYPE_OUTPUT) && 
			strcmp( ty->type, IM_TYPE_IMAGE ) == 0 ) {
				*ia = j;
				break;
			}
	}
}

/* Print decl for a function. If decl == 1, then print member part, else print
 * full declaration. Print "&" for reference args for other than first
 * output.
 *
 * decl == 1:
 * 	VImage cos();
 *	VImage lin( double, double );
 *
 * decl == 0:
 * 	VImage VImage::cos()
 *	VImage VImage::lin( double a, double b )
 */
static void
print_cppdecl( im_function *fn, int decl )
{
	int j;
	char name[ 4096 ];
	int oa, ia;
	int flg;

	/* If it's not cppable, do nothing.
	 */
	if( !is_cppable( fn ) )
		return;

	/* Make name for function: chop off "im_" prefix.
	 */
	if( isprefix( "im_", fn->name ) )
		strcpy( name, fn->name + 3 );
	else
		strcpy( name, fn->name );

	/* And drop "tra" postfix (eg. so im_costra becomes cos).
	 */
	if( ispostfix( "tra", name ) )
		name[strlen( name ) - 3] = '\0';

	/* Find input and output args. 
	 */
	find_ioargs( fn, &ia, &oa );
	if( ia == -1 && decl ) 
		/* No input image, so make it a static in the class
		 * declaration.
		 */
		printf( "static " );

	/* Print output type.
	 */
	if( oa == -1 )
		printf( "void " );
	else 
		printf( "%s ", vips2cpp( fn->argv[oa].desc ) );

	/* Print function name and start arg list.
	 */
	if( decl )
		printf( "%s(", name );
	else
		printf( "VImage::%s(", name );

	/* Print arg list.
	 */
	flg = 0;
	for( j = 0; j < fn->argc; j++ ) {
		im_type_desc *ty = fn->argv[j].desc;

		/* Skip ia and oa.
		 */
		if( j == ia || j == oa )
			continue;

		/* Print arg type.
		 */
		if( flg )
			printf( ", %s", vips2cpp( ty ) );
		else {
			printf( " %s", vips2cpp( ty ) );
			flg = 1;
		}

		/* If it's an putput arg, print a "&" to make a reference
		 * argument.
		 */
		if( ty->flags & IM_TYPE_OUTPUT )
			printf( "&" );

		/* Print arg name.
		 */
		if( !decl )
			printf( " %s", fn->argv[j].name );
	}

	/* End of arg list!
	 */
	if( flg )
		printf( " " );
	printf( ")" );
	if( decl )
		printf( ";" );
	printf( "\n" );
}

/* Print the definition for a function.
 */
static void
print_cppdef( im_function *fn )
{
	int j;
	int ia, oa;

	/* If it's not cppable, do nothing.
	 */
	if( !is_cppable( fn ) )
		return;

	find_ioargs( fn, &ia, &oa );

	printf( "// %s: %s\n", fn->name, fn->desc );
	print_cppdecl( fn, 0 );
	printf( "{\n" );

	/* Declare the implicit input image.
	 */
	if( ia != -1 )
		printf( "\tVImage %s = *this;\n", fn->argv[ia].name );

	/* Declare return value, if any.
	 */
	if( oa != -1 )
		printf( "\t%s %s;\n\n", 
			vips2cpp( fn->argv[oa].desc ),
			fn->argv[oa].name );

	/* Declare the arg vector.
	 */
	printf( "\tVargv _vec( \"%s\" );\n\n", fn->name );

	/* Create the input args.
	 */
	for( j = 0; j < fn->argc; j++ ) {
		im_type_desc *ty = fn->argv[j].desc;

		/* Images are special - have to init the vector, even
		 * for output args. Have to translate VImage.
		 */
		if( strcmp( ty->type, IM_TYPE_IMAGE ) == 0 ) {
			printf( "\t_vec.data(%d) = %s.image();\n",
				j, fn->argv[j].name );
			continue;
		}

		/* For output masks, we have to set an input filename. Not
		 * freed, so constant string is OK.
		 */
		if( (ty->flags & IM_TYPE_OUTPUT) && 
			(strcmp( ty->type, IM_TYPE_IMASK ) == 0 ||
			strcmp( ty->type, IM_TYPE_DMASK ) == 0) ) {
			printf( "\t((im_mask_object*) _vec.data(%d))->name = "
				"(char*)\"noname\";\n", j );
			continue;
		}

		/* Skip other output args.
		 */
		if( ty->flags & IM_TYPE_OUTPUT )
			continue;

		if( strcmp( ty->type, IM_TYPE_IMASK ) == 0 ||
			strcmp( ty->type, IM_TYPE_DMASK ) == 0 ) 
			/* Mask types are different - have to use
			 * im_mask_object.
			 */
			printf( "\t((im_mask_object*) "
				"_vec.data(%d))->mask = %s.mask();\n",
				j, fn->argv[j].name );
		else if( strcmp( ty->type, IM_TYPE_DISPLAY ) == 0 )
			/* Display have to use VDisplay.
			 */
			printf( "\t_vec.data(%d) = %s.disp();\n",
				j, fn->argv[j].name );
		else if( strcmp( ty->type, IM_TYPE_STRING ) == 0 )
			/* Zap input strings directly into _vec.
			 */
			printf( "\t_vec.data(%d) = (im_object) %s;\n",
				j, fn->argv[j].name );
		else
			/* Just use vips2cpp().
			 */
			printf( "\t*((%s*) _vec.data(%d)) = %s;\n",
				vips2cpp( ty ), j, fn->argv[j].name );
	}

	/* Call function.
	 */
	printf( "\t_vec.call();\n" );

	/* Extract output args.
	 */
	for( j = 0; j < fn->argc; j++ ) {
		im_type_desc *ty = fn->argv[j].desc;

		/* Skip input args.
		 */
		if( !(ty->flags & IM_TYPE_OUTPUT) )
			continue;

		/* Skip images (done on input side, really).
		 */
		if( strcmp( ty->type, IM_TYPE_IMAGE ) == 0 )
			continue;

		if( strcmp( ty->type, IM_TYPE_IMASK ) == 0 ||
			strcmp( ty->type, IM_TYPE_DMASK ) == 0 ) 
			/* Mask types are different - have to use
			 * im_mask_object.
			 */
			printf( "\t%s.embed( (void *)((im_mask_object*)"
				"_vec.data(%d))->mask );\n",
				fn->argv[j].name, j );
		else if( strcmp( ty->type, IM_TYPE_STRING ) == 0 )
			/* Strings are grabbed out of the vec.
			 */
			printf( "\t%s = (char*) _vec.data(%d);\n",
				fn->argv[j].name, j ); 
		else 
			/* Just use vips2cpp().
			 */
			printf( "\t%s = *((%s*)_vec.data(%d));\n",
				fn->argv[j].name, vips2cpp( ty ), j ); 
	}

	/* Note dependancies if out is an image and this function uses
	 * PIO.
	 */
	if( oa != -1 ) {
		im_type_desc *ty = fn->argv[oa].desc;
		
		if( strcmp( ty->type, IM_TYPE_IMAGE ) == 0 &&
			(fn->flags & IM_FN_PIO) ) {
			/* Loop for all input args again ..
			 */
			for( j = 0; j < fn->argc; j++ ) {
				im_type_desc *ty2 = fn->argv[j].desc;

				/* Skip output args.
				 */
				if( ty2->flags & IM_TYPE_OUTPUT )
					continue;

				/* Skip non-image args.
				 */
				if( strcmp( ty2->type, 
					IM_TYPE_IMAGE ) != 0 )
					continue;

				/* Found one!
				 */
				printf( "\t%s._ref->addref( %s._ref );\n",
					fn->argv[oa].name,
					fn->argv[j].name );
			}
		}
	}

	/* Return result.
	 */
	if( oa != -1 )
		printf( "\n\treturn( %s );\n", fn->argv[oa].name );

	printf( "}\n" );
}

/* Generate member function decls for package.
 */
static void *
print_cpppackdelcs( im_package *pack )
{
	int i;

	printf( "// Member function decls for package \"%s\"\n", pack->name );
	for( i = 0; i < pack->nfuncs; i++ ) 
		print_cppdecl( pack->table[i], 1 );
	printf( "\n" );
	
	return( NULL );
}

/* Generate function bodies for package.
 */
static void *
print_cpppackdefs( im_package *pack )
{
	int i;

	printf( "// Member function defs for package \"%s\"\n\n", 
		pack->name );
	for( i = 0; i < pack->nfuncs; i++ ) {
		im_function *fn = pack->table[i];

		if( is_cppable( fn ) ) {
			print_cppdef( fn );
			printf( "\n" );
		}
	}

	return( NULL );
}

/* Print C++ decls for function, package or all.
 */
static void
print_cppdecls( char *name )
{
	im_package *pack = im_find_package( name );
	im_function *fn = im_find_function( name );

	printf( "// this file automatically generated from\n"
		"// VIPS library %s\n", im_version_string() );

	if( strcmp( name, "all" ) == 0 ) 
		/* Do all packages.
		 */
		im_map_packages( (im_list_map_fn) print_cpppackdelcs, NULL );
	else if( pack ) 
		/* Do one package.
		 */
		print_cpppackdelcs( pack );
	else if( fn ) 
		/* Do a single function.
		 */
		print_cppdecl( fn, 1 );
	else
		error_exit( "%s: -c++: no package/function \"%s\"", 
			our_name, name );
}

/* Print C++ bindings for function, package or all.
 */
static void
print_cppdefs( char *name )
{
	im_package *pack = im_find_package( name );
	im_function *fn = im_find_function( name );

	printf( "// this file automatically generated from\n"
		"// VIPS library %s\n", im_version_string() );

	if( strcmp( name, "all" ) == 0 ) 
		/* Do all packages.
		 */
		im_map_packages( (im_list_map_fn) print_cpppackdefs, NULL );
	else if( pack ) 
		/* Do one package.
		 */
		print_cpppackdefs( pack );
	else if( fn ) 
		/* Do a single function.
		 */
		print_cppdef( fn );
	else
		error_exit( "%s: -c++: no package/function \"%s\"", 
			our_name, name );
}

/* VIPS universal main program. 
 */
int
main( int argc, char **argv )
{
	const char *vipshome = im_guess_prefix( argv[0], "VIPSHOME" );
	im_function *fn;
	int i;

	/* Want numeric conversions to be non-locale, for eg. matrix read.
	 */
	setlocale( LC_NUMERIC, "C" );

	/* Load up any standard plugins.
	 */
	if( vipshome )
		if( im_load_plugins( "%s/lib", vipshome ) )
			error_exit( "error loading plugins" );

	/* Remove any leading pathname components from our name, drop ".exe"
	 * suffix.
	 */
	if( (our_name = strrchr( argv[0], IM_DIR_SEP )) )
		our_name++;
	else
		our_name = argv[0];
	if( ispostfix( ".exe", our_name ) )
		our_name[strlen( our_name ) - 4] = '\0';

	/* Should we try to run the thing we are named as?
	 */
	if( strcasecmp( our_name, "vips" ) != 0 ) {
		/* We are not called vips - try to look up ourselves.
		 */
		char name2[ 1024 ];

		/* If our_name is unknown, try with "im_" prepended.
		 */
		if( !(fn = im_find_function( our_name )) ) {
			im_snprintf( name2, 1024, "im_%s", our_name );

			if( (fn = im_find_function( name2 )) )
				our_name = name2;
			else
				error_exit( "unknown function" );
		}

		/* Execute it!
		 */
		if( im_run_command( our_name, argc-1, argv+1 ) ) {
			usage( fn );
			error_exit( "%s: error calling function", our_name );
		}

		return( 0 );
	}

	/* We are called "vips". No more args?
	 */
	if( argc == 1 )
		error_exit( "usage: %s [flags] [command]", our_name );

	/* Look for flags.
	 */
	for( i = 1; argv[i] && argv[i][0] == '-'; i++ )
		if( strcmp( argv[i], "-help" ) == 0 ) {
			print_help( argv[i+1] );
			return( 0 );
		}
		else if( strcmp( argv[i], "-plugin" ) == 0 ) {
			if( !argv[i+1] )
				error_exit( "%s: missing argument to "
					"-plugin", argv[0] );
			if( !im_load_plugin( argv[i+1] ) )
				error_exit( "%s: unable to load plugin",
					argv[0] );
			i++;
		}
		else if( strcmp( argv[i], "-ip" ) == 0 ) {
			if( !argv[i+1] )
				error_exit( "%s: missing argument to "
					"-ip", argv[0] );
			print_defs( argv[++i] );
		}
		else if( strcmp( argv[i], "-c++.h" ) == 0 ) {
			if( !argv[i+1] )
				error_exit( "%s: missing argument to "
					"-c++.h", argv[0] );
			print_cppdecls( argv[++i] );
		}
		else if( strcmp( argv[i], "-c++.cc" ) == 0 ) {
			if( !argv[i+1] )
				error_exit( "%s: missing argument to "
					"-c++.cc", argv[0] );
			print_cppdefs( argv[++i] );
		}
		else if( strcmp( argv[i], "-c++" ) == 0 ) {
			if( !argv[i+1] )
				error_exit( "%s: missing argument to "
					"-c++", argv[0] );
			print_cppdecls( argv[i+1] );
			print_cppdefs( argv[i+1] );
			i++;
		}
		else if( strcmp( argv[i], "-links" ) == 0 ) {
			im_map_packages( (im_list_map_fn) print_links, NULL );
		}
		else 
			error_exit( "%s: unknown flag \"%s\" - try \"-help\"", 
				argv[0], argv[i] );

	/* All flags processed. Any args left?
	 */
	if( i < argc )
		if( im_run_command( argv[i], argc-i-1, argv+i+1 ) ) {
			if( !(fn = im_find_function( argv[i] )) )
				error_exit( "unknown function" );
			usage( fn );
			error_exit( "%s: error calling function", argv[0] );
		}

	im_close_plugins();
	return( 0 );
}
