/*
 * This file is part of Soprano Project
 *
 * Copyright (C) 2009-2010 Sebastian Trueg <trueg@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "virtuosocontroller.h"
#include "sopranodirs.h"

#include <QtCore/QTemporaryFile>
#include <QtCore/QProcess>
#include <QtCore/QSettings>
#include <QtCore/QFileInfo>
#include <QtCore/QFile>
#include <QtCore/QEventLoop>
#include <QtCore/QDir>
#include <QtCore/QDebug>
#include <QtCore/QMutex>
#include <QtCore/QMutexLocker>

Q_DECLARE_METATYPE( QProcess::ExitStatus )

namespace {

#ifdef Q_OS_WIN
    QMutex portNumberMutex;
#endif

    quint16 getFreePortNumber() {
//         QTcpServer server;
//         if ( server.listen() ) {
//             return server.serverPort();
//         }
//         else {
//             qDebug() << "Failed to determine free port. Falling back to default 1111.";
//             return 1111;
//         }
#ifdef Q_OS_WIN
        static quint16 p = 1111;
        QMutexLocker l(&portNumberMutex);
        return p++;
#else
        int p = 1111;
        while ( QFile::exists( QString( "/tmp/virt_%1" ).arg( p ) ) ) {
            ++p;
        }
        return p;
#endif
    }
}

Soprano::VirtuosoController::VirtuosoController()
    : QObject( 0 ),
      m_port( 0 ),
      m_status( NotRunning ),
      m_lastExitStatus( NormalExit ),
      m_initializationLoop( 0 )
{
    connect( &m_virtuosoProcess, SIGNAL(finished(int,QProcess::ExitStatus)),
             this, SLOT(slotProcessFinished(int,QProcess::ExitStatus)) );
    connect( &m_virtuosoProcess, SIGNAL(readyReadStandardError()),
             this, SLOT(slotProcessReadyRead()) );

    // necessary in case we are started from a thread != the main thread
    qRegisterMetaType<QProcess::ExitStatus>();
}


Soprano::VirtuosoController::~VirtuosoController()
{
    if ( isRunning() )
        shutdown();
}


void Soprano::VirtuosoController::setDirsAllowed( const QStringList& dirs )
{
    m_allowedDirs = dirs;
}


bool Soprano::VirtuosoController::start( const QString& virtuosoExe, const QString& storagePath, RunFlags flags )
{
    if ( !isRunning() ) {
        QTemporaryFile tmpFile( QDir::tempPath() + "/virtuoso_XXXXXX.ini" );
        tmpFile.setAutoRemove( false );
        tmpFile.open();
        m_configFilePath = tmpFile.fileName();
        tmpFile.close();
        writeConfigFile( m_configFilePath, storagePath );
        m_runFlags = flags;

        m_status = StartingUp;

        if ( virtuosoExe.isEmpty() ) {
            qDebug() << "Unable to find the Virtuoso binary.";
            return false;
        }

        // remove old lock files to be sure
        QString lockFilePath = storagePath + QLatin1String( "/soprano-virtuoso.lck" );
        if ( QFile::exists( lockFilePath ) )
            QFile::remove( lockFilePath );

        QStringList args;
#ifdef Q_OS_WIN
        args << "+foreground"
             << "+configfile" << QDir::toNativeSeparators(m_configFilePath);
#else
        args << "+foreground"
             << "+config" << m_configFilePath
             << "+wait";
#endif
        qDebug() << "Starting Virtuoso server:" << virtuosoExe << args;

        m_virtuosoProcess.start( virtuosoExe, args, QIODevice::ReadOnly );
        m_virtuosoProcess.setReadChannel( QProcess::StandardError );
        m_virtuosoProcess.closeReadChannel( QProcess::StandardOutput );
        if ( waitForVirtuosoToInitialize() ) {
            m_status = Running;
            qDebug() << "Virtuoso started:" << m_virtuosoProcess.pid();
            return true;
        }
        else {
            qDebug() << "Failed to start Virtuoso";
            return false;
        }
    }
    else {
        qDebug() << "Virtuoso is already running.";
        return false;
    }
}


bool Soprano::VirtuosoController::waitForVirtuosoToInitialize()
{
    // FIXME: timeout
    if ( m_virtuosoProcess.waitForStarted() ) {
        QEventLoop loop;
        m_initializationLoop = &loop;
        loop.exec();
        m_initializationLoop = 0;
        return( m_status == Running );
    }
    else {
        return false;
    }
}


void Soprano::VirtuosoController::slotProcessReadyRead()
{
    // we only wait for the server to tell us that it is ready
    while ( m_virtuosoProcess.canReadLine() ) {
        QString line = QString::fromLatin1( m_virtuosoProcess.readLine() );
        qDebug() << line;
        if ( line.contains( "Server online at" ) ) {
            m_virtuosoProcess.closeReadChannel( QProcess::StandardError );
            m_status = Running;
            m_initializationLoop->exit();
        }
    }
}


int Soprano::VirtuosoController::usedPort() const
{
    return m_port;
}


bool Soprano::VirtuosoController::shutdown()
{
    if ( m_virtuosoProcess.state() == QProcess::Running ) {
        qDebug() << "Shutting down virtuoso instance" << m_virtuosoProcess.pid();
#ifndef Q_OS_WIN
        m_status = ShuttingDown;
        m_virtuosoProcess.terminate();
        if ( !m_virtuosoProcess.waitForFinished( 30*1000 ) ) {
            qDebug() << "Killing virtuoso instance" << m_virtuosoProcess.pid();
            m_status = Killing;
            m_virtuosoProcess.kill();
            m_virtuosoProcess.waitForFinished();
            return false;
        }
        else {
            return true;
        }
#else
        m_status = Killing;
        m_virtuosoProcess.kill();
        m_virtuosoProcess.waitForFinished();
        clearError();
        return true;
#endif
    }
    else {
        qDebug() << "Virtuoso not running. Cannot shutdown.";
        return false;
    }
}


bool Soprano::VirtuosoController::isRunning() const
{
    return m_status == Running;
}


void Soprano::VirtuosoController::slotProcessFinished( int, QProcess::ExitStatus exitStatus )
{
    // clean up config
    if ( !( m_runFlags & DebugMode ) &&
         QFile::exists( m_configFilePath ) ) {
        QFile::remove( m_configFilePath );
    }

    m_lastExitStatus = NormalExit;
    if ( exitStatus == QProcess::CrashExit )
        m_lastExitStatus = CrashExit;
    else if ( m_status == Killing )
        m_lastExitStatus = ForcedExit;
    else if ( m_status != ShuttingDown )
        m_lastExitStatus = ThirdPartyExit;

    m_status = NotRunning;

    qDebug() << "Virtuoso server stopped:" << m_lastExitStatus;

    emit stopped( m_lastExitStatus );

    if ( m_initializationLoop )
        m_initializationLoop->exit();
}


// TODO: optimize the settings
void Soprano::VirtuosoController::writeConfigFile( const QString& path, const QString& storageDir )
{
    qDebug() << Q_FUNC_INFO << path;

    // backwards compatibility
    int numberOfBuffers = 5000;
    int numberOfThreads = 100;

    int checkpointInterval = -1;
    int minAutoCheckpointSize = -1;

    // although we do not actually use a port Virtuoso uses the port number to create
    // the unix socket name.
    m_port = getFreePortNumber();

    QString dir( storageDir );
    if ( !dir.endsWith( '/' ) )
        dir += '/';
    dir = QDir::toNativeSeparators( dir );

    QSettings cfs( path, QSettings::IniFormat );

    cfs.beginGroup( "Database" );
    cfs.setValue( "DatabaseFile", dir + "soprano-virtuoso.db" );
    cfs.setValue( "ErrorLogFile", dir + "soprano-virtuoso.log" );
    cfs.setValue( "TransactionFile", dir + "soprano-virtuoso.trx" );
    cfs.setValue( "xa_persistent_file", dir + "soprano-virtuoso.pxa" );
    cfs.endGroup();

    cfs.beginGroup( "TempDatabase" );
    cfs.setValue( "DatabaseFile", dir + "soprano-virtuoso-temp.db" );
    cfs.setValue( "TransactionFile", dir + "soprano-virtuoso-temp.trx" );
    cfs.setValue( "MaxCheckpointRemap", "1000");
    cfs.endGroup();

    cfs.beginGroup( "Parameters" );
    cfs.setValue( "LiteMode", "1" );
    cfs.setValue( "ServerPort", QString::number( m_port ) );
#ifdef Q_OS_WIN
    cfs.setValue( "DisableUnixSocket", "1" );
    cfs.setValue( "DisableTcpSocket", "0" );
#else
    cfs.setValue( "DisableTcpSocket", "1" );
#endif

    // FIXME: what is this?
    //    cfs.setValue( "DirsAllowed", "." );

    // FIXME: what is this?
    cfs.setValue( "PrefixResultNames", "0" );

    // Number of thread used in the server (default: 10)
    // FIXME: we have a problem here: soprano server is now multithreaded. Thus, we run out of threads very quickly.
    cfs.setValue( "ServerThreads", numberOfThreads );

    // Memory used by Virtuoso, 8k buffers
    cfs.setValue( "NumberOfBuffers", numberOfBuffers );

    // down from 1200
    cfs.setValue( "MaxDirtyBuffers", "50" );

    // down from 10
    cfs.setValue( "SchedulerInterval", "5" );

    // down from 10000000
    cfs.setValue( "FreeTextBatchSize", "1000" );

    // checkpoint interval in minutes (Virtuoso default: 60)
    if ( checkpointInterval >= 0 )
        cfs.setValue( "CheckpointInterval", checkpointInterval );
    if ( minAutoCheckpointSize >= 0 )
        cfs.setValue( "MinAutoCheckpointSize", minAutoCheckpointSize );

    cfs.setValue( "DirsAllowed", m_allowedDirs.join( "," ) );

    cfs.endGroup();
}


// static
QString Soprano::VirtuosoController::locateVirtuosoBinary()
{
    QStringList dirs = Soprano::exeDirs();
#ifdef Q_OS_WIN
    const QString virtuosoHome = QDir::fromNativeSeparators( qgetenv("VIRTUOSO_HOME") );
    if ( !virtuosoHome.isEmpty() )
        dirs << virtuosoHome + QLatin1String("/bin");
#endif

    foreach( const QString& dir, dirs ) {
#ifdef Q_OS_WIN
        QFileInfo info( dir + QLatin1String("/virtuoso-t.exe") );
#else
        QFileInfo info( dir + QLatin1String("/virtuoso-t") );
#endif
        if ( info.isExecutable() && !info.isSymLink() ) {
            return info.absoluteFilePath();
        }
    }
    return QString();
}

#include "virtuosocontroller.moc"
